/*
 * Copyright 2011 Nate Koenig
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
*/
/* Desc: Camera for viewing the world
 * Author: Nate Koenig
 * Date: 19 Jun 2008
 */

#ifndef _USERCAMERA_HH_
#define _USERCAMERA_HH_

#include <string>
#include <vector>

#include "rendering/Camera.hh"
#include "rendering/RenderTypes.hh"
#include "common/CommonTypes.hh"

namespace gazebo
{
  namespace rendering
  {
    class OrbitViewController;
    class FPSViewController;
    class Visual;
    class GUIOverlay;
    class SelectionBuffer;

    /// \addtogroup gazebo_rendering
    /// \{

    /// \class UserCamera UserCamera.hh rendering/rendering.hh
    /// \brief A camera used for user visualization of a scene
    class UserCamera : public Camera
    {
      /// \brief Constructor
      /// \param[in] _name Name of the camera.
      /// \param[in] _scene Scene to put the camera in.
      public: UserCamera(const std::string &_name, Scene *_scene);

      /// \brief Destructor
      public: virtual ~UserCamera();

      /// \brief Load the user camera.
      /// \param[in] _sdf Parameters for the camera.
      public: void Load(sdf::ElementPtr _sdf);

      /// \brief Generic load function
      public: void Load();

      /// \brief Initialize
      public: void Init();

      /// \brief Render the camera
      public: virtual void Update();

      /// \brief Post render
      public: virtual void PostRender();

      /// \brief Finialize
      public: void Fini();

      /// \brief Set the pose in the world coordinate frame.
      /// \param[in] _pose New pose of the camera.
      public: virtual void SetWorldPose(const math::Pose &_pose);

      /// \brief Handle a mouse event.
      /// \param[in] _evt The mouse event.
      public: void HandleMouseEvent(const common::MouseEvent &_evt);

      /// \brief Handle a key press.
      /// \param[in] _key The key pressed.
      public: void HandleKeyPressEvent(const std::string &_key);

      /// \brief Handle a key release.
      /// \param[in] _key The key released.
      public: void HandleKeyReleaseEvent(const std::string &_key);

      /// \brief Set view controller.
      /// \param[in] _type The type of view controller: "orbit", "fps"
      public: void SetViewController(const std::string &_type);

      /// \brief Set view controller
      /// \param[in] _type The type of view controller: "orbit", "fps"
      /// \param[in] _pos The initial pose of the camera.
      public: void SetViewController(const std::string &_type,
                                     const math::Vector3 &_pos);

      /// \brief Resize the camera.
      /// \param[in] _w Width of the camera image.
      /// \param[in] _h Height of the camera image.
      public: void Resize(unsigned int _w, unsigned int _h);

      /// \brief Set the dimensions of the viewport.
      /// \param[in] _x X position of the viewport.
      /// \param[in] _y Y position of the viewport.
      /// \param[in] _w Width of the viewport.
      /// \param[in] _h Height of the viewport.
      public: void SetViewportDimensions(float _x, float _y,
                                         float _w, float _h);

      /// \brief Get the average frames per second
      /// \return The average rendering frames per second
      public: float GetAvgFPS() const;

      /// \brief Get the triangle count.
      /// \return The number of triangles currently being rendered.
      public: float GetTriangleCount() const;

      /// \brief Move the camera to focus on a visual.
      /// \param[in] _visual Visual to move the camera to.
      public: void MoveToVisual(VisualPtr _visual);

      // Doxygen automatically pulls in the correct documentation.
      public: virtual bool MoveToPosition(const math::Pose &_pose,
                                          double _time);

      /// \brief Move the camera to focus on a visual.
      /// \param[in] _visualName Name of the visual to move the camera to.
      public: void MoveToVisual(const std::string &_visualName);

      /// \brief Set to true to enable rendering
      ///
      /// Use this only if you really know what you're doing.
      /// \param[in] _target The new rendering target.
      public: virtual void SetRenderTarget(Ogre::RenderTarget *_target);

      /// \brief Get the GUI overlay
      ///
      /// An overlay allows you to draw 2D elements on the viewport.
      /// \return Pointer to the GUIOverlay.
      public: GUIOverlay *GetGUIOverlay();

      /// \brief Set whether the view controller is enabled.
      ///
      /// The view controller is used to handle user camera movements.
      /// \param[in] _value True to enable viewcontroller, False to
      /// disable.
      public: void EnableViewController(bool _value) const;

      /// \brief Get an entity at a pixel location using a camera. Used for
      /// mouse picking.
      /// \param[in] _mousePos The position of the mouse in screen coordinates
      /// \param[out] _mod Used for object manipulation
      /// \return The selected entity, or NULL
      public: VisualPtr GetVisual(const math::Vector2i &mousePos,
                                  std::string &mod);

      /// \brief Get a visual at a mouse position
      /// \param[in] _mousePos 2D position of the mouse in pixels.
      public: VisualPtr GetVisual(const math::Vector2i &_mousePos) const;

      /// \brief Set the point the camera should orbit around.
      /// \param[in] _pt The focal point
      public: void SetFocalPoint(const math::Vector3 &_pt);

      /// \brief Set the camera to be attached to a visual.
      ///
      /// This causes the camera to move in relation to the specified visual.
      /// \param[in] _visual The visual to attach to.
      /// \param[in] _inheritOrientation True if the camera should also
      /// rotate when the visual rotates.
      /// \param[in] _minDist Minimum distance the camera can get to the
      /// visual.
      /// \param[in] _maxDist Maximum distance the camera can get from the
      /// visual.
      /// \return True if successfully attach to the visual.
      protected: virtual bool AttachToVisualImpl(VisualPtr _visual,
                     bool _inheritOrientation, double _minDist = 0,
                     double _maxDist = 0);

      /// \brief Set the camera to track a scene node.
      ///
      /// Tracking just causes the camera to rotate to follow the visual.
      /// \param[in] _visual Visual to track.
      /// \return True if the camera is now tracking the visual.
      protected: virtual bool TrackVisualImpl(VisualPtr _visual);


      /// \brief Toggle whether to show the visual.
      private: void ToggleShowVisual();

      /// \brief Set whether to show the visual.
      /// \param[in] _show True to show the visual representation for this
      /// camera. Currently disabled.
      private: void ShowVisual(bool _show);

      /// \brief Callback used when the camera has finished moving to
      /// a visual.
      private: void OnMoveToVisualComplete();

      /// \brief The visual used to render the camera's appearance.
      private: Visual *visual;

      /// \brief The currently active view controller.
      private: ViewController *viewController;

      /// \brief An orbit view controller.
      private: OrbitViewController *orbitViewController;

      /// \brief A FPS view controller.
      private: FPSViewController *fpsViewController;

      /// \brief The GUI overlay.
      private: GUIOverlay *gui;

      /// \brief Draws a 3D axis in the viewport.
      // private: Ogre::SceneNode *axisNode;

      /// \brief Used to select objects from mouse clicks.
      private: SelectionBuffer *selectionBuffer;
    };
    /// \}
  }
}
#endif
