/*
 *                         OpenSplice DDS
 *
 *   This software and documentation are Copyright 2006 to TO_YEAR PrismTech
 *   Limited, its affiliated companies and licensors. All rights reserved.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
#ifndef U_DATAREADER_H
#define U_DATAREADER_H

#include "u_types.h"
#include "u_reader.h"

#if defined (__cplusplus)
extern "C" {
#endif

#ifdef OSPL_BUILD_CORE
#define OS_API OS_API_EXPORT
#else
#define OS_API OS_API_IMPORT
#endif
/* !!!!!!!!NOTE From here no more includes are allowed!!!!!!! */

#define u_dataReader(o) \
        ((u_dataReader)u_objectCheckType(u_object(o), U_READER))

typedef v_copyin_result (*u_copyIn)(c_type type, const void *data, void *to);
typedef void (*u_copyOut)(const void *from, void *to);

/* The DataReader copy function signature:
 *
 * All Read and Take operations on the DataReader that need to copy out data
 * will accept a copy function implementing this signature.
 * The copy function will be invoked on all available samples with respect to any
 * selection criteria.
 * The processing of samples will continue until all samples are read or this
 * function returns an action result telling to stop processing samples.
 * The first parameter 'o' is a reference to the internal representation of the
 * sample, so the copy function needs to understand the internal representation.
 * The copy routines will therefore be generated by the preprocessor to assure
 * that the internal representation is always understood.
 * The second parameter is void to the read and take operations, it will only be
 * passed to the copy function and the copy function can use it as a in/out parameter.
 * Typically this parameter will hold a list in which the copy function copies the
 * data.
 * Altough not required to know about for understanding it might be useful to
 * understand that the action result value consists of two bit:
 * V_PROCEED : continue with the next sample, so if not set the read will stop.
 * V_SKIP    : do not modify the state of this sample, i.e. set sample state to READ.
 * In addition the action function will always be called once after reading the samples
 * without providing a sampe (i.e. 'o' == NULL) to indicate that the read has finished.
 */
typedef u_actionResult (*u_dataReaderAction)(c_object o, void *arg);

OS_API u_dataReader
u_dataReaderNew(
    const u_subscriber _scope,
    const os_char *name,
    const os_char *expr,
    const c_value params[],
    const u_readerQos qos,
    u_bool enable);

OS_API u_result
u_dataReaderGetQos (
    const u_dataReader _this,
    u_readerQos *qos);

OS_API u_result
u_dataReaderSetQos (
    const u_dataReader _this,
    const u_readerQos qos);

OS_API u_result
u_dataReaderRead(
    const u_dataReader _this,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderTake(
    const u_dataReader _this,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderReadInstance(
    const u_dataReader _this,
    u_instanceHandle handle,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderTakeInstance(
    const u_dataReader _this,
    u_instanceHandle handle,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderReadNextInstance(
    const u_dataReader _this,
    u_instanceHandle handle,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderTakeNextInstance(
    const u_dataReader _this,
    u_instanceHandle handle,
    u_sampleMask mask,
    u_dataReaderAction action,
    void *actionArg,
    const os_duration timeout);

OS_API u_result
u_dataReaderGetInstanceHandles (
    const u_dataReader _this,
    u_result (*action)(u_instanceHandle *buf, os_uint32 length, void *arg),
    void *arg);

OS_API u_result
u_dataReaderWaitForHistoricalData(
    const u_dataReader _this,
    os_duration timeout);

OS_API u_result
u_dataReaderWaitForHistoricalDataWithCondition(
    const u_dataReader _this,
    const os_char* filter,
    const os_char* params[],
    os_uint32 paramsLength,
    os_timeW min_source_time,
    os_timeW max_source_time,
    os_int32 max_samples,
    os_int32 max_instances,
    os_int32 max_samples_per_instance,
    os_duration timeout);

OS_API u_result
u_dataReaderLookupInstance(
    const u_dataReader _this,
    void *keyTemplate,
    u_copyIn copyIn,
    u_instanceHandle *handle);

OS_API u_result
u_dataReaderGetInstanceUserData (
    const u_dataReader _this,
    u_instanceHandle handle,
    c_voidp* userData_out);

OS_API u_result
u_dataReaderSetInstanceUserData (
    const u_dataReader _this,
    u_instanceHandle handle,
    c_voidp userData);

OS_API u_result
u_dataReaderTopic(
    const u_dataReader _this,
    u_topic *topic);

OS_API u_result
u_dataReaderCopyKeysFromInstanceHandle (
    const u_dataReader _this,
    u_instanceHandle handle,
    u_copyOut action,
    void *copyArg);

OS_API c_ulong
u_dataReaderGetNumberOpenTransactions(
    u_dataReader _this);

#undef OS_API

#if defined (__cplusplus)
}
#endif

#endif
