/*
 * Copyright 2017 Fraunhofer Institute for Manufacturing Engineering and Automation (IPA)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0

 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef CANDRIVEITF_INCLUDEDEF_H
#define CANDRIVEITF_INCLUDEDEF_H

//-----------------------------------------------
#include <cob_generic_can/CanItf.h>
#include <cob_canopen_motor/DriveParam.h>
#include <cob_canopen_motor/SDOSegmented.h>
//-----------------------------------------------

/**
 * Interface for a drive.
 *  \ingroup DriversCanModul
 */
class CanDriveItf
{
public:
	/**
	 * Motion type of the controller.
	 */
	enum MotionType
	{
		MOTIONTYPE_VELCTRL,
        MOTIONTYPE_TORQUECTRL,
		MOTIONTYPE_POSCTRL
	};

	/**
	 * Sets the CAN interface.
	 */
	virtual void setCanItf(CanItf* pCanItf) = 0;

	/**
	 * Initializes the driver.
	 * Call this function once after construction.
	 */
	virtual bool init() = 0;

	/**
	 * Check if the driver is already initialized.
	 * This is necessary if a drive gets switched off during runtime.
	 * @return true if initialization occured already, false if not.
	 */
	virtual bool isInitialized() = 0;

	/**
	 * Enables the motor.
	 * After calling the drive accepts velocity and position commands.
	 */
	virtual bool start() = 0;

	/**
	 * Disables the motor.
	 * After calling the drive won't accepts velocity and position commands.
	 */
	virtual bool stop() = 0;

	/**
	 * Resets the drive.
	 * The drive changes into the state after initializaton.
	 */
	virtual bool reset() = 0;

	/**
	 * Shutdowns the motor.
	 */
	virtual bool shutdown() = 0;

	/**
	 * Disables the brake.
	 * This function is not implemented for Harmonica,
	 * because brakes are released upon power on and
	 * shut only at emergency stop.
	 */
	virtual bool disableBrake(bool bDisabled) = 0;

	/**
	 * Inits homing procedure.
	 * Used only in position mode.
	 */
	virtual bool initHoming() = 0;

	/**
	 * Performs homing procedure.
	 * Used only in position mode.
	 */
	virtual bool execHoming() = 0;

	/**
	 * Returns the elapsed time since the last received message.
	 */
	virtual double getTimeToLastMsg() = 0;

	/**
	 * Returns the status of the limit switch needed for homing.
	 * true = limit switch is reached; false = not reached
	 */
	virtual bool getStatusLimitSwitch() = 0;

	/**
	 * Starts the watchdog.
	 * The Harmonica provides watchdog functionality which means the drive stops if the watchdog
	 * becomes active. To keep the watchdog inactive a heartbeat message has to be sent
	 * periodically. The update rate is set to 1s.
	 * The update is is done in function setGearVelRadS().
	 */
	virtual bool startWatchdog(bool bStarted) = 0;

	/**
	 * Evals a received message.
	 * Only messages with fitting identifiers are evaluated.
	 * @param msg message to be evaluated.
	 */
	virtual bool evalReceivedMsg(CanMsg& msg) = 0;

	/**
	 * Evals received messages in OBJECT mode.
	 * The CAN drives have to implement which identifiers they are interested in.
	 * @return true on success, false on failure.
	 */
	virtual bool evalReceivedMsg() = 0;

	/**
	 * Sets required position and veolocity.
	 * Use this function only in position mode.
	 * By calling the function the status is requested, too.
	 */
	virtual void setGearPosVelRadS(double dPosRad, double dVelRadS) = 0;

	/**
	 * Sets the velocity.
	 * By calling the function the status is requested, too.
	 */
	virtual void setGearVelRadS(double dVelRadS) = 0;

	/**
	 * Sets the motion type drive.
	 * The function is not implemented for Harmonica.
	 * The harmonica drive is configured just for velocity mode.
	 */
	virtual bool setTypeMotion(int iType) = 0;

	/**
	 * Returns the position and the velocity of the drive.
	 */
	virtual void getGearPosVelRadS(double* pdAngleGearRad, double* pdVelGearRadS) = 0;

	/**
	 * Returns the change of the position and the velocity.
	 * The given delta position is given since the last call of this function.
	 */
	virtual void getGearDeltaPosVelRadS(double* pdDeltaAngleGearRad, double* pdVelGearRadS) = 0;

	/**
	 * Returns the current position.
	 */
	virtual void getGearPosRad(double* pdPosGearRad) = 0;

	/**
	 * Sets the drive parameter.
	 */
	virtual void setDriveParam(DriveParam driveParam) = 0;

	/**
	 * Returns true if an error has been detected.
	 */
	virtual bool isError() = 0;

	/**
	 * Return a bitfield containing information about the pending errors.
	 */
	virtual unsigned int getError() = 0;

	/**
	 * Requests position and velocity.
	 */
	virtual void requestPosVel() = 0;

	/**
	 * Requests status.
	 */
	virtual void requestStatus() = 0;

	/**
	 * Returns the measured temperature.
	 */
	virtual void getStatus(int* piStatus, int* piTempCel) = 0;

	/**
	 * Enable the emergency stop.
	 */
	virtual bool setEMStop() = 0;

	/**
	 * Disable the emergency stop.
	 */
	virtual bool resetEMStop() = 0;

	/**
	 * Sends an integer value to the Harmonica using the built in interpreter. cpc-kischa
	 */
	virtual void IntprtSetInt(int iDataLen, char cCmdChar1, char cCmdChar2, int iIndex, int iData) = 0;

    /**
     *Provides several functions for drive information recording purposes. By now, only implemented for the Elmo-recorder. cpc-pk
     */
    virtual	int setRecorder(int iFlag, int iParam = 0, std::string sParam = "/home/MyLog") = 0;

	/**
	 * Sends Requests for "active current" to motor via CAN
	 */
	virtual void requestMotorTorque() = 0;

	/**
	 * Returns member variable m_MotorCurrent
	 * To update this value call requestMotorCurrent before
	 * and evaluate CAN buffer, or wait one cycle
	 */
	virtual void getMotorTorque(double* dTorqueNm) = 0;

    /**
     * Sends command for motor Torque (in Nm)
     */
    virtual void setMotorTorque(double dTorqueNm) = 0;
};


//-----------------------------------------------
#endif

