#ifndef MAPTAB_H
#define MAPTAB_H

// QT
#include <QWidget>
#include <QGridLayout>
#include <QGroupBox>
#include <QPushButton>
#include <QLabel>
#include <QLineEdit>
#include <QScrollArea>
#include <QScrollBar>
#include <QComboBox>
#include <QTreeWidget>
#include <QFileDialog>
#include <QCheckBox>
#include <QAccessible>
#include <QStyle>
#include <QTreeWidgetItem>
#include <QRadioButton>
#include <QButtonGroup>
#include <QPrinter>
#include <QPrintDialog>
#include <QGraphicsSimpleTextItem>
#include <QListWidgetItem>

// STL
#include <list>
#include <sstream>


// Project
#include "MapScrollArea.h"

#include "../../Widgets/MapDisplay/MapDisplay.h"
#include "../../Widgets/MapDisplay/PoiIcon.h"
//#include "GUI/Containers/SemanticMapTab/SemanticMapTab.h" // TODO add later

#include <ros/ros.h>
#include <homer_mapnav_msgs/PointOfInterest.h>
#include <nav_msgs/OccupancyGrid.h>
#include <std_msgs/Bool.h>

#include <iostream>

/**
 * @class MapTab
 *
 * @author Malte Knauf (2014), Christian Feinen (RX), David Gossow (RX), Nicolai Wojke (R14)
 *
 * @brief This class implements a QT widget that contains the top level map and localization functionality.
 *        It also contains a further QT widget that shows the map and implements the second part of the
 *        functionality as MousePressEvents. In addition the user is able to save, load and modify
 *        created maps.
 *
 * This class is a MainWindow's QT frame that provides the needed map functionality as adding, editing,
 * and erasing of points of interest (poi). It shows all added poi in a QTreeWidget and the user can
 * select always one of them. The robot may be send to a selected poi. Further, a poi may be used to
 * initialize the SlamModule.
 */
class MapTab : public QWidget
{
    Q_OBJECT
public:

  /**
  * The constructor passes the given parent to the constructor of the base class.
  * It creates the used widgets (scrollArea, checkboxes and buttons) and connects the signals to
  * the slots. Furthermore it creates the poi icons, fetches all config entries and initialize the map
  * widget (links and views).
  * @param parent Parent widget
  * @see getConfigEntries(), createWidgetsAndSetupLayout(), installSignalSlotConnections(), MapWidget
  */
  MapTab (  QWidget *parent , ros::NodeHandle *nodeHandle ) ;


  MapDisplay* getMapWidget() { return m_MapWidget; }

  /** @brief the destructor */
  ~MapTab();

private:

  /**
  * It creates all poi icons defined by the user
  */
  void createUserDefinedPoiIcons();

  /**
  * It fetches all config entries as iSize and iCellSize
  */
  void getConfigEntries();

  /**
  * It creates the used QT widgets.
  */
  void createWidgetsAndSetupLayout();

  /**
  * It installs all signal slot connections between the widgets
  */
  void installSignalSlotConnections();

  /**
  * The function is called when a new poi list arrived and updates the QTreeWidget
  */
  void fillPoiTreeWidget();

  /**
  * It is called to clear and disable the line edit fields. Also it clears a added poi
  * in the MapWidget and disables the apply Button
  */
  void deactivatePoiApply();

  /**
  * It is called to clear and disable the line edit fields. But also to deactivate the
  * delete Button
  */
  void deactivatePoiDelete();

  /**
  * It disables the drive to position button
  */
  void deactivateDriveToPosition();

  /**
  * It sends a message with the id of the point that shall be deleted. Also it calls
  * deactivatePoiDelete and deactivateDriveToPosition
  */
  void deletePoi();

  /**
  * reimplantation of QT's keyPressEvent
  *
  * @param event type: QKeyEvent; it contains the pressed keys
  */
  virtual void keyPressEvent ( QKeyEvent * event );

  /**  Used to zoom into the map. */
  void wheelEvent( QWheelEvent* event );

  /**
  * reimplantation of QT's sizeHint
  *
  * @return optimal widget size
  */
  virtual QSize sizeHint() const;

  /**
  * reimplantation of QT's minimumSizeHint
  *
  * @return minimum widget size
  */
  virtual QSize minimumSizeHint() const;

  ros::NodeHandle *m_NodeHandle;

  /**
   * @brief Publishers of MapTab
   */
  ros::Publisher* m_UserDefPosePublisher;
  ros::Publisher* m_AddPOIPublisher;
  ros::Publisher* m_ModifyPOIPublisher;
  ros::Publisher* m_DeletePOIPublisher;
  ros::Publisher* m_SaveMapPublisher;
  ros::Publisher* m_LoadMapPublisher;
  ros::Publisher* m_DoMappingPublisher;
  ros::Publisher* m_ResetMapsPublisher;
  ros::Publisher* m_MapVisibilityPublisher;
  ros::Publisher* m_StopNavigationPublisher;
  ros::Publisher* m_StartNavigationPublisher;

  /**
  * instance of the MapDisplay to display the map, poi and the robot position
  */
  MapDisplay* m_MapWidget;

  /**
  * instance of a QT scroll area to center and to show the whole map
  */
  MapScrollArea* m_ScrollArea;

  /**
  * a QT vector filled with the user-defined poi icons
  */
  QVector< PoiIcon > m_PoiIcons;

  /**
  * a STL vector filled with all current poi
  */
  std::vector < homer_mapnav_msgs::PointOfInterest > m_PoiList;

  /**
  * the user-defined orientation angle; value: radian {-PI:PI}
  */
  double m_UserDefinedRobotOrientation;

  /**
  * the user-defined robot position in map coordinates as QT point
  */
  QPoint m_UserDefinedRobotPos;

  /**
  * the unique name of the current selected poi
  */
  std::string m_CurrentPoiSelectionName;

  /**
  * the id of the currently selected map layer
  * no selection = -1
  */
  int m_CurrentlySelectedMapID;

  /**
  * the config entry of the map size value of one side in m
  */
  int m_MapSize;

  /**
   * the size of the map in cell count
   */
  int m_PixelSize;

  /**
  * the config entry cell size value (resolution)
  */
  float m_CellSize;

  /**
   * resolution of the ma (same as m_CellSize)
   */
  double m_Resolution;

  /**
   * real-world origin of the point (0,0) of the map
   */
  geometry_msgs::Pose m_Origin;

  /**
  * last seen pose of the robot
  */
  geometry_msgs::Pose m_LastPose;

  /**
  * position of the user-defined poi that isn't saved yet
  */
  QPoint m_TmpPoiPos;

  /**
  * orientation of the user-defined poi that isn't saved yet
  */
  double m_TmpPoiOrientation;

  /**
  * instance of a QT line edit field for entering/editing the poi name
  */
  QLineEdit* m_PoiNameLineEdit;

  /**
  * instance of a QT line edit field for entering/editing poi remarks
  */
  QLineEdit* m_PoiMiscLineEdit;

  /**
  * instance of a QT text label to indicate the current mouse position in the MapWidget
  */
  QLabel* m_ShowMousePosition;

  /**
  * instance of a QT text label to indicate the current zoom factor of the MapWidget
  */
  QLabel* m_ShowZoomFactor;

  /**
  * instance of a QT push button to save a user-defined poi
  */
  QPushButton* m_PoiApplyButton;

  /**
  * instance of a QT push button to delete a selected poi
  */
  QPushButton* m_PoiDeleteButton;

  /**
  * instance of a QT push button to send the robot to a selected poi
  */
  QPushButton* m_DriveToPoiButton;
  QPushButton* m_StopNavigationButton;
  QPushButton* m_GrabPoiButton;
  QPushButton* m_DeliverToPoiButton;
  QPushButton* m_CenterPTUButton;

  /**
   * instance of a QT check box to switch mapping state
   */
  QCheckBox* m_DoMappingCheckBox;

  /**
  * instance of a QT tree widget to show all saved poi
  */
  QTreeWidget* m_PoiTreeWidget;

  /**
  * instance of a QT combobox to show all user-defined poi icons
  */
  QComboBox* m_PoiSymbolsComboBox;

  /**
  * instance of a QT button group to check which mask mode is selected
  */
  QButtonGroup* m_MaskModeButtonGroup;

  QComboBox* m_MapSelector;

  /**
   * instance of a QT list widget to list each available map layer (which is defined in MapDisplay::MapId)
   */
  QListWidget* m_MapLayersListWidget;

  /**
   * QWidget that contains the semantic map tab.
   */
  // SemanticMapTab* m_SemanticsTab; // TODO include later

public slots:


  /**
  * It is called by the PoseCallback function in MainWindow to update the pose (x,y,theta)
  * @param x in m
  * @param y in m
  * @param theta in radiants
  */
  void updatePose(double x, double y, double theta);

  /**
   * @brief The method is called by the callbackOccupancyGrid function in QtRosNode to update the map
   * @param map Map data
   * @param size Size of one side in cells
   * @param resolution Size of one cell
   * @param origin Origin of the map
   */
  void updateMap(unsigned char *map, uint size, double resolution, geometry_msgs::Pose origin);
  
   /**
   * @brief The method is called by the callbackMaskSlam function in QtRosNode to update the mask
   */
  void updateMask(const nav_msgs::OccupancyGrid::ConstPtr &msg);

  /**
   * @brief This method is called by the callbackPOIList function to update the poi list
   * @param poiList
   */
  void updatePOIList(std::vector<homer_mapnav_msgs::PointOfInterest> poiList);

  /**
   * @brief This method is called by the callbackPath function to show the current path of the navigation node
   * @param path
   */
  void updatePath(std::vector<geometry_msgs::PoseStamped> path);

  /**
   * @brief This method is called by the callbackFollowingPath function to show the current path of the following node
   * @param path
   */
  void updateFollowingPath(std::vector<geometry_msgs::PoseStamped> path);

  /**
  * @brief This slot is used to set the current user-defined robot position and orientation
  * These information will be send by clicking on the self-loc button
  * @param [in] robotPos A QPoint that contains the robot position
  * @param [in] robotOrientation Value that contains the robot orientation as radian measure between -PI and PI
  */
  void userdefinedRobotorPos ( QPoint robotPos, double robotOrientation ) ;

  /**
  * @brief This slot is used to set the current user-defined poi
  * If he decides to save this poi these information will be needed to save the poi
  * @param [in]
  */
  void addedNewPoi ( QPoint addedNewPoiPos ) ;

  /**
  * @brief This slot is used to save the map
  */
  void clickedSaveMapButton();

  /**
  * @brief This slot is used to open a saved map
  */
  void clickedOpenMapButton();

  /**
  * @brief This slot is used to reset the map
  */
  void clickedResetMapsButton();

  /**
  * @brief This slot is used to save a poi that was set by the user
  * If the apply button is clicked it emits the clickedPoiApplyButton slot and the slot sends either an AddPointOfInterestM or a ModifyPointOfInterestM message depending on the status of m_CurrentPoiSelectionID
  */
  void clickedPoiApplyButton();

  /**
  * @brief This slot is used to delete a poi
  * It can emits by clicking the delete button or by pressing the remove key
  */
  void clickedPoiDeleteButton();

  /**
  * @brief This slot is used to send the robot to the current selected point
  */
  void clickedDriveToPositionButton();

  /**
  * @brief This slot is used to stop an ongoing navigation task
  */
  void clickedStopNavigationButton();

  /**
  * @brief This slot is used to send the masked region in the map to the slam module
  * Thereby the mask-mode will be transmitted
  */
  void clickedMaskButton();

  /**
  * @brief This slot is used to toggle the robot's view in the MapWidget
  * This slot changes the visibility of the robot position in the map
  * @param [in] state 0 if the robot shall be hidden otherwise 1
  */
  void toggleRobotView ( int state ) ;

  /**
  * @brief This slot is used to toggle the poi's view in the MapWidget
  * This slot changes the visibility of the poi presentation in the map
  * @param [in] state 0 if the poi shall be hidden otherwise 1
  */
  void togglePoiView ( int state ) ;

  /**
  * @brief This slot is used to control if the user has actually entered a name for the poi
  */
  void modifyPoiNameLineEdit();

  /**
  * @brief This slot is used to show a new poi icon if the user has chosen an other on
  * @param [in] index Index of the position in the combobox
  */
  void changedCurrentPoiIcon ( int index ) ;

  /**
  * @brief This slot is used to update the poi selection if the user changes it by the tree widget
  * This slot fetches the id of the selected poi and calls switchToEditMode
  * @see switchToEditMode
  */
  void changedCurrentPoiSelectionByTreeWidget();

  /**
  * @brief This slot is used to update the poi selection if the user changes it by the map widget
  * This slot calls switchToEditMode
  * @param [in] id The id of the selected point
  * @param [in] poiIconFileName Filename of the selected point
  * @see switchToEditMode
  */
  void changedCurrentPoiSelectionByMapWidget ( QString name, QString poiIconFileName ) ;

  /**
  * @brief This slot is used to provide the poi edit functionality
  * It will be called by changedCurrentPoiSelectionByTreeWidget or by changedCurrentPoiSelectionByTreeWidget
  * @param [in] id The id of the selected poi
  * @see changedCurrentPoiSelectionByTreeWidget, changedCurrentPoiSelectionByTreeWidget
  */
  void switchToEditMode ( std::string name ) ;

  /**
  * @brief This slot is used to show the current poi position in the MapWidget
  * It will only invoked if the mouse cursor moves over the MapWidget
  * @param [in] mousePosition QMouseEvent that contains the mouse position
  */
  void showMousePoition( QPoint mousePos ) ;

  /**
  * @brief This slot is used to show the current zoomFactor of the MapWidget
  * It will only bw invoked if a new zoom level ist applied
  * @param [in] zoomFactor new zoom factor
  */
  void showZoomFactor( float zoomFactor ) ;

  /**
  * @brief This slot is used to center the MapWidget always to the robot position
  * @param [in] xy Point to which shall be centered
  */
  void centerMap ( QPoint xy ) ;

  /**
  * @brief This slot is used to buffer the position of a new poi that isn't saved yet
  * @param [in] pos QPoint with the position information in pixel coordinates
  */
  void saveTmpPoiPosition ( QPoint pos ) ;

  /**
  * @brief This slot is used to buffer the orientation of a new or modified poi that isn't saved yet
  * @param [in] orientation doouble with the orientation information in radians
  */
  void saveTmpPoiOrientation ( double orientation ) ;

  /**
  * @brief This slot is used when the user toggle the state of the do mapping option
  * It sends a DoMappingM message with the current state
  * @param [in] state The current state of the checkbox
  */
  void toggleDoMappingState( int state ) ;

  /**
    * @brief This slot is sends a message to map_manager to toggle the visibility of a map layer
    * when it is checked or unchecked in the map layers tab
    * @param [in] item Pointer to respective map layer QListWidgetItem
    */
  void toggleMapLayerVisibility( QListWidgetItem* item);

  /**
   * @brief This slot is called whenever a new item in the map layers list is selected.
   * The index of the newly selected map is stored and used for editing the current layer
   */
  void changeCurrentMapLayer();

};


#endif
