#include <homer_nav_libs/tools/tools.h>

#include <tf/transform_datatypes.h>
#include <cmath>

#include <QtCore/QVariant>
#include <QtGui/QCursor>
#include <QtGui/QPen>
#include <QtGui/QBrush>
#include <QtGui/QColor>

#include "MapDisplay.h"

//#include "Workers/Semantics/Location/Area.h"

#include <homer_mapnav_msgs/ModifyMap.h>
#include <homer_mapnav_msgs/MapLayers.h>

#include <homer_nav_libs/tools/loadRosConfig.h>

#include <sstream>

MapDisplay::MapDisplay( ros::NodeHandle* nodeHandle, QWidget *parent )
: QWidget( parent ), m_MovablePoi( true )
{
  m_NodeHandle = nodeHandle;

  //add ros publishers here
  m_ModifyMapPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::ModifyMap>("/map_manager/modify_map", 10));

  resetRobotPos();
  clearTmpPoi();
  m_DefaultIconSize = QSize( 16, 16 );
  setMouseTracking( true );
  m_TmpMouseOperationLabel = 0;
  m_LastSelectedPoiLabelAddress = 0;
  m_SelectedRegionIndex = -1;
  m_RegionVisibility = false;

  m_PaintGrid = false;

  // Added by CNL, Masterarbeit 2011
  m_DisplayRooms = true;
  m_DisplayIntArtifacts = true;
  m_DisplayNonIntArtifacts = true;
  m_RoomCenterIcon = QPixmap("icons/cross.png");
  m_InteractiveIcon = QPixmap("icons/connected.png");
  m_NonInteractiveIcon = QPixmap("icons/disconnected.png");
  //m_SelectedArtifactIndex = -1;
  //m_SelectedRoomIndex = -1;
  m_SelectedListItemIndex = -1;
  m_SelectedLayer = -1;

  m_ModifyRegion = false;
  m_ScaleMode = Qt::FastTransformation;

  m_MapNames[ homer_mapnav_msgs::MapLayers::SLAM_LAYER ] = "SLAM Map";
  m_MapNames[ homer_mapnav_msgs::MapLayers::KINECT_LAYER ] = "Kinect Map";
  m_MapNames[ homer_mapnav_msgs::MapLayers::MASKING_LAYER ] = "Masking Map";
  m_MapNames[ homer_mapnav_msgs::MapLayers::SICK_LAYER ] = "SICK laser data";

  m_CurrentMap = SlamMap;
  m_ZoomFactor = 1.0;

  m_MapPalette.resize( 205);
  m_MapPalette[ 100 ] = qRgb( 255, 0, 0   );	// blocked TODO
  m_MapPalette[ 101 ] = qRgb( 0, 104, 176 ); 	// obstacle
  m_MapPalette[ 102 ] = qRgb( 0,   0, 80  );    // Kinect
  m_MapPalette[ 203 ] = qRgb( 255, 0, 0   ); 	// blocked sensitive
  m_MapPalette[ 204 ] = qRgb( 0, 200, 176 ); 	// obstacle sensitive
  //create color palette
  int r, g, b;
  for ( int i = 1; i <= 100; i++ )
  {
    r = int( i * 2.55 );
    g = int( i * 2.55 );
    b = int( i * 2.55 );
    if ( r > 255 ) r = 255;
    if ( g > 255 ) g = 255;
    if ( b > 255 ) b = 255;
     m_MapPalette[ 100 - i ] = qRgb( r, g, b ); //index 0 = white, index 100 = black 
  }
  for ( int i = 1; i <= 100; i++ )
  {
    r = int( i * 2.55 );
    b = int( i * 2.55 );
    if ( r > 255 ) r = 255;
    g = 200;
    if ( b > 255 ) b = 255;
     m_MapPalette[ 203 - i ] = qRgb( r, g, b ); //index 103 = free, index 203 = occupied
  }
  
  m_MapPalette[201] = qRgb(0, 0, 0);
  

  m_DistDetermination = false;
  
  float mapSize;
  float resolution;
  unsigned pixelSize;
  unsigned byteSize;
  
  loadConfigValue("/homer_mapping/size", mapSize);
  loadConfigValue("/homer_mapping/resolution", resolution);

  //add one safety pixel
  pixelSize = mapSize / resolution + 1;
  byteSize = pixelSize * pixelSize;
  m_HighSensitive = new unsigned short[byteSize];
   
  for ( unsigned i=0; i<byteSize; i++ )
  {
    m_HighSensitive[i] = 0;
  }
}

MapDisplay::~MapDisplay()
{
  // delete all existing poi labels
  QVectorIterator< PoiLabel* > q_it( m_PoiLabelList );
  while ( q_it.hasNext() )
  {
    delete q_it.next();
    // remove vector field
    m_PoiLabelList.pop_front();
  }
  if ( m_HighSensitive ) 
  {
    delete[] m_HighSensitive;
  }
}

/** NOTE The two factors cancel each other (1.25 * 0.8 = 1.0), so don't change them unthoughtfully. */
float MapDisplay::zoomIn()
{
  if (m_ZoomFactor < 4.0)
  {
    m_ZoomFactor += 1.0;
    emit changedZoomFactor( m_ZoomFactor );
    // resize the map region modification polygon too
    for ( int i = 0; i < m_Region.size(); i++ )
    {
        m_Region[ i ] *= 1.25;
    }

    // causes the parent QScrollArea to adjust its scroll bars to the map's new display size
    resize( sizeHint() );
    update();
    updatePoiPresentation();

    if ( m_RobotPos != QPoint( -1, -1 ) )
    {
        // center over robot position
        emit centerMap( m_RobotPos * m_ZoomFactor );
    }
    else
    {
        // center over map center
        int x = m_MapImages[ m_CurrentMap ].width() / 2 * m_ZoomFactor;
        emit centerMap( QPoint( x, x ) );
    }

    // to prevent the GUI from crashing when the user zoomes while moving a PoI
    m_TmpMouseOperationLabel = 0;
  }
  return m_ZoomFactor;
}

float MapDisplay::zoomOut()
{
    if ( m_ZoomFactor > 1.0 )
    {
        m_ZoomFactor -= 1.0;

        emit changedZoomFactor( m_ZoomFactor );

        for ( int i = 0; i < m_Region.size(); i++ )
        {
            m_Region[ i ] *= 0.8;
        }

        resize( sizeHint() );
        update();
        updatePoiPresentation();

        if ( m_RobotPos != QPoint( -1, -1 ) )
        {
            emit centerMap( m_RobotPos * m_ZoomFactor );
        }
        else
        {
            int x = m_MapImages[ m_CurrentMap ].width() / 2 * m_ZoomFactor;
            emit centerMap( QPoint( x, x ) );
        }

        m_TmpMouseOperationLabel = 0;
    }

    return m_ZoomFactor;
}

void MapDisplay::selectMap( int mapId )
{
    m_CurrentMap = MapId( mapId );
    m_ZoomFactor = 1.0;
    resize( sizeHint() );
    emit mapSelected( mapId );
    update();
}

void MapDisplay::printPath(std::vector<geometry_msgs::PoseStamped> path)
{
    m_Path = path;
}

void MapDisplay::printFollowingPath(std::vector<geometry_msgs::PoseStamped> path)
{
    m_FollowingPath = path;
}


void MapDisplay::updateMap( MapId mapId, unsigned char * pMap, unsigned size)
{

    if ( ( m_MapImages.find( mapId ) == m_MapImages.end() ) ||
            ( m_MapImages[ mapId ].width() != int( size ) ) )
    {
        std::ostringstream stream;
        stream << "Resizing map image '" << m_MapNames[ mapId ] << "' to " << size << "x" << size;
        ROS_INFO_STREAM( stream.str() );
        m_MapImages[ mapId ] = QImage( size, size, QImage::Format_Indexed8 );
        m_MapImages[ mapId ].setNumColors( m_MapPalette.size() );
        m_MapImages[ mapId ].setColorTable( m_MapPalette );
        m_MapImages[ mapId ].fill( MapDisplay::UNKNOWN );
        resize( sizeHint() );
    }

    QImage& targetImage = m_MapImages[ mapId ];

    //is there a faster way (memcpy?)
    for ( unsigned x = 0; x < size; x++ ) {
        int x_offset = x * size;
        for ( unsigned y = 0; y < size; y++)
        {
            int i = x_offset + y;
            if(m_HighSensitive[i]==1)
            {
	        	targetImage.setPixel(QPoint(size-x-1, size-y-1), pMap[i] + 103);
            }
            else
            {
    	        targetImage.setPixel(QPoint(size-x-1, size-y-1), pMap[i]);
            }
            //memcpy( targetImage.scanLine( x ), pMap + ( size * x ), size );
        }

    }
    //m_MapUpdateTimer.submit();
    m_CheckerPainted[ mapId ] = false;
    update();
}


void MapDisplay::updateMask(const nav_msgs::OccupancyGrid::ConstPtr &msg)
{
    int size = msg->info.height;
    for ( unsigned x = 0; x < size ; x++ ) {
        int x_offset = x * size;
        for ( unsigned y = 0; y < size; y++)
        {
            int i = x_offset + y;
            if(msg->data[i] == homer_mapnav_msgs::ModifyMap::HIGH_SENSITIV)
            {
    			m_HighSensitive[i] = 1;
            }
        }
    }
}


void MapDisplay::mousePressEvent( QMouseEvent* event )
{



  int x = event->pos().x();
  int y =  event->pos().y();

  if ( !childAt( x, y ) )
  {
    if ( m_RegionVisibility )
    {
      for ( int i = 0; i < m_Region.size(); i++ )
      {
        QRect pbb( m_Region[ i ].x() - 4, m_Region[ i ].y() - 4, 8, 8 );
        if ( pbb.contains( x, y ) )
        {
          m_SelectedRegionIndex = i;
          update();
          m_ModifyRegion = true;
          break;
        }
      }
    }

    if ( m_SelectedRegionIndex == -1 )
    {
      if ( ( event->buttons() & Qt::LeftButton ) && ( event->modifiers() & Qt::ControlModifier ) )
      {
        emit robotPositionChanged( event->pos() / m_ZoomFactor, m_RobotOrientation );
      }
      else if ( ( event->buttons() & Qt::LeftButton ) && ( event->modifiers() & Qt::ShiftModifier ) )
      {
        // creates an initialize unreachable-region-rectangle
        m_RegionVisibility = true;
        m_Region.putPoints( 0, 4, x - 8, y - 8, x - 8, y + 8, x + 8, y + 8, x + 8, y - 8 );
        update();
      }
      else if ( event->buttons() & Qt::LeftButton )
      {
        // save new poi position
        m_CurrentlySelectedPoi = event->pos() / m_ZoomFactor;
        // deactivate poi selection rectangle
        showActivePoiLabelSelection( "" );
        // toggle icon settings to new poi
        m_LastSelectedPoiLabelAddress = 0;
        // send message
        emit addedNewPoi( m_CurrentlySelectedPoi );
        // update painter and poi presentation
        update();
        updatePoiPresentation();
      }

      if ( ( event->buttons() & Qt::RightButton ) && ( event->modifiers() & Qt::ControlModifier ) )
      {
        if(m_LastSelectedPoiLabelAddress != 0)
        {
            //m_LastSelectedPoiLabelAddress->setVisible(false);
            m_CurrentPoiOrientation = computeOrientationAngle((m_LastSelectedPoiLabelAddress->getPoiPosition() + QPoint( 8, 8 ))/m_ZoomFactor, QPoint(event->pos())/m_ZoomFactor);

            //m_LastSelectedPoiLabelAddress->setOrientation();
            //m_LastSelectedPoiLabelAddress->update();
            emit modifiedPoiOrientation(-m_CurrentPoiOrientation);
            update();
        }
        else
        {
            m_RobotOrientation = computeOrientationAngle( m_RobotPos, QPoint( event->pos() ) / m_ZoomFactor );
            emit robotPositionChanged( m_RobotPos, m_RobotOrientation );
        }
      }
      else if ( event->button() == Qt::RightButton )
      {
        QApplication::setOverrideCursor( QCursor( Qt::CrossCursor ) );
        m_DistDetermination = true;
        m_DistDetermStart = event->pos();
        m_DistDetermCurrent = event->pos();
        repaint();
      }
    }
  }
  else
  {
    m_TmpMouseOperationLabel = dynamic_cast< PoiLabel* >( childAt( x, y ) );
    // get the poi name
    QString name = m_TmpMouseOperationLabel->getPoiName();
    m_CurrentlySelectedPoiName = name.toStdString();
    m_CurrentPoiOrientation = m_TmpMouseOperationLabel->getPoiOrientation();
    // get the poi icon file name
    QString poiIconFileName = m_TmpMouseOperationLabel->getPoiIconFileName();
    // pass the information to the mapframe
    emit modifiedPoi( name, poiIconFileName );
    // update painter and poi presentation
  }
}

void MapDisplay::mouseMoveEvent( QMouseEvent* event )
{
  if ( m_DistDetermination )
  {
    m_DistDetermCurrent = event->pos();
    repaint();
  }
  else
  {
    emit changedMousePosition( event->pos() / m_ZoomFactor );

    if ( m_TmpMouseOperationLabel && m_MovablePoi )
    {
        m_TmpMouseOperationLabel->move( event->pos() - QPoint( 8, 8 ) );
    }

    if ( m_ModifyRegion )
    {
        m_Region.setPoint( m_SelectedRegionIndex, event->pos().x(), event->pos().y() );
        update();
    }
  }
}

void MapDisplay::mouseReleaseEvent( QMouseEvent* event )
{
  if( m_DistDetermination )
  {
    QApplication::restoreOverrideCursor();

    QVariant xPress = m_DistDetermStart.x();
    QVariant yPress = m_DistDetermStart.y();
    QVariant xRelease = event->x();
    QVariant yRelease = event->y();
    QVariant zoomFactor = m_ZoomFactor;
    QVariant a = std::max( xPress.toInt(), xRelease.toInt() )
                 - std::min( xPress.toInt(), xRelease.toInt() );
    QVariant b = std::max( yPress.toInt(), yRelease.toInt() )
                 - std::min( yPress.toInt(), yRelease.toInt() );
    QVariant lineLength = sqrt( a.toInt() * a.toInt() + b.toInt() * b.toInt() );
    QVariant zoomFtmp = m_ZoomFactor;
    QVariant result = ( lineLength.toDouble() * 50.0 )
                        / ( zoomFtmp.toDouble() * 1000.0 );
    int x = result.toDouble() * 100.0;
    QVariant distance = x / 100.0;

    QString messageBoxcontent = "Start point:\t\t"  + xPress.toString()     + ":"
                                                    + yPress.toString()     + "\n"   +
                                "End point:\t\t"    + xRelease.toString()   + ":"
                                                    + yRelease.toString()   + "\n"   +
                                "Line:\t\t"         + lineLength.toString() + "\n"   +
                                "Scaling factor:\t" + zoomFactor.toString() + "\n\n" +
                                "Dinstance:\t\t"    + distance.toString()   + " m";

    QMessageBox::information( this, tr( "measured distance" ),
                              messageBoxcontent, QMessageBox::Ok );

    m_DistDetermination = false;
    //TODO : scaling
  }

  // so there is no longer poi moving if the mouse is moving
  if ( m_TmpMouseOperationLabel )
  {
    emit modifiedPoiPosition( ( m_LastSelectedPoiLabelAddress->pos() + QPoint( 8, 8 ) )
                              / m_ZoomFactor );
  }

  m_TmpMouseOperationLabel = 0;
  m_ModifyRegion = false;
}

void MapDisplay::resetPoiPresentation()
{
    if ( m_LastSelectedPoiLabelAddress )
    {
        m_LastSelectedPoiLabelAddress->move( m_LastModifiedPoiPositionBackup );
        m_LastSelectedPoiLabelAddress = 0;
    }
}

void MapDisplay::resetRegion()
{
    // deletes the rectangle whether modified or not
    m_RegionVisibility = false;
    m_SelectedRegionIndex = -1;
    m_Region.clear();
    update();
}

void MapDisplay::commitRegion( int maskAction , int mapLayer)
{
    if ( m_RegionVisibility )
    {
        homer_mapnav_msgs::ModifyMap msg;
        msg.maskAction = maskAction;
        msg.mapLayer = mapLayer;

        for ( int i = 0; i < m_Region.size(); i++ )
        {
          Eigen::Vector2i rmp( m_Region[i].x(), m_Region[i].y() );

          //Point (0,0) in ROS map starts in lower right position
          //Pixel (0,0) in upper left position
          //x and y is swapped
          int height = m_MapImages[m_CurrentMap].height();
          int width = m_MapImages[m_CurrentMap].width();
          geometry_msgs::Point rosPoint;
          rosPoint.x = width - 1 - rmp.y() / m_ZoomFactor;
          rosPoint.y = height -1 - rmp.x() / m_ZoomFactor;
          rosPoint.z = 0.0;

          msg.region.push_back(rosPoint);
        }

        m_ModifyMapPublisher->publish(msg);
        resetRegion();
    }
}

void MapDisplay::paintGrid( QImage& mapImage )
{
    unsigned char* currentScanLine;

    //TODO add config value
    double gridCellWidthM = 1;
    double cellSize = 0.05;//Config::getInt("Map.iCellSize");
    int gridCellWidth = gridCellWidthM / cellSize;

    for(int y = 0; y < mapImage.height(); y++)
    {
        currentScanLine = mapImage.scanLine( y );


        if( y % gridCellWidth == 0)
        {
            for (int x = 0; x < mapImage.width(); x++ )
            {
                currentScanLine[x] = MapDisplay::UNKNOWN;
            }
        }
        else
        {
            for ( int x = 0; x < mapImage.width(); x++ )
            {
                if( x % gridCellWidth == 0)
                {
                    currentScanLine[x] = MapDisplay::UNKNOWN;
                }
            }

        }
    }
}

void MapDisplay::paintCheckerTexture( QImage& mapImage )
{
    //Replace unknown areas by checker image
   unsigned char* currentScanLine;
    int x,y;
    const unsigned char grayDiff = 0; // was 8

    double checkerWidthM = 0.01; // TODO add config value: Config::getInt( "Gui.MapDisplay.iCheckerWidth" );
    double cellSize = 0.05; // TODO add config value Config::getInt( "Map.iCellSize" );
    int checkerWidth = checkerWidthM / cellSize;

    for ( y = 0; y < mapImage.height(); y++ )
    {
        currentScanLine = mapImage.scanLine( y );

        if ( y / checkerWidth % 2 )
        {
            //even rows:
            for ( x = 0; x < mapImage.width(); x++ )
            {
                if ( ( x / checkerWidth % 2 ) && ( currentScanLine[ x ] == MapDisplay::UNKNOWN ) )
                {
                    currentScanLine[ x ] = MapDisplay::UNKNOWN + grayDiff;
                }
            }
        }
        else
        {
            //uneven rows:
            for ( x = 0; x < mapImage.width(); x++ )
            {
                if ( ! ( x / checkerWidth % 2 ) && ( currentScanLine[ x ] == MapDisplay::UNKNOWN ) )
                {
                    currentScanLine[ x ] = MapDisplay::UNKNOWN + grayDiff;
                }
            }
        }
    }
}

void MapDisplay::paintEvent( QPaintEvent* )
{
  QPainter painter( this );

  if ( m_MapImages.find( m_CurrentMap ) != m_MapImages.end() )
  {
    if ( !m_CheckerPainted[ m_CurrentMap ] )
    {
      //paintCheckerTexture( m_MapImages[ m_CurrentMap ] );
    }
    if(m_PaintGrid)
        paintGrid(m_MapImages[m_CurrentMap]);

    QSize newSize = m_MapImages[ m_CurrentMap ].size() * m_ZoomFactor;
    painter.drawImage( QPoint( 0, 0 ), m_MapImages[ m_CurrentMap ].scaled( newSize, Qt::KeepAspectRatio, m_ScaleMode ) );
  }

  //draw path
  if ( m_Path.size() > 0 )
  {
    painter.setPen ( QPen ( QColor ( Qt::blue ), 0.5, Qt::SolidLine ) );
    // painter.setBrush(QColor(Qt::blue));
    Eigen::Vector2i tmpCoord, nextTmpCoord;
    tmpCoord = map_tools::toMapCoords(m_Path[0].pose.position, m_Origin, m_Resolution);
    painter.drawEllipse( (m_PixelSize - 1 - tmpCoord.y()) * m_ZoomFactor - 2, (m_PixelSize - 1 - tmpCoord.x()) * m_ZoomFactor - 2, 4, 4 );
    for ( unsigned int i = 0; i < m_Path.size() - 1; i++ )
    {
      tmpCoord =  map_tools::toMapCoords(m_Path[i].pose.position, m_Origin, m_Resolution);
      nextTmpCoord = map_tools::toMapCoords(m_Path[i+1].pose.position, m_Origin, m_Resolution);
      painter.drawLine( (m_PixelSize - 1 - tmpCoord.y()) * m_ZoomFactor, (m_PixelSize - 1 - tmpCoord.x()) * m_ZoomFactor,
                        (m_PixelSize - 1 - nextTmpCoord.y()) * m_ZoomFactor, (m_PixelSize - 1 - nextTmpCoord.x()) * m_ZoomFactor );
      painter.drawEllipse( (m_PixelSize - 1 - nextTmpCoord.y()) * m_ZoomFactor - 2, (m_PixelSize - 1 - nextTmpCoord.x()) * m_ZoomFactor - 2, 4, 4 );
    }
  }

  if ( m_FollowingPath.size() > 0 )
  {
    painter.setPen ( QPen ( QColor ( Qt::red ), 0.5, Qt::SolidLine ) );
    // painter.setBrush(QColor(Qt::blue));
    Eigen::Vector2i tmpCoord, nextTmpCoord;
    tmpCoord = map_tools::toMapCoords(m_FollowingPath[0].pose.position, m_Origin, m_Resolution);
    painter.drawEllipse( (m_PixelSize - 1 - tmpCoord.y()) * m_ZoomFactor - 2, (m_PixelSize - 1 - tmpCoord.x()) * m_ZoomFactor - 2, 4, 4 );
    for ( unsigned int i = 0; i < m_FollowingPath.size() - 1; i++ )
    {
      tmpCoord =  map_tools::toMapCoords(m_FollowingPath[i].pose.position, m_Origin, m_Resolution);
      nextTmpCoord = map_tools::toMapCoords(m_FollowingPath[i+1].pose.position, m_Origin, m_Resolution);
      painter.drawLine( (m_PixelSize - 1 - tmpCoord.y()) * m_ZoomFactor, (m_PixelSize - 1 - tmpCoord.x()) * m_ZoomFactor,
                        (m_PixelSize - 1 - nextTmpCoord.y()) * m_ZoomFactor, (m_PixelSize - 1 - nextTmpCoord.x()) * m_ZoomFactor );
      painter.drawEllipse( (m_PixelSize - 1 - nextTmpCoord.y()) * m_ZoomFactor - 2, (m_PixelSize - 1 - nextTmpCoord.x()) * m_ZoomFactor - 2, 4, 4 );
    }
  }

  // Added by Carmen Navarro Luzón, MA 2011
  if (m_DisplayRooms ) {
      for (unsigned int i=0; i<m_Rooms.size(); ++i) {
          if (i == m_SelectedListItemIndex) {
              painter.setPen( QPen( QColor( 0, 150, 120 ), 1, Qt::SolidLine ) );
              painter.setBrush( QBrush(QColor (0, 150, 120), Qt::SolidPattern ));
          }
          else {
              painter.setPen( QPen( QColor( 0, 200, 150 ), 1, Qt::SolidLine ) );
              painter.setBrush( QBrush(QColor (0, 200, 150), Qt::SolidPattern ));
          }
          painter.drawPolygon( m_Rooms.at(i) );

          // Paint the cross
          painter.drawPixmap( QRect( m_Centers.at(i) * m_ZoomFactor - QPoint( 8, 8 ), QSize( 16, 16 ) ), m_RoomCenterIcon);

      }

      painter.setBrush(Qt::NoBrush);

  }
  if (m_DisplayIntArtifacts) {
      // TODO draw artifacts
  }

  if (m_DisplayNonIntArtifacts) {

  }
  //draw selected poi
  painter.setPen( QPen( QColor( 0, 0, 0 ), 1, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin ) );
  if ( m_CurrentlySelectedPoi != QPoint( -1, -1 ) )
  {
    painter.drawPixmap( QRect( m_CurrentlySelectedPoi * m_ZoomFactor - QPoint( 8, 8 ), QSize( 16, 16 ) ), m_CurrentPoiIconSelection );
    painter.save();

    //translate and rotate painter to poi position to draw orientation arrow
    painter.setBrush( Qt::blue );
    painter.translate(m_CurrentlySelectedPoi * m_ZoomFactor);
    painter.rotate(m_CurrentPoiOrientation * 180 / M_PI);

    //draw current poi orientation
    QPolygon triangle;
    triangle << QPoint( -3, -2 );
    triangle << QPoint( 0, -8 );
    triangle << QPoint( 3, -2 );
    triangle << QPoint( -3, -2 );
    painter.drawPolygon( triangle );

    painter.restore();
  }

  // draw region polygon
  if ( m_RegionVisibility )
  {
    painter.setPen( QPen( QColor( 255, 0, 0 ), 1, Qt::SolidLine ) );
    painter.drawPolygon( m_Region );
  }


  // highlight selected region polygon point
  if ( m_SelectedRegionIndex != -1 )
  {
    painter.setBrush( QColor( Qt::red ) );
    painter.drawEllipse( QRect( m_Region.point( m_SelectedRegionIndex ) - QPoint( 3, 3 ), QSize( 6, 6 ) ) );
  }

  painter.save();
  painter.translate( m_RobotPos * m_ZoomFactor );
  int qtRobotOrientation = int( m_RobotOrientation / M_PI * 180.0 );
  // draw the robot's position and orientation
  if ( m_RobotPos != QPoint( -1, -1 ) && m_RobotVisibility )
  {
    painter.setOpacity ( 0.5 );
    // draws the robot circle
    painter.setBrush( QColor( 0,0,255 ) );
    painter.setPen( Qt::NoPen );
    painter.drawEllipse( QRectF( QPointF( -m_DefaultIconSize.width() / 2, -m_DefaultIconSize.height() / 2 ), m_DefaultIconSize ) );
    // draws the orientation arrow
    painter.rotate( qtRobotOrientation );
    painter.setBrush( QColor( 255, 255, 255 ) );
    QPolygon triangle;
    triangle << QPoint( -m_DefaultIconSize.width() / 3, m_DefaultIconSize.height() / 3 );
    triangle << QPoint( 0, -m_DefaultIconSize.height() / 2 );
    triangle << QPoint( m_DefaultIconSize.width() / 3, m_DefaultIconSize.height() / 3 );
    triangle << QPoint( -m_DefaultIconSize.width() / 3, m_DefaultIconSize.height() / 3 );
    painter.drawPolygon( triangle );
  }

  painter.restore();

  // draw the Distance
  if ( m_DistDetermination )
  {
    painter.setPen( QPen( QColor( Qt::black ), 2, Qt::SolidLine ) );
    painter.drawLine( m_DistDetermStart, m_DistDetermCurrent );
  }

  painter.setOpacity( 1 );

  // m_PaintEventTimer.submit();
}


// Added by Carmen Navarro Luzón, MasterArbeit 2011
void MapDisplay::deleteRoomByIndex(unsigned i)
{
    if (i<m_Rooms.size()) {
        m_Rooms.erase(m_Rooms.begin() + i);
        m_Centers.erase(m_Centers.begin() + i);
        if (m_SelectedListItemIndex == i && m_SelectedLayer == 0) {
            m_SelectedListItemIndex = -1;
        }
    }
}

void MapDisplay::toggleRoomView ( bool visibility )
{
    //TRACE_INFO("Room view change"); // TODO use ROS
    m_DisplayRooms = visibility;
}

void MapDisplay::toggleArtifactView ( bool visibility )
{
   // TRACE_INFO("NonInteractive Artifact view change"); // TODO use ROS
    m_DisplayNonIntArtifacts = visibility;
}

void MapDisplay::toggleInteractiveArtView ( bool visibility )
{
    // TRACE_INFO("Interactive Artifact view change"); // TODO use ROS
    m_DisplayIntArtifacts = visibility;
}

void MapDisplay::deleteArtifactByIndex(unsigned i)
{
    if (i<m_InteractiveArtifacts.size()) {
        m_InteractiveArtifacts.erase(m_InteractiveArtifacts.begin() + i);
        if (m_SelectedListItemIndex == i && m_SelectedLayer > 0) {
            m_SelectedListItemIndex = -1;
        }
        else {
            // TRACE_WARNING("Tried to remove artifact when selected layer is 0"); // TODO use ROS
        }
    }
}

void MapDisplay::printQPolygon(QPolygon p) {
    //TRACE_INFO("Polygon:"); // TODO use ROS
    for (int i = 0; i<p.size(); i++) {
        printQPoint(p.at(i));
    }
}

void MapDisplay::printQPoint(QPoint p) {
    ROS_INFO_STREAM("(" << p.x() << "," << p.y() << ")");
}

void MapDisplay::updateSelectedLayer(unsigned i)
{
    m_SelectedLayer = i;
}


void MapDisplay::updateSelectedItemIndex(unsigned i)
{
    ROS_INFO_STREAM("Selected list item index");
    m_SelectedListItemIndex = i;
}

geometry_msgs::Point MapDisplay::commitArtifact() {

    std::cout<<"Pixel : "<<m_CurrentlySelectedPoi.x() <<","<< m_CurrentlySelectedPoi.y() << std::endl;
    Eigen::Vector2i rmp( m_PixelSize - 1 - m_CurrentlySelectedPoi.y()/m_ZoomFactor, m_PixelSize - 1 - m_CurrentlySelectedPoi.x()/m_ZoomFactor );

    geometry_msgs::Point worldPoint = map_tools::fromMapCoords(rmp, m_Origin, m_Resolution);
    m_InteractiveArtifacts.push_back(QPoint(m_CurrentlySelectedPoi));

    ROS_INFO_STREAM("Artifact added to mapDisplay");
    update();
    return worldPoint;
}

std::vector<geometry_msgs::Point> MapDisplay::commitRoom() {
    std::vector< geometry_msgs::Point > worldRegion;
    ROS_INFO_STREAM("region size: " << m_Region.size() );
    for ( int i = 0; i < m_Region.size(); i++ )
    {
      std::cout<<"Pixel "<<i<<" : "<<m_Region[i].x() <<","<< m_Region[i].y() << std::endl;
        Eigen::Vector2i rmp( m_PixelSize - 1 -m_Region[i].y()/m_ZoomFactor, m_PixelSize - 1 - m_Region[i].x()/m_ZoomFactor );

        std::ostringstream stream;
        stream << " Point " << i << ": "<< m_Region[i].x() <<" , "<< m_Region[i].y() << std::endl;

        geometry_msgs::Point worldPoint = map_tools::fromMapCoords(rmp, m_Origin, m_Resolution);
        worldRegion.push_back( worldPoint );
    }
    m_Rooms.push_back(QPolygon(m_Region));
    m_Centers.push_back(QPoint(m_CurrentlySelectedPoi));
    ROS_INFO_STREAM("Room added to mapDisplay");
    resetRegion();
    update();
    return worldRegion;
}

std::vector<geometry_msgs::Point> MapDisplay::getPolygonFromRegion() {
    std::vector< geometry_msgs::Point > worldRegion;
    if ( m_RegionVisibility )
    {

        for ( int i = 0; i < m_Region.size(); i++ )
        {
            Eigen::Vector2i rmp( m_PixelSize - 1 - m_Region[i].y()/m_ZoomFactor, m_PixelSize - 1 - m_Region[i].x()/m_ZoomFactor );

            geometry_msgs::Point worldPoint = map_tools::fromMapCoords(rmp, m_Origin, m_Resolution);
            worldRegion.push_back( worldPoint );
        }

    }

   // Area result(worldRegion, m_CurrentPoi);
    return worldRegion;
}

geometry_msgs::Point MapDisplay::getSelectedPOI()
{
    geometry_msgs::Point poiWorldCoordinate = map_tools::fromMapCoords(Eigen::Vector2i(m_PixelSize - 1 - m_CurrentlySelectedPoi.y()/m_ZoomFactor, m_PixelSize - 1 - m_CurrentlySelectedPoi.x()/m_ZoomFactor), m_Origin, m_Resolution);
    //Location result(poiWorldCoordinate);

    return poiWorldCoordinate;
}

// End of the code added by Carmen Navarro Luzón

void MapDisplay::updatePoiPresentation()
{
    // doesn't exist any longer
    m_LastSelectedPoiLabelAddress = 0;

    // delete all existing poi labels
    QVectorIterator< PoiLabel* > q_it( m_PoiLabelList );
    while ( q_it.hasNext() )
    {
        delete q_it.next();
        // remove vector field
        m_PoiLabelList.pop_front();
    }

    // create new labels
    std::vector< homer_mapnav_msgs::PointOfInterest >::iterator it;
    for ( it = m_PoiList->begin(); it != m_PoiList->end(); it++ )
    {
        // convert PoiPosition to map coordinates
        geometry_msgs::Point poiPoint = it->pose.position;
        Eigen::Vector2i mapPoint = map_tools::toMapCoords(poiPoint, m_Origin, m_Resolution);
        QPoint displayPosition( ( (m_PixelSize - 1 - mapPoint.y())*m_ZoomFactor - (m_DefaultIconSize.width() / 2) ), ( (m_PixelSize - 1 - mapPoint.x())*m_ZoomFactor - (m_DefaultIconSize.height() / 2) ) );
        double theta = -tf::getYaw(it->pose.orientation);
        // get the suitable poi icon filename
        QString iconFileName = getPoiIconFileName( it->type );
        // get poi's name
        QString poiName = it->name.c_str();
        // get the remarks
        QString remarks = it->remarks.c_str();

        // create the poilabel
        PoiLabel* tmp = new PoiLabel(displayPosition , theta, iconFileName , poiName, remarks, it->type, this, 1.0 );

        // push it into the poi label list
        m_PoiLabelList.push_back( tmp );
    }

    togglePoiView( m_PoiVisibility );
}

void MapDisplay::setRobotPos( QPoint position, double angle )
{

    m_RobotPos = position;
    m_RobotOrientation = angle;

    update();
}

double MapDisplay::computeOrientationAngle (QPoint point, QPoint orientation )
{
    QPoint tmp = orientation - point;
    double orientationAngle = atan2( tmp.x(), -tmp.y() );
    return orientationAngle;
}

void MapDisplay::resetRobotPos()
{
    m_RobotPos = QPoint ( -1,-1 );
    m_RobotOrientation = 0;
    update();
}

void MapDisplay::linkPoiList ( std::vector< homer_mapnav_msgs::PointOfInterest >& poiList )
{
    m_PoiList = &poiList;
}

void MapDisplay::linkPoiIconList ( QVector< PoiIcon >& poiIconList )
{
    m_PoiIcons = &poiIconList;
}

void MapDisplay::setCurrentPoiIcon ( QString poiIconFileName )
{
    // modify the selected poi's icon
    if ( m_LastSelectedPoiLabelAddress )
    {
        m_LastSelectedPoiLabelAddress->setNewPoiIcon ( poiIconFileName );
    }
    // modify the added poi's icon
    m_CurrentPoiIconSelection = QPixmap ( poiIconFileName );
    update();
}

void MapDisplay::clearTmpPoi()
{
    ROS_INFO_STREAM("cleared POI selection");
    m_CurrentlySelectedPoi = QPoint ( -1,-1 );
    m_CurrentlySelectedPoiName = "";
    m_CurrentPoiOrientation = 0;
    update();
}

QString MapDisplay::getPoiIconFileName (int poiType )
{
    QString fileName="";
    for ( int i = 0; i < m_PoiIcons->size(); i++ )
    {
        if ( poiType == m_PoiIcons->at ( i ).getPoiType() )
        {
            fileName = m_PoiIcons->at ( i ).getFileName();
            break;
        }
    }
    if ( fileName == "" ) {
      ROS_ERROR_STREAM( "Could not find filename for this POI type" );
      fileName = "icons/DefaultPoi.png"; // TODO check correct path to icons
    }
    return fileName;
}

void MapDisplay::toggleRobotView ( bool visibility )
{
    m_RobotVisibility = visibility;
    update();
}

void MapDisplay::togglePoiView ( bool visibility )
{
    m_PoiVisibility = ( visibility ) ? true : false;
    QVectorIterator<PoiLabel* > it ( m_PoiLabelList );
    while ( it.hasNext() )
    {
        if ( visibility )
        {
            it.next()->show();
        }
        else
        {
            it.next()->hide();
        }
    }
}

void MapDisplay::toggleAntiAliasing( int state )
{
    if ( state == Qt::Checked )
    {
        m_ScaleMode = Qt::SmoothTransformation;
    }
    else
    {
        m_ScaleMode = Qt::FastTransformation;
    }

    update();
}

void MapDisplay::toggleMovablePoi( int state )
{
    m_MovablePoi = ( state == Qt::Checked );
}

void MapDisplay::showActivePoiLabelSelection ( QString name )
{
    QVectorIterator<PoiLabel* > it ( m_PoiLabelList );
    while ( it.hasNext() )
    {
        it.peekNext()->setActive ( false );
        if ( it.peekNext()->getPoiName() == name )
        {
            /* the mouse release event will be set m_TmpMouseOperationLabel to 0
            but if the user wants to change the icon of the current selection you need the pointer.
            Also you need it to reset the poi position modifications */
            m_LastSelectedPoiLabelAddress = it.peekNext();
            // backup of the poi position to reset it if necessary
            m_LastModifiedPoiPositionBackup = it.peekNext()->pos();
            // set active
            it.peekNext()->setActive ( true );
        }
        it.next();
    }
}

const QImage* MapDisplay::getMapImagePtr()
{
    return &m_MapImages[ SlamMap ];
}

QSize MapDisplay::minimumSizeHint()
{
    return sizeHint();
}

QSize MapDisplay::sizeHint()
{
    if ( m_MapImages.find( m_CurrentMap ) != m_MapImages.end() )
    {
        return m_MapImages[ m_CurrentMap ].size() * m_ZoomFactor;
    }
    else
    {
        return QSize( 0, 0 );
    }
}

void MapDisplay::updatePose(double x, double y, double theta)
{
    geometry_msgs::Point p;
    p.x = x;
    p.y = y;
    p.z = 0.0;
    Eigen::Vector2i robotPos = map_tools::toMapCoords(p, m_Origin, m_Resolution);
    QPoint newPos( m_PixelSize - 1 - robotPos.y(), m_PixelSize - 1 - robotPos.x() );
    setRobotPos( newPos, -theta );
}

void MapDisplay::toggleGridVisibility( int state )
{
m_PaintGrid = (bool)state;
}
