#ifndef ROI_MANAGER_H
#define ROI_MANAGER_H

#include <ros/ros.h>
#include <homer_mapnav_msgs/RegionOfInterest.h>


/**
 * @class RoiManager (Roi = Region of interest)
 * @author Viktor Seib (R23)
 * @brief This class manages the List of regions of interest (ROIs)
  *
  * This class keeps a list of all ROIs within the current map. It provides the usual functions
  * to edit the list.
  */
class RoiManager
{

  public:

    /** The constructor of the class. */
    RoiManager(ros::NodeHandle* nh);

    /** constructor initializing the roi list */
    RoiManager( std::vector< homer_mapnav_msgs::RegionOfInterest > rois );

    /** Does nothing. */
    ~RoiManager() {}

    /** Adds a new ROI to the list, in contrast to POIs, several ROIs with the same name are allowed
      * @param roi RegionOfInterest message with the ROI to be added
      * @return true if successful, false otherwise
      */
    bool addRegionOfInterest( const homer_mapnav_msgs::RegionOfInterest::ConstPtr& roi );

    /** Replaces a ROI with a new one
      * @param roi RegionOfInterest to be inserted
      *            the ROI with the same ID as the new one is first deleted
      * @return true if the old ROI was found and could be deleted
      *         false otherwise
      */
    bool modifyRegionOfInterest( const homer_mapnav_msgs::RegionOfInterest::ConstPtr& roi );

    /** Deletes all ROIs with a certain name from the list
      * @param name Name of the ROIs to be deleted
      * @return true if the ROI was found and could be deleted
      *         false otherwise
      */
    bool deleteRegionOfInterest( std::string name );

    /** Deltes ROI with the given id
     * @param id ID of ROI to be deleted
     * @return true if the ROI was found and could be deleted
     *         false otherwise
     */
    bool deleteRegionOfInterest( int id );

    /**
     * @brief place the current roi list
     * @param roilist new roi list
     */
    void replaceROIList(std::vector<homer_mapnav_msgs::RegionOfInterest> roilist);


    /** Returns current ROI list
      * @return the ROI list
      */
    std::vector< homer_mapnav_msgs::RegionOfInterest > getList();

  private:

    /** Looks for ROI with name in the list
      * @param id ID of the ROI
      */
    bool roiExists(int id );

    /**
     * @brief Publishes a RegionsOfInterest Message with current ROIs

     */
    void broadcastRoiList();

    /** The copy constructor of the class.
      * It's kept private, because it will never be used.
      */
    RoiManager( const RoiManager& instance );

    /** Holds the ROI vector */
    std::vector< homer_mapnav_msgs::RegionOfInterest > m_Rois;

    /** publisher that publishes the current roi list */
    ros::Publisher m_ROIsPublisher;
};
#endif
