#ifndef FastNavigationModule_H
#define FastNavigationModule_H

#include <vector>
#include <string>
#include <map>
#include <deque>

#include <Eigen/Geometry>

#include <ros/ros.h>
#include <tf/transform_listener.h>

#include <robbie_architecture/Architecture/StateMachine/StateMachine.h>

#include <nav_msgs/OccupancyGrid.h>
#include <geometry_msgs/PoseStamped.h>
#include <sensor_msgs/LaserScan.h>
#include <sensor_msgs/PointCloud2.h>
#include <homer_mapnav_msgs/StartNavigation.h>
#include <homer_mapnav_msgs/StopNavigation.h>
#include <homer_mapnav_msgs/NavigateToPOI.h>
#include <std_msgs/Int8.h>

class Explorer;
/**
 * @class  HomerNavigationNode
 * @author Malte Knauf, Stephan Wirth, David Gossow (RX)
 * @brief  Performs autonomous exploration and navigation
 */
class HomerNavigationNode {

  public:

    /**
     * @brief   States of the state machines
     */
    enum MapType
    {
      SLAM_MAP,
      NAVIGATION_MAP
    };

    enum ProcessState
    {
      IDLE,
      AWAITING_EXPLORATION_MAP,
      AWAITING_PATHPLANNING_MAP,
      FOLLOWING_PATH,
      AVOIDING_COLLISION,
      FINAL_TURN,
      TARGET_REACHED,
      TILTED,
      STALLED
    };

     /**
     * The constructor
     */
    HomerNavigationNode();

     /**
     * The destructor
     */
    virtual ~HomerNavigationNode();
    
    /** @brief Is called in constant intervals. */
    void idleProcess();


  protected:

    /** @brief Handles incoming messages. */
    //virtual std::set<Message*> processMessages();
    void mapCallback(const nav_msgs::OccupancyGrid::ConstPtr& msg);
    void poseCallback(const geometry_msgs::PoseStamped::ConstPtr& msg);
    void laserDataCallback(const sensor_msgs::LaserScan::ConstPtr& msg);
    void backLaserCallback(const sensor_msgs::LaserScan::ConstPtr& msg);
    void startNavigationCallback(const homer_mapnav_msgs::StartNavigation::ConstPtr& msg);
    void stopNavigationCallback(const homer_mapnav_msgs::StopNavigation::ConstPtr& msg);
    void navigateToPOICallback(const homer_mapnav_msgs::NavigateToPOI::ConstPtr& msg);
    void unknownThresholdCallback(const std_msgs::Int8::ConstPtr& msg);
	void moveBaseSimpleGoalCallback(const geometry_msgs::PoseStamped::ConstPtr& msg);


    /** @brief initializes and refreshs parameters */
	void loadParameters();

    /** @brief Is called when all modules are loaded and thread has started. */
    virtual void init();


    /** @brief Detect & handle possible collision */
    void handleCollision();

  private:
  

    /** @brief Start navigation to m_Target on  last_map_data_ */
    void startNavigation();

    /** @brief Start exploration on last_map_data_ */
    void startExploration();

    /** @brief Check if obstacles are blocking the way in last_map_data_ */
    void checkPath();

    /** @brief calculate path from current robot position to target approximation */
    void calculatePath();

    /** @brief Send message containing current navigation path */
    void sendPathData();

    /** @brief Sends target reached and stops the robot. */
    void sendTargetReachedMsg();

    /**
     * @brief Sends a target unreachable with given reason and stops the robot.
     * @param reason reason for unreachable target (see homer_mapnav_msgs::TargetUnreachable for possible reasons)
     */
    void sendTargetUnreachableMsg( int8_t reason );
	
	void refreshParamsCallback(const std_msgs::Empty::ConstPtr& msg);

    /** @brief Navigate robot to next waypoint */
    void performNextMove();

    /** @brief Finishes navigation or starts turning to target direction if the target position has been reached */
    void targetPositionReached();

    /** @return Distance from robot_pose_ to point */
    double distanceTo(geometry_msgs::Point point);

    /** @return Angle from robot_pose_ to point in degrees */
    int angleToPointDeg(geometry_msgs::Point point);

    /** @brief Set status info */
    void actualizeStatusInfo();

    /** @brief Calculates current maximal speed based on laser data */
    float calcSpeedFactor();
    
    /** @brief Calculates current maximal backwards distance on map Data */
    float obstacleBackwardDistance();

    
    /**
     * @brief Send move message
     * @param distance_m distance to drive in m
     * @param speed_mPerSec driving speed in m/s
     * @param drive permanently until stopped
     */

    void sendMoveMessage(double distance_m , double speed_mPerSec, bool permanent = false);
    
    /**
     * @brief Send TurnMessage and set current turn action
     * @param theta angle to turn (in degrees)
     * @param speed turning speed
     * @param turn permanently until stopped
     */
    void sendTurnMessage(double theta, double speed, bool permanent = false);

	void sendStopRobot();

    /**
     * @brief Sets each cell of the map to -1 outside the bounding box
     *        containing the robot pose and the current target
     */
    void maskMap();

    /**
     * @brief Current path was finished (either successful or not),
     *        sets state machine to path planning to check if the robot is already at the goal
     */
    void currentPathFinished();

    //convenience math functions
    /**
     * Computes mean of given input values
     * @param values Container with values from which to compute mean
     * @return mean value
     */
    template<class ContainerT>
    static double mean ( const ContainerT& values );

    /**
     * Computes minimum turn angle from angle 1 to angle 2
     * @param angle1 from angle
     * @param angle2 to angle
     * @return minimal angle needed to turn from angle 1 to angle 2 [-Pi..Pi]
     */
    static float minTurnAngle ( float angle1, float angle2 );

    /**
     * converts value from degree to radiant
     * @param deg Value in degree
     * @return value in radiants
     */
    static float deg2Rad ( float deg ) { return deg / 180.0*M_PI; }

    /**
     * converts value from radiants to degrees
     * @param rad Value in radiants
     * @return value in degrees
     */
    static float rad2Deg ( float rad ) { return rad / M_PI*180.0; }

    /// @brief Worker instances

    Explorer* explorer_;

    /// @brief State machines

    StateMachine<MapType> m_MapTypeMachine;
    StateMachine<ProcessState> m_MainMachine;

    /// @brief Navigation options & data

    /** list of waypoints subsampled from m_PixelPath */
    std::vector<geometry_msgs::PoseStamped> waypoints_;

    /** Path planned by Explorer, pixel accuracy */
    std::vector<Eigen::Vector2i> pixel_path_;

    /** target point */
    geometry_msgs::Point target_point_;

    /** approximation of the target point is target is in an occupied cell */
    Eigen::Vector2i target_approx_;

    /** orientation the robot should have at the target point */
    double target_orientation_;

    /** allowed distance to target */
    float desired_distance_;

    /** check if the final turn should be skipped */
    bool skip_final_turn_;

    /**
     *  check if navigation should perform fast planning. In this mode a path is only planned within
     *  a bounding box containing the robot pose and the target point
     */
    bool fast_path_planning_;

    /** current pose of the robot */
    geometry_msgs::Pose robot_pose_;

    /** current laser scan */
    std::vector<geometry_msgs::Point> laser_points_;
    std::vector<geometry_msgs::Point> m_laser_points_map;
    std::vector<geometry_msgs::Point> m_back_laser;

    /** time stamp of the last incoming laser scan */
    ros::Time last_laser_time_;

    /** Distance factor of a frontier cell considered save for exploration */
    float m_FrontierSafenessFactor;

    /** stores the last m_SpeedFactorMeanFilterSize speed factors */
    std::deque<float> m_LastSpeedFactors;
    /** stores the mean of the last speed factors */
    std::deque<float> m_LastMeanSpeedFactors;
    /** maximal count of stored last speed factors bevore oldest one will be overwritten */
    int m_SpeedFactorMeanFilterSize;

    /** minimal speed factor while moving */
    float m_MinMoveSpeedFactor;

    /** minimal turn speed factor while moving */
    float m_MinTurnSpeedFactorMoving;

    /** minimal turn speed factor while standing */
    float m_MinTurnSpeedFactorStanding;

    double m_SafePathWeight;

    /** Number of subsequent times that an obstacle was detected in the planned path */
    int invalid_path_count_;

    ///map parameters
    double resolution_;
    double width_;
    double height_;
    geometry_msgs::Pose origin_;

    /// @brief Configuration parameters

    /** maximum move speed of the robot */
    float m_MaxTransVel;
    /** maximum turn speed of the robot */
    int m_MaxRotVel;

	bool m_use_cmd_vel_;

    /** Allowed distances of obstacles to robot. Robot must move within these bounds */
    std::pair<float,float> m_AllowedObstacleDistance;
    /** Safe distances of obstacles to robot. If possible, robot should move within these bounds */
    std::pair<float,float> m_SafeObstacleDistance;

    /** threshold to sample down waypoints */
    float waypoint_sampling_threshold_;

    /** if distance to nearest obstacle is below collision distance trigger collision avoidance */
    float collision_distance_;
    /** do not drive back in collision avoidance when this near target */
    float collision_distance_near_target_;
    /** drive this distance backwards when avoiding collision */
    float backward_distance_;

    /** threshold angle between robot and next waypoint when to turn instead of driving forward */
    double turn_threshold_angle_;

    /** if true, obstacles in path will be detected and path will be replanned */
    bool check_path_;
    /** path will be replanned, if obstacle is present in at least check_path_max_errors consecutive incoming maps  */
    int check_path_max_errors_;

    /** waypoints will only be checked for obstacles if they are closer than check_path_max_distance to robot */
    float check_path_max_distance_;

	bool m_avoided_collision;

	double m_min_turn_angle;
	double m_max_turn_speed;
	double m_max_move_speed;
	double m_max_drive_angle;
	
	float m_act_speed;
	float m_act_angle; 

	bool m_path_reaches_target;

    /** timestamp of last incoming map */
    ros::Time last_map_timestamp_;

    /** last map data */
    std::vector<int8_t> * last_map_data_;

    //ros specific members
    tf::TransformListener transform_listener_;

    //subscribers
    ros::Subscriber map_sub_;
    ros::Subscriber pose_sub_;
    ros::Subscriber laser_data_sub_;
    ros::Subscriber laser_back_data_sub_;
    ros::Subscriber robot_data_sub_;
    ros::Subscriber task_finished_sub_;
    ros::Subscriber start_navigation_sub_;
    ros::Subscriber stop_navigation_sub_;
    ros::Subscriber navigate_to_poi_sub_;
    ros::Subscriber unknown_threshold_sub_;
	ros::Subscriber refresh_param_sub_;
	ros::Subscriber m_move_base_simple_goal_sub_;

    //publishers
    ros::Publisher move_robot_pub_;
    ros::Publisher turn_robot_pub_;
    ros::Publisher stop_robot_pub_;
	ros::Publisher cmd_vel_pub_;
    ros::Publisher target_reached_pub_;
    ros::Publisher target_unreachable_pub_;
    ros::Publisher path_pub_;


    //service clients
    ros::ServiceClient get_POIs_client_;
};

#endif
