#!/usr/bin/env roseus

;; $ rtmlaunch hrpsys_ros_bridge samplerobot.launch
;; $ roseus samplerobot-auto-balancer.l "(samplerobot-auto-balancer-demo)"

(load "package://hrpsys_ros_bridge/euslisp/samplerobot-interface.l")

(defun samplerobot-auto-balancer-init ()
  (samplerobot-init)
  (setq *robot* *sr*)
  (initialize-pose-list)
  t)

(defun test-Pose-List (pose-list initial-pose)
  (dolist (pose pose-list)
    (send *sr* :angle-vector pose)
    (send *ri* :angle-vector (send *sr* :angle-vector) 1000)
    (send *ri* :wait-interpolation)
    (send *sr* :angle-vector initial-pose)
    (send *ri* :angle-vector (send *sr* :angle-vector) 1000)
    (send *ri* :wait-interpolation)))

(defun initialize-pose-list
  ()
  (setq *initial-pose* (send *sr* :reset-pose))
  (send *sr* :arms :shoulder-p :joint-angle -90)
  (send *sr* :arms :elbow-p :joint-angle -20)
  (setq *arm-front-pose* (send *sr* :angle-vector))
  (send *sr* :reset-pose)
  (send *sr* :legs :move-end-pos #f(0 0 70))
  (setq *half-sitting-pose* (send *sr* :angle-vector))
  (send *sr* :reset-pose)
  (send *sr* :fix-leg-to-coords (make-coords))
  (send *sr* :legs :move-end-pos #f(0 0 70))
  (let ((lc (send *sr* :legs :end-coords :copy-worldcoords)))
    (send *sr* :move-coords (send (send (car (send *sr* :links)) :copy-worldcoords) :rotate (deg2rad 10) :x) (car (send *sr* :links)))
    (mapcar #'(lambda (l c) (send *sr* l :inverse-kinematics c)) '(:lleg :rleg) lc)
    (send *sr* :move-centroid-on-foot :both '(:rleg :lleg))
    (setq *root-rot-x-pose* (send *sr* :angle-vector)))
  (send *sr* :reset-pose)
  (send *sr* :fix-leg-to-coords (make-coords))
  (send *sr* :legs :move-end-pos #f(0 0 70))
  (let ((lc (send *sr* :legs :end-coords :copy-worldcoords)))
    (send *sr* :move-coords (send (send (car (send *sr* :links)) :copy-worldcoords) :rotate (deg2rad 20) :y) (car (send *sr* :links)))
    (mapcar #'(lambda (l c) (send *sr* l :inverse-kinematics c)) '(:lleg :rleg) lc)
    (send *sr* :move-centroid-on-foot :both '(:rleg :lleg))
    (setq *root-rot-y-pose* (send *sr* :angle-vector)))
  (setq *pose-list* (list *half-sitting-pose* *root-rot-x-pose* *root-rot-y-pose*))
  (send *sr* :angle-vector *initial-pose*)
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000)
  (send *ri* :wait-interpolation)
  )

(defun samplerobot-auto-balancer-demo0 ()
  (print "1. AutoBalancer mode by fixing feet")
  (send *ri* :start-auto-balancer :limbs '(:rleg :lleg))
  (test-pose-list (list *arm-front-pose*) *initial-pose*)
  (send *ri* :stop-auto-balancer)
  (print "  Start and Stop AutoBalancer by fixing feet=>OK")
  t)

(defun samplerobot-auto-balancer-demo1 ()
  (print "2. AutoBalancer mode by fixing hands and feet")
  (send *ri* :start-auto-balancer)
  (test-pose-list (list *arm-front-pose*) *initial-pose*)
  (send *ri* :stop-auto-balancer)
  (print "  Start and Stop AutoBalancer by fixing hands and feet=>OK")
  t)

(defun samplerobot-auto-balancer-demo2 ()
  (print "3. getAutoBalancerParam")
  (pprint (send (send *ri* :get-auto-balancer-param) :slots))
  (print "  getAutoBalancerParam() => OK")
  t)

(defun samplerobot-auto-balancer-demo3 ()
  (print "4. setAutoBalancerParam")
  (let ((zmpoff (list (float-vector 100 0 0) (float-vector 100 0 0))))
    (send *ri* :set-auto-balancer-param :default-zmp-offsets zmpoff)
    (let ((dd (send (send (send *ri* :get-auto-balancer-param) :default_zmp_offsets) :data)))
      (if (and (eps-v= (car zmpoff) (scale 1e3 (subseq dd 0 3))) (eps-v= (cadr zmpoff) (scale 1e3 (subseq dd 3)))) ;; [m] => [mm]
          (print "  setAutoBalancerParam() => OK"))))
  (print "  default_zmp_offsets setting check in start and stop")
  (send *ri* :start-auto-balancer :limbs '(:rleg :lleg))
  (send *ri* :stop-auto-balancer)
  (send *ri* :set-auto-balancer-param :default-zmp-offsets (list (float-vector 0 0 0) (float-vector 0 0 0)))
  t)

(defun samplerobot-auto-balancer-demo4 ()
  (print "5. change base height, base rot x, base rot y, and upper body while AutoBalancer mode")
  (send *ri* :start-auto-balancer :limbs '(:rleg :lleg))
  (test-pose-list *pose-list* *initial-pose*)
  (send *ri* :stop-auto-balancer)
  )

(defun samplerobot-auto-balancer-demo5 ()
  (print "6. start stop check")
  (send *ri* :set-auto-balancer-param :default-zmp-offsets (list (float-vector -50 50 0) (float-vector -50 50 0)))
  (dolist (pose *pose-list*)
    (send *sr* :angle-vector pose)
    (send *ri* :angle-vector (send *sr* :angle-vector) 1000)
    (send *ri* :wait-interpolation)
    (send *ri* :start-auto-balancer :limbs '(:rleg :lleg))
    (send *ri* :stop-auto-balancer)
    )
  (send *ri* :set-auto-balancer-param :default-zmp-offsets (list (float-vector 0 0 0) (float-vector 0 0 0)))
  (send *ri* :angle-vector (send *sr* :angle-vector) 1000)
  (send *ri* :wait-interpolation)
  )

(defun samplerobot-auto-balancer-demo6 ()
  (print "7. balance against hand force")
  (send *ri* :start-auto-balancer :limbs '(:rleg :lleg))
  (send *ri* :set-ref-force (float-vector 0 0 -50) 1000 :rarm)
  (send *ri* :wait-interpolation-seq)
  (send *ri* :set-ref-force (float-vector 0 0 0) 1000 :rarm)
  (send *ri* :wait-interpolation-seq)
  (send *ri* :stop-auto-balancer)
  )

(defun samplerobot-auto-balancer-demo7 ()
  (print "1. goPos")
  (send *ri* :go-pos 0.1 0.05 20)
  (print "  goPos()=>OK")
  t)

(defun samplerobot-auto-balancer-demo8 ()
  (print "2. goVelocity and goStop")
  (send *ri* :go-velocity -0.1 -0.05 -20)
  (unix:sleep 1)
  (send *ri* :go-stop)
  (print "  goVelocity()=>OK")
  t)

(defun samplerobot-auto-balancer-demo9 ()
  (print "3. setFootSteps")
  (send *ri* :set-foot-steps
        (list (make-coords :pos (float-vector 0 -90 0) :name :rleg)
              (make-coords :pos (float-vector 0 90 0) :name :lleg)))
  (send *ri* :wait-foot-steps)
  (send *ri* :set-foot-steps
        (list (make-coords :pos (float-vector 0 -90 0) :name :rleg)
              (make-coords :pos (float-vector 150 90 0) :name :lleg)
              (make-coords :pos (float-vector 300 -90 0) :name :rleg)
              (make-coords :pos (float-vector 300 90 0) :name :lleg)))
  (send *ri* :wait-foot-steps)
  (print "  setFootSteps()=>OK")
  t)

(defun samplerobot-auto-balancer-demo10 ()
  (print "4. change base height, base rot x, base rot y, and upper body while walking")
  (send *ri* :start-auto-balancer)
  (send *ri* :go-velocity 0 0 0)
  (test-pose-list *pose-list* *initial-pose*)
  (send *ri* :go-stop)
  (send *ri* :stop-auto-balancer)
  t)

(defun samplerobot-auto-balancer-demo11 ()
  (print "5. getGaitGeneratorParam")
  (pprint (send (send *ri* :get-gait-generator-param) :slots))
  (print "  getGaitGeneratorParam() => OK")
  t)

(defun samplerobot-auto-balancer-demo12 ()
  (print "6. setGaitGeneratorParam")
  (let ((org-gp (send *ri* :get-gait-generator-param))
        (default-step-time 0.7)
        (default-step-height 0.15)
        (default-double-support-ratio 0.4)
        (default-orbit-type :RECTANGLE))
    (send *ri* :set-gait-generator-param
          :default-step-time default-step-time
          :default-step-height default-step-height
          :default-double-support-ratio default-double-support-ratio
          :default-orbit-type default-orbit-type)
    (let ((gp (send *ri* :get-gait-generator-param)))
      (if (and (eps= (send gp :default_step_time) default-step-time)
               (eps= (send gp :default_step_height) default-step-height)
               (eps= (send gp :default_double_support_ratio) default-double-support-ratio)
               (eq (send gp :default_orbit_type) 2)) ;; rectangle
          (print "  setGaitGeneratorParam() => OK")))
    (send *ri* :go-velocity 0.1 0 0)
    (unix:sleep 1)
    (send *ri* :go-stop)
    ;; stride-parameter
    (send *ri* :set-gait-generator-param :stride-parameter #f(0.15 0.05 10 0.05))
    (send *ri* :set-gait-generator-param :stride-parameter #f(0.15 0.05 10 0.05 0.025 5))
    (send *ri* :set-gait-generator-param
          :default-step-time (send org-gp :default_step_time)
          :default-step-height (send org-gp :default_step_height)
          :default-double-support-ratio (send org-gp :default_double_support_ratio)
          :default-orbit-type (send org-gp :default_orbit_type)
          )
    t))

(defun samplerobot-auto-balancer-demo13 ()
  (print "7. non-default stride")
  (send *ri* :start-auto-balancer)
  (send *ri* :set-foot-steps
        (list (make-coords :pos (float-vector 0 -90 0) :name :rleg)
              (make-coords :pos (float-vector 150 90 0) :name :lleg)))
  (send *ri* :wait-foot-steps)
  (send *ri* :set-foot-steps
        (list (make-coords :pos (float-vector 0 -90 0) :name :rleg)
              (make-coords :pos (float-vector 0 90 0) :name :lleg)))
  (send *ri* :wait-foot-steps)
  (send *ri* :stop-auto-balancer)
  (print "  Non default Stride()=>OK")
  t)

(defun samplerobot-auto-balancer-demo14 ()
  (print "8. Use toe heel contact")
  (send *ri* :start-auto-balancer)
  (send *ri* :set-gait-generator-param
        :toe-pos-offset-x (* 1e-3 182.0) :heel-pos-offset-x (* 1e-3 -72.0)
        :toe-zmp-offset-x (* 1e-3 182.0) :heel-zmp-offset-x (* 1e-3 -72.0)
        :toe-angle 20 :heel-angle 10)
  (send *ri* :go-pos 0.3 0 0)
  (send *ri* :stop-auto-balancer)
  (send *ri* :set-gait-generator-param
        :toe-angle 0 :heel-angle 0)
  (print "  Toe heel contact=>OK")
  t)

(defun samplerobot-auto-balancer-demo15 ()
  (print "9. Stop and start auto balancer sync check2")
  t)

(defun samplerobot-auto-balancer-demo16 ()
  (print "10. Emergency stop")
  (send *ri* :start-auto-balancer)
  (send *ri* :go-pos-no-wait 0 0 90)
  (print "  Start goPos and wait for 4 steps")
  (dotimes (i 4) ;; Wait for 4 steps including initial double support phase
    ;; Wait for 1 steps
    (send *ri* :angle-vector (send *sr* :angle-vector) (* 1e3 (send (send *ri* :get-gait-generator-param) :default_step_time)))
    (send *ri* :wait-interpolation))
  (print "  Emergency stoping")
  (send *ri* :emergency-walking-stop)
  (print "  Align foot steps")
  (send *ri* :go-pos 0 0 0)
  t)

(defun samplerobot-auto-balancer-demo17 ()
  (print "11. Get remaining foot steps")
  (send *sr* :fix-leg-to-coords (send *rI* :get-foot-step-param :dst-foot-midcoords))
  (objects (list *sr*))
  (send *ri* :go-pos-no-wait 0.3 0.1 15)
  (let ((fs-list (send *ri* :get-remaining-foot-step-sequence)))
    (while fs-list
      (setq fs-list (send *ri* :get-remaining-foot-step-sequence))
      (unless (or (null x::*display*) (= x::*display* 0))
        (send *irtviewer* :draw-objects :flush nil)
        (send *ri* :draw-remaining-foot-step-sequence (send *irtviewer* :viewer) :flush t))
      (send *ri* :angle-vector (send *sr* :angle-vector) (* 1e3 (send (send *ri* :get-gait-generator-param) :default_step_time)))
      (send *ri* :wait-interpolation))
    t))

(defun samplerobot-auto-balancer-demo18 ()
  (print "12. Change step param with setFootSteps")
  (send *ri* :get-gait-generator-param)
  ;; Dummy settting just for testing
  (send *ri* :set-gait-generator-param :toe-angle 50 :heel-angle 50 :default-step-time 4.0 :default-step-height 0.0)
  ;; Step time change
  (send *ri* :set-foot-steps-with-param
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :lleg))
        (list 0 50 50 50)
        (list 1.0 2.0 1.0 2.0)
        (list 0 0 0 0)
        (list 0 0 0 0))
  ;; Step height change
  (send *ri* :set-foot-steps-with-param
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :lleg))
        (list 0 100 50 100)
        (list 1.0 1.0 1.0 1.0)
        (list 0 0 0 0)
        (list 0 0 0 0))
  ;; Toe heel change
  (send *ri* :set-foot-steps-with-param
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :lleg))
        (list 0 50 50 50)
        (list 1.0 1.0 1.0 1.0)
        (list 0 0 20 10)
        (list 0 0 50 10))
  (send *ri* :set-gait-generator-param :toe-angle 0 :heel-angle 0 :default-step-time 1.0 :default-step-height 0.05)
  )

(defun samplerobot-auto-balancer-demo19-old ()
  (print "13. Overwrite footsteps during walking.")
  (send *ri* :set-foot-steps-no-wait
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              ))
  ;; used for waiting
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000)
  (send *ri* :wait-interpolation)
  ;; overwriting
  (send *ri* :set-foot-steps-no-wait
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              ))
  ;; used for waiting
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000)
  (send *ri* :wait-interpolation)
  ;; overwrite
  (send *ri* :set-foot-steps-no-wait
        (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :rleg)
              (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 200 100 0)) :name :lleg)
              ))
  (send *ri* :wait-foot-steps)
  ;; memo
  ;; Current support leg is obtained by (send *ri* :get-foot-step-param :support-leg) and current support leg coords is obtained by (send *ri* :get-foot-step-param :support-leg-coords)
  ;; In normal case, next support leg is obtained (case (send *ri* :get-foot-step-param :support-leg) (:rleg :lleg) (:lleg :rleg)) and next support leg coords is obtained by (send *ri* :get-foot-step-param :swing-leg-dst-coords)
  )

(defun samplerobot-auto-balancer-demo19 (&key (overwrite-offset-idx 1))
  (print "13. Overwrite footsteps during walking.")
  (send *ri* :start-auto-balancer)
  (samplerobot-auto-balancer-demo19-base :x overwrite-offset-idx t)
  (send *ri* :angle-vector (send *sr* :angle-vector) (* 1000 overwrite-offset-idx))
  (send *ri* :wait-interpolation)
  (samplerobot-auto-balancer-demo19-base :y overwrite-offset-idx t)
  (send *ri* :angle-vector (send *sr* :angle-vector) (* 1000 overwrite-offset-idx))
  (send *ri* :wait-interpolation)
  (samplerobot-auto-balancer-demo19-base :x overwrite-offset-idx t)
  (send *ri* :angle-vector (send *sr* :angle-vector) (* 1000 overwrite-offset-idx))
  (send *ri* :wait-interpolation)
  (send *ri* :wait-foot-steps)
  )

(defun samplerobot-auto-balancer-demo19-base (axis overwrite-offset-idx init-fs)
  (when init-fs
    (send *ri* :set-foot-steps-no-wait
          (list (make-coords :coords (send *sr* :rleg :end-coords :copy-worldcoords) :name :rleg)
                (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
                (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
                (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 300 0 0)) :name :lleg)
                (make-coords :coords (send (send *sr* :rleg :end-coords :copy-worldcoords) :translate (float-vector 400 0 0)) :name :rleg)
                (make-coords :coords (send (send *sr* :lleg :end-coords :copy-worldcoords) :translate (float-vector 400 0 0)) :name :lleg)
                )))
  (print ";;   Overwrite footsteps")
  (let* ((ret (send *ri* :get-remaining-foot-step-sequence-current-index))
         (remain-fs (car ret))
         (current-index (cadr ret)))
    (format t ";;    Remaining legs = ~A~%" (send-all remain-fs :name))
    (format t ";;    Remaining idx = ~A~%" (let ((idx -1)) (mapcar #'(lambda (x) (+ current-index (incf idx))) remain-fs)))
    (let* ((overwrite-footstep-index (+ current-index overwrite-offset-idx))
           (support-fs (elt remain-fs (- overwrite-offset-idx 1))) ;; support fs before ovewritten fs
           (pos-offset1) (pos-offset2))
      (format t ";;    Overwrite index = ~A, leg = ~A, current index = ~A~%"
              overwrite-footstep-index (send (elt remain-fs overwrite-offset-idx) :name)
              current-index)
      (case axis
            (:x
             (setq pos-offset1 (float-vector 100 0 0) pos-offset2 (float-vector 200 0 0)))
            (t
             (setq pos-offset1 (float-vector 0 (case (send support-fs :name) (:rleg 100) (:lleg -100)) 0)
                   pos-offset2 pos-offset1)))
      (let ((new-foot-steps
             (list
              (make-coords :coords (send support-fs :copy-worldcoords) :name (send support-fs :name))
              (make-coords :coords (send (send support-fs :copy-worldcoords)
                                         :translate (v+ pos-offset1 (float-vector 0 (case (send support-fs :name) (:rleg (* 2 90)) (:lleg (* 2 -90))) 0)))
                           :name (case (send support-fs :name) (:rleg :lleg) (:lleg :rleg)))
              (make-coords :coords (send (send support-fs :copy-worldcoords)
                                         :translate pos-offset1)
                           :name (send support-fs :name))
              (make-coords :coords (send (send support-fs :copy-worldcoords)
                                         :translate (v+ pos-offset2 (float-vector 0 (case (send support-fs :name) (:rleg (* 2 90)) (:lleg (* 2 -90))) 0)))
                           :name (case (send support-fs :name) (:rleg :lleg) (:lleg :rleg)))
              (make-coords :coords (send (send support-fs :copy-worldcoords)
                                         :translate pos-offset2)
                           :name (send support-fs :name)))))
        (send *ri* :set-foot-steps-no-wait
              new-foot-steps :overwrite-footstep-index overwrite-footstep-index))))
  ;; memo
  ;; Current support leg is obtained by (send *ri* :get-foot-step-param :support-leg) and current support leg coords is obtained by (send *ri* :get-foot-step-param :support-leg-coords)
  ;; In normal case, next support leg is obtained (case (send *ri* :get-foot-step-param :support-leg) (:rleg :lleg) (:lleg :rleg)) and next support leg coords is obtained by (send *ri* :get-foot-step-param :swing-leg-dst-coords)
  )

(defun samplerobot-auto-balancer-demo20 (&key (reset-standing-pos nil))
  (print ";; 15. Overwrite current footstep")
  (if reset-standing-pos (demo-reset-standing-pos))
  (send *ri* :angle-vector (send *sr* :angle-vector) 1000)
  (send *ri* :wait-interpolation)
  (send *ri* :start-auto-balancer)
  (send *ri* :set-gait-generator-param :overwritable-footstep-index-offset 0 :default-orbit-type :rectangle)
  (send *ri* :go-velocity 0 0 0)
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000) (send *ri* :wait-interpolation) ;; wait 2 step using dummy waitInterpolation
  (send *ri* :go-velocity 0.1 0 0)
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000) (send *ri* :wait-interpolation) ;; wait 2 step using dummy waitInterpolation
  (send *ri* :go-velocity 0 0.1 0)
  (send *ri* :angle-vector (send *sr* :angle-vector) 2000) (send *ri* :wait-interpolation) ;; wait 2 step using dummy waitInterpolation
  (send *ri* :go-stop)
  (send *ri* :set-gait-generator-param :overwritable-footstep-index-offset 1 :default-orbit-type :cycloiddelay)
  )

(defun samplerobot-auto-balancer-demo21 ()
  (print ";; 16. goPos overwriting")
  (send *ri* :start-auto-balancer)
  (print ";;  Overwrite goPos by goPos")
  (let ((goalx 0.3) (goaly 0.1) (goalth 15.0))
    (send *ri* :go-pos-no-wait 0.2 -0.1 -5) ;; initial gopos
    (send *ri* :angle-vector (send *sr* :angle-vector) 2000) (send *ri* :wait-interpolation) ;; wait 2 step using dummy waitInterpolation
    (send *ri* :go-pos-no-wait goalx goaly goalth) ;; overwrite gopos
    (send *ri* :wait-foot-steps)
    )
  (print ";;  Overwrite setFootSteps by goPos")
  (let ((goalx 0.3) (goaly 0.1) (goalth 15.0))
    (send *ri* :set-foot-steps-no-wait
          (list (make-coords :coords (send *robot* :rleg :end-coords :copy-worldcoords) :name :rleg)
                (make-coords :coords (send (send *robot* :lleg :end-coords :copy-worldcoords) :translate (float-vector 100 0 0)) :name :lleg)
                (make-coords :coords (send (send *robot* :rleg :end-coords :copy-worldcoords) :translate (float-vector 200 0 0)) :name :rleg)
                (make-coords :coords (send (send *robot* :lleg :end-coords :copy-worldcoords) :translate (float-vector 300 0 0)) :name :lleg)
                (make-coords :coords (send (send *robot* :rleg :end-coords :copy-worldcoords) :translate (float-vector 300 0 0)) :name :rleg)))
    (send *ri* :angle-vector (send *sr* :angle-vector) 2000) (send *ri* :wait-interpolation) ;; wait 2 step using dummy waitInterpolation
    (send *ri* :go-pos-no-wait goalx goaly goalth) ;; overwrite gopos
    (send *ri* :wait-foot-steps)
    )
  )

(defun samplerobot-auto-balancer-demo22 ()
  (print ";; 17. Graspless manip mode")
  (labels ((ik-pose-gen
            (rarm-target-coords larm-target-coords)
            (send *robot* :reset-manip-pose)
            (send *robot* :fix-leg-to-coords (make-coords))
            (send *robot* :fullbody-inverse-kinematics
                  (list rarm-target-coords larm-target-coords
                        (send *robot* :rleg :end-coords :copy-worldcoords)
                        (send *robot* :lleg :end-coords :copy-worldcoords))
                  :move-target (mapcar #'(lambda (x) (send *robot* x :end-coords)) '(:rarm :larm :rleg :lleg))
                  :link-list (mapcar #'(lambda (x) (send *robot* :link-list (send *robot* x :end-coords :parent))) '(:rarm :larm :rleg :lleg)))))
    ;; gen pose
    (let ((dualarm-push-pose) (av-fwd) (av-bwd) (av-left) (av-right) (av-lturn) (av-rturn))
      (let ((r-handle-coords (make-coords :pos (float-vector 480 -220 870) :rpy (list 0 0 pi/2)))
            (l-handle-coords (make-coords :pos (float-vector 480  220 870) :rpy (list 0 0 -pi/2))))
        (ik-pose-gen
         (make-coords :pos (float-vector 480 -220 870) :rpy (list 0 0 pi/2))
         (make-coords :pos (float-vector 480  220 870) :rpy (list 0 0 -pi/2)))
        (setq dualarm-push-pose (send *robot* :angle-vector))
        (ik-pose-gen
         (make-coords :pos (float-vector (+ 50 480) -220 870) :rpy (list 0 0 pi/2))
         (make-coords :pos (float-vector (+ 50 480)  220 870) :rpy (list 0 0 -pi/2)))
        (setq av-fwd (send *robot* :angle-vector))
        (ik-pose-gen
         (make-coords :pos (float-vector (+ -50 480) -220 870) :rpy (list 0 0 pi/2))
         (make-coords :pos (float-vector (+ -50 480)  220 870) :rpy (list 0 0 -pi/2)))
        (setq av-bwd (send *robot* :angle-vector))
        (ik-pose-gen
         (make-coords :pos (float-vector 480 (+ 50 -220) 870) :rpy (list 0 0 pi/2))
         (make-coords :pos (float-vector 480 (+ 50  220) 870) :rpy (list 0 0 -pi/2)))
        (setq av-left (send *robot* :angle-vector))
        (ik-pose-gen
         (make-coords :pos (float-vector 480 (+ -50 -220) 870) :rpy (list 0 0 pi/2))
         (make-coords :pos (float-vector 480 (+ -50  220) 870) :rpy (list 0 0 -pi/2)))
        (setq av-right (send *robot* :angle-vector))
        (let* ((rc (make-cascoords :pos (float-vector 480 -220 870) :rpy (list 0 0 pi/2)))
               (lc (make-cascoords :pos (float-vector 480  220 870) :rpy (list 0 0 -pi/2)))
               (mc (make-cascoords :pos (midpoint 0.5 (send rc :worldpos) (send lc :worldpos)))))
          (dolist (cc (list rc lc)) (send mc :assoc cc))
          (send mc :rotate (deg2rad 10) :z)
          (ik-pose-gen rc lc)
          (setq av-lturn (send *robot* :angle-vector))
          (send mc :rotate (deg2rad -20) :z)
          (ik-pose-gen rc lc)
          (setq av-rturn (send *robot* :angle-vector))
          (dolist (cc (list rc lc)) (send mc :dissoc cc))
          ))
      (send *ri* :start-auto-balancer)
      (send *robot* :angle-vector dualarm-push-pose)
      (send *ri* :angle-vector (send *robot* :angle-vector) 500)
      (send *ri* :wait-interpolation)
      (send *ri* :stop-collision-detection)
      (send *ri* :start-graspless-manip-mode *robot* :arms)
      (send *ri* :set-auto-balancer-param :graspless-manip-p-gain #f(1 1 1))
      (send *ri* :wait-foot-steps)
      (mapcar #'(lambda (av)
                  (send *ri* :go-velocity 0 0 0)
                  (send *robot* :angle-vector av)
                  (send *ri* :angle-vector (send *robot* :angle-vector) 400)
                  (send *ri* :wait-interpolation)
                  ;; dummy 2step
                  (send *ri* :angle-vector (send *robot* :angle-vector) 1600)
                  (send *ri* :wait-interpolation)
                  (send *ri* :go-stop))
              (list av-fwd av-bwd av-left av-right av-lturn av-rturn))
      (send *ri* :start-collision-detection)
      (send *robot* :angle-vector dualarm-push-pose)
      (send *ri* :angle-vector (send *robot* :angle-vector) 0.5)
      (send *ri* :wait-interpolation)
      )))

(defun samplerobot-auto-balancer-demo23 ()
  (print ";; 18. Get footsteps of go-pos")
  (let* ((goalx 0.3) (goaly 0.15) (goalth 60)
         (fsll (send *ri* :get-go-pos-footsteps-sequence goalx goaly goalth)))
    (pprint fsll)
    (print ";; Execute this footsteps")
    (send *ri* :set-foot-steps fsll)
    ))

(defun demo-reset-standing-pos ()
  (send *ri* :go-pos 0 0 -45)
  (send *ri* :go-pos 0 -1.0 0)))

(defun samplerobot-auto-balancer-demo ()
  (samplerobot-auto-balancer-init)

  ;; sample for AutoBalancer mode
  (samplerobot-auto-balancer-demo0)
  (samplerobot-auto-balancer-demo1)
  (samplerobot-auto-balancer-demo2)
  (samplerobot-auto-balancer-demo3)
  (samplerobot-auto-balancer-demo4)
  (samplerobot-auto-balancer-demo5)
  (samplerobot-auto-balancer-demo6)

  ;; sample for walk pattern generation by AutoBalancer RTC
  (samplerobot-auto-balancer-demo7)
  (samplerobot-auto-balancer-demo8)
  (samplerobot-auto-balancer-demo9)
  (samplerobot-auto-balancer-demo10)
  (samplerobot-auto-balancer-demo11)
  (samplerobot-auto-balancer-demo12)
  (samplerobot-auto-balancer-demo13)
  (samplerobot-auto-balancer-demo14)
  (samplerobot-auto-balancer-demo15)
  (samplerobot-auto-balancer-demo16)
  (samplerobot-auto-balancer-demo17)
  (samplerobot-auto-balancer-demo18)
  (samplerobot-auto-balancer-demo19)
  (samplerobot-auto-balancer-demo19 :overwrite-offset-idx 2)
  (samplerobot-auto-balancer-demo20 :reset-standing-pos t)
  (samplerobot-auto-balancer-demo21)
  (samplerobot-auto-balancer-demo22)
  (samplerobot-auto-balancer-demo23)
  )

(warn ";; (samplerobot-auto-balancer-demo)~%")
