// License: Apache 2.0. See LICENSE file in root directory.
// Copyright(c) 2015 Intel Corporation. All Rights Reserved.

#ifndef LIBREALSENSE_RS_HPP
#define LIBREALSENSE_RS_HPP

#include "rsutil.h"
#include "rscore.hpp"
#include <cmath>
#include <cstdint>
#include <cstring>
#include <sstream>
#include <stdexcept>
#include <functional>
#include <vector>

namespace rs
{
    /* streams are different types of data provided by RealSense devices */
    enum class stream : int32_t
    {
        depth                           ,  /**< Native stream of depth data produced by RealSense device */
        color                           ,  /**< Native stream of color data captured by RealSense device */
        infrared                        ,  /**< Native stream of infrared data captured by RealSense device */
        infrared2                       ,  /**< Native stream of infrared data captured from a second viewpoint by RealSense device */
        fisheye                         ,  /**< Native stream of fish-eye (wide) data captured from the dedicate motion camera */
        points                          ,  /**< Synthetic stream containing point cloud data generated by deprojecting the depth image */
        rectified_color                 ,  /**< Synthetic stream containing undistorted color data with no extrinsic rotation from the depth stream */
        color_aligned_to_depth          ,  /**< Synthetic stream containing color data but sharing intrinsic of depth stream */
        infrared2_aligned_to_depth      ,  /**< Synthetic stream containing second viewpoint infrared data but sharing intrinsic of depth stream */
        depth_aligned_to_color          ,  /**< Synthetic stream containing depth data but sharing intrinsic of color stream */
        depth_aligned_to_rectified_color,  /**< Synthetic stream containing depth data but sharing intrinsic of rectified color stream */
        depth_aligned_to_infrared2         /**< Synthetic stream containing depth data but sharing intrinsic of second viewpoint infrared stream */
    };

    /*  formats define how each stream can be encoded
        format is closely relateed to Linux pixel-formats, but is trying to abstract away the platform specific constructs */
    enum class format : int32_t
    {
        any         ,  /**< When passed to enable stream, librealsense will try to provide best suited format */
        z16         ,  /**< 16 bit linear depth values. The depth is meters is equal to depth scale * pixel value */
        disparity16 ,  /**< 16 bit linear disparity values. The depth in meters is equal to depth scale / pixel value */
        xyz32f      ,  /**< 32 bit floating point 3D coordinates. */
        yuyv        ,  /**< Standard YUV pixel format as described in https://en.wikipedia.org/wiki/YUV */
        rgb8        ,  /**< 8-bit Red, Green and Blue channels */
        bgr8        ,  /**< 8-bit Blue, Green and Red channels, suitable for OpenCV */
        rgba8       ,  /**< 8-bit Red, Green, Blue channels + constant alpha channel equal to FF */
        bgra8       ,  /**< 8-bit Blue, Green, Red channels + constant alpha channel equal to FF */
        y8          ,  /**< 8-bit per-pixel grayscale image */
        y16         ,  /**< 16-bit per-pixel grayscale image */
        raw10       ,  /**< Four 10-bit luminance values encoded into a 5-byte macropixel */
        raw16       ,  /**< 16-bit raw image */
        raw8           /**< 8-bit raw image */
    };

    /* Output buffer format sets how librealsense will work with frame memory */
    enum class output_buffer_format : int32_t
    {
        continous      , /**< Makes sure that the output frame is exposed as a single continuous buffer */
        native           /**< Don't convert buffer to continuous, the user has to handle pitch manually */
    };

    /* Presets hint general preference that is translated by librealsense into concrete resultion and fps */
    enum class preset : int32_t
    {
        best_quality     ,/**< Prefer best overall quality */
        largest_image    ,/**< Prefer largest image size */
        highest_framerate /**< Prefer highest framerate */
    };

    /* Distortion model define how pixel coordinates should be mapped to sensor coordinates */
    enum class distortion : int32_t
    {
        none                  , /**< Rectilinear images, no distortion compensation required */
        modified_brown_conrady, /**< Equivalent to Brown-Conrady distortion, except that tangential distortion is applied to radially distorted points */
        inverse_brown_conrady,  /**< Equivalent to Brown-Conrady distortion, except undistorts image instead of distorting it */
        distortion_ftheta       /**< Distortion model of the fish-eye camera */
    };

    /* Camera options define general configuration controls.
    These can generally be mapped to camera UVC controls, and unless stated otherwise can be set/queried at any time */
    enum class option : int32_t
    {
        color_backlight_compensation                    ,  /**< Enable / disable color backlight compensation*/
        color_brightness                                ,  /**< Color image brightness*/
        color_contrast                                  ,  /**< Color image contrast*/
        color_exposure                                  ,  /**< Controls exposure time of color camera. Setting any value will disable auto exposure*/
        color_gain                                      ,  /**< Color image gain*/
        color_gamma                                     ,  /**< Color image gamma setting*/
        color_hue                                       ,  /**< Color image hue*/
        color_saturation                                ,  /**< Color image saturation setting*/
        color_sharpness                                 ,  /**< Color image sharpness setting*/
        color_white_balance                             ,  /**< Controls white balance of color image. Setting any value will disable auto white balance*/
        color_enable_auto_exposure                      , /**< Enable / disable color image auto-exposure*/
        color_enable_auto_white_balance                 , /**< Enable / disable color image auto-white-balance*/
        f200_laser_power                                , /**< Power of the F200 / SR300 projector, with 0 meaning projector off*/
        f200_accuracy                                   , /**< Set the number of patterns projected per frame. The higher the accuracy value the more patterns projected. Increasing the number of patterns help to achieve better accuracy. Note that this control is affecting the Depth FPS */
        f200_motion_range                               , /**< Motion vs. Range trade-off, with lower values allowing for better motion sensitivity and higher values allowing for better depth range*/
        f200_filter_option                              , /**< Set the filter to apply to each depth frame. Each one of the filter is optimized per the application requirements*/
        f200_confidence_threshold                       , /**< The confidence level threshold used by the Depth algorithm pipe to set whether a pixel will get a valid range or will be marked with invalid range*/
        f200_dynamic_fps                                , /**< (F200-only) Allows to reduce FPS without restarting streaming. Valid values are {2, 5, 15, 30, 60}*/
        sr300_auto_range_enable_motion_versus_range     , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_enable_laser                   , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_min_motion_versus_range        , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_max_motion_versus_range        , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_start_motion_versus_range      , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_min_laser                      , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_max_laser                      , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_start_laser                    , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_upper_threshold                , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        sr300_auto_range_lower_threshold                , /**< Configures SR300 Depth Auto-Range setting. Should not be used directly but through set IVCAM preset method*/
        r200_lr_auto_exposure_enabled                   , /**< Enables / disables R200 auto-exposure. This will affect both IR and depth image.*/
        r200_lr_gain                                    , /**< IR image gain*/
        r200_lr_exposure                                , /**< This control allows manual adjustment of the exposure time value for the L/R imagers*/
        r200_emitter_enabled                            , /**< Enables / disables R200 emitter*/
        r200_depth_units                                , /**< Micrometers per increment in integer depth values, 1000 is default (mm scale). Set before streaming*/
        r200_depth_clamp_min                            , /**< Minimum depth in current depth units that will be output. Any values less than Min Depth will be mapped to 0 during the conversion between disparity and depth. Set before streaming*/
        r200_depth_clamp_max                            , /**< Maximum depth in current depth units that will be output. Any values greater than Max Depth will be mapped to 0 during the conversion between disparity and depth. Set before streaming*/
        r200_disparity_multiplier                       , /**< The disparity scale factor used when in disparity output mode. Can only be set before streaming*/
        r200_disparity_shift                            , /**< {0 - 512}. Can only be set before streaming starts*/
        r200_auto_exposure_mean_intensity_set_point     , /**< (Requires LR-Auto-Exposure ON) Mean intensity set point*/
        r200_auto_exposure_bright_ratio_set_point       , /**< (Requires LR-Auto-Exposure ON) Bright ratio set point*/
        r200_auto_exposure_kp_gain                      , /**< (Requires LR-Auto-Exposure ON) Kp Gain*/
        r200_auto_exposure_kp_exposure                  , /**< (Requires LR-Auto-Exposure ON) Kp Exposure*/
        r200_auto_exposure_kp_dark_threshold            , /**< (Requires LR-Auto-Exposure ON) Kp Dark Threshold*/
        r200_auto_exposure_top_edge                     , /**< (Requires LR-Auto-Exposure ON) Auto-Exposure region-of-interest top edge (in pixels)*/
        r200_auto_exposure_bottom_edge                  , /**< (Requires LR-Auto-Exposure ON) Auto-Exposure region-of-interest bottom edge (in pixels)*/
        r200_auto_exposure_left_edge                    , /**< (Requires LR-Auto-Exposure ON) Auto-Exposure region-of-interest left edge (in pixels)*/
        r200_auto_exposure_right_edge                   , /**< (Requires LR-Auto-Exposure ON) Auto-Exposure region-of-interest right edge (in pixels)*/
        r200_depth_control_estimate_median_decrement    , /**< Value to subtract when estimating the median of the correlation surface*/
        r200_depth_control_estimate_median_increment    , /**< Value to add when estimating the median of the correlation surface*/
        r200_depth_control_median_threshold             , /**< A threshold by how much the winning score must beat the median*/
        r200_depth_control_score_minimum_threshold      , /**< The minimum correlation score that is considered acceptable*/
        r200_depth_control_score_maximum_threshold      , /**< The maximum correlation score that is considered acceptable*/
        r200_depth_control_texture_count_threshold      , /**< A parameter for determining whether the texture in the region is sufficient to justify a depth result*/
        r200_depth_control_texture_difference_threshold , /**< A parameter for determining whether the texture in the region is sufficient to justify a depth result*/
        r200_depth_control_second_peak_threshold        , /**< A threshold on how much the minimum correlation score must differ from the next best score*/
        r200_depth_control_neighbor_threshold           , /**< Neighbor threshold value for depth calculation*/
        r200_depth_control_lr_threshold                 , /**< Left-Right threshold value for depth calculation*/
        fisheye_exposure                                , /**< Fisheye image exposure time in msec*/
        fisheye_gain                                    , /**< Fisheye image gain*/
        fisheye_strobe                                  , /**< Enables / disables fisheye strobe. When enabled this will align timestamps to common clock-domain with the motion events*/
        fisheye_external_trigger                        , /**< Enables / disables fisheye external trigger mode. When enabled fisheye image will be aquired in-sync with the depth image*/
        fisheye_color_auto_exposure                     , /**< Enable / disable fisheye auto-exposure */
        fisheye_color_auto_exposure_mode                , /**< 0 - static auto-exposure, 1 - anti-flicker auto-exposure, 2 - hybrid */
        fisheye_color_auto_exposure_rate                , /**< Fisheye auto-exposure anti-flicker rate, can be 50 or 60 Hz */
        fisheye_color_auto_exposure_sample_rate         , /**< In Fisheye auto-exposure sample frame every given number of pixels */
        fisheye_color_auto_exposure_skip_frames         , /**< In Fisheye auto-exposure sample every given number of frames */
        frames_queue_size                               , /**< Number of frames the user is allowed to keep per stream. Trying to hold-on to more frames will cause frame-drops.*/
        hardware_logger_enabled                         , /**< Enable / disable fetching log data from the device */
        total_frame_drops                               , /**< Total number of detected frame drops from all streams*/
    };

    /* frame metadata enum discriminates between the different types of values provided from the device with each frame */
    enum class frame_metadata
    {
        actual_exposure, /**< actual exposure that the frame was captured with */
        actual_fps       /**< actual fps at the time of capture */
    };

    /* rs_capabilities defines the full set of functionality that a RealSense device might provide
    to check what functionality is supported by a particular device at runtime call dev->supports(capability) */
    enum class capabilities : int32_t
    {
        depth,                      /**< provides depth stream */
        color,                      /**< provides color stream */
        infrared,                   /**< provides infrared stream */
        infrared2,                  /**< provides second infrared stream */
        fish_eye,                   /**< provides wide field of view (fish-eye) stream */
        motion_events,              /**< provides gyro and accelorometer events */
        motion_module_fw_update,    /**< provides method for upgrading motion module firmware */
        adapter_board,              /**< interanlly includes MIPI to USB adapter */
        enumeration,                /**< provides enough basic functionality to be considered supported. this to catch at runtime various outdated engineering samples */
    };

    /* This type defines proprietary formats for direct communication with device firmware */
    enum class blob_type 
    {
        motion_module_firmware_update /**< By using this option, new firmware can be uploaded to the ZR300 motion-module */
    };

    /* Camera Info feilds are read-only strings that can be queried from the device
       Not all info feilds are available on all camera types
       This information is mainly available for camera debug and troubleshooting and should not be used in applications */
    enum class camera_info {
        device_name                   , /**< Device friendly name */
        serial_number                 , /**< Device serial number */
        camera_firmware_version       , /**< Primary firmware version */
        adapter_board_firmware_version, /**< MIPI-to-USB Adapter board firmware version if such board is present */
        motion_module_firmware_version, /**< Motion module firmware version if motion module is present */
        camera_type                   , /**< R200 / LR200 / ZR300 camera type */
        oem_id                        , /**< OEM ID */
        isp_fw_version                , /**< ISP firmware version when available */
        content_version               , /**< R200 / LR200 / ZR300 content version */
        module_version                , /**< R200 / LR200 / ZR300 module version */
        imager_model_number           , /**< Primary imager model number */
        build_date                    , /**< Device build date */
        calibration_date              , /**< Primary calibration date */
        program_date                  , /**< R200 / LR200 / ZR300 Program date */
        focus_alignment_date          , /**< Focus calibration date */
        emitter_type                  , /**< R200 / LR200 / ZR300 emitter type */
        focus_value                   , /**< Result of the focus calibration */
        lens_type                     , /**< Primary lens type */
        third_lens_type               , /**< Color imager lens type */
        lens_coating_type             , /**< Lens coating type */
        third_lens_coating_type       , /**< Color coating type */
        lens_nominal_baseline         , /**< Nominal baseline */
        third_lens_nominal_baseline     /**< Color Nominal baseline */
    };

    /* Source allows the user to choose between available hardware subdevices */
    enum class source : uint8_t
    {
        video      ,/**< Video streaming of depth, infrared, color or fish-eye */
        motion_data,/**< Motion tracking from Gyro and Accelerometer */
        all_sources,/**< Enable everything together */
    };

    /* Source device that tiggered specific timestamp event from the motion module */
    enum class event : uint8_t
    {
        event_imu_accel     , /**< Event from accelerometer */
        event_imu_gyro      , /**< Event from the gyroscope */
        event_imu_depth_cam , /**< Event from depth camera (depth / IR frame) */
        event_imu_motion_cam, /**< Event from the fish-eye camera */
        event_imu_g0_sync   , /**< Event from external GPIO 0 */
        event_imu_g1_sync   , /**< Event from external GPIO 1 */
        event_imu_g2_sync     /**< Event from external GPIO 2 */
    };

    /* When working with motion microcontroller, motion data timestamps are always in microcontroller timestamp domain.
       Some frames, however, might not succesfully receive microcontroller timestamp and will be marked as camera domain */
    enum class timestamp_domain
    {
        camera,         /**< Frame timestamp was measured in relation to the camera clock */
        microcontroller /**< Frame timestamp was measured in relation to the microcontroller clock */
    };

    struct float2 { float x,y; };
    struct float3 { float x,y,z; };

    /* Video stream intrinsics */
    struct intrinsics : rs_intrinsics
    {
        float       hfov() const                                                        { return (atan2f(ppx + 0.5f, fx) + atan2f(static_cast<float>(width) - (ppx + 0.5f), fx)) * 57.2957795f; }
        float       vfov() const                                                        { return (atan2f(ppy + 0.5f, fy) + atan2f(static_cast<float>(height) - (ppy + 0.5f), fy)) * 57.2957795f; }
        distortion  model() const                                                       { return (distortion)rs_intrinsics::model; }

                    // Helpers for mapping between pixel coordinates and texture coordinates
        float2      pixel_to_texcoord(const float2 & pixel) const                       { return {(pixel.x+0.5f)/static_cast<float>(width), (pixel.y+0.5f)/static_cast<float>(height)}; }
        float2      texcoord_to_pixel(const float2 & coord) const                       { return {coord.x*static_cast<float>(width) - 0.5f, coord.y*static_cast<float>(height) - 0.5f}; }

                    // Helpers for mapping from image coordinates into 3D space
        float3      deproject(const float2 & pixel, float depth) const                  { float3 point = {}; rs_deproject_pixel_to_point(&point.x, this, &pixel.x, depth); return point; }
        float3      deproject_from_texcoord(const float2 & coord, float depth) const    { return deproject(texcoord_to_pixel(coord), depth); }

                    // Helpers for mapping from 3D space into image coordinates
        float2      project(const float3 & point) const                                 { float2 pixel = {}; rs_project_point_to_pixel(&pixel.x, this, &point.x); return pixel; }
        float2      project_to_texcoord(const float3 & point) const                     { return pixel_to_texcoord(project(point)); }

        bool        operator == (const intrinsics & r) const                            { return memcmp(this, &r, sizeof(r)) == 0; }

    };

    /* represents motion device intrinsic - scale, bias and variances */
    struct motion_intrinsics : rs_motion_intrinsics
    {
        motion_intrinsics(){};
    };

    /* Cross-stream extrinsics, encode the topology of how the different devices are connected */
    struct extrinsics : rs_extrinsics
    {
        bool        is_identity() const                                                 { return (rotation[0] == 1) && (rotation[4] == 1) && (translation[0] == 0) && (translation[1] == 0) && (translation[2] == 0); }
        float3      transform(const float3 & point) const                               { float3 p = {}; rs_transform_point_to_point(&p.x, this, &point.x); return p; }
    };

    /* Timestamp data from the motion microcontroller */
    struct timestamp_data : rs_timestamp_data
    {
        timestamp_data(rs_timestamp_data orig) : rs_timestamp_data(orig) {}
        timestamp_data() {}
    };

    /* Motion data from Gyro / Accel from the microcontroller */
    struct motion_data : rs_motion_data
    {
        motion_data(rs_motion_data orig) : rs_motion_data(orig) {}
        motion_data() {}
    };

    class context;
    class device;
    
    class error : public std::runtime_error
    {
        std::string function, args;
    public:
        error(rs_error * err) : std::runtime_error(rs_get_error_message(err))
        { 
            function = (nullptr != rs_get_failed_function(err)) ? rs_get_failed_function(err) : std::string();
            args = (nullptr != rs_get_failed_args(err)) ? rs_get_failed_args(err) : std::string();
            rs_free_error(err);
        }
        const std::string & get_failed_function() const { return function; }
        const std::string & get_failed_args() const { return args; }
        static void handle(rs_error * e) { if(e) throw error(e); }
    };

    class context
    {
        rs_context * handle;
        context(const context &) = delete;
        context & operator = (const context &) = delete;
    public:

        // create realsense context required for rest of the API
        context()
        {
            rs_error * e = nullptr;
            handle = rs_create_context(RS_API_VERSION, &e);
            error::handle(e);
        }

        explicit context(rs_context * handle) : handle(handle) {}

        ~context()
        {
            rs_delete_context(handle, nullptr);
        }

        /// determine number of connected devices
        /// \return  the count of devices
        int get_device_count() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_count(handle, &e);
            error::handle(e);
            return r;
        }

        /// retrieve connected device by index
        /// \param[in] index  the zero based index of device to retrieve
        /// \return           the requested device
        device * get_device(int index)
        {
            rs_error * e = nullptr;
            auto r = rs_get_device(handle, index, &e);
            error::handle(e);
            return (device *)r;
        }
    };  

    class motion_callback : public rs_motion_callback
    {
        std::function<void(motion_data)> on_event_function;
    public:
        explicit motion_callback(std::function<void(motion_data)> on_event) : on_event_function(on_event) {}

        void on_event(rs_motion_data e) override
        {
            on_event_function(motion_data(e));
        }

        void release() override { delete this; }
    };

    class timestamp_callback : public rs_timestamp_callback
    {
        std::function<void(timestamp_data)> on_event_function;
    public:
        explicit timestamp_callback(std::function<void(timestamp_data)> on_event) : on_event_function(on_event) {}

        void on_event(rs_timestamp_data data) override
        {
            on_event_function(std::move(data));
        }

        void release() override { delete this; }
    };

    class frame
    {
        rs_device * device;
        rs_frame_ref * frame_ref;

        frame(const frame &) = delete;

    public:
        frame() : device(nullptr), frame_ref(nullptr) {}
        frame(rs_device * device, rs_frame_ref * frame_ref) : device(device), frame_ref(frame_ref) {}
        frame(frame&& other) : device(other.device), frame_ref(other.frame_ref) { other.frame_ref = nullptr; }
        frame& operator=(frame other)
        {
            swap(other);
            return *this;
        }
        void swap(frame& other)
        {
            std::swap(device, other.device);
            std::swap(frame_ref, other.frame_ref);
        }

        ~frame()
        {
            if (device && frame_ref)
            {
                rs_error * e = nullptr;
                rs_release_frame(device, frame_ref, &e);
                error::handle(e);
            }
        }

        /// retrieve the time at which the frame was captured
        /// \return            the timestamp of the frame, in milliseconds since the device was started
        double get_timestamp() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_timestamp(frame_ref, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the timestamp domain 
        /// \return            timestamp domain (clock name) for timestamp values
        timestamp_domain get_frame_timestamp_domain() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_timestamp_domain(frame_ref, &e);
            error::handle(e);
            return static_cast<timestamp_domain>(r);
        }

        /// retrieve the current value of a single frame_metadata
        /// \param[in] frame_metadata  the frame_metadata whose value should be retrieved
        /// \return            the value of the frame_metadata
        double get_frame_metadata(rs_frame_metadata frame_metadata) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_metadata(frame_ref, (rs_frame_metadata)frame_metadata, &e);
            error::handle(e);
            return r;
        }

        /// determine if the device allows a specific metadata to be queried
        /// \param[in] frame_metadata  the frame_metadata to check for support
        /// \return            true if the frame_metadata can be queried
        bool supports_frame_metadata(rs_frame_metadata frame_metadata) const
        {
            rs_error * e = nullptr;
            auto r = rs_supports_frame_metadata(frame_ref, frame_metadata, &e);
            error::handle(e);
            return r != 0;
        }

        // retrieve the frame number
        // \return            the frame number
        unsigned long long get_frame_number() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_number(frame_ref, &e);
            error::handle(e);
            return r;
        }

        // retrieve the frame content
        // \return   frame content
        const void * get_data() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_data(frame_ref, &e);
            error::handle(e);
            return r;
        }

        // returns image width in pixels
        int get_width() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_width(frame_ref, &e);
            error::handle(e);
            return r;
        } 

        // returns image height in pixels
        int get_height() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_height(frame_ref, &e);
            error::handle(e);
            return r;
        }

        // returns the configured framerate
        int get_framerate() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_framerate(frame_ref, &e);
            error::handle(e);
            return r;
        }

        // retrive frame stride, meaning the actual line width in memory in bytes (not the logical image width)
        int get_stride() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_stride(frame_ref, &e);
            error::handle(e);
            return r;
        }

        /// retrieve bits per pixel
        /// \return            number of bits per one pixel
        int get_bpp() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_bpp(frame_ref, &e);
            error::handle(e);
            return r;
        }

        // retrieve the frame format
        // \return    frame format
        format get_format() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_detached_frame_format(frame_ref, &e);
            error::handle(e);
            return static_cast<format>(r);
        }

        // retrieve the frame stream type
        // \return    frame stream type
        stream get_stream_type() const
        {
            rs_error * e = nullptr;
            auto s = rs_get_detached_frame_stream_type(frame_ref, &e);
            error::handle(e);
            return static_cast<stream>(s);
        }
    };

    class frame_callback : public rs_frame_callback
    {
        std::function<void(frame)> on_frame_function;
    public:
        explicit frame_callback(std::function<void(frame)> on_frame) : on_frame_function(on_frame) {}

        void on_frame(rs_device * device, rs_frame_ref * fref) override
        {
            on_frame_function(std::move(frame(device, fref)));
        }

        void release() override { delete this; }
    };

    class device
    {
        device() = delete;
        device(const device &) = delete;
        device & operator = (const device &) = delete;
        ~device() = delete;


    public:
        /// retrieve a human readable device model string
        /// \return  the model string, such as "Intel RealSense F200" or "Intel RealSense R200"
        const char * get_name() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_name((const rs_device *)this, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the unique serial number of the device
        /// \return  the serial number, in a format specific to the device model
        const char * get_serial() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_serial((const rs_device *)this, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the USB port number of the device
        /// \return  the USB port number, in a format specific to the device model
        const char * get_usb_port_id() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_usb_port_id((const rs_device *)this, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the version of the firmware currently installed on the device
        /// \return  firmware version string, in a format is specific to device model
        const char * get_firmware_version() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_firmware_version((const rs_device *)this, &e);
            error::handle(e);
            return r;
        }

        /// retrieve camera specific information like the versions of the various componnents
        /// \return  camera info string, in a format specific to the device model
        const char * get_info(camera_info info) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_info((const rs_device *)this, (rs_camera_info)info, &e);
            error::handle(e);
            return r;
        }

        /// retrieve extrinsic transformation between the viewpoints of two different streams
        /// \param[in] from_stream  stream whose coordinate space we will transform from
        /// \param[in] to_stream    stream whose coordinate space we will transform to
        /// \return                 the transformation between the two streams
        extrinsics get_extrinsics(stream from_stream, stream to_stream) const
        {
            rs_error * e = nullptr;
            extrinsics extrin;
            rs_get_device_extrinsics((const rs_device *)this, (rs_stream)from_stream, (rs_stream)to_stream, &extrin, &e);
            error::handle(e);
            return extrin;
        }

        /// retrieve extrinsic transformation between the viewpoints of specific stream and the motion module
        /// \param[in] from_stream  stream whose coordinate space we will transform from
        /// \return                 the transformation between the  specific stream and motion module
        extrinsics get_motion_extrinsics_from(stream from_stream) const
        {
            rs_error * e = nullptr;
            extrinsics extrin;
            rs_get_motion_extrinsics_from((const rs_device *)this, (rs_stream)from_stream, &extrin, &e);
            error::handle(e);
            return extrin;
        }

        /// retrieve mapping between the units of the depth image and meters
        /// \return  depth in meters corresponding to a depth value of 1
        float get_depth_scale() const
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_depth_scale((const rs_device *)this, &e);
            error::handle(e);
            return r;
        }

        /// determine if the device allows a specific option to be queried and set
        /// \param[in] option  the option to check for support
        /// \return            true if the option can be queried and set
        bool supports_option(option option) const
        {
            rs_error * e = nullptr;
            auto r = rs_device_supports_option((const rs_device *)this, (rs_option)option, &e);
            error::handle(e);
            return r != 0;
        }

        /// determine the number of streaming modes available for a given stream
        /// \param[in] stream  the stream whose modes will be enumerated
        /// \return            the count of available modes
        int get_stream_mode_count(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_stream_mode_count((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// determine the properties of a specific streaming mode
        /// \param[in] stream      the stream whose mode will be queried
        /// \param[in] index       the zero based index of the streaming mode
        /// \param[out] width      the width of a frame image in pixels
        /// \param[out] height     the height of a frame image in pixels
        /// \param[out] format     the pixel format of a frame image
        /// \param[out] framerate  the number of frames which will be streamed per second
        void get_stream_mode(stream stream, int index, int & width, int & height, format & format, int & framerate) const
        {
            rs_error * e = nullptr;
            rs_get_stream_mode((const rs_device *)this, (rs_stream)stream, index, &width, &height, (rs_format *)&format, &framerate, &e);
            error::handle(e);
        }

        /// enable a specific stream and request specific properties
        /// \param[in] stream                   the stream to enable
        /// \param[in] width                    the desired width of a frame image in pixels, or 0 if any width is acceptable
        /// \param[in] height                   the desired height of a frame image in pixels, or 0 if any height is acceptable
        /// \param[in] format                   the pixel format of a frame image, or ANY if any format is acceptable
        /// \param[in] framerate                the number of frames which will be streamed per second, or 0 if any framerate is acceptable
        /// \param[in] output_buffer_type       output buffer format (continous in memory / native with pitch)
        void enable_stream(stream stream, int width, int height, format format, int framerate, output_buffer_format output_buffer_type = output_buffer_format::continous)
        {
            rs_error * e = nullptr;
            rs_enable_stream_ex((rs_device *)this, (rs_stream)stream, width, height, (rs_format)format, framerate, (rs_output_buffer_format)output_buffer_type, &e);
            error::handle(e);
        }

        /// enable a specific stream and request properties using a preset
        /// \param[in] stream  the stream to enable
        /// \param[in] preset  the preset to use to enable the stream
        void enable_stream(stream stream, preset preset)
        {
            rs_error * e = nullptr;
            rs_enable_stream_preset((rs_device *)this, (rs_stream)stream, (rs_preset)preset, &e);
            error::handle(e);
        }

        /// disable a specific stream
        /// \param[in] stream  the stream to disable
        void disable_stream(stream stream)
        {
            rs_error * e = nullptr;
            rs_disable_stream((rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
        }

        /// determine if a specific stream is enabled
        /// \param[in] stream  the stream to check
        /// \return            true if the stream is currently enabled
        bool is_stream_enabled(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_is_stream_enabled((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r != 0;
        }

        /// retrieve the width in pixels of a specific stream, equivalent to the width field from the stream's intrinsic
        /// \param[in] stream  the stream whose width to retrieve
        /// \return            the width in pixels of images from this stream
        int get_stream_width(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_stream_width((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the height in pixels of a specific stream, equivalent to the height field from the stream's intrinsic
        /// \param[in] stream  the stream whose height to retrieve
        /// \return            the height in pixels of images from this stream
        int get_stream_height(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_stream_height((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the pixel format for a specific stream
        /// \param[in] stream  the stream whose format to retrieve
        /// \return            the pixel format of the stream
        format get_stream_format(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_stream_format((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return (format)r;
        }

        /// retrieve the framerate for a specific stream
        /// \param[in] stream  the stream whose framerate to retrieve
        /// \return            the framerate of the stream, in frames per second
        int get_stream_framerate(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_stream_framerate((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// retrieve intrinsic camera parameters for a specific stream
        /// \param[in] stream  the stream whose parameters to retrieve
        /// \return            the intrinsic parameters of the stream
        intrinsics get_stream_intrinsics(stream stream) const
        {
            rs_error * e = nullptr;
            intrinsics intrin;
            rs_get_stream_intrinsics((const rs_device *)this, (rs_stream)stream, &intrin, &e);
            error::handle(e);
            return intrin;
        }

        /// retrieve intrinsic camera parameters for the motion module
        /// \return            the intrinsic parameters of the stream
        motion_intrinsics get_motion_intrinsics() const
        {
            rs_error * e = nullptr;
            motion_intrinsics intrinsics;
            rs_get_motion_intrinsics((const rs_device *)this, &intrinsics, &e);
            error::handle(e);
            return intrinsics;
        }

        /// sets the callback for frame arrival event. provided callback will be called the instant new frame of given stream becomes available
        /// once callback is set on certain stream type, frames of this type will no longer be available throuhg wait/poll methods (those two approaches are mutually exclusive) 
        /// while wait/poll methods provide consistent set of syncronized frames at the expense of extra latency,
        /// set frame callbacks provides low latency solution with no syncronization
        /// \param[in] stream    the stream 
        /// \param[in] on_frame  frame callback to be invoke on every new frame
        /// \return            the framerate of the stream, in frames per second
        void set_frame_callback(rs::stream stream, std::function<void(frame)> frame_handler)
        {
            rs_error * e = nullptr;
            rs_set_frame_callback_cpp((rs_device *)this, (rs_stream)stream, new frame_callback(frame_handler), &e);
            error::handle(e);
        }

        /// sets the callback for motion module event. provided callback will be called the instant new motion or timestamp event is available. 
        /// Note: rs_enable_motion_tracking  is responsible for activating the motion module on-board the device. One of the services it provides is producing shared and high-resolution timestamps for all component hooked-up to it. Usually, librealsense takes care of that and copies the timestamps to the relevant frames.
        /// However, when the user has an external device(like a compass) and wishes to synchronize it precisely with image and motion stream he can connect the sensor to a GPIO avaialbe on some devices. Every time sensor will signal, the user will get a timestamp callback with a frame number, source ID and a timestamp.
        /// This would allow advanced user to synchronize his compass events(presumably coming though I2C or some other method) with realsense data.
        /// \param[in] stream             the stream 
        /// \param[in] motion_handler     frame callback to be invoke on every new motion event
        /// \param[in] timestamp_handler  frame callback to be invoke on every new timestamp event (can be left-out)
        /// \return                       the framerate of the stream, in frames per second
        void enable_motion_tracking(std::function<void(motion_data)> motion_handler, std::function<void(timestamp_data)> timestamp_handler)
        {
            rs_error * e = nullptr;            
            rs_enable_motion_tracking_cpp((rs_device *)this, new motion_callback(motion_handler), new timestamp_callback(timestamp_handler), &e);
            error::handle(e);
        }

        /// sets the callback for motion module event. provided callback will be called the instant new motion event is available. 
        /// \param[in] stream             the stream 
        /// \param[in] motion_handler     frame callback to be invoke on every new motion event
        /// \return                       the framerate of the stream, in frames per second
        void enable_motion_tracking(std::function<void(motion_data)> motion_handler)
        {
            rs_error * e = nullptr;            
            rs_enable_motion_tracking_cpp((rs_device *)this, new motion_callback(motion_handler), new timestamp_callback([](rs::timestamp_data data) {}), &e);
            error::handle(e);
        }

        /// disable events polling
        void disable_motion_tracking(void)
        {
            rs_error * e = nullptr;
            rs_disable_motion_tracking((rs_device *)this, &e);
            error::handle(e);
        }

        /// check if data acquisition is active        
        int is_motion_tracking_active()
        {
            rs_error * e = nullptr;
            auto result = rs_is_motion_tracking_active((rs_device *)this,&e);
            error::handle(e);
            return result;
        }


        /// begin streaming on all enabled streams for this device
        void start(rs::source source = rs::source::video)
        {            
            rs_error * e = nullptr;
            rs_start_source((rs_device *)this, (rs_source)source, &e);
            error::handle(e);
        }

        /// end streaming on all streams for this device
        void stop(rs::source source = rs::source::video)
        {
            rs_error * e = nullptr;
            rs_stop_source((rs_device *)this, (rs_source)source, &e);
            error::handle(e);
        }

        /// determine if the device is currently streaming
        /// \return  true if the device is currently streaming
        bool is_streaming() const
        {
            rs_error * e = nullptr;
            auto r = rs_is_device_streaming((const rs_device *)this, &e);
            error::handle(e);
            return r != 0;
        }

        /// retrieve the available range of values of a supported option
        /// \param[in] option  the option whose range should be queried
        /// \param[out] min    the minimum value which will be accepted for this option
        /// \param[out] max    the maximum value which will be accepted for this option
        /// \param[out] step   the granularity of options which accept discrete values, or zero if the option accepts continuous values
        void get_option_range(option option, double & min, double & max, double & step)
        {
            rs_error * e = nullptr;
            rs_get_device_option_range((rs_device *)this, (rs_option)option, &min, &max, &step, &e);
            error::handle(e);
        }

        /// retrieve the available range of values of a supported option
        /// \param[in] option  the option whose range should be queried
        /// \param[out] min    the minimum value which will be accepted for this option
        /// \param[out] max    the maximum value which will be accepted for this option
        /// \param[out] step   the granularity of options which accept discrete values, or zero if the option accepts continuous values
        /// \param[out] def    the default value of the option
        void get_option_range(option option, double & min, double & max, double & step, double & def)
        {
            rs_error * e = nullptr;
            rs_get_device_option_range_ex((rs_device *)this, (rs_option)option, &min, &max, &step, &def, &e);
            error::handle(e);
        }

        /// efficiently retrieve the value of an arbitrary number of options, using minimal hardware IO
        /// \param[in] options  the array of options which should be queried
        /// \param[in] count    the length of the options and values arrays
        /// \param[out] values  the array which will receive the values of the queried options
        void get_options(const option * options, size_t count, double * values)
        {
            rs_error * e = nullptr;
            rs_get_device_options((rs_device *)this, (const rs_option *)options, (unsigned int)count, values, &e);
            error::handle(e);
        }

        /// efficiently set the value of an arbitrary number of options, using minimal hardware IO
        /// \param[in] options  the array of options which should be set
        /// \param[in] count    the length of the options and values arrays
        /// \param[in] values   the array of values to which the options should be set
        void set_options(const option * options, size_t count, const double * values)
        {
            rs_error * e = nullptr;
            rs_set_device_options((rs_device *)this, (const rs_option *)options, (unsigned int)count, values, &e);
            error::handle(e);
        }

        /// retrieve the current value of a single option
        /// \param[in] option  the option whose value should be retrieved
        /// \return            the value of the option
        double get_option(option option)
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_option((rs_device *)this, (rs_option)option, &e);
            error::handle(e);
            return r;
        }
        
        /// retrieve the device specific option description
        /// \param[in] option  the option whose description should be retrieved
        /// \return            the value of the option
        const char * get_option_description(option option)
        {
            rs_error * e = nullptr;
            auto r = rs_get_device_option_description((rs_device *)this, (rs_option)option, &e);
            error::handle(e);
            return r;
        }

        /// set the current value of a single option
        /// \param[in] option  the option whose value should be set
        /// \param[in] value   the value of the option
        void set_option(option option, double value)
        {
            rs_error * e = nullptr;
            rs_set_device_option((rs_device *)this, (rs_option)option, value, &e);
            error::handle(e);
        }

        /// block until new frames are available
        ///
        void wait_for_frames()
        {
            rs_error * e = nullptr;
            rs_wait_for_frames((rs_device *)this, &e);
            error::handle(e);
        }

        /// check if new frames are available, without blocking
        /// \return  true if new frames are available, false if no new frames have arrived
        bool poll_for_frames()
        {
            rs_error * e = nullptr;
            auto r = rs_poll_for_frames((rs_device *)this, &e);
            error::handle(e);
            return r != 0;
        }

        /// determine device capabilities
        /// \param[in] capability  the capability to check for support
        /// \return                true if device has this capability
        bool supports(capabilities capability) const
        {
            rs_error * e = nullptr;
            auto r = rs_supports((rs_device *)this, (rs_capabilities)capability, &e);
            error::handle(e);
            return r? true: false;
        }


        /// determine device capabilities
        /// \param[in] capability  the capability to check for support
        /// \return                true if device has this capability
        bool supports(camera_info info_param) const
        {
            rs_error * e = nullptr;
            auto r = rs_supports_camera_info((rs_device *)this, (rs_camera_info)info_param, &e);
            error::handle(e);
            return r ? true : false;
        }

        /// retrieve the time at which the latest frame on a stream was captured
        /// \param[in] stream  the stream whose latest frame we are interested in
        /// \return            the timestamp of the frame, in milliseconds since the device was started
        double get_frame_timestamp(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_frame_timestamp((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the frame number
        /// \param[in] stream  the stream whose latest frame we are interested in
        /// \return            the number of the frame, since the device was started
        unsigned long long get_frame_number(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_frame_number((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// retrieve the contents of the latest frame on a stream
        /// \param[in] stream  the stream whose latest frame we are interested in
        /// \return            the pointer to the start of the frame data
        const void * get_frame_data(stream stream) const
        {
            rs_error * e = nullptr;
            auto r = rs_get_frame_data((const rs_device *)this, (rs_stream)stream, &e);
            error::handle(e);
            return r;
        }

        /// send device specific data to the device
        /// \param[in] type  describes the content of the memory buffer, how it will be interpreted by the device
        /// \param[in] data  raw data buffer to be sent to the device
        /// \param[in] size  size in bytes of the buffer
        void send_blob_to_device(rs::blob_type type, void * data, int size)
        {
            rs_error * e = nullptr;
            rs_send_blob_to_device((rs_device *)this, (rs_blob_type)type, data, size, &e);
            error::handle(e);
        }
    };

    inline std::ostream & operator << (std::ostream & o, stream stream) { return o << rs_stream_to_string((rs_stream)stream); }
    inline std::ostream & operator << (std::ostream & o, format format) { return o << rs_format_to_string((rs_format)format); }
    inline std::ostream & operator << (std::ostream & o, preset preset) { return o << rs_preset_to_string((rs_preset)preset); }
    inline std::ostream & operator << (std::ostream & o, distortion distortion) { return o << rs_distortion_to_string((rs_distortion)distortion); }
    inline std::ostream & operator << (std::ostream & o, option option) { return o << rs_option_to_string((rs_option)option); }    
    inline std::ostream & operator << (std::ostream & o, capabilities capability) { return o << rs_capabilities_to_string((rs_capabilities)capability); }
    inline std::ostream & operator << (std::ostream & o, source src) { return o << rs_source_to_string((rs_source)src); }
    inline std::ostream & operator << (std::ostream & o, event evt) { return o << rs_event_to_string((rs_event_source)evt); }

    /* Severity of the librealsense logger */
    enum class log_severity : int32_t
    {
        debug = 0, /**< Detailed information about ordinary operations */
        info  = 1, /**< Terse information about ordinary operations */
        warn  = 2, /**< Indication of possible failure */
        error = 3, /**< Indication of definite failure */
        fatal = 4, /**< Indication of unrecoverable failure */
        none  = 5, /**< No logging will occur */
    };

    class log_callback : public rs_log_callback
    {
        std::function<void(log_severity, const char *)> on_event_function;
    public:
        explicit log_callback(std::function<void(log_severity, const char *)> on_event) : on_event_function(on_event) {}

        void on_event(rs_log_severity severity, const char * message) override
        {
            on_event_function((log_severity)severity, message);
        }

        void release() override { delete this; }
    };

    inline void log_to_console(log_severity min_severity)
    {
        rs_error * e = nullptr;
        rs_log_to_console((rs_log_severity)min_severity, &e);
        error::handle(e);
    }

    inline void log_to_file(log_severity min_severity, const char * file_path)
    {
        rs_error * e = nullptr;
        rs_log_to_file((rs_log_severity)min_severity, file_path, &e);
        error::handle(e);
    }

    inline void log_to_callback(log_severity min_severity, std::function<void(log_severity, const char *)> callback)
    {
        rs_error * e = nullptr;
        rs_log_to_callback_cpp((rs_log_severity)min_severity, new log_callback(callback), &e);
        error::handle(e);
    }

    // Additional utilities
    inline void apply_depth_control_preset(device * device, int preset) { rs_apply_depth_control_preset((rs_device *)device, preset); }
    inline void apply_ivcam_preset(device * device, rs_ivcam_preset preset) { rs_apply_ivcam_preset((rs_device *)device, preset); }
    inline void apply_ivcam_preset(device * device, int preset) { rs_apply_ivcam_preset((rs_device *)device, (rs_ivcam_preset)preset); } // duplicate for better backward compatibility with existing applications
}
#endif
