/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef SPCOM_H
#define SPCOM_H

#include <queue>
#include <memory>
#include <thread>
#include <mutex>
#include <functional>
#include <opencv2/opencv.hpp>

#include <visiontransfer/asynctransfer.h>
#include <visiontransfer/reconstruct3d.h>
#include "ratelimit.h"
#include "imagequeue.h"
#include "colorcoder.h"
#include "settings.h"

/*
 * All functionality of SpCom except the GUI
 */
class SpCom {
public:
    SpCom(const Settings& newSettings);
    ~SpCom();

    void terminate();
    void connect();

    unsigned int getWriteQueueSize() {return writeQueue.size();}
    void updateSettings(const Settings& newSettings);

    void captureSingleFrame() {captureNextFrame = true;}
    void setCaptureSequence(bool capture) {captureSequence = capture;}
    void resetCaptureIndex() {captureIndex = 0;}

    // Methods for setting callback functions
    void setFrameDisplayCallback(const std::function<void(const cv::Mat_<cv::Vec3b>& left,
            const cv::Mat_<cv::Vec3b>& right, bool resize)>& f) {
        frameDisplayCallback = f;
    }
    void setExceptionCallback(const std::function<void(const std::exception& ex)>& f) {
        exceptionCallback = f;
    }
    void setSendCompleteCallback(const std::function<void()>& f) {
        sendCompleteCallback = f;
    }
    void setConnectedCallback(const std::function<void()>& f) {
        connectedCallback = f;
    }

    // A mutex that should be used when accessing the display frames
    // outside the callback
    std::mutex& getDisplayMutex() {return displayMutex;}

private:
    Settings settings;

    volatile bool terminateThreads;
    volatile bool captureNextFrame;
    volatile bool captureSequence;
    int captureIndex;
    int seqNum;
    int minDisparity;
    int maxDisparity;

    std::unique_ptr<StereoImageQueue8U> stereoQueue;
    std::unique_ptr<RateLimit> frameRateLimit;
    std::unique_ptr<AsyncTransfer> asyncTrans;
    std::queue<std::pair<std::string, cv::Mat> > writeQueue;
    std::unique_ptr<ColorCoder> redBlueCoder;
    std::unique_ptr<ColorCoder> rainbowCoder;
    Reconstruct3D recon3d;

    cv::Size2i lastFrameSize;
    cv::Mat convertedLeftFrame;
    cv::Mat convertedRightFrame;

    // Callback functions
    std::function<void(const cv::Mat_<cv::Vec3b>&, const cv::Mat_<cv::Vec3b>&, bool)> frameDisplayCallback;
    std::function<void(const std::exception& ex)> exceptionCallback;
    std::function<void()> sendCompleteCallback;
    std::function<void()> connectedCallback;

    // Variables for main loop thread
    std::thread mainLoopThread;
    std::mutex displayMutex;

    // Variables for writing thread
    std::thread writingThread;
    std::mutex writingMutex;
    std::condition_variable writingCond;

    void mainLoop();
    void writeLoop();

    void framePause();
    void scheduleWrite(const cv::Mat& frame, int index, int camera);
    void captureFrameIfRequested(const ImagePair& imagePair, const cv::Mat& receivedLeftFrame,
        const cv::Mat& receivedRightFrame);
    bool transmitFrame();
    void receiveFrame(ImagePair& imagePair, cv::Mat& receivedLeftFrame, cv::Mat& receivedRightFrame);
    void convertFrame(const cv::Mat& src, cv::Mat& dst);
    void colorCodeAndDisplay(const cv::Mat& receivedLeftFrame, const cv::Mat& receivedRightFrame);
    void joinAllThreads();
};

#endif
