#include "hough_feature_clustering.h"

#include <ros/package.h>
#include <std_msgs/String.h>
#include <fstream>
#include <boost/algorithm/string.hpp>

#include "or_msgs/MatchResult.h"
#include "or_nodes/src/Modules/ORLearningModule.h" // TODO get rid of this include
#include "or_nodes/src/Modules/ORControlModule.h" // TODO get rid of this include
#include "color_recognizer/GetHistogram.h"

HoughFeatureClustering::HoughFeatureClustering(ros::NodeHandle *nh, int thread_id, bool training_mode)
{
    m_nh = new ros::NodeHandle(); //nh
    m_queue = new ros::CallbackQueue();
    m_nh->setCallbackQueue(m_queue);

    m_thread_id = thread_id;
    m_sleep_time = 0.25;
    m_ready = true;
    m_ready_shape = true;
    m_one_object_per_image = true;

    m_path = ros::package::getPath("or_nodes");
    m_path += "/";

    m_path_shape = ros::package::getPath("or_shape");
    m_path_shape += "/";

    // prepare thread id for topic namespace
    std::ostringstream oss;
    if(training_mode)
    {
        oss << "/0";
    }
    else
    {
        oss << "/" << thread_id;
    }


    // init random numbers
    srand (time(NULL));

    // advertise topics
    m_learn_command_pub = m_nh->advertise<or_msgs::OrLearnCommand>(oss.str() + "/or/learn_commands", 10);
    m_command_pub = m_nh->advertise<or_msgs::OrCommand>(oss.str() + "/or/commands", 10);

    m_shape_instance_name_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/instance_name", 10);
    m_shape_class_name_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/class_name", 10);
    m_shape_file_input_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/file_input", 10);
    m_shape_with_color_file_input_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/file_input2", 10);
    m_shape_with_surf_file_input_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/file_input3", 10);
    m_shape_save_file_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/save_file", 10);
    m_shape_load_object_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/load_object", 10);
    m_shape_recognize_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/recognize_object", 10);
    m_shape_with_color_recognize_pub = m_nh->advertise<std_msgs::String>(oss.str() + "/or_shape/recognize_object2", 10);

    // subscribe topics
    m_notify_object_saved_sub = m_nh->subscribe<std_msgs::Empty>(oss.str() + "/or/notify_object_saved", 10, &HoughFeatureClustering::object_saved_callback, this);
    m_or_result_sub = m_nh->subscribe<or_msgs::OrMatchResult>(oss.str() + "/or/match_result", 10, &HoughFeatureClustering::recognition_result_callback, this);
    m_or_object_loaded_sub = m_nh->subscribe<or_msgs::OrObjectNames>(oss.str() + "/or/obj_names", 10, &HoughFeatureClustering::object_loaded_callback, this);

    m_shape_object_saved_sub = m_nh->subscribe<std_msgs::Empty>(oss.str() + "/or_shape/object_saved", 10, &HoughFeatureClustering::shape_object_saved_callback, this);
    m_shape_object_loaded_sub = m_nh->subscribe<std_msgs::Empty>(oss.str() + "/or_shape/object_loaded", 10, &HoughFeatureClustering::shape_object_loaded_callback, this);
    m_shape_object_results_sub = m_nh->subscribe<or_msgs::OrMatchResult>(oss.str() + "/or_shape/results", 10, &HoughFeatureClustering::shape_object_results, this);

    // services
    m_histogram_service = m_nh->serviceClient<color_recognizer::GetHistogram>(oss.str() + "/color_recognizer/get_color_histogram");

    ROS_INFO_STREAM("Created hfc object with thread id " << thread_id);
}

HoughFeatureClustering::~HoughFeatureClustering()
{
    ROS_WARN_STREAM("-------------- hfc destructor ------------------");
    //    delete m_nh;
    //    delete m_queue;
}


void HoughFeatureClustering::call_available()
{
    // check for incoming messages in callback queue (i.e. "ros::spin()")
    m_queue->callAvailable();
}


void HoughFeatureClustering::train(std::string class_name, std::string instance_name,
                                   std::vector<std::string> &image_file_names, bool has_background)
{
    int min_num_images = has_background ? 1 : 2;
    if(image_file_names.size() < min_num_images)
    {
        ROS_ERROR_STREAM("Not enough images! There have to be at least 2 images: one background image and one object image,");
        ROS_ERROR_STREAM("or at least 1 image if background is generated automatically.");
        return;
    }

    // use largest segment as object image
    send_learn_command_message(ORLearningModule::SetIsolateLargestSegment, "true");

    if(has_background)
    {
        ROS_INFO_STREAM("reading background image");
        // load each file for training: first file is background ...
        send_learn_command_message(ORLearningModule::LoadBackgroundImage, m_path + image_file_names.at(0));
        ros::Duration(m_sleep_time).sleep();
    }

    int start_index = has_background ? 1 : 0;
    // load each file for training: ... all other files are foreground
    for(unsigned i = start_index; i < image_file_names.size(); i++)
    {
        //if(i % 3 != 0) continue; // use only each third image for training

        ROS_INFO_STREAM("reading foreground image " << (i+1) << " of " << image_file_names.size());

        // add object image
        send_learn_command_message(ORLearningModule::LoadForegroundImage, m_path + image_file_names.at(i));
        ros::Duration(m_sleep_time).sleep();

        // save extracted object
        std::ostringstream oss;
        oss << instance_name << "_" << i;
        send_learn_command_message(ORLearningModule::SaveImage, oss.str());
        ros::Duration(m_sleep_time).sleep();
    }

    // save complete object
    send_learn_command_message(ORLearningModule::SetObjectType, class_name);
    send_learn_command_message(ORLearningModule::SaveObject, instance_name);
    m_ready = false;
}


void HoughFeatureClustering::train_shape(std::string class_name, std::string instance_name,
                                         std::vector<std::string> &image_file_names)
{
    std_msgs::String class_name_msg;
    class_name_msg.data = class_name;
    m_shape_class_name_pub.publish(class_name_msg);

    std_msgs::String instance_name_msg;
    instance_name_msg.data = instance_name;
    m_shape_instance_name_pub.publish(instance_name_msg);

    std_msgs::String file_name_msg;

    for(unsigned i = 0; i < image_file_names.size(); i++)
    {
        //if(i % 3 != 0) continue; // use only each third image for training

        std::string name = image_file_names.at(i);
        int len = name.find_last_of('.');
        std::string new_name = name.substr(0,len);
        new_name.append(".pcd");

        ROS_INFO_STREAM("Processing file " << new_name);
        file_name_msg.data = new_name;
        m_shape_file_input_pub.publish(file_name_msg);
        ros::Duration(m_sleep_time).sleep();
    }

    std_msgs::String save_file_msg;
    save_file_msg.data = instance_name + ".shape";
    m_shape_save_file_pub.publish(save_file_msg);

    m_ready_shape = false;
}


void HoughFeatureClustering::train_shape_with_color(std::string class_name, std::string instance_name,
                                         std::vector<std::string> &image_file_names)
{
    std_msgs::String class_name_msg;
    class_name_msg.data = class_name;
    m_shape_class_name_pub.publish(class_name_msg);

    std_msgs::String instance_name_msg;
    instance_name_msg.data = instance_name;
    m_shape_instance_name_pub.publish(instance_name_msg);

    std_msgs::String file_name_msg;

    for(unsigned i = 0; i < image_file_names.size(); i++)
    {
        //if(i % 3 != 0) continue; // use only each third image for training

        std::string name = image_file_names.at(i);
        int len = name.find_last_of('_');
        std::string new_name = name.substr(0,len);
        new_name.append(".pcd");

        ROS_INFO_STREAM("Processing file " << new_name);
        file_name_msg.data = new_name;

        // switch between learning color shot and surf+shot
#if 0
        m_shape_with_color_file_input_pub.publish(file_name_msg); // color shot
#else
        m_shape_with_surf_file_input_pub.publish(file_name_msg); // surf+shot
#endif
        ros::Duration(m_sleep_time).sleep();
    }

    std_msgs::String save_file_msg;
    save_file_msg.data = instance_name + ".shape";
    m_shape_save_file_pub.publish(save_file_msg);

    m_ready_shape = false;
}



void HoughFeatureClustering::shape_object_saved_callback(const std_msgs::Empty::ConstPtr &msg)
{
    //ROS_INFO_STREAM("Received message: object saved");
    m_ready_shape = true;
}

void HoughFeatureClustering::train_color_histogram(std::string class_name, std::string instance_name,
                                                   std::vector<std::string> &image_file_names, bool has_background)
{
    // prepare service
    std::vector<std::pair<std::string, float> > result;
    color_recognizer::GetHistogram srv;
    srv.request.background_color = "black";
    srv.request.request_type = color_recognizer::GetHistogramRequest::NORM_HISTOGRAM;

    int start_index = has_background ? 1 : 0;
    // load each file for training color histogram
    for(unsigned idx = start_index; idx < image_file_names.size(); idx++)
    {
        //if(idx % 3 != 0) continue; // use only each third image for training

        ROS_INFO_STREAM("reading foreground image " << (idx+1) << " of " << image_file_names.size());
        srv.request.image = imgToMsg(cv::imread(m_path + image_file_names.at(idx)));

        if(m_histogram_service.call(srv))
        {
            Histogram hist;
            for(int i = 0; i < srv.response.colors.size(); i++)
            {
                hist.data.push_back(std::make_pair<std::string,float>(srv.response.colors.at(i),srv.response.dominance.at(i)));
                // TODO use this with cpp 11
                //hist.data.push_back({srv.response.colors.at(i), srv.response.dominance.at(i)});
                //ROS_INFO_STREAM(srv.response.colors.at(i) << ": " << srv.response.dominance.at(i));
            }
            std::ostringstream strm;
            strm << instance_name << "_" << idx;
            m_color_histograms.insert(std::make_pair<std::string, Histogram>(strm.str(),hist));
        }
    }
}


void HoughFeatureClustering::object_saved_callback(const std_msgs::Empty::ConstPtr &msg)
{
    m_ready = true;
}


void HoughFeatureClustering::load_object(std::string file_name)
{
    send_command_message(ORControlModule::LoadObject, file_name);
    m_ready = false;
}

void HoughFeatureClustering::load_shape_object(std::string file_name)
{
    int len = file_name.find_last_of('.');
    std::string new_name = file_name.substr(0,len);
    new_name.append(".shape");

    std_msgs::String load_msg;
    load_msg.data = new_name;
    m_shape_load_object_pub.publish(load_msg);

    ROS_INFO_STREAM("request to load: " << new_name);

    m_ready_shape = false;
}

void HoughFeatureClustering::shape_object_loaded_callback(const std_msgs::Empty::ConstPtr &msg)
{
    ROS_INFO_STREAM("shape object loaded");
    m_ready_shape = true;
}


void HoughFeatureClustering::unload_object(std::string file_name)
{
    // to unload the object's name is needed, not the filename
    std::string file_extension = ".objprop";
    std::string object_name = file_name.substr(0, file_name.length()-file_extension.length());
    ROS_WARN_STREAM("sending msg to delete " << object_name);
    send_command_message(ORControlModule::UnloadObject, object_name);
    sleep(1);
}

void HoughFeatureClustering::object_loaded_callback(const or_msgs::OrObjectNames::ConstPtr &msg)
{
    //ROS_INFO_STREAM("Thread " << m_thread_id << ": object loaded");
    m_ready = true;
}


void HoughFeatureClustering::recognize(std::string image_file_name, bool one_object_per_image)
{
    send_command_message(ORControlModule::LoadSingleImage, m_path + image_file_name);
    m_one_object_per_image = one_object_per_image;

    // save histogram of current image
    color_recognizer::GetHistogram srv;
    srv.request.background_color = "black";
    srv.request.request_type = color_recognizer::GetHistogramRequest::NORM_HISTOGRAM;
    srv.request.image = imgToMsg(cv::imread(m_path + image_file_name));
    if(m_histogram_service.call(srv))
    {
        Histogram hist;
        for(int i = 0; i < srv.response.colors.size(); i++)
        {
            hist.data.push_back(std::make_pair<std::string,float>(srv.response.colors.at(i),srv.response.dominance.at(i)));
        }
        m_current_hist = hist;
    }
    m_ready = false;
}


void HoughFeatureClustering::recognize_shape(std::string image_file_name, bool one_object_per_image)
{
    //std::string input_path = m_path_shape + "input/";   
    boost::replace_all(image_file_name, "rgb", "points");

    int pos = image_file_name.find_last_of('.');
    std::string new_name = image_file_name.substr(0,pos);
    new_name.append(".pcd");
    //input_path.append(new_name);

   // ROS_INFO_STREAM("request to recognize: " << new_name);

    std_msgs::String msg;
    msg.data = new_name;
    m_shape_recognize_pub.publish(msg);

    m_one_object_per_image = one_object_per_image;
    m_ready_shape = false;
}


void HoughFeatureClustering::recognize_shape_with_color(std::string image_file_name, bool one_object_per_image)
{
    //std::string input_path = m_path_shape + "input/";
    int pos = image_file_name.find_last_of('_');
    std::string new_name = image_file_name.substr(0,pos);
    new_name.append(".pcd");
    //input_path.append(new_name);

    ROS_INFO_STREAM("request to recognize: " << new_name);

    std_msgs::String msg;
    msg.data = new_name;
    m_shape_with_color_recognize_pub.publish(msg);

    m_one_object_per_image = one_object_per_image;
    m_ready_shape = false;
}


std::vector<std::vector<ResultData> > HoughFeatureClustering::get_last_combined_results()
{
    // preprocessor switch between category and instance evaluation
    #define CATEGORY_EVAL false

    //check
#if CATEGORY_EVAL
    ROS_ERROR_STREAM("--- performing CATEGORY evaluation ---");
#else
    ROS_ERROR_STREAM("--- performing INSTANCE evaluation ---");
#endif

ROS_ERROR_STREAM("--- performing --- 1");
    // calculate combined results
    m_last_combined_results.clear();

    // 1st combined strategy: take class from shape and determine instance from rgb
    // here: take first class from knn1 and knn2
    {
        std::vector<ResultData> temp_result;
        std::vector<std::string> possible_classes;
        if(m_last_shape_results.size() > 0)
        {
            if(m_last_shape_results.at(0).size() > 0)
            {
                // take first result from first list (knn = 1)
                possible_classes.push_back(m_last_shape_results.at(0).at(0).class_name);
            }
        }
        if(m_last_shape_results.size() > 1)
        {
            if(m_last_shape_results.at(1).size() > 0)
            {
                // take first result from second list (knn = 2)
                possible_classes.push_back(m_last_shape_results.at(1).at(0).class_name);
            }
        }
        possible_classes = remove_dublicates(possible_classes);

        // find instances for classes (dublicate code)
        float best_similarity = 0;
        std::string best_instance = "";
        std::string best_class = "";
        std::vector<std::string> possible_instances;
        std::vector<std::string> corresponding_classes;
        for(unsigned i = 0; i < possible_classes.size(); i++)
        {
            std::string current_class = possible_classes.at(i);
            ROS_ERROR_STREAM("cur class: " << current_class);
            std::set<std::string> current_set = m_class_instance_map.at(current_class);
            for(std::set<std::string>::iterator it = current_set.begin(); it != current_set.end(); it++)
            {
                possible_instances.push_back(*it);
                corresponding_classes.push_back(current_class);
            }
        }

        if(possible_instances.size() == 0)
        {
            if(m_last_results.size() > 2)
            {
                std::vector<ResultData> res_rgb_2= m_last_results.at(2);
                if(res_rgb_2.size() > 0)
                {
                    possible_instances.push_back(res_rgb_2.at(0).instance_name);
                    corresponding_classes.push_back(res_rgb_2.at(0).class_name);
                }
            }
        }

        for(unsigned i = 0; i < possible_instances.size(); i++)
        {
            for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
            {
                if(it->first.find(possible_instances.at(i)) != std::string::npos)
                {
                    float similarity = m_current_hist.compute_similarity(it->second);

                    // EVALUATION: BEST
                    if(similarity > best_similarity)
                    {
                        best_similarity = similarity;
                        best_instance = possible_instances.at(i);
                        best_class = corresponding_classes.at(i);
                    }
                }
            }
        }

        ResultData res;
        res.class_name = best_class;
        res.instance_name = best_instance;
        temp_result.push_back(res);
        m_last_combined_results.push_back(temp_result);
    }

    // 2nd combined strategy: take class from shape and determine instance from rgb
    // here: take first class from knn4
    {
        std::vector<ResultData> temp_result;
        std::vector<std::string> possible_classes;
        if(m_last_shape_results.size() > 3)
        {
            if(m_last_shape_results.at(3).size() > 0)
            {
                // take first result from 4th list (knn = 4)
                possible_classes.push_back(m_last_shape_results.at(3).at(0).class_name);
            }
        }
        possible_classes = remove_dublicates(possible_classes);

        // find instaces for classes (dublicate code)
        float best_similarity = 0;
        std::string best_instance = "";
        std::string best_class = "";
        std::vector<std::string> possible_instances;
        std::vector<std::string> corresponding_classes;
        for(unsigned i = 0; i < possible_classes.size(); i++)
        {
            std::string current_class = possible_classes.at(i);
            std::set<std::string> current_set = m_class_instance_map.at(current_class);
            for(std::set<std::string>::iterator it = current_set.begin(); it != current_set.end(); it++)
            {
                possible_instances.push_back(*it);
                corresponding_classes.push_back(current_class);
            }
        }

        if(possible_instances.size() == 0)
        {
            if(m_last_results.size() > 2)
            {
                std::vector<ResultData> res_rgb_2= m_last_results.at(2);
                if(res_rgb_2.size() > 0)
                {
                    possible_instances.push_back(res_rgb_2.at(0).instance_name);
                    corresponding_classes.push_back(res_rgb_2.at(0).class_name);
                }
            }
        }


        for(unsigned i = 0; i < possible_instances.size(); i++)
        {
            for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
            {
                if(it->first.find(possible_instances.at(i)) != std::string::npos)
                {
                    float similarity = m_current_hist.compute_similarity(it->second);

                    // EVALUATION: BEST
                    if(similarity > best_similarity)
                    {
                        best_similarity = similarity;
                        best_instance = possible_instances.at(i);
                        best_class = corresponding_classes.at(i);
                    }
                }
            }
        }
        ResultData res;
        res.class_name = best_class;
        res.instance_name = best_instance;
        temp_result.push_back(res);
        m_last_combined_results.push_back(temp_result);
    }

    // 3rd combined strategy: take class from shape and determine instance from rgb
    // here: take all classes from knn1 and knn2
    {
        std::vector<ResultData> temp_result;
        std::vector<std::string> possible_classes;
        if(m_last_shape_results.size() > 0)
        {
            for(unsigned i = 0; i < m_last_shape_results.at(0).size(); i++)
            {
                // take all results from first list (knn = 1)
                possible_classes.push_back(m_last_shape_results.at(0).at(i).class_name);
            }
        }
        if(m_last_shape_results.size() > 1)
        {
            for(unsigned i = 0; i < m_last_shape_results.at(1).size(); i++)
            {
                // take all results from first list (knn = 2)
                possible_classes.push_back(m_last_shape_results.at(1).at(i).class_name);
            }
        }
        possible_classes = remove_dublicates(possible_classes);

        // find instaces for classes (dublicate code)
        float best_similarity = 0;
        std::string best_instance = "";
        std::string best_class = "";
        std::vector<std::string> possible_instances;
        std::vector<std::string> corresponding_classes;
        for(unsigned i = 0; i < possible_classes.size(); i++)
        {
            std::string current_class = possible_classes.at(i);
            std::set<std::string> current_set = m_class_instance_map.at(current_class);
            for(std::set<std::string>::iterator it = current_set.begin(); it != current_set.end(); it++)
            {
                possible_instances.push_back(*it);
                corresponding_classes.push_back(current_class);
            }
        }

        if(possible_instances.size() == 0)
        {
            if(m_last_results.size() > 2)
            {
                std::vector<ResultData> res_rgb_2= m_last_results.at(2);
                if(res_rgb_2.size() > 0)
                {
                    possible_instances.push_back(res_rgb_2.at(0).instance_name);
                    corresponding_classes.push_back(res_rgb_2.at(0).class_name);
                }
            }
        }


        for(unsigned i = 0; i < possible_instances.size(); i++)
        {
            for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
            {
                if(it->first.find(possible_instances.at(i)) != std::string::npos)
                {
                    float similarity = m_current_hist.compute_similarity(it->second);

                    // EVALUATION: BEST
                    if(similarity > best_similarity)
                    {
                        best_similarity = similarity;
                        best_instance = possible_instances.at(i);
                        best_class = corresponding_classes.at(i);
                    }
                }
            }
        }
        ResultData res;
        res.class_name = best_class;
        res.instance_name = best_instance;
        temp_result.push_back(res);
        m_last_combined_results.push_back(temp_result);
    }


    // 4th combined strategy: take class from shape and determine instance from rgb
    // here: take all classes from knn4
    {
        std::vector<ResultData> temp_result;
        std::vector<std::string> possible_classes;
        if(m_last_shape_results.size() > 3)
        {
            for(unsigned i = 0; i < m_last_shape_results.at(3).size(); i++)
            {
                // take all results from first list (knn = 1)
                possible_classes.push_back(m_last_shape_results.at(3).at(i).class_name);
            }
        }
        possible_classes = remove_dublicates(possible_classes);

        // find instaces for classes (dublicate code)
        float best_similarity = 0;
        std::string best_instance = "";
        std::string best_class = "";
        std::vector<std::string> possible_instances;
        std::vector<std::string> corresponding_classes;
        for(unsigned i = 0; i < possible_classes.size(); i++)
        {
            std::string current_class = possible_classes.at(i);
            std::set<std::string> current_set = m_class_instance_map.at(current_class);
            for(std::set<std::string>::iterator it = current_set.begin(); it != current_set.end(); it++)
            {
                possible_instances.push_back(*it);
                corresponding_classes.push_back(current_class);
            }
        }

        if(possible_instances.size() == 0)
        {
            if(m_last_results.size() > 2)
            {
                std::vector<ResultData> res_rgb_2= m_last_results.at(2);
                if(res_rgb_2.size() > 0)
                {
                    possible_instances.push_back(res_rgb_2.at(0).instance_name);
                    corresponding_classes.push_back(res_rgb_2.at(0).class_name);
                }
            }
        }


        for(unsigned i = 0; i < possible_instances.size(); i++)
        {
            for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
            {
                if(it->first.find(possible_instances.at(i)) != std::string::npos)
                {
                    float similarity = m_current_hist.compute_similarity(it->second);

                    // EVALUATION: BEST
                    if(similarity > best_similarity)
                    {
                        best_similarity = similarity;
                        best_instance = possible_instances.at(i);
                        best_class = corresponding_classes.at(i);
                    }
                }
            }
        }
        ResultData res;
        res.class_name = best_class;
        res.instance_name = best_instance;
        temp_result.push_back(res);
        m_last_combined_results.push_back(temp_result);
    }

    // 5th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_instance_best is in shape list knn1, take rgb result, else take best shape result from knn1
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 0 && m_last_shape_results.size() > 0)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_1 = m_last_results.at(0); // color_instance_best
            if(res_rgb_1.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_1.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(0); // knn 1
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    if(res_shape_1.size() > 0)
                    {
                        res = res_shape_1.at(0);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }


    // 6th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_instance_best is in shape list knn4, take rgb result, else take best shape result from knn4
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 0 && m_last_shape_results.size() > 3)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_1 = m_last_results.at(0); // color_instance_best
            if(res_rgb_1.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_1.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(3); // knn 4
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    if(res_shape_1.size() > 0)
                    {
                        res = res_shape_1.at(0);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }


    // 7th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_class_best is in shape list knn2, take rgb result, else take best shape result from knn2
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 2 && m_last_shape_results.size() > 1)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_2= m_last_results.at(2); // color_class_best
            if(res_rgb_2.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_2.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(1); // knn 2
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    if(res_shape_1.size() > 0)
                    {
                        res = res_shape_1.at(0);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }

    // 8th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_class_best is in shape list knn4, take rgb result, else take best shape result from knn4
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 2 && m_last_shape_results.size() > 3)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_2= m_last_results.at(2); // color_class_best
            if(res_rgb_2.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_2.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(3); // knn 4
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    if(res_shape_1.size() > 0)
                    {
                        res = res_shape_1.at(0);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }


    // 9th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_instance_best is in shape list knn1, take rgb result, else take random shape result from knn1
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 0 && m_last_shape_results.size() > 0)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_1 = m_last_results.at(0); // color_instance_best
            if(res_rgb_1.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_1.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(0); // knn 1
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    int size = res_shape_1.size();
                    if(size != 0)
                    {
                        int random = rand() % size;
                        res = res_shape_1.at(random);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }


    // 10th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_class_best is in shape list knn2, take rgb result, else take random shape result from knn2
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 2 && m_last_shape_results.size() > 1)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_2= m_last_results.at(2); // color_class_best
            if(res_rgb_2.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_2.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(1); // knn 2
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {
#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    int size = res_shape_1.size();
                    if(size != 0)
                    {
                        int random = rand() % size;
                        res = res_shape_1.at(random);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }


    // 11th combined strategy: take best from shape and rgb recognition and decide between these two
    // here: if rgb from color_class_best is in shape list knn4, take rgb result, else take random shape result from knn4
    {
        std::vector<ResultData> temp_result;
        if(m_last_results.size() > 2 && m_last_shape_results.size() > 3)
        {
            ResultData res;
            std::vector<ResultData> res_rgb_2= m_last_results.at(2); // color_class_best
            if(res_rgb_2.size() > 0)
            {
                bool found = false;
                ResultData rgb_candidate = res_rgb_2.at(0);
                std::vector<ResultData> res_shape_1 = m_last_shape_results.at(3); // knn 4
                for(unsigned i = 0; i < res_shape_1.size(); i++)
                {

#if CATEGORY_EVAL // category evaluation
                    if(rgb_candidate.class_name == res_shape_1.at(i).class_name)
#else // instance evaluation
                    if(rgb_candidate.instance_name == res_shape_1.at(i).instance_name)
#endif
                    {
                        res = rgb_candidate;
                        found = true;
                        break;
                    }
                }
                if(!found)
                {
                    int size = res_shape_1.size();
                    if(size != 0)
                    {
                        int random = rand() % size;
                        res = res_shape_1.at(random);
                    }
                    else
                    {
                        res = rgb_candidate;
                    }
                }
            }
            temp_result.push_back(res);
            m_last_combined_results.push_back(temp_result);
        }
    }

    return m_last_combined_results;
}


void HoughFeatureClustering::shape_object_results(const or_msgs::OrMatchResult::ConstPtr &msg)
{
    // save new results
    m_last_shape_results.clear();
    std::vector<ResultData> temp_result;

    ROS_INFO_STREAM("Shape Results: " << msg->match_results.size()); // there are 4 strategies and 3 results per strategy
    int results_per_strategy = 3;
    for(unsigned i = 0; i < msg->match_results.size(); i++)
    {
        or_msgs::MatchResult results = msg->match_results.at(i);
        ResultData res;

        // somehow at rockin type is always same as name
        results.object_type = results.object_type.substr(0,1); // TODO rockin hack

        res.class_name = results.object_type;
        res.instance_name = results.object_name;
        if(res.instance_name != "") // remove dummy values again
        {
            temp_result.push_back(res);
        }
        ROS_INFO_STREAM("     object: " << res.instance_name << ", type: " << res.class_name);

        if(i % results_per_strategy == 2) // called after each "results_per_strategy" loop runs
        {
            m_last_shape_results.push_back(temp_result);
            temp_result.clear();
        }
    }
    //ROS_INFO_STREAM("last shape results size: " << m_last_shape_results.size());
    m_ready_shape = true;
}



void HoughFeatureClustering::recognition_result_callback(const or_msgs::OrMatchResult::ConstPtr &msg)
{
    //ROS_INFO_STREAM("Thread " << m_thread_id << ": received or results: " << msg->match_results.size());

    // save new results
    m_last_results.clear();
    std::vector<ResultData> temp_result;

    ROS_INFO_STREAM("Results: ");
    for(unsigned i = 0; i < msg->match_results.size(); i++)
    {
        or_msgs::MatchResult results = msg->match_results.at(i);
        ResultData res;
        res.class_name = results.object_type;
        res.instance_name = results.object_name;
        res.confidence = results.object_key_points.size() / (float)(results.scene_key_points_within_outline.size()+results.object_key_points.size());

        temp_result.push_back(res);
        ROS_INFO_STREAM("     object: " << res.instance_name << ", type: " << res.class_name << ", confidence: " << res.confidence);
    }

    if(m_one_object_per_image)
    {
        int num_color_strategies = 4;
        for(int i = 0; i < num_color_strategies; i++)
        {
            m_last_results.push_back(temp_result);
        }

        // push additional result without a color histogram strategy (only pure hough recognition)
        for(unsigned i = 0; i < temp_result.size(); i++)
        {
            float conf = temp_result.at(i).confidence;
            temp_result.at(i).confidence = 1 - conf;

        }
        m_last_results.push_back(temp_result);


        // find histograms of possible instances
        std::pair<std::string, std::string> best_instances_strategy = get_instance_of_best_matching_histogram("instances");
        std::pair<std::string, std::string> best_classes_strategy = get_instance_of_best_matching_histogram("classes");

        // save histogram results
        std::vector<std::string> best_instances;
        best_instances.push_back(best_classes_strategy.first);
        best_instances.push_back(best_classes_strategy.second);
        best_instances.push_back(best_instances_strategy.first);
        best_instances.push_back(best_instances_strategy.second);


        ROS_INFO_STREAM("   best color hist votes for: ");
        ROS_INFO_STREAM("       instance best: " << best_instances_strategy.first);
        ROS_INFO_STREAM("       instance average: " << best_instances_strategy.second);
        ROS_INFO_STREAM("       classes best: " << best_classes_strategy.first);
        ROS_INFO_STREAM("       classes average: " << best_classes_strategy.second);

        int index = -1;
        // assign different color results
        for(int col_str = 0; col_str < num_color_strategies; col_str++)
        {
            // strategy: just assume closest color hist in list of hypothesis is correct result (overwrite confidence)
            ResultData new_res;
            int pos = best_instances.at(col_str).find_last_of('_');
            new_res.class_name = best_instances.at(col_str).substr(0, pos);
            new_res.instance_name = best_instances.at(col_str);
            new_res.confidence = 1;

            // loop over all normal results
            for(unsigned i = 0; i < m_last_results.at(col_str).size(); i++)
            {
                // find this instances and overwrite confidence
                if(m_last_results.at(col_str).at(i).instance_name == best_instances.at(col_str))
                {
                    index = i;
                    m_last_results.at(col_str).at(i).confidence = 1;
                    break;
                }
            }
            if(index == -1)
            {
                m_last_results.at(col_str).push_back(new_res);
            }
        }
    }
    m_ready = true;
}


sensor_msgs::Image HoughFeatureClustering::imgToMsg(const cv::Mat &img, std::string encoding)
{
    cv_bridge::CvImage cv_image;
    cv_image.image = img;
    cv_image.encoding = encoding;
    sensor_msgs::Image img_msg;
    cv_image.toImageMsg(img_msg);
    return img_msg;
}


void HoughFeatureClustering::send_learn_command_message(int command, std::string value)
{
    or_msgs::OrLearnCommand msg;
    msg.command = command;
    msg.string_value = value;
    m_learn_command_pub.publish(msg);
}

void HoughFeatureClustering::send_command_message(int command, std::string value)
{
    or_msgs::OrCommand msg;
    msg.command = command;
    msg.string_value = value;
    m_command_pub.publish(msg);
}




std::pair<std::string, std::string> HoughFeatureClustering::get_instance_of_best_matching_histogram(std::string strategy)
{
    // TODO this should be params
    //std::string strategy = "instances"; // only instances from hypotheses list
    //std::string strategy = "classes"; // all instances from classes in hypotheses list
    //std::string evaluation = "best";  // best match is selected
    //std::string evaluation = "average";  // average highest score is selected
    //float confidence_threshold = 0;


    // variables for best-only strategy
    float best_similarity = 0;
    std::string best_instance = "";

    // variables for average best strategy
    std::map<std::string, std::vector<float> > instance_score_map;

    // there have to be some results
    if(m_last_results.size() == 0)
            return std::make_pair<std::string, std::string>("","");

    // only check instances from hypotheses list
    std::vector<std::string> possible_instances;
    if(m_last_results.at(0).size() > 0)
    {
        ROS_INFO_STREAM("checking color histograms of hypotheses list");

        // check color histograms to find most probable result
        // select all possible instances
        for(unsigned i = 0; i < m_last_results.at(0).size(); i++)
        {
            if(strategy == "instances")
            {
                // STRATEGY instances: only use instances from results list
//                if(m_last_results.at(0).at(i).confidence > confidence_threshold)
//                {
                    possible_instances.push_back(m_last_results.at(0).at(i).instance_name);
//                }
            }
            else if(strategy == "classes")
            {
                // STRATEGY classes: use all instances from all classes in the results list
//                if(m_last_results.at(0).at(i).confidence > confidence_threshold)
//                {
                    std::string current_class = m_last_results.at(0).at(i).class_name;
                    std::set<std::string> current_set = m_class_instance_map.at(current_class);
                    for(std::set<std::string>::iterator it = current_set.begin(); it != current_set.end(); it++)
                    {
                        possible_instances.push_back(*it);
                    }
//                }
            }
            else
            {
                ROS_ERROR_STREAM("Invalid color histogram strategy!");
            }
        }

        // loop over all possible instances and find the best according to the chosen EVALUATION
        for(unsigned i = 0; i < possible_instances.size(); i++)
        {
            for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
            {
                if(it->first.find(possible_instances.at(i)) != std::string::npos)
                {
                    float similarity = m_current_hist.compute_similarity(it->second);

                    // EVALUATION: BEST
                    if(similarity > best_similarity)
                    {
                        best_similarity = similarity;
                        best_instance = possible_instances.at(i);
                    }

                    // EVALUATION: AVERAGE
                    if(instance_score_map.find(it->first) != instance_score_map.end())
                    {
                        // insert to existing set in map
                        instance_score_map.at(it->first).push_back(similarity);
                    }
                    else // add new object to map
                    {
                        std::vector<float> similarities;
                        similarities.push_back(similarity);
                        instance_score_map.insert(std::make_pair<std::string, std::vector<float> >(possible_instances.at(i), similarities));
                    }
                }
            }
        }
    }


    if(m_last_results.at(0).size() == 0 || possible_instances.size() == 0)
    {
        ROS_INFO_STREAM("checking all color histograms");
        for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
        {
            float similarity = m_current_hist.compute_similarity(it->second);
            // EVALUATION: BEST
            if(similarity > best_similarity)
            {
                int pos = it->first.find_last_of('_');
                best_instance = it->first.substr(0, pos);
                best_similarity = similarity;
            }

            int pos = it->first.find_last_of('_');
            std::string inst_name = it->first.substr(0, pos);

            // EVALUATION: AVERAGE
            if(instance_score_map.find(inst_name) != instance_score_map.end())
            {
                // insert to existing set in map
                instance_score_map.at(inst_name).push_back(similarity);
            }
            else // add new object to map
            {
                std::vector<float> similarities;
                similarities.push_back(similarity);
                instance_score_map.insert(std::make_pair<std::string, std::vector<float> >(inst_name, similarities));
            }
        }
    }

    // variables for best-average strategy
    float best_average_similarity = 0;
    std::string best_average_instance = "";
    std::string last_class;

    float sum = 0;
    float size = 0;

    // now compute the average best result
//    if(evaluation == "average")
//    {
        for(std::map<std::string, std::vector<float> >::iterator it = instance_score_map.begin(); it != instance_score_map.end(); it++)
        {
#if 1 // instance evaluation: average over each instance
            float sum = 0;
            std::vector<float> temp_vec = it->second;
            for(unsigned i = 0; i < temp_vec.size(); i++)
            {
                sum += temp_vec.at(i);
            }
            sum /= temp_vec.size();
#else // category evaluation: average over all instances of a class

            std::string class_name_temp = it->first;
            int pos = class_name_temp.find_last_of('_');
            std::string class_name = class_name_temp.substr(0,pos);

            if(last_class != class_name)
            {
                // we are at a new class, so check last result
                if(size != 0)
                    sum /= size;

                if(sum > best_average_similarity)
                {
                    best_average_similarity = sum;
                    best_average_instance = it->first;
                }

                // init values for new class
                last_class = class_name;
                sum = 0;
                size = 0;

                std::vector<float> temp_vec = it->second;
                for(unsigned i = 0; i < temp_vec.size(); i++)
                {
                    sum += temp_vec.at(i);
                }
                size += temp_vec.size();
            }
            else
            {
                std::vector<float> temp_vec = it->second;
                for(unsigned i = 0; i < temp_vec.size(); i++)
                {
                    sum += temp_vec.at(i);
                }
                size += temp_vec.size();
            }
#endif
        }
//    }

    return std::make_pair<std::string, std::string>(best_instance, best_average_instance);
}






std::vector<std::string> HoughFeatureClustering::remove_dublicates(std::vector<std::string> &in_list)
{
    std::vector<std::string> results;
    for(unsigned i = 0; i < in_list.size(); i++)
    {
        std::string temp = in_list.at(i);
        bool found = false;
        for(unsigned j = 0; j < results.size(); j++)
        {
            if(temp == results.at(j))
            {
                found = true;
                break;
            }
        }
        if(!found)
        {
            results.push_back(temp);
        }
    }

    return results;
}


void HoughFeatureClustering::save_color_hist_file()
{
    std::string filename = m_path + "color_histograms.hist";
    std::ofstream out ( filename.c_str() );
    boost::archive::text_oarchive oa(out);

    unsigned size = m_color_histograms.size();
    oa << size;
    // TODO use auto type
    for(std::map<std::string, Histogram>::iterator it = m_color_histograms.begin(); it != m_color_histograms.end(); it++)
    {
        oa << it->first;
        oa << it->second;
    }
}

void HoughFeatureClustering::load_color_hist_file()
{
    ROS_INFO_STREAM("Loading color histogram in thread " << m_thread_id);

    std::string filename = m_path + "color_histograms.hist";
    std::ifstream ifs( filename.c_str() );
    boost::archive::text_iarchive ia(ifs);

    m_color_histograms.clear();
    unsigned size;
    ia >> size;
    for(unsigned i = 0; i < size; i++)
    {
        std::string first;
        Histogram second;
        ia >> first;
        ia >> second;
        m_color_histograms.insert(std::make_pair<std::string, Histogram>(first, second));


        // --- this has nothing to do with loading color hist - it is just an additional map that is constructed ----
        // extract instance and class name
        int pos = first.find_last_of('_');
        std::string instance = first.substr(0, pos); // remove image index
        pos = instance.find_last_of('_');
        std::string class_name = instance.substr(0, pos);

        class_name = class_name.substr(0,1); // TODO rockin camp hack
        //ROS_ERROR_STREAM("inst: " << instance << "    class: " << class_name);

        // insert them into map
        if(m_class_instance_map.find(class_name) != m_class_instance_map.end())
        {
            // insert to existing set in map
            m_class_instance_map.at(class_name).insert(instance);
        }
        else // add new object to map
        {
            std::set<std::string> my_set;
            my_set.insert(instance);
            m_class_instance_map.insert(std::make_pair<std::string, std::set<std::string> >(class_name,my_set));
        }

        // add dummy
        std::set<std::string> my_set;
        my_set.insert("");
        m_class_instance_map.insert(std::make_pair<std::string, std::set<std::string> >("",my_set));


    }
    ifs.close();

    //    ROS_WARN_STREAM("all objects");
    //    for(std::map<std::string, std::set<std::string> >::iterator it = m_class_instance_map.begin(); it != m_class_instance_map.end(); it++)
    //    {
    //        ROS_INFO_STREAM("class: " << it->first);
    //        for(std::set<std::string>::iterator it2 = it->second.begin(); it2 != it->second.end(); it2++)
    //        {
    //            ROS_INFO_STREAM("   instance: " << *it2);
    //        }
    //    }
}
