/*******************************************************************************
 *  ORControlModule.cpp
 *
 *  (C) 2006 AG Aktives Sehen <agas@uni-koblenz.de>
 *           Universitaet Koblenz-Landau
 *
 *******************************************************************************/

#include <sstream>

#include <cv_bridge/cv_bridge.h>
#include <sensor_msgs/image_encodings.h>
#include <sensor_msgs/Image.h>
#include <opencv2/highgui/highgui.hpp>

#include <or_msgs/BoundingBox2D.h>
#include <or_msgs/ExtractKeyPoints.h>

#include <robbie_architecture/Architecture/Config/Config.h>

#include <or_nodes/Modules/ORControlModule.h>
#include <or_nodes/Modules/ORMatchingModule.h>

#define THIS ORControlModule

THIS::THIS(ros::NodeHandle *nh, ORMatchingModule* objRecMatchingModule)
{
    m_ORMatchingModule = objRecMatchingModule;

    m_ImagesInPipeline = 0;
    m_MaxImagesInPipeline = Config::getInt( "ObjectRecognition.iMaxImagesInPipeline" );

    m_Continuous = false;

    // subscribe to topics
    m_ORCommandSubscriber = nh->subscribe<or_msgs::OrCommand>("or/commands", 10, &ORControlModule::callbackOrCommand, this);
    m_ORMatchResultSubscriber = nh->subscribe<or_msgs::OrMatchResult>("or/match_result", 10, &ORControlModule::callbackOrMatchResult, this);

    // advertise topics
    m_ExtractKeyPointsPublisher = nh->advertise<or_msgs::ExtractKeyPoints>("or/extract", 10);
    m_DebugImagePublisherGray = nh->advertise<sensor_msgs::Image>("or/debug_image_gray", 10);
    m_DebugImagePublisherColor = nh->advertise<sensor_msgs::Image>("or/debug_image_color", 10);
}


THIS::~THIS()
{}


void THIS::callbackOrCommand( const or_msgs::OrCommand::ConstPtr& or_command_msg )
{
    ROS_ERROR_STREAM("or_command message received");

    switch (or_command_msg->command)
    {
    case ORControlModule::GrabSingleImage:
    {
        // send message for keypoint extraction
        or_msgs::ExtractKeyPoints extract_msg;
        m_ExtractKeyPointsPublisher.publish(extract_msg);

        m_ImagesInPipeline++;
        m_Continuous = false;
                ROS_ERROR_STREAM("leaving the or_command_msg case GrabSingleImage");
        break;
    }

    case ORControlModule::LoadSingleImage:
    {
        std::string fileName = or_command_msg->string_value;
        ROS_ERROR_STREAM("Loading image file: " << fileName);

        m_Continuous = false;

        //read and publish image
        cv_bridge::CvImage gray_image, color_image;
        gray_image.image = cv::imread(fileName.c_str(), CV_LOAD_IMAGE_GRAYSCALE);
        gray_image.encoding = sensor_msgs::image_encodings::MONO8;
        color_image.image = cv::imread(fileName.c_str(), CV_LOAD_IMAGE_COLOR);
        color_image.encoding = sensor_msgs::image_encodings::RGB8;

        // TODO here we have dublicate publishing
        sensor_msgs::Image image_msg;
        image_msg = *(gray_image.toImageMsg());
        m_DebugImagePublisherGray.publish(image_msg);
        image_msg = *(color_image.toImageMsg());
        m_DebugImagePublisherColor.publish(image_msg);

        // pass loaded image directly to the matching module
        m_ORMatchingModule->processImageMessage(color_image.toImageMsg());

        break;
    }

    case ORControlModule::StartRecognitionLoop:
    {
        for ( int i=0; i<m_MaxImagesInPipeline; i++ )
        {
            // send message for keypoint extraction
            or_msgs::ExtractKeyPoints extract_msg;
            m_ExtractKeyPointsPublisher.publish(extract_msg);
            m_ImagesInPipeline++;
        }
        m_Continuous = true;
        break;
    }

    case ORControlModule::StopRecognitionLoop:
    {
        m_Continuous = false;
        break;
    }


     case ORControlModule::LoadObject:
    {
        ROS_INFO_STREAM("I am in Loadobject callback");
        // NOTE: this is handled in ORLoaderModule

        break;
    }
    default:
        break;
    }
}

void THIS::callbackOrMatchResult( const or_msgs::OrMatchResult::ConstPtr& or_match_result_msg)
{
    // TODO only for debug
    if(or_match_result_msg->match_results.size() == 0)
        ROS_WARN_STREAM("no objects recognized");
    else
    {
        ROS_WARN_STREAM("recognized objects: ");
        for(unsigned i = 0; i < or_match_result_msg->match_results.size(); i++)
            ROS_WARN_STREAM(or_match_result_msg->match_results.at(i).object_name);
    }


    if ( m_Continuous )
    {
        // send message for keypoint extraction
        or_msgs::ExtractKeyPoints extract_msg;
        m_ExtractKeyPointsPublisher.publish(extract_msg);
        m_ImagesInPipeline++;
    }
    m_ImagesInPipeline--;
}


#undef THIS
