/*******************************************************************************
 *  ORLoaderModule.cpp
 *
 *  (C) 2006 AG Aktives Sehen <agas@uni-koblenz.de>
 *           Universitaet Koblenz-Landau
 *
 *******************************************************************************/

#include <or_nodes/Modules/ORLoaderModule.h>
#include <or_nodes/Modules/ORControlModule.h>
#include <or_nodes/Modules/ORMatchingModule.h>

#include <robbie_architecture/Architecture/Config/Config.h>

#include <or_libs/ObjectRecognition/ObjectProperties.h>
#include <robbie_architecture/Workers/String/String.h>

#include <sstream>
#include <fstream>

#include <ros/package.h>
#include <homer_nav_libs/tools/loadRosConfig.h>



ORLoaderModule::ORLoaderModule(ros::NodeHandle *nh, ORMatchingModule* objRecMatchingModule)
{
    m_ORMatchingModule = objRecMatchingModule;

    // subscribe to messages
    m_ORCommandSubscriber = nh->subscribe<or_msgs::OrCommand>("or/commands", 100, &ORLoaderModule::callbackOrCommand, this);

    loadDefaultObjects();
}

ORLoaderModule::~ORLoaderModule()
{}


void ORLoaderModule::callbackOrCommand( const or_msgs::OrCommand::ConstPtr& or_command_msg )
{
    switch( or_command_msg->command )
    {
    case ORControlModule::LoadObject:
        loadObjectProperties( or_command_msg->string_value );
        break;

    default:
        break;
    }
}

void ORLoaderModule::loadDefaultObjects( )
{
    std::string objectListStr = "";
    loadConfigValue("or_objects", objectListStr);
    if(objectListStr == "")
    {
        ROS_WARN_STREAM("No object names set on parameter server. Loading default objects from deprecated old robbie config.");
        objectListStr = Config::getString( "ObjectRecognition.sLoadObjects" );
    }


    ROS_INFO_STREAM( "Loading ObjectRecognition.sLoadObjects: " <<objectListStr <<"\n");

    std::vector<std::string> objectList = String::explode( objectListStr, ",;" );
    for ( unsigned i=0; i<objectList.size(); i++ )
    {
        loadObjectProperties( objectList[i] );
    }
}


void ORLoaderModule::loadObjectProperties( std::string filename )
{
    std::string path = ros::package::getPath("or_nodes");
    std::string dir = Config::getString( "ObjectRecognition.sDataPath" );
    filename = path + dir + filename;

    // if there is no file extension: add it
    if(filename.find_last_of('.') == filename.npos)
    {
        filename = filename  + ".objprop";
    }

    ROS_INFO_STREAM("Loading object " << filename << " from " << (path+dir));

    // Return if file does not exist
    if(!fileExists(filename))
    {
        ROS_WARN_STREAM("File not found: " + filename);
        return;
    }

    ObjectProperties* objectProperties;

    try
    {
        ROS_INFO_STREAM( "Loading " + filename );
        std::ifstream ifs( filename.c_str() );
        boost::archive::text_iarchive ia(ifs);
        objectProperties = new ObjectProperties();
        ia >> objectProperties;
        ifs.close();
    }
    catch ( const char* c )
    {
        ROS_ERROR_STREAM( "Cannot load object: " << c );
        return;
    }

    m_ORMatchingModule->addObjectProperties(objectProperties);
    delete objectProperties;
}



bool ORLoaderModule::fileExists(const std::string& file)
{
    struct stat buf;
    if (stat(file.c_str(), &buf) != -1)
    {
        return true;
    }
    return false;
}



#undef THIS
