/**
 * @file pomp_timer_posix.c
 *
 * @brief Timer implementation, posix implementation.
 *
 * @author yves-marie.morgan@parrot.com
 *
 * Copyright (c) 2014 Parrot S.A.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name of the <organization> nor the
 *     names of its contributors may be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "pomp_priv.h"

#ifdef POMP_HAVE_TIMER_POSIX

/**
 * Function called when the posix timer signal is triggered.
 * @param signo : signal number.
 * @param si : signal information.
 * @param unused : unused parameter.
 */
static void pomp_timer_posix_sighandler(int signo, siginfo_t *si, void *unused)
{
	ssize_t res = 0;
	uint8_t dummy = 0xff;
	struct pomp_timer *timer = NULL;

	/* Extract timer structure */
	if (si->si_code != SI_TIMER)
		return;
	timer = si->si_value.sival_ptr;
	if (timer == NULL)
		return;

	/* Write in notification pipe */
	do {
		res = write(timer->pipefds[1], &dummy, sizeof(dummy));
	} while (res < 0 && errno == EINTR);
}

/**
 * Function called when the read notification pipe fd is ready for events.
 * @param fd : triggered fd.
 * @param revents : event that occurred.
 * @param userdata : timer object.
 */
static void pomp_timer_posix_cb(int fd, uint32_t revents, void *userdata)
{
	struct pomp_timer *timer = userdata;
	ssize_t res = 0;
	uint8_t dummy = 0;

	/* Read from notification pipe */
	do {
		res = read(timer->pipefds[0], &dummy, sizeof(dummy));
	} while (res < 0 && errno == EINTR);

	/* Notify callback */
	(*timer->cb)(timer, timer->userdata);
}

/**
 * @see pomp_timer_destroy.
 */
static int pomp_timer_posix_destroy(struct pomp_timer *timer)
{
	POMP_RETURN_ERR_IF_FAILED(timer != NULL, -EINVAL);

	/* Free resources */
	/* from 'timer_delete' man page : The treatment of any pending signal
	 * generated by the deleted timer is unspecified */
	if (timer->id != 0)
		timer_delete(timer->id);
	if (timer->pipefds[0] >= 0) {
		pomp_loop_remove(timer->loop, timer->pipefds[0]);
		close(timer->pipefds[0]);
	}
	if (timer->pipefds[1] >= 0)
		close(timer->pipefds[1]);
	free(timer);
	return 0;
}

/**
 * @see pomp_timer_new.
 */
static struct pomp_timer *pomp_timer_posix_new(struct pomp_loop *loop,
		pomp_timer_cb_t cb, void *userdata)
{
	int res = 0;
	struct pomp_timer *timer = NULL;
	struct sigevent sigev;
	struct sigaction act;
	POMP_RETURN_VAL_IF_FAILED(loop != NULL, -EINVAL, NULL);
	POMP_RETURN_VAL_IF_FAILED(cb != NULL, -EINVAL, NULL);

	/* Allocate timer structure */
	timer = calloc(1, sizeof(*timer));
	if (timer == NULL)
		goto error;
	timer->loop = loop;
	timer->cb = cb;
	timer->userdata = userdata;
	timer->pipefds[0] = -1;
	timer->pipefds[1] = -1;

	/* Create timer */
	memset(&sigev, 0, sizeof(sigev));
	sigev.sigev_notify = SIGEV_SIGNAL;
	sigev.sigev_signo = POMP_TIMER_POSIX_SIGNO;
	sigev.sigev_value.sival_ptr = timer;
	res = timer_create(CLOCK_MONOTONIC, &sigev, &timer->id);
	if (res < 0) {
		POMP_LOG_ERRNO("timer_create");
		goto error;
	}

	/* Create pipe for notification */
	res = pipe(timer->pipefds);
	if (res < 0) {
		POMP_LOG_ERRNO("pipe");
		goto error;
	}

	/* Setup fd flags */
	res = fd_setup_flags(timer->pipefds[0]);
	if (res < 0)
		goto error;
	res = fd_setup_flags(timer->pipefds[1]);
	if (res < 0)
		goto error;

	/* Add read side in loop */
	res = pomp_loop_add(timer->loop, timer->pipefds[0], POMP_FD_EVENT_IN,
			&pomp_timer_posix_cb, timer);
	if (res < 0)
		goto error;

	/* Register signal handler */
	memset(&act, 0, sizeof(act));
	act.sa_flags = SA_SIGINFO;
	act.sa_sigaction = &pomp_timer_posix_sighandler;
	res = sigaction(POMP_TIMER_POSIX_SIGNO, &act, NULL);
	if (res < 0) {
		POMP_LOG_ERRNO("sigaction");
		goto error;
	}

	/* Success */
	return timer;

	/* Cleanup in case of error */
error:
	if (timer != NULL)
		pomp_timer_posix_destroy(timer);
	return NULL;
}

/**
 * @see pomp_timer_set.
 */
static int pomp_timer_posix_set(struct pomp_timer *timer, uint32_t delay,
		uint32_t period)
{
	int res = 0;
	struct itimerspec newval, oldval;
	POMP_RETURN_ERR_IF_FAILED(timer != NULL, -EINVAL);

	/* Setup timeout */
	newval.it_interval.tv_sec = (time_t)(period / 1000);
	newval.it_interval.tv_nsec = (long int)((period % 1000) * 1000 * 1000);
	newval.it_value.tv_sec = (time_t)(delay / 1000);
	newval.it_value.tv_nsec = (long int)((delay % 1000) * 1000 * 1000);
	if (timer_settime(timer->id, 0, &newval, &oldval) < 0) {
		res = -errno;
		POMP_LOG_ERRNO("timer_settime");
	}

	return res;
}

/**
 * @see pomp_timer_clear.
 */
static int pomp_timer_posix_clear(struct pomp_timer *timer)
{
	POMP_RETURN_ERR_IF_FAILED(timer != NULL, -EINVAL);
	return pomp_timer_posix_set(timer, 0, 0);
}

/** Timer operations for 'posix' implementation */
const struct pomp_timer_ops pomp_timer_posix_ops = {
	.timer_new = &pomp_timer_posix_new,
	.timer_destroy = &pomp_timer_posix_destroy,
	.timer_set = &pomp_timer_posix_set,
	.timer_clear = &pomp_timer_posix_clear,
};

#endif /* POMP_HAVE_TIMER_POSIX */
