// Copyright 2018 Slightech Co., Ltd. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
#include <opencv2/highgui/highgui.hpp>

#include "mynteye/api.h"
#include "mynteye/logger.h"

#include "util/cv_painter.h"

MYNTEYE_USE_NAMESPACE

int main(int argc, char *argv[]) {
  auto &&api = API::Create(argc, argv);
  if (!api)
    return 1;

  // auto-exposure: 0
  api->SetOptionValue(Option::EXPOSURE_MODE, 0);

  // max_gain: range [0,48], default 48
  api->SetOptionValue(Option::MAX_GAIN, 48);
  // max_exposure_time: range [0,240], default 240
  api->SetOptionValue(Option::MAX_EXPOSURE_TIME, 240);
  // desired_brightness: range [0,255], default 192
  api->SetOptionValue(Option::DESIRED_BRIGHTNESS, 192);

  LOG(INFO) << "Enable auto-exposure";
  LOG(INFO) << "Set MAX_GAIN to " << api->GetOptionValue(Option::MAX_GAIN);
  LOG(INFO) << "Set MAX_EXPOSURE_TIME to "
            << api->GetOptionValue(Option::MAX_EXPOSURE_TIME);
  LOG(INFO) << "Set DESIRED_BRIGHTNESS to "
            << api->GetOptionValue(Option::DESIRED_BRIGHTNESS);

  api->Start(Source::VIDEO_STREAMING);

  CVPainter painter(api->GetOptionValue(Option::FRAME_RATE));

  cv::namedWindow("frame");

  while (true) {
    api->WaitForStreams();

    auto &&left_data = api->GetStreamData(Stream::LEFT);
    auto &&right_data = api->GetStreamData(Stream::RIGHT);

    cv::Mat img;
    cv::hconcat(left_data.frame, right_data.frame, img);
    painter.DrawImgData(img, *left_data.img);
    cv::imshow("frame", img);

    char key = static_cast<char>(cv::waitKey(1));
    if (key == 27 || key == 'q' || key == 'Q') {  // ESC/Q
      break;
    }
  }

  api->Stop(Source::VIDEO_STREAMING);
  return 0;
}
