/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "interface/interface.h"
#include "system/system.h"
#include "device/physicaldevice.h"
#include "device/logicaldevice.h"
#include "misc/infoquery.h"
#include <cstring>
#include <cstdio>

namespace GenTL {

Interface::Interface(System* system): Handle(TYPE_INTERFACE), system(system), updateCalled(false),
    portImpl(this), port("eth", "interface.xml", "InterfacePort", "NerianGentTLInterface", &portImpl) {

    const char* devices = std::getenv("NERIAN_GENTL_DEVICES");
    if(devices == nullptr || std::string(devices) == "") {
        deviceList.push_back("udp://0.0.0.0:7681/");
        deviceList.push_back("udp://0.0.0.0:7681/left");
        deviceList.push_back("udp://0.0.0.0:7681/right");
        deviceList.push_back("udp://0.0.0.0:7681/disparity");
        deviceList.push_back("udp://0.0.0.0:7681/pointcloud");
    } else {
        std::string remainingDevices(devices);
        while(remainingDevices != "") {
            std::string device = "";
            size_t separator = remainingDevices.find(';');
            if(separator == std::string::npos) {
                device = remainingDevices;
                remainingDevices = "";
            } else {
                device = remainingDevices.substr(0, separator);
                remainingDevices = remainingDevices.substr(separator + 1);
            }
            deviceList.push_back(device);
        }
    }
}

GC_ERROR Interface::close() {
    return GC_ERR_SUCCESS;
}

GC_ERROR Interface::updateDeviceList(bool8_t* pbChanged, uint64_t iTimeout) {
    if (pbChanged != nullptr) {
        // We signal an update on the first call
        if (updateCalled) {
            *pbChanged = 0;
        }
        else {
            *pbChanged = 1;
        }
    }
    updateCalled = true;
    return GC_ERR_SUCCESS;
}

GC_ERROR Interface::getNumDevices(uint32_t* piNumDevices) {
    if(piNumDevices == nullptr) {
        return GC_ERR_INVALID_PARAMETER;
    } else {
        *piNumDevices = static_cast<unsigned int>(deviceList.size());
        return GC_ERR_SUCCESS;
    }
}

GC_ERROR Interface::getDeviceID(uint32_t iIndex, char* sDeviceID, size_t* piSize) {
    if(iIndex > deviceList.size()) {
        return GC_ERR_INVALID_INDEX;
    }

    InfoQuery info(nullptr, sDeviceID, piSize);
    info.setString(deviceList[iIndex]);
    return info.query();
}

GC_ERROR Interface::getDeviceInfo(const char* sDeviceID, DEVICE_INFO_CMD iInfoCmd,
        INFO_DATATYPE* piType, void* pBuffer, size_t* piSize) {
    if(sDeviceID == nullptr) {
        return GC_ERR_INVALID_ID;
    } else if(iInfoCmd == DEVICE_INFO_SERIAL_NUMBER) {
        return GC_ERR_NOT_AVAILABLE;
    }

    InfoQuery info(piType, pBuffer, piSize);
    switch(iInfoCmd) {
        case DEVICE_INFO_ID:
            info.setString(sDeviceID);
            break;
        case DEVICE_INFO_VENDOR:
            info.setString("Nerian Vision Technologies");
            break;
        case DEVICE_INFO_MODEL:
            info.setString("SP1");
            break;
        case DEVICE_INFO_TLTYPE:
            info.setString("Ethernet");
            break;
        case DEVICE_INFO_DISPLAYNAME:
            info.setString(std::string("SP1 - ") + sDeviceID);
            break;
        case DEVICE_INFO_ACCESS_STATUS:
            if(!checkDeviceInUse(sDeviceID)) {
                info.setInt(DEVICE_ACCESS_STATUS_READWRITE);
            } else {
                info.setInt(DEVICE_ACCESS_STATUS_BUSY);
            }
            break;
        case DEVICE_INFO_USER_DEFINED_NAME:
        case DEVICE_INFO_SERIAL_NUMBER:
        case DEVICE_INFO_VERSION:
            return GC_ERR_NOT_AVAILABLE;
        case DEVICE_INFO_TIMESTAMP_FREQUENCY:
            info.setUInt64(1000000);
            break;
        default:
            return GC_ERR_NOT_IMPLEMENTED;
    }

    return info.query();
}


GC_ERROR Interface::getParentTL(TL_HANDLE* phSystem) {
    if(phSystem == nullptr) {
        return GC_ERR_INVALID_PARAMETER;
    } else {
        *phSystem = reinterpret_cast<TL_HANDLE>(system);
        return GC_ERR_SUCCESS;
    }
}

GC_ERROR Interface::getInfo(INTERFACE_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    return system->getInterfaceInfo("eth", iInfoCmd, piType, pBuffer, piSize);
}

bool Interface::parseDeviceUrl(const char* sDeviceID, bool& udp, std::string& host, std::string& service, std::string& stream) {
    // Device URL have the format:
    // [protocol]://[host]:[port]/[stream]

    std::string id(sDeviceID);
    std::string protocol = id.substr(0, 6);
    host = id.substr(6, id.rfind(':') - 6);
    service = id.substr(id.rfind(':') + 1); // Still has the stream part
    stream = service.substr(service.rfind('/') + 1);
    service = service.substr(0, service.rfind('/'));

    if(protocol == "udp://") {
        udp = true;
    } else if(protocol == "tcp://") {
        udp = false;
    } else {
        return false;
    }

    if(host == "" || service == "") {
        return false;
    } else {
        return true;
    }
}

GC_ERROR Interface::openDevice(const char* sDeviceID, DEVICE_ACCESS_FLAGS iOpenFlags,
        DEV_HANDLE* phDevice) {
    if(sDeviceID == nullptr || strlen(sDeviceID) <= 6) {
        return GC_ERR_INVALID_ID;
    }

    if(checkDeviceInUse(sDeviceID)) {
        return GC_ERR_RESOURCE_IN_USE;
    }

    // Parse the URL
    bool udp = true;
    std::string host, service, stream;
    if(!parseDeviceUrl(sDeviceID, udp, host, service, stream)) {
        return GC_ERR_INVALID_ID;
    }

    // Get index for logical device
    int index = PhysicalDevice::logicalIdToIndex(stream.c_str());
    if(index < 0) {
        return GC_ERR_INVALID_ID;
    }

    PhysicalDevice* physicalDevice = system->findPhysicalDeviceByAddress(
        udp, host.c_str(), service.c_str());

    if(physicalDevice == nullptr) {
        // We could not find an existing physical device. We have to open one
        std::shared_ptr<PhysicalDevice> device(new PhysicalDevice(this));
        if(device->open(udp, host.c_str(), service.c_str()) == GC_ERR_SUCCESS) {
            system->addPhysicalDevice(device);
            physicalDevice = device.get();
        } else {
            return GC_ERR_IO;
        }
    }

    physicalDevice->getLogicalDevice(index)->open();
    *phDevice = reinterpret_cast<DEV_HANDLE>(physicalDevice->getLogicalDevice(index));

    return GC_ERR_SUCCESS;
}

bool Interface::checkDeviceInUse(const char* sDeviceID) {
    bool udp = true;
    std::string host, service, stream;
    if(!parseDeviceUrl(sDeviceID, udp, host, service, stream)) {
        return false;
    }

    PhysicalDevice* physicalDevice = system->findPhysicalDeviceByAddress(
        udp, host.c_str(), service.c_str());

    if(physicalDevice != nullptr) {
        int logicalIndex = PhysicalDevice::logicalIdToIndex(stream.c_str());
        if(logicalIndex < 0) {
            return false; // Invalid id;
        }

        return physicalDevice->getLogicalDevice(logicalIndex)->isOpen();
    } else {
        return false;
    }
}

}
