#include <genicam/gentl.h>
#include <gtest/gtest.h>
#include <iostream>
#include "test-common.h"

using namespace std;
using namespace GenTL;

class DeviceFixture: public ::testing::Test {
public:
   DeviceFixture(): hSystem(nullptr), hIface(nullptr), hDevice0(nullptr), hDevice1(nullptr) {
   }

   ~DeviceFixture() {
   }

   virtual void SetUp( ) {
        ASSERT_EQ(GC_ERR_SUCCESS, GCInitLib());
        ASSERT_EQ(GC_ERR_SUCCESS, TLOpen(&hSystem));
        ASSERT_EQ(GC_ERR_SUCCESS, TLOpenInterface(hSystem, "eth", &hIface));
        ASSERT_EQ(GC_ERR_SUCCESS, IFOpenDevice(hIface, "udp://0.0.0.0:7681/left", DEVICE_ACCESS_READONLY, &hDevice0));
        ASSERT_EQ(GC_ERR_SUCCESS, IFOpenDevice(hIface, "udp://0.0.0.0:7681/right", DEVICE_ACCESS_READONLY, &hDevice1));
   }

   virtual void TearDown( ) {
        if(hDevice0 != nullptr) {
            ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDevice0));
        }
        if(hDevice1 != nullptr) {
            ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDevice1));
        }
        ASSERT_EQ(GC_ERR_SUCCESS, IFClose(hIface));
        ASSERT_EQ(GC_ERR_SUCCESS, TLClose(hSystem));
        ASSERT_EQ(GC_ERR_SUCCESS, GCCloseLib());
   }

protected:
    TL_HANDLE hSystem;
    IF_HANDLE hIface;
    DEV_HANDLE hDevice0;
    DEV_HANDLE hDevice1;
};

TEST_F(DeviceFixture, ParentIF) {
    IF_HANDLE hParentIf = nullptr;
    EXPECT_EQ(GC_ERR_SUCCESS, DevGetParentIF(hDevice0, &hParentIf));
    EXPECT_EQ(hParentIf, hIface);
}

TEST_F(DeviceFixture, NumStreams) {
    uint32_t numStreams = 0;
    EXPECT_EQ(GC_ERR_SUCCESS, DevGetNumDataStreams(hDevice0, &numStreams));
    EXPECT_EQ(numStreams, 1);
}

TEST_F(DeviceFixture, StreamID) {
    char buffer[100];
    size_t size = sizeof(buffer);
    EXPECT_EQ(GC_ERR_SUCCESS, DevGetDataStreamID(hDevice0, 0, buffer, &size));
    EXPECT_EQ(string(buffer), string("default"));

    size = sizeof(buffer);
    EXPECT_EQ(GC_ERR_SUCCESS, DevGetDataStreamID(hDevice1, 0, buffer, &size));
    EXPECT_EQ(string(buffer), string("default"));
}

TEST_F(DeviceFixture, OpenMultiple) {
    DEVICE_ACCESS_STATUS accessStatus;
    size_t size = sizeof(accessStatus);
    INFO_DATATYPE type;

    // Check access status
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/left", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_BUSY);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/right", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_BUSY);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/disparity", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/pointcloud", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);

    // Try re-open 0 device
    DEV_HANDLE hDummy;
    ASSERT_EQ(GC_ERR_RESOURCE_IN_USE, IFOpenDevice(hIface, "udp://0.0.0.0:7681/left", DEVICE_ACCESS_READONLY, &hDummy));

    // Open disparity device
    ASSERT_EQ(GC_ERR_SUCCESS, IFOpenDevice(hIface, "udp://0.0.0.0:7681/disparity", DEVICE_ACCESS_READONLY, &hDummy));
    ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDummy));

    // Open point cloud device
    ASSERT_EQ(GC_ERR_SUCCESS, IFOpenDevice(hIface, "udp://0.0.0.0:7681/pointcloud", DEVICE_ACCESS_READONLY, &hDummy));
    ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDummy));

    // Open multipart device
    ASSERT_EQ(GC_ERR_SUCCESS, IFOpenDevice(hIface, "udp://0.0.0.0:7681/", DEVICE_ACCESS_READONLY, &hDummy));
    ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDummy));

    // Close devices
    ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDevice0));
    hDevice0 = nullptr;
    ASSERT_EQ(GC_ERR_SUCCESS, DevClose(hDevice1));
    hDevice1 = nullptr;

    // Check access status
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/left", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/right", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/disparity", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
    EXPECT_EQ(GC_ERR_SUCCESS, IFGetDeviceInfo(hIface, "udp://0.0.0.0:7681/pointcloud", DEVICE_INFO_ACCESS_STATUS, &type, &accessStatus, &size));
    EXPECT_EQ(accessStatus, DEVICE_ACCESS_STATUS_READWRITE);
}
