/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_ASYNCTRANSFER_H
#define VISIONTRANSFER_ASYNCTRANSFER_H

#include "visiontransfer/common.h"
#include "visiontransfer/imagetransfer.h"
#include "visiontransfer/imagepair.h"

/**
 * \brief Class for asynchronous transfer of image pairs.
 *
 * This class opens a network socket for delivering or receiving image pairs. All
 * operations are performed asynchronously, which means that they do not block.
 * The class encapsulates ImageTransfer.
 */
class VT_EXPORT AsyncTransfer {
public:
    /**
     * \brief Creates a new transfer object.
     *
     * \param mode The transfer mode. This determines which network protocol to
     *        use, and whether to operate in server or client mode.
     * \param remoteAddress Address of the remote host. In TCP server mode this
     *        parameter can be set to a null pointer.
     * \param remoteService The remote port number as string or as textual service name.
     *        In TCP server mode this parameter can be set to a null pointer.
     * \param localAddress Local interface address that is used for communication.
     *        If set to a null pointer, the default interface is used.
     * \param localService The local port number as string or a textual service name.
     *        If set to a null pointer an automatically assigned port number is used.
     * \param bufferSize Buffer size for sending / receiving network data.
     *
     * Please see ImageTransfer::ImageTransfer() for further details.
     */
    AsyncTransfer(ImageTransfer::OperationMode mode, const char* remoteAddress, const char* remoteService,
        const char* localAddress, const char* localService, int bufferSize = 800000);
    ~AsyncTransfer();

    /**
     * \brief Starts an asynchronous transmission of the given image pair
     *
     * \param imagePair The image pair that shall be transmitted.
     * \param deleteData If set to true, the pointers to the pixel data that
     *        are contained in \c imagePair, will be deleted after the
     *        image pair has been transmitted.
     *
     * If deleteData is set to false, the pixel data contained in \c imagePair
     * must not be freed before the data has been transmitted. As transmission
     * happens asynchronously, it is recommended to let AsyncTransfer delete
     * the data pointers.
     */
    void sendImagePairAsync(const ImagePair& imagePair, bool deleteData);

    /**
     * \brief Collects the asynchronously received image.
     *
     * \param imagePair The received image pair.
     * \param timeout The maximum time in seconds for which to wait if no
     *        image pair has been received yet.
     * \return True if an image pair has been received before the timeout.
     *
     * If no image pair has been received, this method might block or return false.
     * Otherwise the returned image pair is valid until the next call.
     *
     * If timeout is set to a value < 0, the function will block indefinitely.
     * If timeout = 0, the function will return immediately, and if timeout is > 0 then
     * the function will block for the given amount of time in seconds. The received
     * image pair is only valid until the next call of collectReceivedImagePair().
     */
    bool collectReceivedImagePair(ImagePair& imagePair, double timeout = -1);

private:
    // We follow the pimpl idiom
    class Pimpl;
    Pimpl* pimpl;

    // This class cannot be copied
    AsyncTransfer(const AsyncTransfer& other);
    AsyncTransfer& operator=(const AsyncTransfer&);
};

#endif
