/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include <algorithm>
#include <iostream>
#include <cstring>
#include "visiontransfer/datablockprotocol.h"
#include "visiontransfer/exceptions.h"

// Network headers
#ifdef _WIN32
#include <winsock2.h>
#else
#include <arpa/inet.h>
#endif

using namespace std;

DataBlockProtocol::DataBlockProtocol(ProtocolType protType): protType(protType),
        transferDone(true), rawData(nullptr), rawValidBytes(0),
        transferOffset(0), transferSize(0), transferSeqNum(0), overwrittenTransferData(0),
        restoreTransferData(false), receiveDataSize(0), receiveSeqNum(0),
        unprocessedMsgLength(0), receiveOffset(0), receptionDone(true) {
    // Determine the maximum allowed payload size
    if(protType == PROTOCOL_TCP) {
        maxPayloadSize = MAX_TCP_BYTES_TRANSFER;
        minPayloadSize = 0;
    } else {
        maxPayloadSize = MAX_UDP_BYTES_TRANSFER - sizeof(unsigned short);
        minPayloadSize = MIN_UDP_BYTES_TRANSFER;
    }
}

void DataBlockProtocol::startTransfer() {
    transferSeqNum = 0;
    transferDone = false;
    restoreTransferData = false;
}

void DataBlockProtocol::resetTransfer() {
    transferDone = true;
    restoreTransferData = false;
}

void DataBlockProtocol::setTransferData(unsigned char* data, int size, int validBytes) {
    rawData = data;
    transferSize = size;
    transferOffset = 0;
    restoreTransferData = false;
    rawValidBytes = min(transferSize, validBytes);
}

void DataBlockProtocol::setTransferValidBytes(int validBytes) {
    if(validBytes >= transferSize) {
        rawValidBytes = transferSize;
    } else if(validBytes < static_cast<int>(sizeof(unsigned short))) {
        rawValidBytes = 0;
    } else {
        // Always leave a small trailer of additional bytes to folow,
        // as this one will be buffered for the next transfer.
        rawValidBytes = validBytes - sizeof(unsigned short);
    }
}

const unsigned char* DataBlockProtocol::getTransferMessage(int& length) {
    if(transferDone) {
        length = 0;
        return nullptr;
    }

    // Determine the payload length
    length = min(maxPayloadSize, rawValidBytes - transferOffset);
    if(length == 0 || (length < minPayloadSize && rawValidBytes != transferSize)) {
        length = 0;
        return nullptr; //Not enough to send yet
    }

    unsigned char* buffer = &rawData[transferOffset];
    transferOffset += length;

    if(restoreTransferData) {
        // Restore the original memory content that was overwritten with
        // the previous sequence number
        *reinterpret_cast<unsigned short*>(&buffer[0]) = overwrittenTransferData;
    }

    if(protType == PROTOCOL_UDP) {
        // For udp, we always append a sequence number
        unsigned short* seqNumPtr = reinterpret_cast<unsigned short*>(&buffer[length]);
        overwrittenTransferData = *seqNumPtr;
        *seqNumPtr = htons(transferSeqNum);
        length += sizeof(unsigned short);
        restoreTransferData = true;
    }

    transferSeqNum++;

    return buffer;
}

bool DataBlockProtocol::transferComplete() {
    return transferOffset >= transferSize;
}

void DataBlockProtocol::setReceiveDataSize(int size) {
    if(size != receiveDataSize) {
        receiveDataSize = size;

        int bufferSize;

        // We increase the requested size to allow for one
        // additional network message and the protocol overhead
        if(protType == PROTOCOL_TCP) {
            bufferSize = size + MAX_TCP_BYTES_TRANSFER + sizeof(unsigned short);
        } else {
            bufferSize = size + MAX_UDP_BYTES_TRANSFER + sizeof(unsigned short);
        }

        // Resize the buffer
        receiveBuffer.resize(bufferSize);
    }
}

unsigned char* DataBlockProtocol::getNextReceiveBuffer(int maxLength) {
    if(static_cast<int>(receiveBuffer.size() - receiveOffset) < maxLength) {
        throw ProtocolException("No more receive buffers available!");
    }

    if(receptionDone) {
        // Start receiving a new data block
        receptionDone = false;
        receiveOffset = 0;
        receiveSeqNum = 0;
    }

    return &receiveBuffer[receiveOffset];
}

bool DataBlockProtocol::processReceivedMessage(int length) {
    if(receiveOffset + length > static_cast<int>(receiveBuffer.size())) {
        throw ProtocolException("Received message size is invalid!");
    }

    if(length == 0) {
        // Nothing received. Let's not call this an error.
        return true;
    } else if(protType == PROTOCOL_UDP) {
        // UDP specific payload handling

        // Correct the length
        length -= sizeof(unsigned short);

        // Extract the sequence number
        unsigned short seqNum = ntohs(
            *reinterpret_cast<unsigned short*>(&receiveBuffer[receiveOffset + length]));

        if(seqNum != static_cast<unsigned short>(receiveSeqNum)) {
            // Sequence numbers don't match. Probably dropped a packet
            resetReception();
            return false;
        }

        // Update sequence number
        receiveSeqNum++;
    } else {
        // For TCP we might have some outstanding bytes from the
        // previous transfer. Lets copy that part from a separate buffer.
        if(unprocessedMsgLength != 0) {
            if(length + unprocessedMsgLength > MAX_OUTSTANDING_BYTES) {
                throw ProtocolException("Received too much data!");
            }

            ::memmove(&receiveBuffer[unprocessedMsgLength], &receiveBuffer[0], length);
            ::memcpy(&receiveBuffer[0], &unprocessedMsgPart[0], unprocessedMsgLength);
            length += unprocessedMsgLength;
            unprocessedMsgLength = 0;
        }

        // The message might also contain extra bytes for the next
        // transfer. Lets copy that part into a separate buffer.
        if(receiveOffset + length > receiveDataSize) {
            int newLength = static_cast<int>(receiveDataSize - receiveOffset);

            if(unprocessedMsgLength != 0 || length - newLength > MAX_OUTSTANDING_BYTES) {
                throw ProtocolException("Received too much data!");
            }

            unprocessedMsgLength = length - newLength;
            ::memcpy(unprocessedMsgPart, &receiveBuffer[receiveOffset + newLength], unprocessedMsgLength);

            length = newLength;
        }
    }

    // Update the receive buffer offset
    receiveOffset += length;

    if(receiveOffset > static_cast<int>(receiveBuffer.size())) {
        throw ProtocolException("Receive buffer overflow!");
    }

    return true;
}

void DataBlockProtocol::finishReception() {
    receptionDone = true;
}

void DataBlockProtocol::resetReception() {
    receiveOffset = 0;
    receiveSeqNum = 0;
}

unsigned char* DataBlockProtocol::getReceivedData(int& length) {
    length = receiveOffset;
    return &receiveBuffer[0];
}
