/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_DATABLOCKPROTOCOL_H
#define VISIONTRANSFER_DATABLOCKPROTOCOL_H

#include <map>
#include <vector>
#include <memory>
#include "visiontransfer/alignedallocator.h"

/**
 * \brief A protocol for transmitting large blocks of data over a network.
 *
 * The protocol slices the large data block into several smaller chunks
 * that can be transmitted over a network. In UDP mode, each chunk contains
 * a sequence number.
 *
 * This class is intended to be used by ImageProtocol and should normally
 * not be used directly.
 */

class DataBlockProtocol {
public:
    enum ProtocolType {
        PROTOCOL_TCP,
        PROTOCOL_UDP
    };

    // Constants that are also used in other places.
    static const int MAX_UDP_BYTES_TRANSFER = 1472;
    //static const int MAX_UDP_BYTES_TRANSFER = 8972;
    static const int MAX_TCP_BYTES_TRANSFER = 0xFFFF; //64K - 1
    static const int MAX_OUTSTANDING_BYTES = 2*MAX_TCP_BYTES_TRANSFER;

    DataBlockProtocol(ProtocolType protType);

    /**
     * \brief Returns the size of the overhead data that is required for
     * transferring a single network message.
     */
    int getProtocolOverhead() const {
        return protType == PROTOCOL_UDP ? sizeof(unsigned short) : 0;
    }

    /**
     * \brief Returns the maximum paload size that can be transferred / received.
     */
    int getMaxPayloadSize() const {return maxPayloadSize;}

    /**
     * \brief Starts the transfer of a new data block
     *
     * Call this method before setTransferData().
     */
    void startTransfer();

    /**
     * \brief Stops the current transfer
     */
    void resetTransfer();

    /**
     * \brief Sets a new chunk of data that should be transferred.
     *
     * \param data Pointer to the data that should be transferred.
     * \param size Total size of the data transfer.
     * \param validBytes The number of bytes that are currently
     *        valid in \c data.
     *
     * Part of \c data will be overwritten. There must be at least 2 additional
     * bytes before the start of \c data.
     *
     * If \c validBytes is set to a value smaller than the total transfer
     * size, only a partial transfer is performed. Subsequent calls to
     * setTransferValidBytes() are then necessary.
    */
    void setTransferData(unsigned char* data, int size, int validBytes = 0x7FFFFFFF);

    /**
     * \brief Updates the number of valid bytes in a partial transmission.
     *
     * \param validBytes The number of already valid bytes in the previously
     *        set data pointer.
     *
     * This method has to be called whenever new data is available in a
     * partial transfer. \see setTransferData()
     */
    void setTransferValidBytes(int validBytes);

    /**
     * \brief Gets the next network message for the current transfer.
     *
     * \param length The length of the network message.
     * \return Pointer to the network message data.
     *
     * If the transfer has already been completed or if there are currently
     * no more valid bytes to be transmitted, a null pointer is returned.
     */
    const unsigned char* getTransferMessage(int& length);

    /**
     * \brief Returns true if the current transfer has been completed.
     */
    bool transferComplete();

    /**
     * \brief Sets the total size of the data that shall be received
     *
     * \param size New data size.
     *
     * This method has to be called before receiving the first data item.
     * It can be called within an active transfer. This is typically done to
     * set the expected data size after the header has been received.
     */

    void setReceiveDataSize(int size);

    /**
     * \brief Gets a buffer for receiving the next network message
     *
     * The returned buffer is a subsection of the internal receive buffer
     */
    unsigned char* getNextReceiveBuffer(int maxLength);

    /**
     * \brief Resets the message reception.
     */
    void resetReception();

    /**
     * \brief Handles a received network message
     *
     * Please see ImageProtocol::processReceivedMessage() for further details.
     */
    bool processReceivedMessage(int length);

    /**
     * \brief Returns the data that has been received for the current data block.
     *
     * \param length Will be set to the number of bytes that have been received.
     * \return Pointer to the buffer containing the received data.
     *
     * The received data is valid until receiving the first byte of a new
     * data block.
     */
    unsigned char* getReceivedData(int& length);

    /**
     * \brief Finishes reception of the current data block.
     *
     * The first network message that is added after calling this method
     * will start the reception of a new data block.
     */
    void finishReception();

private:
    // The pimpl idiom is not necessary here, as this class is usually not
    // used directly
    static const int MIN_UDP_BYTES_TRANSFER = 512;

    ProtocolType protType;
    int maxPayloadSize;
    int minPayloadSize;

    // Transfer related variables
    bool transferDone;
    unsigned char* rawData;
    int rawValidBytes;
    int transferOffset;
    int transferSize;
    unsigned short transferSeqNum;
    unsigned short overwrittenTransferData;
    bool restoreTransferData;

    // Reception related variables
    std::vector<unsigned char, AlignedAllocator<unsigned char> > receiveBuffer;
    int receiveDataSize;
    unsigned short receiveSeqNum;
    unsigned char unprocessedMsgPart[MAX_OUTSTANDING_BYTES];
    int unprocessedMsgLength;
    int receiveTotalSize;
    int receiveOffset;
    bool receptionDone;

    const unsigned char* extractPayload(const unsigned char* data, int& length, bool& error);
};

#endif
