/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_RECONSTRUCT3D_H
#define VISIONTRANSFER_RECONSTRUCT3D_H

#include <limits>
#include "visiontransfer/common.h"
#include "visiontransfer/imagepair.h"

/**
 * \brief Transforms a disparity map into a set of 3D points.
 *
 * Use this class for reconstructing the 3D location for each valid
 * point in a disparity map.
 */
class VT_EXPORT Reconstruct3D {
public:
    /**
     * \brief Constructs a new object for 3D reconstructing.
     */
    Reconstruct3D();

    ~Reconstruct3D();

    /**
     * \brief Reconstructs the 3D location of each pixel in the given
     * disparity map.
     *
     * \param dispMap Pointer to the data of the disparity map. The disparity map is
     *        assumed to have a 4-bit subpixel resolution. This means that each
     *        value needs to be divided by 16 to receive the true disparity.
     * \param width Width of the disparity map.
     * \param height Height of the disparity map.
     * \param rowStride Row stride (i.e. distance between two rows in bytes) of the
     *        disparity map.
     * \param q Disparity-to-depth mapping matrix of size 4x4. The matrix is
     *        stored in a row-wise alignment. Obtain this matrix from your
     *        camera calibration data.
     * \param minDisparity The minimum disparity, again with 4-bit subpixel
     *        resolution. Lower disparities will be clamped to this value before
     *        computing the 3D location.
     *
     * The output map will have a size of exactly 4*width*height float values. For each
     * point the x, y and z coordinates are stored consecutively, plus one additional
     * float (four bytes) as padding. Invalid disparities will be set to the given minimum disparity.
     *
     * If the minimum disparity is set to 0, points with a disparity of 0 or an invalid
     * disparity will receive a z coordinate of +inf.
     *
     * The returned point map is valid until the next call of createPointMap() or
     * writePlyFile().
     */
    float* createPointMap(const unsigned short* dispMap, int width, int height,
        int rowStride, const float* q, unsigned short minDisparity = 1);

    /**
     * \brief Reconstructs the 3D location of each pixel in the given
     * disparity map.
     *
     * \param imagePair Image pair containing the disparity map.
     * \param minDisparity The minimum disparity with 4-bit subpixel resolution.
     *
     * The second image in \c imagePair is interpreted as the disparity map.
     *
     * Please see createPointMap(const unsigned short*, int, int, int, const float*, unsigned short)
     * for further details.
     */
    float* createPointMap(const ImagePair& imagePair, unsigned short minDisparity = 1);

    /**
     * \brief Projects the given disparity map to 3D points and exports the result to
     * a PLY file.
     *
     * \param file The output file
     * \param dispMap Pointer to the data of the disparity map. The disparity map is
     *        assumed to have a 4-bit subpixel resolution. This means that each
     *        value needs to be divided by 16 to receive the true disparity.
     * \param left Rectified left input image in 8-bit gray scale format.
     * \param width Width of the disparity map and left image.
     * \param height Height of the disparity map and left image.
     * \param dispRowStride Row stride (i.e. distance between two rows in bytes) of the
     *        disparity map.
     * \param imageRowStride Row stride (i.e. distance between two rows in bytes) of the
     *        left image.
     * \param q Disparity-to-depth mapping matrix of size 4x4. The matrix is
     *        stored in a row-wise alignment. Obtain this matrix from your
     *        camera calibration data.
     * \param maxZ Maximum allowed z-coordinate. Points with a higher z-coordinate
     *        are not exported.
     *
     * The left image has to be provided for assigning a color to each point.
     */
    void writePlyFile(const char* file, const unsigned short* dispMap,
        const unsigned char* image,  int width, int height,
        int dispRowStride, int imageRowStride, const float* q,
        double maxZ = std::numeric_limits<double>::max());

    /**
     * \brief Projects the given disparity map to 3D points and exports the result to
     * a PLY file.
     *
     * \param imagePair Image pair containing camera image and disparity map.
     * \param maxZ Maximum allowed z-coordinate.
     *
     * The first image in \c imagePair is interpreted as camera image and the second
     * image as disparity map.
     *
     * Please see writePlyFile(const char*, const unsigned short*, const unsigned char*,  int, int, int, int, const float*, double)
     * for further details.
     */
    void writePlyFile(const char* file, const ImagePair& imagePair,
        double maxZ = std::numeric_limits<double>::max());

private:
    // We follow the pimpl idiom
    class Pimpl;
    Pimpl* pimpl;
    
    // This class cannot be copied
    Reconstruct3D(const Reconstruct3D& other);
    Reconstruct3D& operator=(const Reconstruct3D&);
};

#endif
