/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "displaywidget.h"
#include "spcom.h"

constexpr int DisplayWidget::IMAGE_PADDING;

DisplayWidget::DisplayWidget(QWidget * parent): QWidget(parent),
        resizeWidget(false), displayLeft(true), displayRight(true), zoomPercent(100) {
}

void DisplayWidget::setSpCom(std::shared_ptr<SpCom> spcom) {
    this->spcom = spcom;
}

void DisplayWidget::setDisplayFrame(const cv::Mat_<cv::Vec3b>& left,
        const cv::Mat_<cv::Vec3b>& right, bool resize) {
    displayLeftFrame = left;
    displayRightFrame = right;
    if(resize) {
        resizeWidget = true;
    }
}

void DisplayWidget::setZoom(int percent) {
    zoomPercent = percent;
    resizeWidget = true;
}

void DisplayWidget::setImagesToDisplay(bool displayLeft, bool displayRight) {
    this->displayLeft = displayLeft;
    this->displayRight = displayRight;
}

void DisplayWidget::paintEvent(QPaintEvent *) {
    if(spcom == nullptr) {
        // Should never happen
        return;
    }

    std::unique_lock<std::mutex> lock(spcom->getDisplayMutex());
    if(displayLeftFrame.data == nullptr) {
        return; // No frame available yet
    }

    double scale = zoomPercent/100.0;

    // Resize widget if necessary
    if(resizeWidget) {
        int width = 0;
        if(displayLeft && displayRight) {
            width = displayLeftFrame.cols*scale + displayRightFrame.cols*scale + IMAGE_PADDING;
        } else if(displayLeft) {
            width = displayLeftFrame.cols*scale;
        } else if(displayRight) {
            width = displayRightFrame.cols*scale;
        }

        setFixedSize(width, displayLeftFrame.rows*scale);
        resizeWidget = false;
    }
    
    painter.begin(this);
    painter.scale(scale, scale);

    // Display the received images
    QImage img1(displayLeftFrame.data, displayLeftFrame.cols, displayLeftFrame.rows,
        displayLeftFrame.step[0], QImage::Format_RGB888);

    QImage img2(displayRightFrame.data, displayRightFrame.cols, displayRightFrame.rows,
        displayRightFrame.step[0], QImage::Format_RGB888);

    if(displayLeft && displayRight) {
        painter.drawImage(0, 0, img1);
        painter.drawImage(displayLeftFrame.cols + IMAGE_PADDING/scale, 0, img2);
    } else if(displayLeft) {
        painter.drawImage(0, 0, img1);
    } else if(displayRight) {
        painter.drawImage(0, 0, img2);
    }

    painter.end();
}
