/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include <iostream>
#include <iomanip>
#include <cstdio>
#include <functional>
#include <stdexcept>
#include <string>
#include <chrono>

#include "filequeue.h"
#include "spcom.h"
#include "ui_mainwindow.h"
#include "preferencesdialog.h"

#include <QPainter>
#include <QFileDialog>
#include <QInputDialog>
#include <QLabel>
#include <QSettings>
#include <QMessageBox>

#ifdef _WIN32
#include <windows.h>
#endif

using namespace std;
using namespace std::chrono;
using namespace cv;

SpCom::SpCom(const Settings& newSettings):
    terminateThreads(false), captureNextFrame(false), captureSequence(false),
    captureIndex(0), seqNum(0), minDisparity(0), maxDisparity(0) {

    updateSettings(newSettings);
}

SpCom::~SpCom() {
    terminate();
}

void SpCom::terminate() {
    joinAllThreads();
    stereoQueue.reset(nullptr);
    asyncTrans.reset();
}

void SpCom::joinAllThreads() {
    terminateThreads = true;

    if(writingThread.joinable()) {
        writingCond.notify_all();
        writingThread.join();
    }

    if(mainLoopThread.joinable()) {
        mainLoopThread.join();
    }
}

void SpCom::connect() {
    // Just need to start the threads
    terminateThreads = false;
    mainLoopThread = std::thread(std::bind(&SpCom::mainLoop, this));
    writingThread = std::thread(std::bind(&SpCom::writeLoop, this));
}

void SpCom::mainLoop() {
    cv::Mat receivedLeftFrame, receivedRightFrame;
    int endTransmissionCtr = 0;

    try {
        // First establish network connection
        if(asyncTrans == nullptr) {
            asyncTrans.reset(new AsyncTransfer(settings.tcp ? ImageTransfer::TCP_CLIENT : ImageTransfer::UDP,
                settings.remoteHost.c_str(), to_string(settings.remotePort).c_str(),
                settings.disableReception ? nullptr : settings.localHost.c_str(),
                settings.disableReception ? "0" : to_string(settings.localPort).c_str()));
            connectedCallback();
        }

        while(!terminateThreads) {
            if(!transmitFrame() && endTransmissionCtr < 10) {
                // Try to yet receive frames for a little bit longer
                settings.readImages = false;
                endTransmissionCtr++;
                if(endTransmissionCtr == 10) {
                    sendCompleteCallback();
                }
            }

            if(!settings.disableReception) {
                do {
                    ImagePair imagePair;
                    receiveFrame(imagePair, receivedLeftFrame, receivedRightFrame);
                    if(receivedLeftFrame.data == nullptr || receivedRightFrame.data == nullptr) {
                        break;
                    }

                    int minDisp = 0, maxDisp = 0;
                    imagePair.getDisparityRange(minDisp, maxDisp);

                    if(minDisp != minDisparity || maxDisp != maxDisparity) {
                        minDisparity = minDisp;
                        maxDisparity = maxDisp;

                        // Force update of legend
                        convertedLeftFrame = cv::Mat();
                        convertedRightFrame = cv::Mat();
                        redBlueCoder.reset(new ColorCoder(ColorCoder::COLOR_RED_BLUE_RGB, minDisp*16,
                            maxDisp*16, true, true));
                        rainbowCoder.reset(new ColorCoder(ColorCoder::COLOR_RAINBOW_RGB, minDisp*16,
                            maxDisp*16, true, true));
                    }

                    colorCodeAndDisplay(receivedLeftFrame, receivedRightFrame);
                    captureFrameIfRequested(imagePair, receivedLeftFrame, receivedRightFrame);
                } while(settings.readImages && receivedLeftFrame.data != nullptr && !terminateThreads);
            }

            // Sleep for a while if we are processing too fast
            if(settings.readImages) {
                frameRateLimit->next();
            }
        }
    } catch(const std::exception& ex) {
        exceptionCallback(ex);
    }
}

void SpCom::colorCodeAndDisplay(const cv::Mat& receivedLeftFrame, const cv::Mat& receivedRightFrame) {
    if(receivedRightFrame.data == nullptr) {
        return; // Can't display anything
    }

    // Prepare window resizing
    bool resize = false;
    if(lastFrameSize != receivedLeftFrame.size()) {
        // Size of buffers will change
        convertedLeftFrame = cv::Mat();
        convertedRightFrame = cv::Mat();
        resize = true;
        lastFrameSize = receivedLeftFrame.size();
    }

    // Perform color coding or just use the grayscale image
    {
        unique_lock<mutex> lock(displayMutex);
        convertFrame(receivedLeftFrame, convertedLeftFrame);
        convertFrame(receivedRightFrame, convertedRightFrame);
        frameDisplayCallback(convertedLeftFrame, convertedRightFrame, resize);
    }
}

bool SpCom::transmitFrame() {
    StereoFrame8U::ConstPtr stereoSendFrame;

    // Get frame to send if an image queue exists
    if(stereoQueue != nullptr) {
        stereoSendFrame = stereoQueue->pop();
        if(stereoSendFrame == nullptr) {
            // No more frames
            return false;
        }
    }

    // Transmit frame
    if(stereoSendFrame != nullptr) {
        ImagePair pair;
        pair.setWidth(stereoSendFrame->first.cols);
        pair.setHeight(stereoSendFrame->first.rows);
        pair.setRowStride(0, stereoSendFrame->first.step[0]);
        pair.setRowStride(1, stereoSendFrame->first.step[0]);
        pair.setPixelFormat(0, ImagePair::FORMAT_8_BIT);
        pair.setPixelFormat(1, ImagePair::FORMAT_8_BIT);
        pair.setSequenceNumber(seqNum++);

        steady_clock::time_point time = steady_clock::now();
        long long microSecs = duration_cast<microseconds>(time.time_since_epoch()).count();
        pair.setTimestamp(microSecs / 1000000, microSecs % 1000000);

        // Clone image data such that we can delete the original
        unsigned char* leftPixel = new unsigned char[stereoSendFrame->first.step[0] * stereoSendFrame->first.rows];
        unsigned char* rightPixel = new unsigned char[stereoSendFrame->second.step[0] * stereoSendFrame->second.rows];
        memcpy(leftPixel, stereoSendFrame->first.data, stereoSendFrame->first.step[0] * stereoSendFrame->first.rows);
        memcpy(rightPixel, stereoSendFrame->second.data, stereoSendFrame->second.step[0] * stereoSendFrame->second.rows);

        pair.setPixelData(0, leftPixel);
        pair.setPixelData(1, rightPixel);

        asyncTrans->sendImagePairAsync(pair, true);
    }

    return true;
}

void SpCom::receiveFrame(ImagePair& imagePair, cv::Mat& receivedLeftFrame, cv::Mat& receivedRightFrame) {
    if(!asyncTrans->collectReceivedImagePair(imagePair, settings.readImages ? 0 : 0.1)) {
        // No image received yet
        imagePair = ImagePair();
        receivedLeftFrame = cv::Mat();
        receivedRightFrame = cv::Mat();
        return;
    }

    if(settings.printTimestamps) {
        int secs = 0, microsecs = 0;
        imagePair.getTimestamp(secs, microsecs);
        cout << (secs + (microsecs * 1e-6)) << endl;
    }

    // Convert received data to opencv images
    receivedLeftFrame = cv::Mat_<unsigned char>(imagePair.getHeight(), imagePair.getWidth(),
        imagePair.getPixelData(0), imagePair.getRowStride(0));

    if(imagePair.getPixelFormat(1) == ImagePair::FORMAT_12_BIT) {
        receivedRightFrame = cv::Mat_<unsigned short>(imagePair.getHeight(), imagePair.getWidth(),
            reinterpret_cast<unsigned short*>(imagePair.getPixelData(1)), imagePair.getRowStride(1));
    } else {
        receivedRightFrame = cv::Mat_<unsigned char>(imagePair.getHeight(), imagePair.getWidth(),
            imagePair.getPixelData(1), imagePair.getRowStride(1));
    }
}

void SpCom::convertFrame(const cv::Mat& src, cv::Mat& dst) {
    if(src.type() == CV_16U) {
        if(settings.colorScheme == Settings::COLOR_SCHEME_NONE) {
            // Convert 16 to 8 bit
            cv::Mat_<unsigned char> image8Bit;
            src.convertTo(image8Bit, CV_8U, 1.0/16.0);
            cvtColor(image8Bit, dst, CV_GRAY2RGB);
        } else {
            std::unique_ptr<ColorCoder>& coder = (
                settings.colorScheme == Settings::COLOR_SCHEME_RED_BLUE ? redBlueCoder : rainbowCoder);
            if(coder != nullptr) {
                // Perform color coding
                if(dst.data == nullptr) {
                    dst = coder->createLegendBorder(src.cols, src.rows, 1.0/16.0);
                }
                cv::Mat_<cv::Vec3b> dstSection = dst(Rect(0, 0, src.cols, src.rows));
                coder->codeImage((cv::Mat_<unsigned short>)src, dstSection);
            }
        }
    } else {
        // Just convert grey to 8 bit
        cvtColor(src, dst, CV_GRAY2RGB);
    }
}

void SpCom::captureFrameIfRequested(const ImagePair& imagePair, const cv::Mat& receivedLeftFrame,
        const cv::Mat& receivedRightFrame) {

    if(captureNextFrame || captureSequence) {
        if(!captureSequence) {
            cout << "Writing frame " << captureIndex << endl;
        }

        captureNextFrame = false;

        if(receivedRightFrame.data != nullptr) {
            // Schedule write for left frame
            if(receivedLeftFrame.type() == CV_8U || (receivedLeftFrame.type() == CV_16U && settings.writeRaw16Bit)) {
                scheduleWrite(receivedLeftFrame.clone(), captureIndex, 0);
            } else {
                cv::Mat_<cv::Vec3b> rgbImage;
                cvtColor(convertedLeftFrame, rgbImage, CV_RGB2BGR);
                scheduleWrite(rgbImage, captureIndex, 0);
            }

            // Schedule write for right frame
            if(receivedRightFrame.type() == CV_8U || (receivedRightFrame.type() == CV_16U && settings.writeRaw16Bit)) {
                scheduleWrite(receivedRightFrame.clone(), captureIndex, 1);
            } else {
                cv::Mat_<cv::Vec3b> rgbImage;
                cvtColor(convertedRightFrame, rgbImage, CV_RGB2BGR);
                scheduleWrite(rgbImage, captureIndex, 1);
            }

            // For simplicity, point clouds are not written asynchroneously
            if(settings.writePointCloud) {
                char fileName[17];
                snprintf(fileName, sizeof(fileName), "image%04d_3d.ply", captureIndex);
                recon3d.writePlyFile((settings.writeDir + "/" + fileName).c_str(), imagePair, settings.pointCloudMaxDist);
            }
        }
        captureIndex++;
    }
}

void SpCom::writeLoop() {
    try {
        while(!terminateThreads) {
            pair<string, Mat> frame;
            {
                unique_lock<mutex> lock(writingMutex);
                while(writeQueue.size() == 0 && !terminateThreads) {
                    writingCond.wait(lock);
                }
                if(terminateThreads) {
                    return;
                }
                frame = writeQueue.front();
                writeQueue.pop();
            }
            if(frame.second.data != NULL && frame.second.cols != 0) {
                string filePath = settings.writeDir + "/" + string(frame.first);
                if(!imwrite(filePath, frame.second))
                    cerr << "Error writing file: " << frame.first << endl;
            }
        }
    } catch(const std::exception& ex) {
        exceptionCallback(ex);
    }
}

void SpCom::scheduleWrite(const cv::Mat& frame, int index, int camera) {
    char fileName[17];
    snprintf(fileName, sizeof(fileName), "image%04d_c%d.png", index, camera);

    unique_lock<mutex> lock(writingMutex);
    writeQueue.push(pair<string, Mat>(fileName, frame));
    writingCond.notify_one();
}

void SpCom::updateSettings(const Settings& newSettings) {
    bool restartThreads = false;
    if(mainLoopThread.joinable() || writingThread.joinable()) {
        restartThreads = true;
        joinAllThreads();
    }

    if(newSettings.colorScheme != settings.colorScheme) {
        unique_lock<mutex> lock(displayMutex);

        // Make sure that a new color legend is created
        convertedLeftFrame = cv::Mat();
        convertedRightFrame = cv::Mat();
        lastFrameSize = cv::Size2i(0,0);
    }

    if(newSettings.readImages != settings.readImages ||
            newSettings.readDir != settings.readDir) {
        if(newSettings.readDir != "" && newSettings.readImages) {
            stereoQueue.reset(new StereoFileQueue8U(newSettings.readDir.c_str(), 10));
        } else {
            stereoQueue.reset();
        }
    }

    frameRateLimit.reset(new RateLimit(newSettings.maxFrameRate));

    settings = newSettings;

    if(restartThreads) {
        connect();
    }
}
