/** @file     logging.h
 *  @author   Adi Singh
 *  @date     August 2019
 */

#ifndef LOGGING_H
#define LOGGING_H

#define PCL_NO_PRECOMPILE

#include <ros/package.h>
#include <ros/ros.h>

#include <pcl/io/io_exception.h>
#include <pcl/io/pcd_io.h>
#include <pcl_ros/point_cloud.h>

#include "toposens_msgs/TsPoint.h"
#include "toposens_pointcloud/mapping.h"
#include "toposens_pointcloud/ts_pointcloud_ros.h"

namespace toposens_pointcloud
{
/**
 * @brief Demonstrates basic TF and PCL integrations for TsScan data.
 * @details Subscribes to a topic publishing TsPoints and maintains an
 * cummulative pointcloud structure of all points.
 *
 * Maintains a persistent PointCloud structure holding the cumulative
 * of all scans received during a run. This cumulative data is saved
 * periodically as a PCD file in the package folder for
 * subsequent use with other PCL tools (like pcl_octree_viewer).
 */
class Logging
{
public:
  /**
   * Subscribes to a #kCloudTopic and prepares a PointCloud structure
   * for persistent storage.
   * @param nh Public nodehandle for pub-sub ops on ROS topics.
   * @param private_nh Private nodehandle for accessing launch parameters.
   * @param pcd_save_rate defines save interval of cumulative pointcloud.
   */
  Logging(ros::NodeHandle nh, ros::NodeHandle private_nh);

  /**
   * Destructor frees memory from heap
   */
  ~Logging();

  /**
   * Saves all pointcloud data contained in #store as a PCD file whose path is
   * specified by #pcd_path.
   * @details Filename is provided as a launch argument. File is saved as a
   * .pcd.
   * @param  event  TimerEvent, currently ignored.
   */
  void save(const ros::TimerEvent &event);

  /** Self-contained Logging class handler that can be threaded.
   *  @param pcd_save_rate defines save interval of cumulative pointcloud.
   */
  static void logPointcloud()
  {
    ros::NodeHandle nh;
    ros::NodeHandle private_nh("~");

    Logging l(nh, private_nh);

    while (true)
    {
      try
      {
        boost::this_thread::sleep(boost::posix_time::milliseconds(10));
      }
      catch (boost::thread_interrupted &)
      {
        return;
      }
    }
  }

private:
  /**
   * Waits for pointcloud updates to become available and accumulates them
   * in a perisistent pointcloud structure.
   * @details Waits for a pointcloud updates to become available and accumulates
   * them in a persistent pointcloud structure.
   * @param  msg  Pointer to an incoming TsScan message.
   */
  void _accumulate(const XYZINCloud::ConstPtr &msg);

  XYZINCloud::Ptr store_; /**< Collection of all pointclouds from a single run.*/
  std::string pcd_path_;  /**< Path of file in which pointcloud is stored by #save().*/

  ros::Subscriber cloud_sub_; /**< Handler for subscribing to TsPoints.*/
  ros::Timer timer_;          /**< Timer for pointcloud saving interval. */

  boost::mutex store_mutex_; /**< Mutex for locking pointcloud.*/
};

}  // namespace toposens_pointcloud

#endif
