#ifndef TS_POINTCLOUD_ROS_H
#define TS_POINTCLOUD_ROS_H

#include <ros/package.h>
#include <ros/ros.h>
#include <tf2_ros/transform_listener.h>
#include <visualization_msgs/Marker.h>

#include "toposens_pointcloud/logging.h"
#include "toposens_pointcloud/mapping.h"

namespace toposens_pointcloud
{
static const std::string kPointCloudTopic = "ts_cloud";

/**
 * @brief ROS interface for transforming the pointclouds
 *
 * @details Implements methods that set up publishers, subscribers and methods
 * for using the Mapping library in a ROS framework.
 *
 * Subscribes to a topic for receiving TsScans and converts incoming data into
 * PointCloud messages using PCL, optionally mapped to another frame of
 * reference. Visual behavior of the pointcloud (scale, color, direction,
 * lifetime, etc) can be controlled via the PointCloud2 display in Rviz.
 *
 * Each incoming scan is converted to a new PointCloud message of template XYZI,
 * which corresponds well with a TsPoint structure. A second persistent
 * PointCloud structure holds the cumulative of all scans received during a run.
 * At the end of each run, this cumulative data is saved as a PCD file in the
 * package folder for subsequent use with other PCL tools (like
 * pcl_octree_viewer).
 *
 * Prior to pointcloud conversion, each scan can also be optionally transformed
 * to a different reference frame (usually, 'world' or 'base_link' frame). This
 * can be used, for instance, to efficiently map an area using sensor or
 * odometry data.
 */
class TSPointCloudROS
{
public:
  /**
   * Initializes subscribers, publishers, transform object
   * @param nh public nodehandle for publishers and subscribers
   * @param private_nh private nodehandle for dynamic reconfigure server
   */
  TSPointCloudROS(ros::NodeHandle nh, ros::NodeHandle private_nh);

  /**
   * Destructor frees memory from heap
   */
  ~TSPointCloudROS();

private:
  /**
   * Converts and broadcasts incoming scans as PointCloud messages.
   * Maintains a local copy of all accumulated scans in their
   * corresponding PointCloud form.
   * @details Waits for a target frame transform to become available and maps
   * incoming TsPoints to this frame. Transformed points are added to a
   * instantaneous pointcloud structure for publishing.
   * @param msg Pointer to an incoming TsScan message.
   */
  void cloudCallback(const toposens_msgs::TsScan::ConstPtr &msg);

  /**
   * Maps a point to the reference frame defined in #target_frame.
   * @details Converts a PointXYZINormal to intermediate Vector Transformation
   * variables. After transformation converts them back to a PointXYZINormal.
   * Transformation is skipped if #target_frame is same as the pcl_point's
   * current frame.
   * @param transform StampedTransform from original frame to target frame.
   * @param p PointXYZINormal that is transformed to #target_frame.
   * @param h Header data of TsScan the point originates from.
   */
  void doTransform(geometry_msgs::TransformStamped transform, pcl::PointXYZINormal &pt,
                   std_msgs::Header h);

  /**
   * Publishes normals as markers for visualiation in RViz.
   * @details  Publishes markers that contain arrows that visualize the normal
   *vectors of points. Each arrow is defined by a starting point and an end
   *point.
   *@param tc The point cloud whose normals are to be published.
   */
  void publishNormals(XYZINCloud::ConstPtr tc);

  std::string scans_topic_;  /**< Topic name to look for TsScan messages.*/
  std::string target_frame_; /**< Target frame for scan transformations.*/

  ros::Publisher cloud_pub_;  /**< Handler for publishing PCL messages.*/
  ros::Subscriber scans_sub_; /**< Handler for subscribing to TsScans.*/
  ros::Publisher marker_pub_; /**< Handler for publishing marker messages.*/

  float lifetime_normals_vis_; /**< Lifetime for visualization of normals.*/
  int marker_id_ = 0;          /**< Id for publishing unique markers for normals.*/

  tf2_ros::Buffer tf2_buffer_;
  tf2_ros::TransformListener *tf2_listener_; /**< Listener for frame mapping.*/

  boost::thread *log_thread_; /**< Accumulates pointcloud and periodically saves
                                 to file. */

  std::unique_ptr<Mapping> pointTransform_; /**< Point Transform object to perform transformation */
  visualization_msgs::Marker marker;        /**< Marker for the arrow(s) to be published.*/
};

}  // namespace toposens_pointcloud

#endif
