#include <iostream>
/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "device/physicaldevice.h"
#include "device/logicaldevice.h"
#include "interface/interface.h"
#include "stream/buffer.h"

// SIMD Headers
#ifdef __SSE2__
#include <emmintrin.h>
#endif

namespace GenTL {


PhysicalDevice::PhysicalDevice(Interface* interface): interface(interface), udp(true), threadRunning(false),
    errorEvent(EVENT_ERROR, sizeof(GC_ERROR), nullptr) {
}

PhysicalDevice::~PhysicalDevice() {
    close();
}

GC_ERROR PhysicalDevice::open(bool udp, const char* host, const char* service) {
    // Open device
    try {
        // Initialize members
        this->udp = udp;
        this->host = host;
        this->service = service;

#ifndef DELIVER_TEST_DATA
        // Initialize network receiver
        if(udp) {
            imageTf.reset(new ImageTransfer(ImageTransfer::UDP, nullptr, nullptr, host, service));
        } else {
            imageTf.reset(new ImageTransfer(ImageTransfer::TCP_CLIENT, host, service, nullptr, nullptr));
        }
#endif

        // Initialize data streams
        std::string baseURL = std::string(udp ? "udp://" : "tcp://" ) + host + ":" + service;
        logicalDevices[ID_MULTIPART].reset(new LogicalDevice(this, baseURL + "/", DataStream::MULTIPART_STREAM));
        logicalDevices[ID_IMAGE_LEFT].reset(new LogicalDevice(this, baseURL + "/left", DataStream::IMAGE_LEFT_STREAM));
        logicalDevices[ID_IMAGE_RIGHT].reset(new LogicalDevice(this, baseURL + "/right", DataStream::IMAGE_RIGHT_STREAM));
        logicalDevices[ID_DISPARITY].reset(new LogicalDevice(this, baseURL + "/disparity", DataStream::DISPARITY_STREAM));
        logicalDevices[ID_POINTCLOUD].reset(new LogicalDevice(this, baseURL + "/pointcloud", DataStream::POINTCLOUD_STREAM));

        bool success = false;
        {
            std::unique_lock<std::mutex> lock(receiveMutex);
            threadRunning = true;
            receiveThread = std::thread(std::bind(&PhysicalDevice::deviceReceiveThread, this));

            // Wait for first frame
            std::chrono::milliseconds duration(1000);
            initializedCondition.wait_for(lock, duration);
            success = (latestMetaData.getHeight() > 0);
        }

        if(!success) {
            threadRunning = false;
            receiveThread.join();
            return GC_ERR_IO;
        } else {
            return GC_ERR_SUCCESS;
        }
    } catch(...) {
        return GC_ERR_IO;
    }
}

void PhysicalDevice::close() {
    if(threadRunning) {
        threadRunning = false;
        if(receiveThread.joinable()) {
            receiveThread.join();
        }
    }
}

void PhysicalDevice::setTestData(ImagePair& receivedPair) {
    // This data can be set for testing purposes
    receivedPair.setWidth(640);
    receivedPair.setHeight(480);
    receivedPair.setRowStride(0, 640);
    receivedPair.setRowStride(1, 2*640);
    receivedPair.setPixelFormat(0, ImagePair::FORMAT_8_BIT);
    receivedPair.setPixelFormat(1, ImagePair::FORMAT_12_BIT);

    float q[] = {
        1., 0., 0., -3.1774090576171875e+02, 0., 1., 0.,
        -2.4968117713928223e+02, 0., 0., 0., 6.9073932990086325e+02, 0., 0.,
        9.9997861072212793e+00, 0.
    };
    receivedPair.setQMatrix(q);

#ifdef _WIN32
    static __declspec(align(32)) unsigned char buffer0[640*480];
    static __declspec(align(32)) unsigned short buffer1[640*480];
#else
    static unsigned char buffer0[640*480] __attribute__((aligned(32)));
    static unsigned short buffer1[640*480] __attribute__((aligned(32)));
#endif

    for(int y = 0; y < 480; y++) {
        for(int x = 0; x < 640; x++) {
            buffer0[y*640 + x] = y + x;
            buffer1[y*640 + x] =( (y + x)*10)%(1<<12);
        }
    }

    receivedPair.setPixelData(0, buffer0);
    receivedPair.setPixelData(1, reinterpret_cast<unsigned char*>(buffer1));
}

void PhysicalDevice::deviceReceiveThread() {
    try {
        bool initialized = false;
        ImagePair receivedPair;
#ifdef DELIVER_TEST_DATA
        setTestData(receivedPair);
#endif

        while(threadRunning) {
#ifdef DELIVER_TEST_DATA
            // Noting to receive in test mode
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
#else
            // Receive new image
            if(!imageTf->receiveImagePair(receivedPair)) {
                // No image available
                continue;
            }
#endif

            {
                std::unique_lock<std::mutex> lock(receiveMutex);
                latestMetaData = receivedPair;

                // Copy raw and 3D data to buffer
                copyRawDataToBuffer(receivedPair);
                copy3dDataToBuffer(receivedPair);
                copyMultipartDataToBuffer(receivedPair);

                if(!initialized) {
                    // Initialization is done once we received the first frame
                    initialized = true;
                    initializedCondition.notify_one();
                }
            }
        }
    } catch(...) {
        // Error has occured
        errorEvent.emitEvent(GC_ERR_IO);
        threadRunning = false;
    }
}

void PhysicalDevice::copyRawDataToBuffer(const ImagePair& receivedPair) {
    for(int i=0; i<=1; i++) {
        // Deterime the correct logical device
        int id;
        if(i == 0) {
            id = ID_IMAGE_LEFT;
        } else {
            if(receivedPair.getPixelFormat(i) == ImagePair::FORMAT_8_BIT) {
                id = ID_IMAGE_RIGHT;
            } else {
                id = ID_DISPARITY;
            }
        }

        // Copy to device buffer
        Buffer* buffer = logicalDevices[id]->getStream()->requestBuffer();
        if(buffer == nullptr) {
            continue;
        }

        int copiedBytes = copyImageToBufferMemory(receivedPair, i, buffer->getData(), buffer->getSize());
        if(copiedBytes < 0) {
            buffer->setIncomplete(true);
        } else {
            buffer->setIncomplete(false);
        }

        buffer->setMetaData(receivedPair);
        logicalDevices[id]->getStream()->queueOutputBuffer();

        if(buffer->isIncomplete()) {
            errorEvent.emitEvent(GC_ERR_BUFFER_TOO_SMALL);
        }
    }
}

int PhysicalDevice::copyImageToBufferMemory(const ImagePair& receivedPair, int id, unsigned char* dst, int dstSize) {
    int bytesPerPixel = receivedPair.getPixelFormat(id) == ImagePair::FORMAT_8_BIT ? 1 : 2;
    int newStride = receivedPair.getWidth() * bytesPerPixel;
    int totalSize = receivedPair.getHeight() * newStride;

    if(totalSize > dstSize) {
        // No more buffer space.
        return -1;
    } else {
        if(newStride == receivedPair.getRowStride(id)) {
            memcpy(dst, receivedPair.getPixelData(id), totalSize);
        } else {
            for(int y = 0; y<receivedPair.getHeight(); y++) {
                memcpy(&dst[y*newStride], &receivedPair.getPixelData(id)[y*receivedPair.getRowStride(id)], newStride);
            }
        }

        return totalSize;
    }
}

void PhysicalDevice::copy3dDataToBuffer(const ImagePair& receivedPair) {
    Buffer* buffer = logicalDevices[ID_POINTCLOUD]->getStream()->requestBuffer();
    if(buffer == nullptr) {
        // No buffer available
        return;
    }

    int copiedBytes = copy3dDataToBufferMemory(receivedPair, buffer->getData(), buffer->getSize());
    if(copiedBytes < 0) {
        buffer->setIncomplete(true);
    } else {
        buffer->setIncomplete(false);
    }

    buffer->setMetaData(receivedPair);
    logicalDevices[ID_POINTCLOUD]->getStream()->queueOutputBuffer();

    if(buffer->isIncomplete()) {
        errorEvent.emitEvent(GC_ERR_BUFFER_TOO_SMALL);
    }
}

int PhysicalDevice::copy3dDataToBufferMemory(const ImagePair& receivedPair, unsigned char* dst, int dstSize) {
    int pixels = receivedPair.getWidth()*receivedPair.getHeight();
    int totalSize = 3*sizeof(float)*pixels;

    if(dstSize < totalSize || receivedPair.getPixelFormat(1) != ImagePair::ImagePair::FORMAT_12_BIT) {
        // Buffer is too small or pixel format doesn't match
        return -1;
    } else {
        // GenTL does not support padding between pixels. Let's only
        // copy the non-padding bytes
        float* inputPtr = reconstruct.createPointMap(receivedPair, 0);
        float* outputPtr = reinterpret_cast<float*>(dst);
#ifdef __SSE2__
        copyPointsSSE(outputPtr, inputPtr, pixels);
#else
        copyPointsFallback(outputPtr, inputPtr, pixels);
#endif
        return totalSize;
    }
}

void PhysicalDevice::copyMultipartDataToBuffer(const ImagePair& receivedPair) {
    Buffer* buffer = logicalDevices[ID_MULTIPART]->getStream()->requestBuffer();
    if(buffer == nullptr) {
        // No buffer available
        return;
    }

    buffer->setIncomplete(false);
    int offset = 0;
    int copiedBytes = copyImageToBufferMemory(receivedPair, 0, buffer->getData(), buffer->getSize());

    if(copiedBytes < 0) {
        buffer->setIncomplete(true);
    } else {
        offset += copiedBytes;
        copiedBytes = copyImageToBufferMemory(receivedPair, 1, &buffer->getData()[offset],
            buffer->getSize() - offset);

        if(copiedBytes < 0) {
            buffer->setIncomplete(true);
        } else {
            offset += copiedBytes;
            copiedBytes = copy3dDataToBufferMemory(receivedPair, &buffer->getData()[offset],
                buffer->getSize() - offset);
            if(copiedBytes < 0) {
                buffer->setIncomplete(true);
            }
        }
    }

    buffer->setMetaData(receivedPair);
    logicalDevices[ID_MULTIPART]->getStream()->queueOutputBuffer();

    if(buffer->isIncomplete()) {
        errorEvent.emitEvent(GC_ERR_BUFFER_TOO_SMALL);
    }
}

void PhysicalDevice::copyPointsFallback(float* dst, float* src, int numPoints) {
    float* endPtr = src + 4*numPoints;
    while(src < endPtr) {
        dst[0] = src[0];
        dst[1] = src[1];
        dst[2] = src[2];
        dst+=3;
        src+=4;
    }
}

#ifdef __SSE2__
void PhysicalDevice::copyPointsSSE(float* dst, float* src, int numPoints) {
    float* endPtr = src + 4*numPoints;

    __m128 point;
    while(src < endPtr) {
        // 1x aligned copy
        point  = _mm_load_ps(src);
        _mm_store_ps(dst, point);
        dst+=3; src+=4;

        // 3x unaligned copy
        point  = _mm_load_ps(src);
        _mm_storeu_ps(dst, point);
        dst+=3; src+=4;

        point  = _mm_load_ps(src);
        _mm_storeu_ps(dst, point);
        dst+=3; src+=4;

        point  = _mm_load_ps(src);
        _mm_storeu_ps(dst, point);
        dst+=3; src+=4;
    }
}
#endif

bool PhysicalDevice::inUse() {
    for(int i=0; i<5; i++) {
        if(logicalDevices[i]->isOpen()) {
            return true;
        }
    }

    return false;
}

int PhysicalDevice::logicalIdToIndex(const char* id) {
    std::string idStr(id);
    if(idStr == "") {
        return ID_MULTIPART;
    } else if(idStr == "left") {
        return ID_IMAGE_LEFT;
    } else if(idStr == "right") {
        return ID_IMAGE_RIGHT;
    } else if(idStr == "disparity") {
        return ID_DISPARITY;
    } else if(idStr == "pointcloud") {
        return ID_POINTCLOUD;
    } else {
        return -1;
    }
}

std::string PhysicalDevice::logicalIndexToId(int index) {
    if(index == ID_MULTIPART) {
        return ""; // Default multipart
    } else if(index == ID_IMAGE_LEFT) {
        return "left";
    } else if(index == ID_IMAGE_RIGHT) {
        return "right";
    } else if(index == ID_DISPARITY) {
        return "disparity";
    } else if(index == ID_POINTCLOUD) {
        return "pointcloud";
    } else {
        return "?"; // Should not happen
    }
}

}
