/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_IMAGETRANSFER_H
#define VISIONTRANSFER_IMAGETRANSFER_H

#include <string>
#include "visiontransfer/common.h"
#include "visiontransfer/imageprotocol.h"
#include "visiontransfer/imagepair.h"

/**
 * \brief Class for synchronous transfer of image pairs.
 *
 * This class opens a network socket for delivering or receiving image pairs. All
 * operations are performed synchronously, which means that they might block.
 * The class encapsulates ImageProtocol.
 */
class VT_EXPORT ImageTransfer {
public:
    /// Supported transfer modes.
    enum OperationMode {
        /// Using UDP for network transfers. Differentiation between server
        /// and client role is not necessary.
        UDP,

        /// Using TCP and acting as communication client.
        TCP_CLIENT,

        /// Using TCP and acting as communication server.
        TCP_SERVER,
    };

    /// The result of a partial image transfer
    enum TransferStatus {
        /// The connection has been closed by the remote host.
        CONNECTION_CLOSED,

        /// The image pair has been transferred completely.
        ALL_TRANSFERRED,

        /// The image pair has been transferred partially. Further
        /// transfers are necessary.
        PARTIAL_TRANSFER,

        /// There is currently no more data that could be transmitted.
        NO_VALID_DATA,

        /// The operation would block and blocking as been disabled.
        WOULD_BLOCK
    };

    /**
     * \brief Creates a new transfer object.
     *
     * \param mode The transfer mode. This determines which network protocol to
     *        use, and whether to operate in server or client mode.
     * \param remoteAddress Address of the remote host. In TCP server mode this
     *        parameter can be set to a null pointer.
     * \param remoteService The remote port number as string or as textual service name.
     *        In TCP server mode this parameter can be set to a null pointer.
     * \param localAddress Local interface address that is used for communication.
     *        If set to a null pointer, the default interface is used.
     * \param localService The local port number as string or a textual service name.
     *        If set to a null pointer an automatically assigned port number is used.
     * \param bufferSize Buffer size for sending / receiving network data.
     *
     * In TCP server mode, a port is opened to which clients can establish connections.
     * In TCP client mode, a connection is established to the given address and service name.
     *
     * In UDP mode, a port is opened to which another host can transmit image data.
     * When calling transferData(), image data is transmitted to remote host given in
     * \c remoteAddress.
     */
    ImageTransfer(OperationMode mode, const char* remoteAddress, const char* remoteService,
        const char* localAddress, const char* localService, int bufferSize = 1000000);

    ~ImageTransfer();

    /**
     * \brief Sets the raw pixel data for a partial image transmission.
     *
     * This method has to be used in conjunction with transferData().
     * Please see ImageProtocol::setRawTransferData() for further details.
     */
    void setRawTransferData(const ImagePair& metaData, unsigned char* rawData,
        int secondTileWidth = 0, int validBytes = 0x7FFFFFFF);

    /**
     * \brief Updates the number of valid bytes in a partial raw transmission.
     *
     * Please see ImageProtocol::setRawValidBytes() for further details.
     */
    void setRawValidBytes(int validBytes);

    /**
     * \brief Sets a new image pair that shall be transmitted.
     *
     * \param imagePair The image pair that shall be transmitted.
     *
     * After setting the image pair, subsequent calls to transferData()
     * are necessary for performing the image transmission.
     *
     * \see ImageProtocol::setTransferImagePair()
     */
    void setTransferImagePair(const ImagePair& imagePair);

    /**
     * \brief Performs a partial (or full) image transmission.
     *
     * \param block If set to true and if the the send buffer is
     *        full, the method will block. Otherwise it will return immediately with
     *        TransferStatus::WOULD_BLOCK in case of full send buffers.
     * \return Status of the transmission. See below.
     *
     * The method transfers up to the specified number of valid bytes. It has to
     * be called in cycles in order to transfer a full image pair. If there
     * is no more data to be transferred, it will return TransferStatus::NO_VALID_DATA .
     *
     * If the transfer is compete, the method will return
     * TransferStatus::ALL_TRANSFERRED. If there remains outstanding data for
     * this transfer, the return value will be TransferStatus::PARTIAL_TRANSFER.
     * If the connection is no longer open, TransferStatus::CONNECTION_CLOSED
     * is returned.
     */
    TransferStatus transferData(bool block);

    /**
     * \brief Waits for and receives a new image pair.
     *
     * \param imagePair Will be set to the received image pair.
     * \param block If set to true, the method will block and wait for further
     *        image data, if a complete image pair hasn't yet been received.
     * \return Returns true if a new image pair has been received. Otherwise
     *         false.
     *
     * The received image pair is only valid until the next call of receiveImagePair().
     * Even if \c block is set to true, the method will not block indefinitely, but return
     * after a short timeout.
     *
     * \see ImageProtocol::getReceivedImagePair()
     */
    bool receiveImagePair(ImagePair& imagePair, bool block = true);

    /**
     * \brief Returns the received image pair, even if it is not yet complete.
     *
     * The received image pair is only valid until calling receivePartialImagePair()
     * for the first time after the current image pair has been received completely.
     * The method returns false if no image data has been received.
     *
     * Please see ImageProtocol::getPartiallyReceivedImagePair() for further details.
     */
    bool receivePartialImagePair(ImagePair& imagePair, int& validRows, bool& complete, bool block = false);

    /**
     * \brief Tries to accept a client connection.
     *
     * \return True if a client has connected..
     *
     * This method can only be used in TCP server mode. It shall be called in
     * regular intervals to allow for client connections. The method is
     * non-blocking.
     */
    bool tryAccept();

    /**
     * \brief Returns true if a client is connected.
     *
     * This method can only be used in TCP server mode.
     */
    bool isClientConnected() const;

    /**
     * \brief Terminates the current connection.
     *
     * If connected to a server or if a client is currently connected,
     * this connection will be closed.
     */
    void disconnect();

    /**
     * \brief Returns the address of the connected client
     *
     * \return Client address or "" if no client is connected.
     *
     * This method can only be used in TCP server mode.
     */
    std::string getClientAddress() const;

private:
    // We follow the pimpl idiom
    class Pimpl;
    Pimpl* pimpl;

    // This class cannot be copied
    ImageTransfer(const ImageTransfer& other);
    ImageTransfer& operator=(const ImageTransfer&);
};

#endif
