/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_IMAGEPAIR_H
#define VISIONTRANSFER_IMAGEPAIR_H

#include <cassert>
#include <cstddef>
#include "visiontransfer/common.h"

namespace visiontransfer {

/**
 * \brief A set of two images, which are usually the left camera image and the
 * disparity map.
 *
 * Both images must be of equal width and height, but are allowed to have a
 * different pixel formats. Please note that the class does not manage the
 * pixel data but only keeps pointers. You thus need to ensure that the pixel
 * data remains valid for as long as this object persists.
 */
class VT_EXPORT ImagePair {
public:
    /**
     * \brief Image formats that can be transferred.
     */
    enum ImageFormat {
        /// 8-bit greyscale format
        FORMAT_8_BIT_MONO,

        /// 8-bit RGB format
        FORMAT_8_BIT_RGB,

        /// 12-bit greyscale format plus 4 bits of padding
        /// (hence a total of 16 bits).
        FORMAT_12_BIT_MONO
    };

    /**
     * \deprecated Please use the new format constants in \c ImageFormat.
     */
    enum ImageFormat_Deprecated {
        FORMAT_8_BIT =  FORMAT_8_BIT_MONO,
        FORMAT_12_BIT = FORMAT_12_BIT_MONO
    };

    /**
     * \brief Default constructor creating an image pair with no pixel data.
     */
    ImagePair();

    /**
     * \brief Copy constructor creating a shallow copy of the image pair.
     */
    ImagePair(const ImagePair& other);

    ~ImagePair();
    ImagePair& operator= (ImagePair const& other);

    /**
     * \brief Sets a new width for both images.
     */
    void setWidth(int w) {width = w;}

    /**
     * \brief Sets a new width for both images.
     */
    void setHeight(int h) {height = h;}

    /**
     * \brief Sets a new row stride for the pixel data of one image.
     *
     * \param imageNumber Number of the image for which to set the
     *        row stride (0 or 1).
     * \param stride The row stride that shall be set.
     */
    void setRowStride(int imageNumber, int stride) {
        assert(imageNumber >= 0 && imageNumber <=1);
        rowStride[imageNumber] = stride;
    }

    /**
     * \brief Sets the pixel format for the given image.
     *
     * \param imageNumber Number of the image for which to set the
     *        pixel format (0 or 1).
     * \param format The pixel format that shall be set.
     */
    void setPixelFormat(int imageNumber, ImageFormat format) {
        assert(imageNumber >= 0 && imageNumber <=1);
        formats[imageNumber] = format;
    }

#ifndef DOXYGEN_SHOULD_SKIP_THIS
    DEPRECATED(void setPixelFormat(int imageNumber, ImageFormat_Deprecated format)) {
        setPixelFormat(imageNumber, static_cast<ImageFormat>(format));
    }
#endif

    /**
     * \brief Sets the pixel data for the given image.
     *
     * \param imageNumber Number of the image for which to set the
     *        pixel data (0 or 1).
     * \param pixelData The pixel data that shall be set.
     */
    void setPixelData(int imageNumber, unsigned char* pixelData) {
        assert(imageNumber >= 0 && imageNumber <=1);
        data[imageNumber] = pixelData;
    }

    /**
     * \brief Sets the pointer to the disparity-to-depth mapping matrix q.
     *
     * No data is copied. The data which q is pointing to has to remain valid
     * for as long as this object exists.
     */
    void setQMatrix(const float* q) {
        qMatrix = q;
    }

    /**
     * \brief Sets the sequence number for this image pair.
     */
    void setSequenceNumber(unsigned int num) {
        seqNum = num;
    }

    /**
     * \brief Sets the time at which this image pair has been captured.
     *
     * \param seconds The time stamp with a resolution of one second.
     * \param microsec The fractional seconds part of the time stamp with
     *        a resolution of 1 microsecond.
     */
    void setTimestamp(int seconds, int microsec) {
        timeSec = seconds;
        timeMicrosec = microsec;
    }

    /**
     * \brief Sets the value range for the disparity map contained in this
     *        image pair.
     *
     * \param minimum Minimum disparity value.
     * \param maximum Maximum disparity value.
     */
    void setDisparityRange(int minimum, int maximum) {
        minDisparity = minimum;
        maxDisparity = maximum;
    }

    /**
     * \brief Sets the subpixel factor for this image pair.
     */
    void setSubpixelFactor(int subpixFact) {
        subpixelFactor = subpixFact;
    }

    /**
     * \brief Sets whether this is a left camera image and disparity
     * map pair, or two raw camera images.
     */
    void setImageDisparityPair(bool dispPair) {
        disparityPair = dispPair;
    }

    /**
     * \brief Returns the width of each image.
     */
    int getWidth() const {return width;}

    /**
     * \brief Returns the height of each image.
     */
    int getHeight() const {return height;}

    /**
     * \brief Returns the row stride for the pixel data of one image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        row stride (0 or 1).
     */
    int getRowStride(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return rowStride[imageNumber];
    }

    /**
     * \brief Returns the pixel format for the given image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        pixel format (0 or 1).
     */
    ImageFormat getPixelFormat(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return formats[imageNumber];
    }

    /**
     * \brief Returns the pixel data for the given image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        pixel data (0 or 1).
     */
    unsigned char* getPixelData(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return data[imageNumber];
    }

    /**
     * \brief Returns a pointer to the disparity-to-depth mapping matrix q.
     */
    const float* getQMatrix() const {
        return qMatrix;
    }

    /**
     * \brief Returns the sequence number for this image pair.
     */
    unsigned int getSequenceNumber() const {return seqNum;}

    /**
     * \brief Returns the time at which this image pair has been captured.
     *
     * \param seconds The time stamp with a resolution of one second.
     * \param microsec The fractional seconds part of the time stamp with
     *        a resolution of 1 microsecond.
     */
    void getTimestamp(int& seconds, int& microsec) const {
        seconds = timeSec;
        microsec = timeMicrosec;
    }

    /**
     * \brief Gets the value range for the disparity map contained in this
     *        image pair. If the image pair does not contain any disparity data
     *        then the disparity range is undefined.
     *
     * \param minimum Minimum disparity value.
     * \param maximum Maximum disparity value.
     */
    void getDisparityRange(int& minimum, int& maximum) const {
        minimum = minDisparity;
        maximum = maxDisparity;
    }

    /**
     * \brief Gets the subpixel factor for this image pair.
     */
    int getSubpixelFactor() const {
        return subpixelFactor;
    }

    /**
     * \brief Writes one image of the pair to a PGM or PPM file.
     *
     * \param imageNumber The number of the image that shall be written.
     * \param File name of the PGM or PPM file that shall be created.
     */
    void writePgmFile(int imageNumber, const char* fileName) const;

    /**
     * \brief Returns true if this is a left camera image and disparity
     * map pair.
     */
    bool isImageDisparityPair() const {
        return disparityPair;
    }

    /**
     * \brief Makes a deep copy of this image pair.
     */
    void copyTo(ImagePair& dest);

    /**
     * \brief Returns the number of bytes that are required to store one
     * image pixel.
     *
     * \param imageNumber The number of the image (0 or 1).
     */
    int getBytesPerPixel(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return getBytesPerPixel(formats[imageNumber]);
    }

    /**
     * \brief Returns the number of bytes that are required to store one
     * image pixel with the given pixel format.
     */
    static int getBytesPerPixel(ImageFormat format);

#ifdef CV_MAJOR_VERSION
    /**
     * \brief Converts one image of the pair to an OpenCV image.
     *
     * \param imageNumber The number of the image that shall be converted (0 or 1).
     * \param convertRgbToBgr If true, then color images will converted from
     *        RGB to BGR in order to comply to OpenCV's convention.
     *
     * For this method to be available, the OpenCV headers need to be
     * included before including headers for libvisiontransfer.
     *
     * Please note that only a shallow copy is performed. The ImagePair object
     * must be kept alive for as long as the OpenCV image is in use.
     */
    inline void toOpenCVImage(int imageNumber, cv::Mat& dest, bool convertRgbToBgr = true);
#endif

private:
    // No pimpl idiom here as almost everything is inlined.

    int width;
    int height;
    int rowStride[2];
    ImageFormat formats[2];
    unsigned char* data[2];
    const float* qMatrix;
    int timeSec;
    int timeMicrosec;
    unsigned int seqNum;
    int minDisparity;
    int maxDisparity;
    int subpixelFactor;
    bool disparityPair;
    int* referenceCounter;

    void copyData(ImagePair& dest, const ImagePair& src, bool countRef);
    void decrementReference();
};

} // namespace

#include "visiontransfer/imagepair-opencv.h"
#endif

