/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include <cstring>
#include <iostream>
#include <limits>
#include <vector>
#include <memory>
#include <algorithm>
#include "visiontransfer/imageprotocol.h"
#include "visiontransfer/alignedallocator.h"
#include "visiontransfer/datablockprotocol.h"
#include "visiontransfer/exceptions.h"
#include "visiontransfer/bitconversions.h"
#include "visiontransfer/internalinformation.h"

// Network headers
#ifdef _WIN32
    #ifndef NOMINMAX
        #define NOMINMAX
    #endif
    #include <winsock2.h>
#else
    #include <arpa/inet.h>
#endif

using namespace std;
using namespace visiontransfer;
using namespace visiontransfer::internal;

namespace visiontransfer {

/*************** Pimpl class containing all private members ***********/

class ImageProtocol::Pimpl {
public:
    Pimpl(bool server, ProtocolType protType, int maxUdpPacketSize);

    // Redeclaration of public members
    void setTransferImagePair(const ImagePair& imagePair);
    void setRawTransferData(const ImagePair& metaData, unsigned char* rawData,
        int firstTileWidth = 0, int secondTileWidth = 0, int validBytes = 0x7FFFFFFF);
    void setRawValidBytes(int validBytes);
    const unsigned char* getTransferMessage(int& length);
    bool transferComplete();
    void resetTransfer();
    bool getReceivedImagePair(ImagePair& imagePair);
    bool getPartiallyReceivedImagePair(ImagePair& imagePair,
        int& validRows, bool& complete);
    bool imagesReceived() const;

    unsigned char* getNextReceiveBuffer(int& maxLength);

    void processReceivedMessage(int length);
    int getProspectiveMessageSize();
    int getNumDroppedFrames() const;
    void resetReception();
    bool isConnected() const;
    const unsigned char* getNextControlMessage(int& length);
    bool newClientConnected();

private:
    unsigned short MAGIC_SEQUECE = 0x3D15;

    // Header data transferred in the first packet
#pragma pack(push,1)
    struct HeaderData{
        unsigned short magic;

        unsigned char protocolVersion;
        unsigned char isRawImagePair;

        unsigned short width;
        unsigned short height;

        unsigned short firstTileWidth;
        unsigned short secondTileWidth;

        unsigned char format0;
        unsigned char format1;
        unsigned short minDisparity;
        unsigned short maxDisparity;
        unsigned char subpixelFactor;

        unsigned int seqNum;
        int timeSec;
        int timeMicrosec;

        float q[16];
    };
#pragma pack(pop)

    // Underlying protocol for data transfers
    DataBlockProtocol dataProt;
    ProtocolType protType;

    // Transfer related variables
    std::vector<unsigned char> headerBuffer;
    std::vector<unsigned char> rawBuffer;
    unsigned char* rawData;

    // Reception related variables
    std::vector<unsigned char, AlignedAllocator<unsigned char> >decodeBuffer[2];
    bool receiveHeaderParsed;
    HeaderData receiveHeader;
    int lastReceivedPayloadBytes[2];
    bool receptionDone;

    // Copies the transmission header to the given buffer
    void copyHeaderToBuffer(const ImagePair& imagePair, int firstTileWidth,
        int secondTileWidth, unsigned char* buffer);

    // Decodes header information from the received data
    void tryDecodeHeader(const unsigned char* receivedData, int receivedBytes);

    // Decodes a received image from an interleaved buffer
    unsigned char* decodeInterleaved(int imageNumber, int receivedBytes,
        unsigned char* data, int& validRows, int& rowStride);

    int getFrameSize(int width, int height, int firstTileWidth, int secondTileWidth,
        ImagePair::ImageFormat format0, ImagePair::ImageFormat format1);

    int getFormatBits(ImagePair::ImageFormat format, bool afterDecode);

    void decodeTiledImage(int imageNumber, int lastReceivedPayloadBytes, int receivedPayloadBytes,
        const unsigned char* data, int firstTileStride, int secondTileStride, int& validRows,
        ImagePair::ImageFormat format);

    void decodeRowsFromTile(int startRow, int stopRow, unsigned const char* src,
        unsigned char* dst, int srcStride, int dstStride, int tileWidth);

    void allocateDecodeBuffer(int imageNumber);
};


/******************** Stubs for all public members ********************/

ImageProtocol::ImageProtocol(bool server, ProtocolType protType, int maxUdpPacketSize)
    : pimpl(new Pimpl(server, protType, maxUdpPacketSize)) {
    // All initializations are done by the Pimpl class
}

ImageProtocol::~ImageProtocol() {
    delete pimpl;
}

void ImageProtocol::setTransferImagePair(const ImagePair& imagePair) {
    pimpl->setTransferImagePair(imagePair);
}

void ImageProtocol::setRawTransferData(const ImagePair& metaData,
        unsigned char* imageData, int firstTileWidth, int secondTileWidth, int validBytes) {
    pimpl->setRawTransferData(metaData, imageData, firstTileWidth, secondTileWidth, validBytes);
}

void ImageProtocol::setRawValidBytes(int validBytes) {
    pimpl->setRawValidBytes(validBytes);
}

const unsigned char* ImageProtocol::getTransferMessage(int& length) {
    return pimpl->getTransferMessage(length);
}

bool ImageProtocol::transferComplete() {
    return pimpl->transferComplete();
}

void ImageProtocol::resetTransfer() {
    pimpl->resetTransfer();
}

bool ImageProtocol::getReceivedImagePair(ImagePair& imagePair) {
    return pimpl->getReceivedImagePair(imagePair);
}

bool ImageProtocol::getPartiallyReceivedImagePair(
        ImagePair& imagePair, int& validRows, bool& complete) {
    return pimpl->getPartiallyReceivedImagePair(imagePair, validRows, complete);
}

bool ImageProtocol::imagesReceived() const {
    return pimpl->imagesReceived();
}

unsigned char* ImageProtocol::getNextReceiveBuffer(int& maxLength) {
    return pimpl->getNextReceiveBuffer(maxLength);
}

void ImageProtocol::processReceivedMessage(int length) {
    pimpl->processReceivedMessage(length);
}

int ImageProtocol::getNumDroppedFrames() const {
    return pimpl->getNumDroppedFrames();
}

void ImageProtocol::resetReception() {
    pimpl->resetReception();
}

bool ImageProtocol::isConnected() const {
    return pimpl->isConnected();
}

const unsigned char* ImageProtocol::getNextControlMessage(int& length) {
    return pimpl->getNextControlMessage(length);
}

bool ImageProtocol::newClientConnected() {
    return pimpl->newClientConnected();
}

/******************** Implementation in pimpl class *******************/

ImageProtocol::Pimpl::Pimpl(bool server, ProtocolType protType, int maxUdpPacketSize)
        :dataProt(server, (DataBlockProtocol::ProtocolType)protType,
        maxUdpPacketSize), protType(protType), rawData(nullptr),
        receiveHeaderParsed(false), lastReceivedPayloadBytes{0, 0},
        receptionDone(false) {
    headerBuffer.resize(sizeof(HeaderData) + 32);
    memset(&headerBuffer[0], 0, sizeof(headerBuffer.size()));
    memset(&receiveHeader, 0, sizeof(receiveHeader));
}

void ImageProtocol::Pimpl::setTransferImagePair(const ImagePair& imagePair) {
    if(imagePair.getPixelData(0) == nullptr || imagePair.getPixelData(1) == nullptr) {
        throw ProtocolException("Image data is null pointer!");
    }

    // Determine the size of a frame
    int rawDataLength = getFrameSize(imagePair.getWidth(), imagePair.getHeight(), 0, 0,
        imagePair.getPixelFormat(0), imagePair.getPixelFormat(1));

    // Set header as first piece of data
    copyHeaderToBuffer(imagePair, 0, 0, &headerBuffer[16]);
    dataProt.resetTransfer();
    dataProt.setTransferHeader(&headerBuffer[16], sizeof(HeaderData), rawDataLength);

    // Perform 12 bit packed encoding if necessary
    int bits[2] = {0, 0};
    int rowSize[2] = {0, 0};
    int rowStride[2] = {0, 0};
    const unsigned char* pixelData[2] = {nullptr, nullptr};
    std::vector<unsigned char> encodingBuffer[2];

    for(int i = 0; i<2; i++) {
        bits[i] = getFormatBits(imagePair.getPixelFormat(i), false);
        rowSize[i] = imagePair.getWidth()*bits[i]/8;

        if(imagePair.getPixelFormat(i) != ImagePair::FORMAT_12_BIT_MONO) {
            pixelData[i] = imagePair.getPixelData(i);
            rowStride[i] = imagePair.getRowStride(i);
        } else {
            encodingBuffer[i].resize(rowSize[i] * imagePair.getHeight());
            BitConversions::encode12BitPacked(0, imagePair.getHeight(), imagePair.getPixelData(i),
                &encodingBuffer[i][0], imagePair.getRowStride(i), rowSize[i], imagePair.getWidth());
            pixelData[i] = &encodingBuffer[i][0];
            rowStride[i] = rowSize[i];
        }
    }

    // Make a interleaved copy
    rawBuffer.resize(imagePair.getWidth()*imagePair.getHeight()*(bits[0] + bits[1])/8 + sizeof(int));
    int bufferOffset = 0;

    for(int y = 0; y<imagePair.getHeight(); y++) {
        memcpy(&rawBuffer[bufferOffset], &pixelData[0][y*rowStride[0]], rowSize[0]);
        bufferOffset += rowSize[0];

        memcpy(&rawBuffer[bufferOffset], &pixelData[1][y*rowStride[1]], rowSize[1]);
        bufferOffset += rowSize[1];
    }

    rawData = &rawBuffer[0];
    int rawValidBytes = static_cast<int>(rawBuffer.size() - sizeof(int));

    dataProt.setTransferData(rawData, rawValidBytes);
}

void ImageProtocol::Pimpl::setRawTransferData(const ImagePair& metaData, unsigned char* rawData,
        int firstTileWidth, int secondTileWidth, int validBytes) {
    if(rawData == nullptr) {
        throw ProtocolException("Image data is null pointer!");
    }

    // Determine the size of a frame
    int rawDataLength = getFrameSize(metaData.getWidth(), metaData.getHeight(),
        firstTileWidth, secondTileWidth, metaData.getPixelFormat(0),
        metaData.getPixelFormat(1));

    // Set header as first piece of data
    copyHeaderToBuffer(metaData, firstTileWidth, secondTileWidth, &headerBuffer[16]);
    dataProt.resetTransfer();
    dataProt.setTransferHeader(&headerBuffer[16], sizeof(HeaderData), rawDataLength);

    this->rawData = rawData;

    dataProt.setTransferData(rawData, validBytes);
}

void ImageProtocol::Pimpl::setRawValidBytes(int validBytes) {
    dataProt.setTransferValidBytes(validBytes);
}

const unsigned char* ImageProtocol::Pimpl::getTransferMessage(int& length) {
    const unsigned char* msg = dataProt.getTransferMessage(length);

    if(msg == nullptr) {
        msg = dataProt.getTransferMessage(length);
    }

    return msg;
}

bool ImageProtocol::Pimpl::transferComplete() {
    return dataProt.transferComplete();
}

int ImageProtocol::Pimpl::getFrameSize(int width, int height, int firstTileWidth,
        int secondTileWidth, ImagePair::ImageFormat format0,
        ImagePair::ImageFormat format1) {

    int bits0 = getFormatBits(format0, false);
    int bits1 = getFormatBits(format1, false);

    int effectiveWidth = firstTileWidth > 0 ? firstTileWidth + secondTileWidth : width;

    return (effectiveWidth * height * (bits0 + bits1)) /8;
}

int ImageProtocol::Pimpl::getFormatBits(ImagePair::ImageFormat format, bool afterDecode) {
    if(afterDecode) {
        return ImagePair::getBytesPerPixel(format)*8;
    } else {
        switch(format) {
            case ImagePair::FORMAT_8_BIT_MONO: return 8;
            case ImagePair::FORMAT_12_BIT_MONO: return 12;
            case ImagePair::FORMAT_8_BIT_RGB: return 24;
            default: throw ProtocolException("Illegal pixel format!");
        }
    }
}

void ImageProtocol::Pimpl::copyHeaderToBuffer(const ImagePair& imagePair,
        int firstTileWidth, int secondTileWidth, unsigned char* buffer) {
    HeaderData* transferHeader = reinterpret_cast<HeaderData*>(buffer);
    memset(transferHeader, 0, sizeof(*transferHeader));
    transferHeader->magic = htons(MAGIC_SEQUECE);
    transferHeader->protocolVersion = InternalInformation::CURRENT_PROTOCOL_VERSION;
    transferHeader->isRawImagePair = imagePair.isImageDisparityPair() ? 0 : 1;
    transferHeader->width = htons(imagePair.getWidth());
    transferHeader->height = htons(imagePair.getHeight());
    transferHeader->firstTileWidth = htons(firstTileWidth);
    transferHeader->secondTileWidth = htons(secondTileWidth);
    transferHeader->format0 = static_cast<unsigned char>(imagePair.getPixelFormat(0));
    transferHeader->format1 = static_cast<unsigned char>(imagePair.getPixelFormat(1));
    transferHeader->seqNum = static_cast<unsigned int>(htonl(imagePair.getSequenceNumber()));

    int minDisp = 0, maxDisp = 0;
    imagePair.getDisparityRange(minDisp, maxDisp);
    transferHeader->minDisparity = minDisp;
    transferHeader->maxDisparity = maxDisp;

    transferHeader->subpixelFactor = imagePair.getSubpixelFactor();

    int timeSec = 0, timeMicrosec = 0;
    imagePair.getTimestamp(timeSec, timeMicrosec);
    transferHeader->timeSec = static_cast<int>(htonl(static_cast<unsigned int>(timeSec)));
    transferHeader->timeMicrosec = static_cast<int>(htonl(static_cast<unsigned int>(timeMicrosec)));

    if(imagePair.getQMatrix() != nullptr) {
        memcpy(transferHeader->q, imagePair.getQMatrix(), sizeof(float)*16);
    }
}

void ImageProtocol::Pimpl::resetTransfer() {
    dataProt.resetTransfer();
}

unsigned char* ImageProtocol::Pimpl::getNextReceiveBuffer(int& maxLength) {
    maxLength = dataProt.getMaxReceptionSize();
    return dataProt.getNextReceiveBuffer(maxLength);
}

void ImageProtocol::Pimpl::processReceivedMessage(int length) {
    receptionDone = false;

    // Add the received message
    dataProt.processReceivedMessage(length, receptionDone);

    int receivedBytes = 0;
    dataProt.getReceivedData(receivedBytes);

    // Immediately try to decode the header
    if(!receiveHeaderParsed) {
        int headerLen = 0;
        unsigned char* headerData = dataProt.getReceivedHeader(headerLen);
        if(headerData != nullptr) {
            tryDecodeHeader(headerData, headerLen);
        }
    }
}

void ImageProtocol::Pimpl::tryDecodeHeader(const
unsigned char* receivedData, int receivedBytes) {
    if(receivedBytes >= static_cast<int>(sizeof(HeaderData))) {
        receiveHeader =  *reinterpret_cast<const HeaderData*>(receivedData);
        if(receiveHeader.magic != htons(MAGIC_SEQUECE)) {
            // Let's not call this an error. Perhaps it's just not a header
            // packet
            return;
        }

        if(receiveHeader.protocolVersion > InternalInformation::CURRENT_PROTOCOL_VERSION ||
                receiveHeader.protocolVersion < 4) {
            throw ProtocolException("Protocol version mismatch!");
        }

        // Convert byte order
        receiveHeader.width = ntohs(receiveHeader.width);
        receiveHeader.height = ntohs(receiveHeader.height);
        receiveHeader.firstTileWidth = ntohs(receiveHeader.firstTileWidth);
        receiveHeader.secondTileWidth = ntohs(receiveHeader.secondTileWidth);
        receiveHeader.timeSec = static_cast<int>(
            htonl(static_cast<unsigned int>(receiveHeader.timeSec)));
        receiveHeader.timeMicrosec = static_cast<int>(
            htonl(static_cast<unsigned int>(receiveHeader.timeMicrosec)));
        receiveHeader.seqNum = htonl(receiveHeader.seqNum);

        receiveHeaderParsed = true;
    }
}

bool ImageProtocol::Pimpl::imagesReceived() const {
    return receptionDone && receiveHeaderParsed;
}

bool ImageProtocol::Pimpl::getReceivedImagePair(ImagePair& imagePair) {
    bool complete = false;
    int validRows;
    bool ok = getPartiallyReceivedImagePair(imagePair, validRows, complete);

    return (ok && complete);
}

bool ImageProtocol::Pimpl::getPartiallyReceivedImagePair(ImagePair& imagePair, int& validRows, bool& complete) {
    imagePair.setWidth(0);
    imagePair.setHeight(0);

    complete = false;

    if(!receiveHeaderParsed) {
        // We haven't even received the image header yet
        return false;
    } else {
        // We received at least some pixel data
        int receivedBytes = 0;
        unsigned char* data = dataProt.getReceivedData(receivedBytes);
        imagePair.setImageDisparityPair(receiveHeader.isRawImagePair == 0);

        validRows = 0;
        imagePair.setWidth(receiveHeader.width);
        imagePair.setHeight(receiveHeader.height);
        imagePair.setPixelFormat(0, static_cast<ImagePair::ImageFormat>(receiveHeader.format0));
        imagePair.setPixelFormat(1, static_cast<ImagePair::ImageFormat>(receiveHeader.format1));

        int rowStride0 = 0, rowStride1 = 0;
        int validRows0 = 0, validRows1 = 0;
        unsigned char* pixel0 = decodeInterleaved(0, receivedBytes, data, validRows0, rowStride0);
        unsigned char* pixel1 = decodeInterleaved(1, receivedBytes, data, validRows1, rowStride1);

        imagePair.setRowStride(0, rowStride0);
        imagePair.setRowStride(1, rowStride1);
        imagePair.setPixelData(0, pixel0);
        imagePair.setPixelData(1, pixel1);
        imagePair.setQMatrix(receiveHeader.q);

        imagePair.setSequenceNumber(receiveHeader.seqNum);
        imagePair.setTimestamp(receiveHeader.timeSec, receiveHeader.timeMicrosec);
        imagePair.setDisparityRange(receiveHeader.minDisparity, receiveHeader.maxDisparity);
        imagePair.setSubpixelFactor(receiveHeader.subpixelFactor);

        validRows = min(validRows0, validRows1);

        if(validRows == receiveHeader.height || receptionDone) {
            complete = true;
            resetReception();
        }

        return true;
    }
}

unsigned char* ImageProtocol::Pimpl::decodeInterleaved(int imageNumber, int receivedBytes,
        unsigned char* data, int& validRows, int& rowStride) {
    ImagePair::ImageFormat format = static_cast<ImagePair::ImageFormat>(
        imageNumber == 0 ? receiveHeader.format0 : receiveHeader.format1);
    int bits0 = getFormatBits(static_cast<ImagePair::ImageFormat>(receiveHeader.format0), false);
    int bits1 = getFormatBits(static_cast<ImagePair::ImageFormat>(receiveHeader.format1), false);

    unsigned char* ret = nullptr;

    if(receiveHeader.secondTileWidth == 0) {
        int bufferOffset = imageNumber*receiveHeader.width * bits0/8;
        int bufferRowStride = receiveHeader.width*(bits0 + bits1) / 8;

        if(format == ImagePair::FORMAT_8_BIT_MONO || format == ImagePair::FORMAT_8_BIT_RGB) {
            // No decoding is necessary. We can just pass through the
            // data pointer
            ret = &data[bufferOffset];
            rowStride = bufferRowStride;
            validRows = receivedBytes / bufferRowStride;
        } else {
            // Perform 12-bit => 16 bit decoding
            allocateDecodeBuffer(imageNumber);
            validRows = receivedBytes / bufferRowStride;
            rowStride = 2*receiveHeader.width;
            int lastRow = lastReceivedPayloadBytes[imageNumber] / bufferRowStride;

            BitConversions::decode12BitPacked(lastRow, validRows, &data[bufferOffset],
                &decodeBuffer[imageNumber][0], bufferRowStride, rowStride, receiveHeader.width);

            ret = &decodeBuffer[imageNumber][0];
        }
    } else {
        // Decode the tiled transfer
        decodeTiledImage(imageNumber,
            lastReceivedPayloadBytes[imageNumber], receivedBytes,
            data, receiveHeader.firstTileWidth * (bits0 + bits1) / 8,
            receiveHeader.secondTileWidth * (bits0 + bits1) / 8,
            validRows, format);
        ret = &decodeBuffer[imageNumber][0];
        rowStride = receiveHeader.width*getFormatBits(
            static_cast<ImagePair::ImageFormat>(format), true)/8;
    }

    lastReceivedPayloadBytes[imageNumber] = receivedBytes;
    return ret;
}

void ImageProtocol::Pimpl::allocateDecodeBuffer(int imageNumber) {
    ImagePair::ImageFormat format = static_cast<ImagePair::ImageFormat>(
        imageNumber == 0 ? receiveHeader.format0 : receiveHeader.format1);
    int bytesPerPixel = getFormatBits(format, true);
    int bufferSize = receiveHeader.width * receiveHeader.height * bytesPerPixel;

    if(decodeBuffer[imageNumber].size() != static_cast<unsigned int>(bufferSize)) {
        decodeBuffer[imageNumber].resize(bufferSize);
    }
}

void ImageProtocol::Pimpl::decodeTiledImage(int imageNumber, int lastReceivedPayloadBytes, int receivedPayloadBytes,
        const unsigned char* data, int firstTileStride, int secondTileStride, int& validRows,
        ImagePair::ImageFormat format) {

    // Allocate a decoding buffer
    allocateDecodeBuffer(imageNumber);

    // Get beginning and end of first tile
    int startFirstTile = lastReceivedPayloadBytes / firstTileStride;
    int stopFirstTile = std::min(receivedPayloadBytes / firstTileStride,
        static_cast<int>(receiveHeader.height));

    // Get beginning and end of second tile
    int secondTileBytes = receivedPayloadBytes - (receiveHeader.height*firstTileStride);
    int lastSecondTileBytes = lastReceivedPayloadBytes - (receiveHeader.height*firstTileStride);
    int startSecondTile = std::max(0, lastSecondTileBytes / secondTileStride);
    int stopSecondTile = std::max(0, secondTileBytes / secondTileStride);
    int firstTileOffset = imageNumber * getFormatBits(
        static_cast<ImagePair::ImageFormat>(receiveHeader.format0), false) * receiveHeader.firstTileWidth / 8;

    // Decode first tile
    int bytesPixel = 0;
    if(format == ImagePair::FORMAT_12_BIT_MONO) {
        bytesPixel = 16;
        BitConversions::decode12BitPacked(startFirstTile, stopFirstTile, &data[firstTileOffset], &decodeBuffer[imageNumber][0],
            firstTileStride, 2*receiveHeader.width, receiveHeader.firstTileWidth);
    } else {
        bytesPixel = (format == ImagePair::FORMAT_8_BIT_RGB ? 3 : 1);
        decodeRowsFromTile(startFirstTile, stopFirstTile, &data[firstTileOffset],
            &decodeBuffer[imageNumber][0], firstTileStride, receiveHeader.width*bytesPixel,
            receiveHeader.firstTileWidth*bytesPixel);
    }

    // Decode second tile
    int secondTileOffset = receiveHeader.height*firstTileStride +
        imageNumber * getFormatBits(static_cast<ImagePair::ImageFormat>(receiveHeader.format0), false) *
        receiveHeader.secondTileWidth / 8;

    if(format == ImagePair::FORMAT_12_BIT_MONO) {
        BitConversions::decode12BitPacked(startSecondTile, stopSecondTile,
            &data[secondTileOffset], &decodeBuffer[imageNumber][2*receiveHeader.firstTileWidth],
            secondTileStride, 2*receiveHeader.width, receiveHeader.secondTileWidth);
    } else {
        decodeRowsFromTile(startSecondTile, stopSecondTile, &data[secondTileOffset],
            &decodeBuffer[imageNumber][receiveHeader.firstTileWidth*bytesPixel],
            secondTileStride, receiveHeader.width*bytesPixel, receiveHeader.secondTileWidth*bytesPixel);
    }

    validRows = stopSecondTile;
}

void ImageProtocol::Pimpl::decodeRowsFromTile(int startRow, int stopRow, unsigned const char* src,
        unsigned char* dst, int srcStride, int dstStride, int tileWidth) {
    for(int y = startRow; y < stopRow; y++) {
        memcpy(&dst[y*dstStride], &src[y*srcStride], tileWidth);
    }
}

void ImageProtocol::Pimpl::resetReception() {
    receiveHeaderParsed = false;
    lastReceivedPayloadBytes[0] = 0;
    lastReceivedPayloadBytes[1] = 0;
    dataProt.resetReception(false);
    receptionDone = false;
}

bool ImageProtocol::Pimpl::isConnected() const {
    return dataProt.isConnected();
}

const unsigned char* ImageProtocol::Pimpl::getNextControlMessage(int& length) {
    return dataProt.getNextControlMessage(length);
}

bool ImageProtocol::Pimpl::newClientConnected() {
    return dataProt.newClientConnected();
}

int ImageProtocol::Pimpl::getNumDroppedFrames() const {
    return dataProt.getDroppedReceptions();
}

} // namespace

