#include <QMessageBox>
#include <QPushButton>

#include <visiontransfer/deviceenumeration.h>
#include "connectiondialog.h"
#include "ui_connectiondialog.h"

using namespace visiontransfer;

ConnectionDialog::ConnectionDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::ConnectionDialog) {

    ui->setupUi(this);

    // Disable the update button until something is selected
    ui->connectButton->setEnabled(false);

    ui->hostsList->sortByColumn(0, Qt::AscendingOrder);

    // Connect signals
    QObject::connect(ui->hostsList, &QTreeWidget::itemDoubleClicked,
        this, [this](QTreeWidgetItem *item, int column){connectToHost();});
    QObject::connect(ui->hostsList, &QTreeWidget::itemSelectionChanged,
        this, [this](){
            if(ui != nullptr) {
                ui->connectButton->setEnabled(ui->hostsList->selectedItems().size() > 0);
            }
        });
    QObject::connect(ui->connectButton, &QPushButton::pressed,
        this, [this](){connectToHost();});
    QObject::connect(ui->cancelButton, &QPushButton::pressed,
        this, [this](){reject();});
    QObject::connect(&updateTimer, &QTimer::timeout,
        this, [this](){queryDevices();});

    updateTimer.start(0);
}

ConnectionDialog::~ConnectionDialog() {
    delete ui;
    ui = nullptr;
}

void ConnectionDialog::queryDevices() {
    DeviceEnumeration devEnum;
    prevDeviceList = currDeviceList;
    currDeviceList = devEnum.discoverDevices();

    if(currDeviceList != prevDeviceList) {
        createHostList();
    }

    updateTimer.start(UPDATE_INTERVAL_MS);
}

void ConnectionDialog::createHostList() {
    // Save selection
    std::string selectedIp = "";
    int selectedIndex = -1;
    QList<QTreeWidgetItem *> selectedItems = ui->hostsList->selectedItems();
    if(selectedItems.size() > 0 && selectedItems[0] != nullptr) {
        selectedIp = selectedItems.first()->data(0, Qt::DisplayRole).toString().toUtf8().constData();
    }

    // Clear all elements
    while (ui->hostsList->topLevelItemCount() > 0) {
        ui->hostsList->takeTopLevelItem(0);
    }
    hostItems.clear();

    // Create item objects
    for(int i=0; i<static_cast<int>(currDeviceList.size()); i++) {
        hostItems.push_back(QTreeWidgetItem());
        hostItems.back().setData(0, Qt::DisplayRole, currDeviceList[i].getIpAddress().c_str());
        hostItems.back().setData(1, Qt::DisplayRole,
            currDeviceList[i].getNetworkProtocol() == DeviceInfo::PROTOCOL_TCP ? "TCP" : "UDP");

        std::string modelString;
        switch(currDeviceList[i].getModel()) {
            case DeviceInfo::SCENESCAN: modelString = "SceneScan"; break;
            case DeviceInfo::SCENESCAN_PRO: modelString = "SceneScan Pro"; break;
            default: modelString = "Unknown"; break;
        }
        hostItems.back().setData(2, Qt::DisplayRole, modelString.c_str());
        hostItems.back().setData(3, Qt::DisplayRole, currDeviceList[i].getFirmwareVersion().c_str());
        if(!currDeviceList[i].isCompatible()) {
            hostItems.back().setIcon(0, QIcon(":/nvcom/icons/warning.png"));
        }
        hostItems.back().setData(4, Qt::DisplayRole, i);
        if(currDeviceList[i].getIpAddress() == selectedIp) {
            selectedIndex = hostItems.size()-1;
        }
    }

    // Insert objects
    for(unsigned int i=0; i<hostItems.size(); i++) {
        ui->hostsList->insertTopLevelItem(i, &hostItems[i]);
    }

    // Select the previously selected item or the first item in the list
    if(selectedIndex != -1) {
        hostItems[selectedIndex].setSelected(true);
    } else if(hostItems.size() > 0 && selectedIp == "") {
        ui->hostsList->itemAt(0,0)->setSelected(true);
    }
}

void ConnectionDialog::connectToHost() {
    QList<QTreeWidgetItem *> selectedItems = ui->hostsList->selectedItems();

    if(selectedItems.size() > 0 && selectedItems[0] != nullptr) {
        int index = selectedItems.first()->data(4, Qt::DisplayRole).toInt();
        if(!currDeviceList[index].isCompatible()) {
            QMessageBox msgBox(QMessageBox::Critical, "Incompatible Firmware!",
                "The selected device has an incompatible firmware version. Please "
                "either update the device firmware or your version of NVCom");
            msgBox.exec();
        } else if(index >= 0 && index < static_cast<int>(currDeviceList.size())) {
            selectedDevice = currDeviceList[index];
            accept();
        }
    }
}
