#!/usr/bin/env python
# -*- coding: euc-jp -*-

##
# @file  ConnectorListener.py
# @brief connector listener class
# @date  $Date$
# @author Noriaki Ando <n-ando@aist.go.jp> and Shinji Kurihara
#
# Copyright (C) 2009
#     Noriaki Ando
#     Task-intelligence Research Group,
#     Intelligent Systems Research Institute,
#     National Institute of
#         Advanced Industrial Science and Technology (AIST), Japan
#     All rights reserved.

from omniORB import *
from omniORB import any

import OpenRTM_aist


##
# @if jp
# @brief ConnectorDataListener Υ
#
# - ON_BUFFER_WRITE:          Хåե񤭹߻
# - ON_BUFFER_FULL:           Хåեե
# - ON_BUFFER_WRITE_TIMEOUT:  Хåե񤭹ߥॢȻ
# - ON_BUFFER_OVERWRITE:      Хåե񤭻
# - ON_BUFFER_READ:           Хåեɤ߽Ф
# - ON_SEND:                  InProtؤ
# - ON_RECEIVED:              InProtؤλ
# - ON_RECEIVER_FULL:         InProt¦Хåեե
# - ON_RECEIVER_TIMEOUT:      InProt¦ХåեॢȻ
# - ON_RECEIVER_ERROR:        InProt¦顼
#
# @else
# @brief The types of ConnectorDataListener
# 
# - ON_BUFFER_WRITE:          At the time of buffer write
# - ON_BUFFER_FULL:           At the time of buffer full
# - ON_BUFFER_WRITE_TIMEOUT:  At the time of buffer write timeout
# - ON_BUFFER_OVERWRITE:      At the time of buffer overwrite
# - ON_BUFFER_READ:           At the time of buffer read
# - ON_SEND:                  At the time of sending to InPort
# - ON_RECEIVED:              At the time of finishing sending to InPort
# - ON_RECEIVER_FULL:         At the time of bufferfull of InPort
# - ON_RECEIVER_TIMEOUT:      At the time of timeout of InPort
# - ON_RECEIVER_ERROR:        At the time of error of InPort
#
# @endif
#
class ConnectorDataListenerType:
  def __init__(self):
    pass

  ON_BUFFER_WRITE              = 0
  ON_BUFFER_FULL               = 1
  ON_BUFFER_WRITE_TIMEOUT      = 2
  ON_BUFFER_OVERWRITE          = 3
  ON_BUFFER_READ               = 4
  ON_SEND                      = 5
  ON_RECEIVED                  = 6
  ON_RECEIVER_FULL             = 7
  ON_RECEIVER_TIMEOUT          = 8
  ON_RECEIVER_ERROR            = 9
  CONNECTOR_DATA_LISTENER_NUM  = 10



##
# @if jp
# @class ConnectorDataListener 饹
#
# ǡݡȤ Connector ˤȯƼ磻٥ȤФ륳
# Хå¸ꥹʥ饹δ쥯饹
#
# åOutPortФƥǡ񤭹ߡInPort¦ǥǡ
# ޤǤδ֤ȯƼ磻٥Ȥեå륳Хå
# ꤹ뤳ȤǤ롣ʤꥹʡ饹2¸ߤХåե
# ΥХåǡλͭʥǡե󥯥ΰ
# ȤƼ ConnectorDataListener Ǥꡢ⤦ϥǡ
# ץƥХåեɤ߹߻ΥॢȤʤɥǡǤʤ
# ʤɤ˥뤵ե󥯥ΰ˲Ȥʤ餤
# ConnecotorListener 롣
# 
# ǡݡȤˤϡ³˥ǡˡˤĤƥǡե
# ֥ץꤹ뤳ȤǤ롣
# ConnectorDaataListener/ConnectorListener ϤȤˡ͡ʥ٥
# Ф륳Хåꤹ뤳ȤǤ뤬ǡե
# ӥ֥ץ󷿤˱ơѲǽʤԲǽ
# ʤΤ䡢ƤӽФ륿ߥ󥰤ۤʤ롣
# ʲˡ󥿡եCORBA CDRξΥХå򼨤
# 
# OutPort:
#  -  Push: Subscription Typeˤꤵ˥٥Ȥμबʬ롣
#    - Flush: FlushˤϥХåեʤ ON_BUFFER ϤΥ٥Ȥȯʤ
#      - ON_SEND
#      - ON_RECEIVED
#      - ON_RECEIVER_FULL
#      - ON_RECEIVER_TIMEOUT
#      - ON_RECEIVER_ERROR
#      - ON_CONNECT
#      - ON_DISCONNECT
#      .
#    - New
#      - ON_BUFFER_WRITE
#      - ON_BUFFER_FULL
#      - ON_BUFFER_WRITE_TIMEOUT
#      - ON_BUFFER_OVERWRITE
#      - ON_BUFFER_READ
#      - ON_SEND
#      - ON_RECEIVED
#      - ON_RECEIVER_FULL
#      - ON_RECEIVER_TIMEOUT
#      - ON_RECEIVER_ERROR
#      - ON_SENDER_ERROR
#      - ON_CONNECT
#      - ON_DISCONNECT
#      .
#    - Periodic
#      - ON_BUFFER_WRITE
#      - ON_BUFFER_FULL
#      - ON_BUFFER_WRITE_TIMEOUT
#      - ON_BUFFER_READ
#      - ON_SEND
#      - ON_RECEIVED
#      - ON_RECEIVER_FULL
#      - ON_RECEIVER_TIMEOUT
#      - ON_RECEIVER_ERROR
#      - ON_BUFFER_EMPTY
#      - ON_SENDER_EMPTY
#      - ON_SENDER_ERROR
#      - ON_CONNECT
#      - ON_DISCONNECT
#      .
#    .
#  - Pull
#    - ON_BUFFER_READ
#    - ON_SEND
#    - ON_BUFFER_EMPTY
#    - ON_BUFFER_READ_TIMEOUT
#    - ON_SENDER_EMPTY
#    - ON_SENDER_TIMEOUT
#    - ON_SENDER_ERROR
#    - ON_CONNECT
#    - ON_DISCONNECT
# 
#  InPort:
#  - Push:
#      - ON_BUFFER_WRITE
#      - ON_BUFFER_FULL
#      - ON_BUFFER_WRITE_TIMEOUT
#      - ON_BUFFER_WRITE_OVERWRITE
#      - ON_RECEIVED
#      - ON_RECEIVER_FULL
#      - ON_RECEIVER_TIMEOUT
#      - ON_RECEIVER_ERROR
#      - ON_CONNECT
#      - ON_DISCONNECT
#      .
#  - Pull
#      - ON_CONNECT
#      - ON_DISCONNECT
# @else
# @class ConnectorDataListener class
#
# This class is abstract base class for listener classes that
# provides callbacks for various events in the data port's
# connectors.
#
# @endif
#
class ConnectorDataListener:
  """
  """

  def __del__(self):
    pass

  # virtual void operator()(const ConnectorInfo& info,
  #                         const cdrMemoryStream& data) = 0;
  def __call__(self, info, data):
    pass

  ##
  # @if jp
  #
  # @brief ConnectorDataListenerType ʸѴ
  #
  # ConnectorDataListenerType ʸѴ
  #
  # @param type Ѵо ConnectorDataListenerType
  #
  # @return ʸѴ
  #
  # @else
  #
  # @brief Convert ConnectorDataListenerType into the string.
  #
  # Convert ConnectorDataListenerType into the string.
  #
  # @param type The target ConnectorDataListenerType for transformation
  #
  # @return Trnasformation result of string representation
  #
  # @endif
  #
  def toString(type):
    typeString = ["ON_BUFFER_WRITE",
                  "ON_BUFFER_FULL",
                  "ON_BUFFER_WRITE_TIMEOUT",
                  "ON_BUFFER_OVERWRITE",
                  "ON_BUFFER_READ", 
                  "ON_SEND", 
                  "ON_RECEIVED",
                  "ON_RECEIVER_FULL", 
                  "ON_RECEIVER_TIMEOUT", 
                  "ON_RECEIVER_ERROR",
                  "CONNECTOR_DATA_LISTENER_NUM"]

    if type < ConnectorDataListenerType.CONNECTOR_DATA_LISTENER_NUM:
      return typeString[type]

    return ""

  toString = staticmethod(toString)


##
# @if jp
# @class ConnectorDataListenerT 饹
#
# ǡݡȤ Connector ˤȯƼ磻٥ȤФ륳
# Хå¸ꥹʥ饹δ쥯饹
# 
# Υ饹ϡoperator()() 2 cdrMemoryStream ǤϤʤ
# ºݤ˥ǡݡȤǻѤѿƥץ졼ȰȤ
# ϤȤǤ롣
#
# @else
# @class ConnectorDataListenerT class
#
# This class is abstract base class for listener classes that
# provides callbacks for various events in the data port's
# connectors.
#
# This class template can have practical data types that are used
# as typed variable for DataPort as an argument of template instead
# of cdrMemoryStream.
#
# @endif
#
class ConnectorDataListenerT(ConnectorDataListener):
  """
  """

  def __del__(self):
    pass


  ##
  # @if jp
  #
  # @brief Хå᥽å
  #
  # ǡǡݡȤǻѤѿѴ ConnectorDataListenerT
  # ΥХå᥽åɤƤӽФ
  #
  # @param info ConnectorInfo 
  # @param cdrdata cdrMemoryStreamΥǡ
  #
  # @else
  #
  # @brief Callback method
  #
  # This method invokes the callback method of ConnectorDataListenerT. 
  # Data is converted into the variable type used in DataPort.
  #
  # @param info ConnectorInfo 
  # @param cdrdata Data of cdrMemoryStream type
  #
  # @endif
  #
  # virtual void operator()(const ConnectorInfo& info,
  #                         const cdrMemoryStream& cdrdata)
  def __call__(self, info, cdrdata, data):
    endian = info.properties.getProperty("serializer.cdr.endian","little")
    if endian is not "little" and endian is not None:
      endian = OpenRTM_aist.split(endian, ",") # Maybe endian is ["little","big"]
      endian = OpenRTM_aist.normalize(endian) # Maybe self._endian is "little" or "big"

    if endian == "little":
      endian = True
    elif endian == "big":
      endian = False
    else:
      endian = True

    _data = cdrUnmarshal(any.to_any(data).typecode(), cdrdata, endian)
    return _data



##
# @if jp
# @brief ConnectorListener Υ
#  
# - ON_BUFFER_EMPTY:       Хåեξ
# - ON_BUFFER_READTIMEOUT: ХåեǥॢȤ
# - ON_SENDER_EMPTY:       OutPort¦Хåե
# - ON_SENDER_TIMEOUT:     OutPort¦ॢȻ
# - ON_SENDER_ERROR:       OutPort¦顼
# - ON_CONNECT:            ³Ω
# - ON_DISCONNECT:         ³ǻ
#
# @else
# @brief The types of ConnectorListener
# 
# - ON_BUFFER_EMPTY:       At the time of buffer empty
# - ON_BUFFER_READTIMEOUT: At the time of buffer read timeout
# - ON_BUFFER_EMPTY:       At the time of empty of OutPort
# - ON_SENDER_TIMEOUT:     At the time of timeout of OutPort
# - ON_SENDER_ERROR:       At the time of error of OutPort
# - ON_CONNECT:            At the time of connection
# - ON_DISCONNECT:         At the time of disconnection
#
# @endif
#
# enum ConnectorListenerType
class ConnectorListenerType:

  def __init__(self):
    pass
  
  ON_BUFFER_EMPTY        = 0
  ON_BUFFER_READ_TIMEOUT = 1
  ON_SENDER_EMPTY        = 2
  ON_SENDER_TIMEOUT      = 3
  ON_SENDER_ERROR        = 4
  ON_CONNECT             = 5
  ON_DISCONNECT          = 6
  CONNECTOR_LISTENER_NUM = 7



##
# @if jp
# @class ConnectorListener 饹
#
# ǡݡȤ Connector ˤȯƼ磻٥ȤФ륳
# Хå¸ꥹʥ饹δ쥯饹
#
# åOutPortФƥǡ񤭹ߡInPort¦ǥǡ
# ޤǤδ֤ȯƼ磻٥Ȥեå륳Хå
# ꤹ뤳ȤǤ롣ʤꥹʡ饹2¸ߤХåե
# ΥХåǡλͭʥǡե󥯥ΰ
# ȤƼ ConnectorDataListener Ǥꡢ⤦ϥǡ
# ץƥХåեɤ߹߻ΥॢȤʤɥǡǤʤ
# ʤɤ˥뤵ե󥯥ΰ˲Ȥʤ餤
# ConnecotorListener 롣
#
# ǡݡȤˤϡ³˥ǡˡˤĤƥǡե
# ֥ץꤹ뤳ȤǤ롣
# ConnectorDaataListener/ConnectorListener ϶ˤˡ͡ʥ٥
# Ф륳Хåꤹ뤳ȤǤ뤬ǡե
# ӥ֥ץ󷿤˱ơѤǤΡǤʤ
# ΡޤƤӽФ륿ߥ󥰤ۤʤ롣ʲˡ󥿡ե
# CORBA CDRξΥХå򼨤
#
# OutPort:
# -  Push: Subscription Typeˤꤵ˥٥Ȥμबʬ롣
#   - Flush: FlushˤϥХåեʤ ON_BUFFER ϤΥ٥Ȥȯʤ
#     - ON_SEND
#     - ON_RECEIVED
#     - ON_RECEIVER_FULL
#     - ON_RECEIVER_TIMEOUT
#     - ON_RECEIVER_ERROR
#     - ON_CONNECT
#     - ON_DISCONNECT
#     .
#   - New
#     - ON_BUFFER_WRITE
#     - ON_BUFFER_FULL
#     - ON_BUFFER_WRITE_TIMEOUT
#     - ON_BUFFER_OVERWRITE
#     - ON_BUFFER_READ
#     - ON_SEND
#     - ON_RECEIVED
#     - ON_RECEIVER_FULL
#     - ON_RECEIVER_TIMEOUT
#     - ON_RECEIVER_ERROR
#     - ON_SENDER_ERROR
#     - ON_CONNECT
#     - ON_DISCONNECT
#     .
#   - Periodic
#     - ON_BUFFER_WRITE
#     - ON_BUFFER_FULL
#     - ON_BUFFER_WRITE_TIMEOUT
#     - ON_BUFFER_READ
#     - ON_SEND
#     - ON_RECEIVED
#     - ON_RECEIVER_FULL
#     - ON_RECEIVER_TIMEOUT
#     - ON_RECEIVER_ERROR
#     - ON_BUFFER_EMPTY
#     - ON_SENDER_EMPTY
#     - ON_SENDER_ERROR
#     - ON_CONNECT
#     - ON_DISCONNECT
#     .
#   .
# - Pull
#   - ON_BUFFER_READ
#   - ON_SEND
#   - ON_BUFFER_EMPTY
#   - ON_BUFFER_READ_TIMEOUT
#   - ON_SENDER_EMPTY
#   - ON_SENDER_TIMEOUT
#   - ON_SENDER_ERROR
#   - ON_CONNECT
#   - ON_DISCONNECT
#
# InPort:
# - Push:
#     - ON_BUFFER_WRITE
#     - ON_BUFFER_FULL
#     - ON_BUFFER_WRITE_TIMEOUT
#     - ON_BUFFER_WRITE_OVERWRITE
#     - ON_RECEIVED
#     - ON_RECEIVER_FULL
#     - ON_RECEIVER_TIMEOUT
#     - ON_RECEIVER_ERROR
#     - ON_CONNECT
#     - ON_DISCONNECT
#     .
# - Pull
#     - ON_CONNECT
#     - ON_DISCONNECT
# @else
# @class ConnectorListener class
#
# This class is abstract base class for listener classes that
# provides callbacks for various events in the data port's
# connectors.
#
# @endif
#
class ConnectorListener:
  """
  """

  def __del__(self):
    pass

  # virtual void operator()(const ConnectorInfo& info) = 0;
  def __call__(self,  info):
    pass

  ##
  # @if jp
  #
  # @brief ConnectorListenerType ʸѴ
  #
  # ConnectorListenerType ʸѴ
  #
  # @param type Ѵо ConnectorListenerType
  #
  # @return ʸѴ
  #
  # @else
  #
  # @brief Convert ConnectorListenerType into the string.
  #
  # Convert ConnectorListenerType into the string.
  #
  # @param type The target ConnectorListenerType for transformation
  #
  # @return Trnasformation result of string representation
  #
  # @endif
  #
  def toString(type):
    typeString = ["ON_BUFFER_EMPTY",
                  "ON_BUFFER_READ_TIMEOUT",
                  "ON_SENDER_EMPTY", 
                  "ON_SENDER_TIMEOUT", 
                  "ON_SENDER_ERROR", 
                  "ON_CONNECT",
                  "ON_DISCONNECT",
                  "CONNECTOR_LISTENER_NUM"]

    if type < ConnectorListenerType.CONNECTOR_LISTENER_NUM:
      return typeString[type]

    return ""

  toString = staticmethod(toString)


##
# @if jp
# @class ConnectorDataListener ۥ饹
#
# ʣ ConnectorDataListener ݻ륯饹
#
# @else
# @class ConnectorDataListener holder class
#
# This class manages one ore more instances of ConnectorDataListener class.
#
# @endif
#
class ConnectorDataListenerHolder:
  """
  """

  ##
  # @if jp
  # @brief 󥹥ȥ饯
  # @else
  # @brief Constructor
  # @endif
  #
  def __init__(self):
    self._listeners = []
    return


  ##
  # @if jp
  # @brief ǥȥ饯
  # @else
  # @brief Destructor
  # @endif
  #
  def __del__(self):
    for listener in self._listeners:
      for (k,v) in listener.iteritems():
        if v:
          del k
    return

    
  ##
  # @if jp
  #
  # @brief ꥹʡɲ
  #
  # ꥹʡɲä롣
  #
  # @param self
  # @param listener ɲäꥹ
  # @param autoclean true:ǥȥ饯Ǻ,
  #                  false:ǥȥ饯Ǻʤ
  # @else
  #
  # @brief Add the listener.
  #
  # This method adds the listener. 
  #
  # @param self
  # @param listener Added listener
  # @param autoclean true:The listener is deleted at the destructor.,
  #                  false:The listener is not deleted at the destructor. 
  # @endif
  #
  # void addListener(ConnectorDataListener* listener, bool autoclean);
  def addListener(self, listener, autoclean):
    self._listeners.append({listener:autoclean})
    return

    
  ##
  # @if jp
  #
  # @brief ꥹʡκ
  #
  # ꥹʤ롣
  #
  # @param self
  # @param listener ꥹ
  # @else
  #
  # @brief Remove the listener. 
  #
  # This method removes the listener. 
  #
  # @param self
  # @param listener Removed listener
  # @endif
  #
  # void removeListener(ConnectorDataListener* listener);
  def removeListener(self, listener):
    for (i, _listener) in enumerate(self._listeners):
      if listener in _listener:
        del self._listeners[i][listener]
        return

    
  ##
  # @if jp
  #
  # @brief ꥹʡΤ
  #
  # ϿƤꥹʤΥХå᥽åɤƤӽФ
  #
  # @param self
  # @param info ConnectorInfo
  # @param cdrdata ǡ
  # @else
  #
  # @brief Notify listeners. 
  #
  # This calls the Callback method of the registered listener. 
  #
  # @param self
  # @param info ConnectorInfo
  # @param cdrdata Data
  # @endif
  #
  # void notify(const ConnectorInfo& info,
  #             const cdrMemoryStream& cdrdata);
  def notify(self, info, cdrdata):
    for listener in self._listeners:
      for (k,v) in listener.iteritems():
        k(info, cdrdata)
    return


##
# @if jp
# @class ConnectorListener ۥ饹
#
# ʣ ConnectorListener ݻ륯饹
#
# @else
# @class ConnectorListener holder class
#
# This class manages one ore more instances of ConnectorListener class.
#
# @endif
#
class ConnectorListenerHolder:
  """
  """

  ##
  # @if jp
  # @brief 󥹥ȥ饯
  # @else
  # @brief Constructor
  # @endif
  #
  def __init__(self):
    self._listeners = []
    return

    
  ##
  # @if jp
  # @brief ǥȥ饯
  # @else
  # @brief Destructor
  # @endif
  #
  def __del__(self):
    for listener in self._listeners:
      for (k,v) in listener.iteritems():
        if v:
          del k
    return
        
    
  ##
  # @if jp
  #
  # @brief ꥹʡɲ
  #
  # ꥹʡɲä롣
  #
  # @param self
  # @param listener ɲäꥹ
  # @param autoclean true:ǥȥ饯Ǻ,
  #                  false:ǥȥ饯Ǻʤ
  # @else
  #
  # @brief Add the listener.
  #
  # This method adds the listener. 
  #
  # @param self
  # @param listener Added listener
  # @param autoclean true:The listener is deleted at the destructor.,
  #                  false:The listener is not deleted at the destructor. 
  # @endif
  #
  # void addListener(ConnectorListener* listener, bool autoclean);
  def addListener(self, listener, autoclean):
    self._listeners.append({listener:autoclean})
    return


  ##
  # @if jp
  #
  # @brief ꥹʡκ
  #
  # ꥹʤ롣
  #
  # @param self
  # @param listener ꥹ
  # @else
  #
  # @brief Remove the listener. 
  #
  # This method removes the listener. 
  #
  # @param self
  # @param listener Removed listener
  # @endif
  #
  # void removeListener(ConnectorListener* listener);
  def removeListener(self, listener):
    for (i, _listener) in enumerate(self._listeners):
      if listener in _listener:
        del self._listeners[i][listener]
        return


  ##
  # @if jp
  #
  # @brief ꥹʡΤ
  #
  # ϿƤꥹʤΥХå᥽åɤƤӽФ
  #
  # @param self
  # @param info ConnectorInfo
  # @else
  #
  # @brief Notify listeners. 
  #
  # This calls the Callback method of the registered listener. 
  #
  # @param self
  # @param info ConnectorInfo
  # @endif
  #
  # void notify(const ConnectorInfo& info);
  def notify(self, info):
    for listener in self._listeners:
      for (k,v) in listener.iteritems():
        k(info)
    return


  
class ConnectorListeners:
  def __init__(self):
    self.connectorData_ = [ OpenRTM_aist.ConnectorDataListenerHolder() for i in range(OpenRTM_aist.ConnectorDataListenerType.CONNECTOR_DATA_LISTENER_NUM) ]
    self.connector_     = [ OpenRTM_aist.ConnectorListenerHolder() for i in range(OpenRTM_aist.ConnectorListenerType.CONNECTOR_LISTENER_NUM) ]
    return
