#!/usr/bin/env python
# -*- coding: euc-jp -*-

##
# @file PortBase.py
# @brief RTC's Port base class
# @date $Date: 2007/09/18 $
# @author Noriaki Ando <n-ando@aist.go.jp> and Shinji Kurihara
#
# Copyright (C) 2006-2008
#     Task-intelligence Research Group,
#     Intelligent Systems Research Institute,
#     National Institute of
#         Advanced Industrial Science and Technology (AIST), Japan
#     All rights reserved.


import threading
import copy

import OpenRTM_aist
import RTC, RTC__POA



##
# @if jp
# @class PortBase
# @brief Port δ쥯饹
#
# RTC::Port δȤʤ륯饹
# RTC::Port Ϥۤ UML Port γǰѾƤꡢۤƱΤΤȤߤʤ
# ȤǤ롣RT ݡͥȤΥ󥻥ץȤˤƤϡ
# Port ϥݡͥȤ°ݡͥȤ¾ΥݡͥȤߺ
# ԤǤꡢ̾ĤΥ󥿡եȴϢդ롣
# ݡͥȤ Port ̤ƳФ󥿡ե󶡤ޤ׵
# 뤳ȤǤPortϤ³ô
# <p>
# Port ζݥ饹ϡ̾ RT ݡͥȥ󥹥Ʊ
# 졢󶡡׵ᥤ󥿡եϿ塢RT ݡͥȤ
# Ͽ졢饢ǽ Port ȤƵǽ뤳ȤꤷƤ롣
# <p>
# RTC::Port  CORBA 󥿡եȤưʲΥڥ졼󶡤롣
#
# - get_port_profile()
# - get_connector_profiles()
# - get_connector_profile()
# - connect()
# - notify_connect()
# - disconnect()
# - notify_disconnect()
# - disconnect_all()
#
# Υ饹ǤϡΥڥ졼μ󶡤롣
# <p>
# Υڥ졼Τget_port_profile(), get_connector_profiles(),
# get_connector_profile(), connect(), disconnect(), disconnect_all() ϡ
# ֥饹ˤä˿񤤤ѹɬפʤᡢС饤
# 뤳ȤϿ侩ʤ
# <p>
# notify_connect(), notify_disconnect() ˤĤƤϡ֥饹󶡡׵
# 륤󥿡եμ˱ơ񤤤ѹɬפ
# ⤷ʤľܥС饤ɤ뤳ȤϿ侩줺
# Ҥ notify_connect(), notify_disconnect() ιˤƤҤ٤̤
# δؿ˴Ϣ ؿ򥪡С饤ɤ뤳Ȥˤ꿶񤤤ѹ
# Ȥ侩롣
#
# @since 0.4.0
#
# @else
# @class PortBase
# @brief Port base class
#
# This class is a base class of RTC::Port.
# RTC::Port inherits a concept of RT-Component, and can be regarded as almost
# the same as it. In the concept of RT-Component, Port is attached to the
# component, can mediate interaction between other components and usually is
# associated with some interfaces.
# Component can provide or require interface for outside via Port, and the
# Port plays a role to manage the connection.
# <p>
# Concrete class of Port assumes to be usually created at the same time that
# RT-Component's instance is created, be registerd to RT-Component after
# provided and required interfaces are registerd, and function as accessible
# Port from outside.
# <p>
# RTC::Port provides the following operations as CORBA interface:
#
# - get_port_profile()
# - get_connector_profiles()
# - get_connector_profile()
# - connect()
# - notify_connect()
# - disconnect()
# - notify_disconnect()
# - disconnect_all()
#
# This class provides implementations of these operations.
# <p>
# In these operations, as for get_port_profile(), get_connector_profiles(),
# get_connector_profile(), connect(), disconnect() and disconnect_all(),
# since their behaviors especially need not to be change in subclass, 
# overriding is not recommended.
# <p>
# As for notify_connect() and notify_disconnect(), you may have to modify
# behavior according to the kind of interfaces that subclass provides and
# requires, however it is not recommended these are overriden directly.
# In the section of notify_connect() and notify_disconnect() as described
# below, it is recommended that you modify behavior by overriding the
# protected function related to these functions.
#
# @since 0.4.0
#
# @endif
class PortBase(RTC__POA.PortService):
  """
  """



  ##
  # @if jp
  # @brief 󥹥ȥ饯
  #
  # PortBase Υ󥹥ȥ饯 Port ̾ name ˼Ԥ
  # ƱˡʬȤ CORBA Object ȤƳȤ PortProfile
  #  port_ref ˼ȤΥ֥ȥե󥹤Ǽ롣
  # ̾ˤϡ"." ʳʸѤ뤳ȤǤ롣
  #
  # @param self
  # @param name Port ̾(ǥե:None)
  #
  # @else
  #
  # @brief Constructor
  #
  # The constructor of the ProtBase class is given the name of this Port
  # and initialized. At the same time, the PortBase activates itself
  # as CORBA object and stores its object reference to the PortProfile's 
  # port_ref member.
  # Characters except "." can be used for the name of the port.
  #
  # @param name The name of Port 
  #
  # @endif
  def __init__(self, name=None):
    self._ownerInstanceName = "unknown"
    self._objref = self._this()

    self._profile = RTC.PortProfile("", [], RTC.PortService._nil, [], RTC.RTObject._nil,[])
    # Now Port name is <instance_name>.<port_name>. r1648
    if name is None:
      self._profile.name = "unknown.unknown"
    else:
      self._profile.name = self._ownerInstanceName+"."+name
      
    self._profile.port_ref = self._objref
    self._profile.owner = RTC.RTObject._nil
    self._profile_mutex = threading.RLock()
    self._connection_mutex = threading.RLock()
    self._rtcout = OpenRTM_aist.Manager.instance().getLogbuf(name)
    self._onPublishInterfaces = None
    self._onSubscribeInterfaces = None
    self._onConnected = None
    self._onUnsubscribeInterfaces = None
    self._onDisconnected = None
    self._onConnectionLost = None
    self._connectionLimit   = -1
    self._portconnListeners = None
    return

  
  ##
  # @if jp
  #
  # @brief ǥȥ饯
  #
  # ǥȥ饯ǤϡPortService CORBA ֥Ȥ deactivate 
  # Ԥdeactivate˺ݤ㳰ꤲ뤳ȤϤʤ
  #
  # @else
  #
  # @brief Destructor
  #
  # In the destructor, PortService CORBA object is deactivated.
  # This function never throws exception.
  #
  # @endif
  #
  def __del__(self):
    self._rtcout.RTC_TRACE("PortBase.__del__()")
    try:
      mgr = OpenRTM_aist.Manager.instance().getPOA()
      oid = mgr.servant_to_id(self)
      mgr.deactivate_object(oid)
    except:
      self._rtcout.RTC_ERROR(OpenRTM_aist.Logger.print_exception())
    

  ##
  # @if jp
  #
  # @brief [CORBA interface] PortProfile
  #
  # PortݻPortProfile֤
  # PortProfile ¤ΤϰʲΥСġ
  #
  # - name              [string ] Port ̾
  # - interfaces        [PortInterfaceProfileList ] Port ݻ
  #                     PortInterfaceProfile Υ
  # - port_ref          [Port Object ] Port ȤΥ֥ȥե
  # - connector_profile [ConnectorProfileList ] Port ݻ
  #                     ConnectorProfile Υ
  # - owner             [RTObject Object ]  Port ͭ
  #                     RTObjectΥե
  # - properties        [NVList ] ¾Υץѥƥ
  #
  # @param self
  #
  # @return PortProfile
  #
  # @else
  #
  # @brief [CORBA interface] Get the PortProfile of the Port
  #
  # This operation returns the PortProfile of the Port.
  # PortProfile struct has the following members,
  #
  # - name              [string ] The name of the Port.
  # - interfaces        [PortInterfaceProfileList] The sequence of 
  #                     PortInterfaceProfile owned by the Port
  # - port_ref          [Port Object] The object reference of the Port.
  # - connector_profile [ConnectorProfileList] The sequence of 
  #                     ConnectorProfile owned by the Port.
  # - owner             [RTObject Object] The object reference of 
  #                     RTObject that is owner of the Port.
  # - properties        [NVList] The other properties.
  #
  # @return the PortProfile of the Port
  #
  # @endif
  # PortProfile* get_port_profile()
  def get_port_profile(self):
    self._rtcout.RTC_TRACE("get_port_profile()")

    self.updateConnectors()

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)

    prof = RTC.PortProfile(self._profile.name,
                           self._profile.interfaces,
                           self._profile.port_ref,
                           self._profile.connector_profiles,
                           self._profile.owner,
                           self._profile.properties)

    return prof


  ##
  # @if jp
  #
  # @brief PortProfile 롣
  #
  # δؿϡ֥ݻƤ PortProfile 
  # const Ȥ֤ const ؿǤ롣
  # 
  # @post δؿƤӽФȤˤ֤ѹ뤳ȤϤʤ
  #
  # @return PortProfile
  #
  # @else
  #
  # @brief Get the PortProfile of the Port
  #
  # This function is a const function that returns a const
  # reference of the PortProfile stored in this Port.
  #
  # @post This function never changes the state of the object.
  #
  # @return PortProfile
  #
  # @endif
  # PortProfile& getPortProfile() const;
  def getPortProfile(self):
    self._rtcout.RTC_TRACE("getPortProfile()")
    return self._profile


  ##
  # @if jp
  #
  # @brief [CORBA interface] ConnectorProfileList
  #
  # Portݻ ConnectorProfile  sequence ֤
  # ConnectorProfile  Port ֤³ץեݻ빽¤ΤǤꡢ
  # ³Port֤Ǿ򴹤ԤϢ뤹٤Ƥ Port Ʊͤ
  # ݻ롣
  # ConnectorProfile ϰʲΥСݻƤ롣
  #
  # - name         [string ] Υͥ̾
  # - connector_id [string ] ˡID
  # - ports        [Port sequnce] Υͥ˴Ϣ Port Υ֥
  #                ե󥹤Υ󥹡
  # - properties   [NVList ] ¾Υץѥƥ
  #
  # @param self
  #
  # @return  Port ݻ ConnectorProfile
  #
  # @else
  #
  # @brief [CORBA interface] Get the ConnectorProfileList of the Port
  #
  # This operation returns a list of the ConnectorProfiles of the Port.
  # ConnectorProfile includes the connection information that describes 
  # relation between (among) Ports, and Ports exchange the ConnectionProfile
  # on connection process and hold the same information in each Port.
  # ConnectionProfile has the following members,
  #
  # - name         [string] The name of the connection.
  # - connector_id [string] Unique identifier.
  # - ports        [Port sequnce] The sequence of Port's object reference
  #                that are related the connection.
  # - properties   [NVList] The other properties.
  #
  # @return the ConnectorProfileList of the Port
  #
  # @endif
  # virtual ConnectorProfileList* get_connector_profiles()
  def get_connector_profiles(self):
    self._rtcout.RTC_TRACE("get_connector_profiles()")

    self.updateConnectors()

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    return self._profile.connector_profiles


  ##
  # @if jp
  #
  # @brief [CORBA interface] ConnectorProfile 
  #
  # connector_id ǻꤵ줿 ConnectorProfile ֤
  # ꤷ connector_id  ConnectorProfile ݻƤʤϡ
  #  ConnectorProfile ֤
  #
  # @param self
  # @param connector_id ConnectorProfile  ID
  #
  # @return connector_id ǻꤵ줿 ConnectorProfile
  #
  # @else
  #
  # @brief [CORBA interface] Get the ConnectorProfile
  #
  # This operation returns the ConnectorProfiles specified connector_id.
  #
  # @param connector_id ID of the ConnectorProfile
  #
  # @return the ConnectorProfile identified by the connector_id
  #
  # @endif
  # ConnectorProfile* get_connector_profile(const char* connector_id)
  def get_connector_profile(self, connector_id):
    self._rtcout.RTC_TRACE("get_connector_profile(%s)", connector_id)

    self.updateConnectors()

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                            self.find_conn_id(connector_id))
    if index < 0:
      conn_prof = RTC.ConnectorProfile("","",[],[])
      return conn_prof

    conn_prof = RTC.ConnectorProfile(self._profile.connector_profiles[index].name,
                                     self._profile.connector_profiles[index].connector_id,
                                     self._profile.connector_profiles[index].ports,
                                     self._profile.connector_profiles[index].properties)
    return conn_prof


  ##
  # @if jp
  #
  # @brief [CORBA interface] Port ³Ԥ
  #
  # Ϳ줿 ConnectoionProfile ξ˴ŤPort֤³Ω
  # 롣δؿϼ˥ץꥱץġ뤫Ƥӽ
  # ȤȤƤ롣
  # 
  # @pre ץꥱץϡݡͥȴ֤ʣ
  # Port ³뤿ˡŬڤͤ򥻥åȤ ConnectorProfile 
  # connect() ΰȤͿƸƤӽФʤФʤʤ
  #
  # @pre connect() Ϳ ConnectorProfile ΥСΤ
  # name, ports, properties СФƥǡ򥻥åȤʤ
  # ʤʤconnector_id ˤ̾ʸꤹ뤫ŬUUID
  # ʸꤹɬפ롣
  #
  # @pre ConnectorProfile::name ³ˤĤ̾ CORBA::string
  # ˳ǼǤǤդʸǤɬפ롣
  # 
  # @pre ConnectorProfile::connector_id Ϥ٤Ƥ³Фưդ
  # ID (̾UUID) Ǽ롣UUID connect() ؿǹ
  # ΤǡƤӽФ¦϶ʸꤹ롣¸³ƱUUID
  # ꤷ connect() ƤӽФˤ PRECONDITION_NOT_MET 顼
  # ֤γĥǴ¸³ץե򹹿뤿
  # ˴¸ UUID ꤷƸƤӽФˡѤǽ롣
  #
  # @pre ConnectorProfile::ports  RTC::PortService Υ󥹤ǡ
  # ³̾2İʾΥݡȤΥ֥ȻȤɬ
  # פ롣㳰ȤơݡȤΥ֥ȻȤ1ĤǼ
  # connect()ƤӽФȤǡݡȤΥ󥿡ե
  # ꡢüʥݡ(CORBARTC::PortServiceʳ)Ф
  # ³Ԥ⤢롣
  #
  # @pre ConnectorProfile::properties ϥݡȤ˴Ϣդ줿󥿡
  # եФץѥƥͿ뤿˻Ѥ롣ץѥƥϡ
  # string 򥭡Any ͤȤƤĥڥΥ󥹤Ǥꡢ
  # ˤǤդCORBAǡǼǤ뤬ǽʸ¤ string Ȥ
  # ƳǼ뤳Ȥ侩롣
  #
  # @pre ʾ connect() ƤӽФꤹ ConnectorProfile Υ
  # ФޤȤȰʲΤ褦ˤʤ롣
  #
  # - ConnectorProfile::name: Ǥդ³̾
  # - ConnectorProfile::connector_id: ʸ
  # - ConnectorProfile::ports: 1İʾΥݡ
  # - ConnectorProfile::properties: 󥿡եФץѥƥ
  #
  # @post connect() ؿϡConnectorProfile::ports˳Ǽ줿ݡ
  # ȥ󥹤ƬΥݡȤФ notify_connect() Ƥ֡
  #
  # @post notify_connect()  ConnectorProfile::ports ˳Ǽ줿ݡ
  # Ƚ notify_connect() 򥫥ɸƤӽФ롣Υ
  # ƤӽФϡnotify_connect() ǥ顼ФƤݡȤΥ
  # ȻȤͭǤ¤ꡢɬ٤ƤΥݡȤФƹԤ
  # Ȥݾڤ롣ͭǤʤ֥ȻȤ¸
  # 硢ΥݡȤ򥹥åפơΥݡȤФ
  # notify_connect() ƤӽФ
  #
  # @post connect() ؿϡnotify_connect()ͤRTC_OKǤС
  # RTC_OK ֤λ³ϴλ롣RTC_OKʳ
  # ξϡ³IDФdisconnect()ƤӽФ³
  # notify_connect() ֤顼꥿󥳡ɤ򤽤Τޤ֤
  # 
  # @post connect() ΰȤϤ ConnectorProfile ˤϡ
  # ConnectorProfile::connector_id ӡΥݡȤ
  # publishInterfaces() Ǹݡȥ󥿡եγƼ
  # ǼƤ롣connect()  notify_connect() 
  # ConnectorProfile::{name, ports} ѹ뤳ȤϤʤ
  #  
  # @param connector_profile ConnectorProfile
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief [CORBA interface] Connect the Port
  #
  # This operation establishes connection according to the given
  # ConnectionProfile inforamtion. This function is premised on
  # calling from mainly application program or tools.
  #
  # @pre To establish the connection among Ports of RT-Components,
  # application programs must call this operation giving
  # ConnectorProfile with valid values as an argument.
  #
  # @pre Out of ConnectorProfile member variables, "name", "ports"
  # and "properties" members shall be set valid
  # data. "connector_id" shall be set as empty string value or
  # valid string UUID value.
  #
  # @pre ConnectorProfile::name that is connection identifier shall
  # be any valid CORBA::string.
  # 
  #
  # @pre ConnectorProfile::connector_id shall be set unique
  # identifier (usually UUID is used) for all connections. Since
  # UUID string value is usually set in the connect() function,
  # caller should just set empty string. If the connect() is called
  # with the same UUID as existing connection, this function
  # returns PRECONDITION_NOT_MET error. However, in order to update
  # the existing connection profile, the "connect()" operation with
  # existing connector ID might be used as valid method by future
  # extension
  #
  # @pre ConnectorProfile::ports, which is sequence of
  # RTC::PortService references, shall store usually two or more
  # ports' references. As exceptions, the "connect()" operation
  # might be called with only one reference in ConnectorProfile, in
  # case of just getting interfaces information from the port, or
  # connecting a special port (i.e. the peer port except
  # RTC::PortService on CORBA).
  #
  # @pre ConnectorProfile::properties might be used to give certain
  # properties to the service interfaces associated with the port.
  # The properties is a sequence variable with a pair of key string
  # and Any type value. Although the A variable can store any type
  # of values, it is not recommended except string.
  #
  # @pre The following is the summary of the ConnectorProfile
  # member to be set when this operation is called.
  #
  # - ConnectorProfile::name: The any name of connection
  # - ConnectorProfile::connector_id: Empty string
  # - ConnectorProfile::ports: One or more port references
  # - ConnectorProfile::properties: Properties for the interfaces
  #
  # @post connect() operation will call the first port in the
  # sequence of the ConnectorProfile.
  #
  # @post "noify_connect()"s perform cascaded call to the ports
  # stored in the ConnectorProfile::ports by order. Even if errors
  # are raised by intermediate notify_connect() operation, as long
  # as ports' object references are valid, it is guaranteed that
  # this cascaded call is completed in all the ports.  If invalid
  # or dead ports exist in the port's sequence, the ports are
  # skipped and notify_connect() is called for the next valid port.
  #
  # @post connect() function returns RTC_OK if all the
  # notify_connect() return RTC_OK. At this time the connection is
  # completed.  If notify_connect()s return except RTC_OK,
  # connect() calls disconnect() operation with the connector_id to
  # destruct the connection, and then it returns error code from
  # notify_connect().
  #
  # @post The ConnectorProfile argument of the connect() operation
  # returns ConnectorProfile::connector_id and various information
  # about service interfaces that is published by
  # publishInterfaces() in the halfway ports. The connect() and
  # halfway notify_connect() functions never change
  # ConnectorProfile::{name, ports}.
  #
  # @param connector_profile The ConnectorProfile.
  # @return ReturnCode_t The return code of ReturnCode_t type.
  #
  # @endif
  #
  # virtual ReturnCode_t connect(ConnectorProfile& connector_profile)
  def connect(self, connector_profile):
    self._rtcout.RTC_TRACE("connect()")
    if self.isEmptyId(connector_profile):
      guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
      self.setUUID(connector_profile)
      assert(not self.isExistingConnId(connector_profile.connector_id))
      del guard
    else:
      guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
      if self.isExistingConnId(connector_profile.connector_id):
        self._rtcout.RTC_ERROR("Connection already exists.")
        return (RTC.PRECONDITION_NOT_MET,connector_profile)
      del guard

    try:
      retval,connector_profile = connector_profile.ports[0].notify_connect(connector_profile)
      if retval != RTC.RTC_OK:
        self._rtcout.RTC_ERROR("Connection failed. cleanup.")
        self.disconnect(connector_profile.connector_id)
    
      return (retval, connector_profile)
      #return connector_profile.ports[0].notify_connect(connector_profile)
    except:
      self._rtcout.RTC_ERROR(OpenRTM_aist.Logger.print_exception())
      return (RTC.BAD_PARAMETER, connector_profile)

    return (RTC.RTC_ERROR, connector_profile)


  ##
  # @if jp
  #
  # @brief [CORBA interface] Port ³ΤԤ
  #
  # Υڥ졼ϡPort֤³ԤݤˡPort֤Ū
  # ˸ƤФ륪ڥ졼Ǥäơ̾異ץꥱץ
  # 䡢PortʳRTCϢ֥ƤӽФ뤳Ȥꤵ
  # Ƥʤ
  #
  # notify_connect() Τϥƥץ졼ȥ᥽åɥѥȤơ
  # 饹Ǽ뤳Ȥ publishInterfaces(),
  # subscribeInterfaces() 2ĤδؿǸƤӽФμ
  # ʲ̤Ǥ롣
  #
  # - publishInterfaces(): 󥿡եθ
  # - connectNext():  Port  notify_connect() θƤӽФ
  # - subscribeInterfaces(): 󥿡եμ
  # - ³¸
  #
  # notify_connect()  ConnectorProfile::ports ˳ǼƤ
  # Port ν˽äơɾ˸ƤӽФԤȤˤꡢ
  # 󥿡եθȼϢ٤ƤΥݡȤФƹԤ
  # ΥɸƤӽФǤ뤳ȤϤʤɬ
  # ConnectorProfile::ports ˳ǼƤݡȤФƹԤ롣
  #
  # @pre notify_connect()  ConnectorProfile::ports ˳Ǽ
  #  Port ȥꥹȤΤ Port ȤλȤμ˳Ǽ
  #  Port Ф notify_connect() ƤӽФä
  # ConnectorProfile::ports ˤ Port λȤǼƤɬ
  # 롣⤷ȤλȤǼƤʤ硢¾νˤ
  # ꥨ顼񤭤ʤСBAD_PARAMETER 顼֤롣
  #
  # @pre ƤӽФ ConnectorProfile::connector_id ˤϰդID
  #  UUID ݻƤɬפ롣̾ connector_id 
  # connect() ؿˤͿ졢ʸξư̤Ǥ롣
  #
  # @post ConnectorProfile::name, ConnectorProfile::connector_id,
  # ConnectorProfile::ports  notify_connect() θƤӽФˤ
  # 񤭴뤳ȤϤʤѤǤ롣
  #
  # @post ConnectorProfile::properties  notify_connect() ǡ
  #  Port ĥӥ󥿡ե˴ؤ¾ Port
  # 뤿ˡץѥƥ󤬽񤭹ޤ롣
  #
  # @post ʤConnectorProfile::ports ΥꥹȤκǽ Port 
  # notify_connet() λǤϡ٤ƤδϢ Port 
  # notify_connect() θƤӽФλ롣publishInterfaces(),
  # connectNext(), subscribeInterfaces() ³¸Τ
  # 줫ʳǥ顼ȯǤ⡢顼ɤŪݻ
  # Ƥꡢǽ顼Υ顼ɤ֤롣
  #
  # @param connector_profile ConnectorProfile
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief [CORBA interface] Notify the Ports connection
  #
  # This operation is usually called from other ports' connect() or
  # notify_connect() operations when connection between ports is
  # established.  This function is not premised on calling from
  # other functions or application programs.
  #
  # According to the template method pattern, the notify_connect()
  # calls "publishInterfaces()" and "subsctiveInterfaces()"
  # functions, which are premised on implementing in the
  # subclasses. The processing sequence is as follows.
  #
  # - publishInterfaces(): Publishing interface information
  # - connectNext(): Calling notify_connect() of the next port
  # - subscribeInterfaces(): Subscribing interface information
  # - Storing connection profile
  #
  # According to the order of port's references stored in the
  # ConnectorProfile::ports, publishing interface information to
  # all the ports and subscription interface information from all
  # the ports is performed by "notify_connect()"s.  This cascaded
  # call never aborts in the halfway operations, and calling
  # sequence shall be completed for all the ports.
  #
  # @pre notify_connect() calls notify_connect() for the port's
  # reference that is stored in next of this port's reference in
  # the sequence of the ConnectorProfile::ports. Therefore the
  # reference of this port shall be stored in the
  # ConnectorProfile::ports. If this port's reference is not stored
  # in the sequence, BAD_PARAMETER error will be returned, except
  # the return code is overwritten by other operations.
  #
  # @pre UUID shall be set to ConnectorProfile::connector_id as a
  # unique identifier when this operation is called.  Usually,
  # connector_id is given by a connect() function and, the behavior
  # is undefined in the case of a null character.
  #
  # @post ConnectorProfile::name, ConnectorProfile::connector_id,
  # ConnectorProfile::ports are invariant, and they are never
  # rewritten by notify_connect() operations.
  #
  # @post In order to transfer interface information to other
  # ports, interface property information is stored into the
  # ConnectorProfile::properties.
  #
  # @post At the end of notify_connect() operation for the first
  # port stored in the ConnectorProfile::ports sequence, the
  # related ports' notify_connect() invocations complete. Even if
  # errors are raised at the halfway of publishInterfaces(),
  # connectNext(), subscribeInterfaces() and storing process of
  # ConnectorProfile, error codes are saved and the first error is
  # returned.
  #
  # @param connector_profile The ConnectorProfile.
  # @return ReturnCode_t The return code of ReturnCode_t type.
  #
  # @endif
  #
  # virtual ReturnCode_t notify_connect(ConnectorProfile& connector_profile)
  def notify_connect(self, connector_profile):
    self._rtcout.RTC_TRACE("notify_connect()")

    guard_connection = OpenRTM_aist.ScopedLock(self._connection_mutex)

    # publish owned interface information to the ConnectorProfile
    retval = [RTC.RTC_OK for i in range(3)]

    self.onNotifyConnect(self.getName(),connector_profile)
    retval[0] = self.publishInterfaces(connector_profile)
    if retval[0] != RTC.RTC_OK:
      self._rtcout.RTC_ERROR("publishInterfaces() in notify_connect() failed.")

    self.onPublishInterfaces(self.getName(), connector_profile, retval[0])
    if self._onPublishInterfaces:
      self._onPublishInterfaces(connector_profile)

    # call notify_connect() of the next Port
    retval[1], connector_profile = self.connectNext(connector_profile)
    if retval[1] != RTC.RTC_OK:
      self._rtcout.RTC_ERROR("connectNext() in notify_connect() failed.")

    self.onConnectNextport(self.getName(), connector_profile, retval[1])
    # subscribe interface from the ConnectorProfile's information
    if self._onSubscribeInterfaces:
      self._onSubscribeInterfaces(connector_profile)

    retval[2] = self.subscribeInterfaces(connector_profile)
    if retval[2] != RTC.RTC_OK:
      self._rtcout.RTC_ERROR("subscribeInterfaces() in notify_connect() failed.")
      #self.notify_disconnect(connector_profile.connector_id)

    self.onSubscribeInterfaces(self.getName(), connector_profile, retval[2])

    self._rtcout.RTC_PARANOID("%d connectors are existing",
                              len(self._profile.connector_profiles))

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    # update ConnectorProfile
    index = self.findConnProfileIndex(connector_profile.connector_id)
    if index < 0:
      OpenRTM_aist.CORBA_SeqUtil.push_back(self._profile.connector_profiles,
                                           connector_profile)
      self._rtcout.RTC_PARANOID("New connector_id. Push backed.")

    else:
      self._profile.connector_profiles[index] = connector_profile
      self._rtcout.RTC_PARANOID("Existing connector_id. Updated.")

    for ret in retval:
      if ret != RTC.RTC_OK:
        self.onConnected(self.getName(), connector_profile, ret)
        return (ret, connector_profile)

    # connection established without errors
    if self._onConnected:
      self._onConnected(connector_profile)
    self.onConnected(self.getName(), connector_profile, RTC.RTC_OK)
    return (RTC.RTC_OK, connector_profile)


  ##
  # @if jp
  #
  # @brief [CORBA interface] Port ³
  #
  # Υڥ졼Ϳ줿 connector_id б³
  # 롣connector_id ̾ƥΤˤưդ UUID ʸ
  # Ǥꡢ connect()/notify_connect() θƤӽФˤ
  # Ω줿³ץե ConnectorProfile::connector_id б
  # 롣
  #
  # @pre connector_id  Port ݻ ConnectorProfile ξʤ
  # Ĥ ID ˰פʸǤʤФʤʤ Port 
  # ConnectorProfile Υꥹ connector_id Ʊ ID 
  # ConnectorProfile ¸ߤʤФδؿ BAD_PARAMETER 顼
  # ֤
  #
  # @pre connector_id Ʊ ID  ConnectorProfile::ports ˤ
  # ͭ Port λȤޤޤƤʤФʤʤ
  #
  # @post disconnect() ؿϡConnectorProfile::ports  Port λ
  # ȥꥹȤƬФơnotify_disconnect() ƤӽФȤ̵
  # Ǥʤɡnotify_disconnect() θƤӽФ˼Ԥˤϡ
  # ȥꥹȤƬ֤ޤ notify_disconnect() θ
  # ӽФnotify_disconnect() θƤӽФ˰ĤǤС
  # notify_disconnect() ֵͤ򤽤Τޤ֤Ĥʤä
  # ˤ RTC_ERROR 顼֤
  # 
  # @param connector_id ConnectorProfile  ID
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief [CORBA interface] Disconnect the Port
  #
  # This operation destroys connection between this port and the
  # peer port according to given connector_id. Usually connector_id
  # should be a UUID string that is unique in the system.  And the
  # connection, which is established by connect()/notify_connect()
  # functions, is identified by the ConnectorProfile::connector_id.
  #
  # @pre connector_id shall be a character string which is same
  # with ID of at least one of the ConnectorProfiles stored in this
  # port. If ConnectorProfile that has same ID with the given
  # connector_id does not exist in the list of ConnectorProfile,
  # this operation returns BAD_PARAMTER error.
  #
  # @pre ConnectorProfile::ports that is same ID with given
  # connector_id shall store the valid ports' references.
  #
  # @post disconnect() function invokes the notify_disconnect() for
  # the port that is stored in the first of the
  # ConnectorProfile::ports. If notify_disconnect() call fails for
  # the first port, It tries on calling "notify_disconnect()" in
  # order for ports stored in ConnectorProfile::ports until the
  # operation call is succeeded. If notify_disconnect() succeeded
  # for at least one port, it returns return code from
  # notify_disconnect(). If none of notify_connect() call
  # succeeded, it returns RTC_ERROR error.
  #
  # @param connector_id The ID of the ConnectorProfile.
  # @return ReturnCode_t The return code of ReturnCode_t type.
  #
  # @endif
  #
  # virtual ReturnCode_t disconnect(const char* connector_id)
  def disconnect(self, connector_id):
    self._rtcout.RTC_TRACE("disconnect(%s)", connector_id)

    index = self.findConnProfileIndex(connector_id)

    if index < 0:
      self._rtcout.RTC_ERROR("Invalid connector id: %s", connector_id)
      return RTC.BAD_PARAMETER

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    if index < len(self._profile.connector_profiles):
      prof = self._profile.connector_profiles[index]
    del guard
    
    if len(prof.ports) < 1:
      self._rtcout.RTC_FATAL("ConnectorProfile has empty port list.")
      return RTC.PRECONDITION_NOT_MET

    for i in range(len(prof.ports)):
      p = prof.ports[i]
      try:
        return p.notify_disconnect(connector_id)
      except:
        self._rtcout.RTC_WARN(OpenRTM_aist.Logger.print_exception())
        continue

    self._rtcout.RTC_ERROR("notify_disconnect() for all ports failed.")
    return RTC.RTC_ERROR


  ##
  # @if jp
  #
  # @brief [CORBA interface] Port ³ΤԤ
  #
  # Υڥ졼ϡPort֤³ԤݤˡPort֤
  # Ū˸ƤФ륪ڥ졼Ǥꡢ̾異ץꥱץ
  # 䡢 Port ʳ RTC Ϣ֥ƤӽФ뤳Ȥ
  # ꤵƤʤ
  #
  # notify_disconnect() Τϥƥץ졼ȥ᥽åɥѥȤơ
  # ֥饹Ǽ뤳Ȥ unsubscribeInterfaces() ؿ
  # ǸƤӽФμϰʲ̤Ǥ롣
  #
  # - ConnectorProfile θ
  # -  Port  notify_disconnect() ƤӽФ
  # - unsubscribeInterfaces()
  # - ConnectorProfile κ
  #
  # notify_disconnect()  ConnectorProfile::ports ˳ǼƤ
  # Port ν˽äơɾ˸ƤӽФԤȤˤꡢ
  # ³β򤹤٤Ƥ Port Τ롣
  #
  # @pre Port Ϳ줿 connector_id б ConnectorProfile
  # ݻƤʤФʤʤ
  #
  # @post connector_id б ConnectorProfile Ĥʤ
  # BAD_PARAMETER 顼֤
  #
  # @post ɸƤӽФԤݤˤ ConnectorProfile::ports 
  # ݻƤ Port λȥꥹȤΤȤλȤμλȤ
  #  notify_disconnect() ƤӽФθƤӽФ㳰ȯ
  # ˤϡƤӽФ򥹥åפꥹȤμλȤФ
  # notify_disconnect() ƤӽФĤƤӽФʤ硢
  # RTC_ERROR 顼ɤ֤
  #
  # @post ʤConnectorProfile::ports ΥꥹȤκǽ Port 
  # notify_disconnet() λǤϡ٤ƤδϢ Port 
  # notify_disconnect() θƤӽФλ롣
  # 
  # @param connector_id ConnectorProfile  ID
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief [CORBA interface] Notify the Ports disconnection
  #
  # This operation is invoked between Ports internally when the
  # connection is destroied. Generally it is not premised on
  # calling from application programs or RTC objects except Port
  # object.
  #
  # According to the template method pattern, the
  # notify_disconnect() calls unsubsctiveInterfaces() function,
  # which are premised on implementing in the subclasses. The
  # processing sequence is as follows.
  #
  # - Searching ConnectorProfile
  # - Calling notify_disconnect() for the next port
  # - Unsubscribing interfaces
  # - Deleting ConnectorProfile
  #
  # notify_disconnect() notifies disconnection to all the ports by
  # cascaded call to the stored ports in the
  # ConnectorProfile::ports in order.
  #
  # @pre The port shall store the ConnectorProfile having same id
  # with connector_id.
  #
  # @post If ConnectorProfile of same ID with connector_id does not
  # exist, it returns BAD_PARAMETER error.
  #
  # @post For the cascaded call, this operation calls
  # noify_disconnect() for the port that is stored in the next of
  # this port in the ConnectorProfile::ports.  If the operation
  # call raises exception for some failure, it tries to call
  # notify_disconnect() and skips until the operation succeeded.
  # If none of operation call succeeded, it returns RTC_ERROR.
  #
  # @post At the end of notify_disconnect() operation for the first
  # port stored in the ConnectorProfile::ports sequence, the
  # related ports' notify_disconnect() invocations complete.
  #
  # @param connector_id The ID of the ConnectorProfile.
  # @return ReturnCode_t The return code of ReturnCode_t type.
  #
  # @endif
  #
  # virtual ReturnCode_t notify_disconnect(const char* connector_id)
  def notify_disconnect(self, connector_id):
    self._rtcout.RTC_TRACE("notify_disconnect(%s)", connector_id)

    guard_connection = OpenRTM_aist.ScopedLock(self._connection_mutex)
    # The Port of which the reference is stored in the beginning of
    # connectorProfile's PortServiceList is master Port.
    # The master Port has the responsibility of disconnecting all Ports.
    # The slave Ports have only responsibility of deleting its own
    # ConnectorProfile.

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)

    index = self.findConnProfileIndex(connector_id)

    if index < 0:
      self._rtcout.RTC_ERROR("Invalid connector id: %s", connector_id)
      return RTC.BAD_PARAMETER

    prof = RTC.ConnectorProfile(self._profile.connector_profiles[index].name,
                                self._profile.connector_profiles[index].connector_id,
                                self._profile.connector_profiles[index].ports,
                                self._profile.connector_profiles[index].properties)
    self.onNotifyDisconnect(self.getName(), prof)

    retval = self.disconnectNext(prof)
    self.onDisconnectNextport(self.getName(), prof, retval)

    if self._onUnsubscribeInterfaces:
      self._onUnsubscribeInterfaces(prof)
    self.onUnsubscribeInterfaces(self.getName(), prof)
    self.unsubscribeInterfaces(prof)

    if self._onDisconnected:
      self._onDisconnected(prof)

    OpenRTM_aist.CORBA_SeqUtil.erase(self._profile.connector_profiles, index)
    
    self.onDisconnected(self.getName(), prof, retval)
    return retval


  ##
  # @if jp
  #
  # @brief [CORBA interface] Port ³
  #
  # Υڥ졼Ϥ Port ˴ϢƤ³롣
  #
  # @param self
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief [CORBA interface] Connect the Port
  #
  # This operation destroys all connection channels owned by the Port.
  #
  # @return ReturnCode_t The return code of this operation.
  #
  # @endif
  # virtual ReturnCode_t disconnect_all()
  def disconnect_all(self):
    self._rtcout.RTC_TRACE("disconnect_all()")
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    plist = copy.deepcopy(self._profile.connector_profiles)
    del guard
    
    retcode = RTC.RTC_OK
    len_ = len(plist)
    self._rtcout.RTC_DEBUG("disconnecting %d connections.", len_)

    # disconnect all connections
    # Call disconnect() for each ConnectorProfile.
    for i in range(len_):
      tmpret = self.disconnect(plist[i].connector_id)
      if tmpret != RTC.RTC_OK:
        retcode = tmpret

    return retcode


  #============================================================
  # Local operations
  #============================================================

  ##
  # @if jp
  # @brief Port ̾ꤹ
  #
  # Port ̾ꤹ롣̾ Port ݻ PortProfile.name
  # ȿǤ롣
  #
  # @param self
  # @param name Port ̾
  #
  # @else
  # @brief Set the name of this Port
  #
  # This operation sets the name of this Port. The given Port's name is
  # applied to Port's PortProfile.name.
  #
  # @param name The name of this Port.
  #
  # @endif
  # void setName(const char* name);
  def setName(self, name):
    self._rtcout.RTC_TRACE("setName(%s)", name)
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    self._profile.name = name
    return

  ##
  # @if jp
  # @brief Port ̾
  # @else
  # @brief Get the name of this Port
  # @return The name of this Port.
  # @endif
  #
  # const char* PortBase::getName() const
  def getName(self):
    self._rtcout.RTC_TRACE("getName() = %s", self._profile.name)
    return self._profile.name


  ##
  # @if jp
  # @brief PortProfile
  #
  # Portݻ PortProfile  const Ȥ֤
  #
  # @param self
  #
  # @return  Port  PortProfile
  #
  # @else
  # @brief Get the PortProfile of the Port
  #
  # This operation returns const reference of the PortProfile.
  #
  # @return the PortProfile of the Port
  #
  # @endif
  # const PortProfile& getProfile() const;
  def getProfile(self):
    self._rtcout.RTC_TRACE("getProfile()")
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    return self._profile


  ##
  # @if jp
  #
  # @brief Port Υ֥ȻȤꤹ
  #
  # Υڥ졼 Port  PortProfile ˤ Port Ȥ
  # ֥ȻȤꤹ롣
  #
  # @param self
  # @param port_ref  Port Υ֥Ȼ
  #
  # @else
  #
  # @brief Set the object reference of this Port
  #
  # This operation sets the object reference itself
  # to the Port's PortProfile.
  #
  # @param The object reference of this Port.
  #
  # @endif
  # void setPortRef(PortService_ptr port_ref);
  def setPortRef(self, port_ref):
    self._rtcout.RTC_TRACE("setPortRef()")
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    self._profile.port_ref = port_ref


  ##
  # @if jp
  #
  # @brief Port Υ֥ȻȤ
  #
  # Υڥ졼 Port  PortProfile ݻƤ
  #  Port ȤΥ֥ȻȤ롣
  #
  # @param self
  #
  # @return  Port Υ֥Ȼ
  #
  # @else
  #
  # @brief Get the object reference of this Port
  #
  # This operation returns the object reference
  # that is stored in the Port's PortProfile.
  #
  # @return The object reference of this Port.
  #
  # @endif
  # PortService_ptr getPortRef();
  def getPortRef(self):
    self._rtcout.RTC_TRACE("getPortRef()")
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    return self._profile.port_ref


  ##
  # @if jp
  #
  # @brief Port  owner  RTObject ꤹ
  #
  # Υڥ졼 Port  PortProfile.owner ꤹ롣
  #
  # @param self
  # @param owner  Port ͭ RTObject λ
  #
  # @else
  #
  # @brief Set the owner RTObject of the Port
  #
  # This operation sets the owner RTObject of this Port.
  #
  # @param owner The owner RTObject's reference of this Port
  #
  # @endif
  # void setOwner(RTObject_ptr owner);
  def setOwner(self, owner):
    prof = owner.get_component_profile()
    self._ownerInstanceName = prof.instance_name
    self._rtcout.RTC_TRACE("setOwner(%s)", self._ownerInstanceName)

    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    plist = self._profile.name.split(".")
    if not self._ownerInstanceName:
      self._rtcout.RTC_ERROR("Owner is not set.")
      self._rtcout.RTC_ERROR("addXXXPort() should be called in onInitialize().")
    portname = self._ownerInstanceName+"."+plist[-1]

    self._profile.owner = owner
    self._profile.name = portname


  #============================================================
  # callbacks
  #============================================================

  ##
  # @if jp
  #
  # @brief 󥿡եݤ˸ƤФ륳Хå򥻥åȤ
  #
  # Υڥ졼ϡΥݡȤ³ˡݡȼȤĥ
  # ӥ󥿡ե륿ߥ󥰤ǸƤФ륳Х
  # ե󥯥򥻥åȤ롣
  #
  # Хåե󥯥νͭϡƤӽФ¦ˤꡢ֥
  # ɬפʤʤä˲ΤΤϸƤӽФ¦ǤǤ롣
  #
  # ΥХåե󥯥ϡPortBase饹β۴ؿǤ
  # publishInterfaces() ƤФ줿ȤˡƱ ConnectorProfile 
  # Ȥ˸ƤӽФ롣ΥХåѤơ
  # publishInterfaces()  ConnectorProfile ѹ뤳Ȥ
  # ǽǤ뤬³ط򾷤ʤ褦ConnectorProfile 
  # ѹˤդפ롣
  #
  # @param on_publish ConnectionCallback Υ֥饹֥ȤΥݥ
  #
  # @else
  #
  # @brief Setting callback called on publish interfaces
  #
  # This operation sets a functor that is called after publishing
  # interfaces process when connecting between ports.
  #
  # Since the ownership of the callback functor object is owned by
  # the caller, it has the responsibility of object destruction.
  # 
  # The callback functor is called after calling
  # publishInterfaces() that is virtual member function of the
  # PortBase class with an argument of ConnectorProfile type that
  # is same as the argument of publishInterfaces() function.
  # Although by using this functor, you can modify the ConnectorProfile
  # published by publishInterfaces() function, the modification
  # should be done carefully for fear of causing connection
  # inconsistency.
  #
  # @param on_publish a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  # void setOnPublishInterfaces(ConnectionCallback* on_publish);
  def setOnPublishInterfaces(self, on_publish):
    self._onPublishInterfaces = on_publish
    return


  ##
  # @if jp
  #
  # @brief 󥿡եݤ˸ƤФ륳Хå򥻥åȤ
  #
  # Υڥ졼ϡΥݡȤ³ˡΥݡȤĥ
  # ӥ󥿡ե륿ߥ󥰤ǸƤФ륳Х
  # ե󥯥򥻥åȤ롣
  #
  # Хåե󥯥νͭϡƤӽФ¦ˤꡢ֥
  # ɬפʤʤä˲ΤΤϸƤӽФ¦ǤǤ롣
  #
  # ΥХåե󥯥ϡPortBase饹β۴ؿǤ
  # subscribeInterfaces() ƤФˡƱ ConnectorProfile 
  # Ȥ˸ƤӽФ롣ΥХåѤơ
  # subscribeInterfaces() Ϳ ConnectorProfile ѹ뤳Ȥ
  # ǽǤ뤬³ط򾷤ʤ褦ConnectorProfile 
  # ѹˤդפ롣
  #
  # @param on_subscribe ConnectionCallback Υ֥饹֥ȤΥݥ
  #
  # @else
  #
  # @brief Setting callback called on publish interfaces
  #
  # This operation sets a functor that is called before subscribing
  # interfaces process when connecting between ports.
  #
  # Since the ownership of the callback functor object is owned by
  # the caller, it has the responsibility of object destruction.
  # 
  # The callback functor is called before calling
  # subscribeInterfaces() that is virtual member function of the
  # PortBase class with an argument of ConnectorProfile type that
  # is same as the argument of subscribeInterfaces() function.
  # Although by using this functor, you can modify ConnectorProfile
  # argument for subscribeInterfaces() function, the modification
  # should be done carefully for fear of causing connection
  # inconsistency.
  #
  # @param on_subscribe a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  #void setOnSubscribeInterfaces(ConnectionCallback* on_subscribe);
  def setOnSubscribeInterfaces(self, on_subscribe):
    self._onSubscribeInterfaces = on_subscribe
    return


  ##
  # @if jp
  #
  # @brief ³λ˸ƤФ륳Хå򥻥åȤ
  #
  # Υڥ졼ϡΥݡȤ³λ˸ƤФ롢Х
  # ե󥯥򥻥åȤ롣
  #
  # Хåե󥯥νͭϡƤӽФ¦ˤꡢ֥
  # ɬפʤʤä˲ΤΤϸƤӽФ¦ǤǤ롣
  #
  # ΥХåե󥯥ϡݡȤ³¹ԴؿǤ
  # notify_connect() νλľˡ³ｪλݤ˸¤ä
  # ƤӽФ륳ХåǤ롣³βǥ顼ȯ
  # ˤϸƤӽФʤ
  # 
  # ΥХåե󥯥 notify_connect()  out ѥ᡼
  # Ȥ֤ΤƱ ConnectorProfile ȤȤ˸ƤӽФΤǡ
  # ³ˤƸ줿٤ƤΥ󥿡ե뤳
  # Ǥ롣ΥХåѤơnotify_connect() ֤
  # ConnectorProfile ѹ뤳ȤǽǤ뤬³ط
  # 򾷤ʤ褦ConnectorProfile ѹˤդפ롣
  #
  # @param on_subscribe ConnectionCallback Υ֥饹֥ȤΥݥ
  #
  # @else
  #
  # @brief Setting callback called on connection established
  #
  # This operation sets a functor that is called when connection
  # between ports established.
  #
  # Since the ownership of the callback functor object is owned by
  # the caller, it has the responsibility of object destruction.
  # 
  # The callback functor is called only when notify_connect()
  # function successfully returns. In case of error, the functor
  # will not be called.
  #
  # Since this functor is called with ConnectorProfile argument
  # that is same as out-parameter of notify_connect() function, you
  # can get all the information of published interfaces of related
  # ports in the connection.  Although by using this functor, you
  # can modify ConnectorProfile argument for out-paramter of
  # notify_connect(), the modification should be done carefully for
  # fear of causing connection inconsistency.
  #
  # @param on_subscribe a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  # void setOnConnected(ConnectionCallback* on_connected);
  def setOnConnected(self, on_connected):
    self._onConnected = on_connected
    return


  ##
  # @if jp
  #
  # @brief 󥿡եݤ˸ƤФ륳Хå򥻥åȤ
  #
  # Υڥ졼ϡΥݡȤ³ˡΥݡȤĥ
  # ӥ󥿡ե륿ߥ󥰤ǸƤФ륳Х
  # ե󥯥򥻥åȤ롣
  #
  # Хåե󥯥νͭϡƤӽФ¦ˤꡢ֥
  # ɬפʤʤä˲ΤΤϸƤӽФ¦ǤǤ롣
  #
  # ΥХåե󥯥ϡPortBase饹β۴ؿǤ
  # unsubscribeInterfaces() ƤФˡƱ ConnectorProfile 
  # Ȥ˸ƤӽФ롣ΥХåѤơ
  # unsubscribeInterfaces() Ϳ ConnectorProfile ѹ뤳Ȥ
  # ǽǤ뤬³ط򾷤ʤ褦ConnectorProfile 
  # ѹˤդפ롣
  #
  # @param on_unsubscribe ConnectionCallback Υ֥饹֥
  # ȤΥݥ
  #
  # @else
  #
  # @brief Setting callback called on unsubscribe interfaces
  #
  # This operation sets a functor that is called before unsubscribing
  # interfaces process when disconnecting between ports.
  #
  # Since the ownership of the callback functor object is owned by
  # the caller, it has the responsibility of object destruction.
  # 
  # The callback functor is called before calling
  # unsubscribeInterfaces() that is virtual member function of the
  # PortBase class with an argument of ConnectorProfile type that
  # is same as the argument of unsubscribeInterfaces() function.
  # Although by using this functor, you can modify ConnectorProfile
  # argument for unsubscribeInterfaces() function, the modification
  # should be done carefully for fear of causing connection
  # inconsistency.
  #
  # @param on_unsubscribe a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  # void setOnUnsubscribeInterfaces(ConnectionCallback* on_subscribe);
  def setOnUnsubscribeInterfaces(self, on_subscribe):
    self._onUnsubscribeInterfaces = on_subscribe
    return


  ##
  # @if jp
  #
  # @brief ³˸ƤФ륳Хå򥻥åȤ
  #
  # Υڥ졼ϡΥݡȤ³˸ƤФ롢Х
  # ե󥯥򥻥åȤ롣
  #
  # Хåե󥯥νͭϡƤӽФ¦ˤꡢ֥
  # ɬפʤʤä˲ΤΤϸƤӽФ¦ǤǤ롣
  #
  # ΥХåե󥯥ϡݡȤ³¹ԴؿǤ
  # notify_disconnect() νλľˡƤӽФ륳ХåǤ롣
  # 
  # ΥХåե󥯥³б ConnectorProfile Ȥ
  # ˸ƤӽФ롣 ConnectorProfile ϤΥե󥯥ƽФ
  # ˴Τǡѹۤ˱ƶͿ뤳ȤϤʤ
  #
  # @param on_disconnected ConnectionCallback Υ֥饹֥
  # ȤΥݥ
  #
  # @else
  #
  # @brief Setting callback called on disconnected
  #
  # This operation sets a functor that is called when connection
  # between ports is destructed.
  #
  # Since the ownership of the callback functor object is owned by
  # the caller, it has the responsibility of object destruction.
  # 
  # The callback functor is called just before notify_disconnect()
  # that is disconnection execution function returns.
  #
  # This functor is called with argument of corresponding
  # ConnectorProfile.  Since this ConnectorProfile will be
  # destructed after calling this functor, modifications never
  # affect others.
  #
  # @param on_disconnected a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  # void setOnDisconnected(ConnectionCallback* on_disconnected);
  def setOnDisconnected(self, on_disconnected):
    self._onDisconnected = on_disconnected
    return

  # void setOnConnectionLost(ConnectionCallback* on_connection_lost);
  def setOnConnectionLost(self, on_connection_lost):
    self._onConnectionLost = on_connection_lost
    return


  ##
  # @if jp
  # @brief PortConnectListeners Υۥ򥻥åȤ
  #
  # ݡȤ³˴ؤꥹʷݻۥ饹ؤΥݥ󥿤
  # åȤ롣δؿ̾ƤRTObjectƤФ졢RTObject
  # ۥ饹ؤΥݥ󥿤åȤ롣
  #
  # @param portconnListeners PortConnectListeners ֥ȤΥݥ
  #
  # @else
  # @brief Setting PortConnectListener holder
  #
  # This operation sets a functor that is called when connection
  # of this port does lost. 
  #
  # @param on_connection_lost a pointer to ConnectionCallback's subclasses
  #
  # @endif
  #
  # void setPortConnectListenerHolder(PortConnectListeners* portconnListeners);
  def setPortConnectListenerHolder(self, portconnListeners):
    self._portconnListeners = portconnListeners
    return


  ##
  # @if jp
  #
  # @brief Interface (֥饹)
  #
  # Υڥ졼ϡnotify_connect() 󥹤λϤ˥
  # ؿǤ롣
  # notify_connect() Ǥϡ
  #
  # - publishInterfaces()
  # - connectNext()
  # - subscribeInterfaces()
  # - updateConnectorProfile()
  #
  # ν protected ؿ뤵³Ԥ롣
  # <br>
  #  Port ǤϤΥڥ졼򥪡С饤ɤȤ
  # Ϳ줿 ConnectorProfile ˽Ԥѥ᡼Ŭ
  # ǤСRteurnCode_t Υ顼ɤ֤
  # ̾ publishInterafaces() ˤƤϡ Port °
  # 󥿡ե˴ؤ ConnectorProfile ФŬڤꤷ
  # ¾ Port ΤʤФʤʤ
  # <br>
  # ޤδؿ뤵ʳǤϡ¾ Port  Interface ˴ؤ
  # Ϥ٤ƴޤޤƤʤΤǡ¾ Port  Interface 
  #  subscribeInterfaces() ǹԤ٤Ǥ롣
  # <br>
  # Υڥ졼ϡ connector_id ФƤ³
  # ¸ connector_id ФƤϹŬڤ˹Ԥɬפ롣<BR>
  # ֥饹Ǥμ
  #
  # @param self
  # @param connector_profile ³˴ؤץե
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief Publish interface information
  #
  # This operation is pure virutal method that would be called at the
  # beginning of the notify_connect() process sequence.
  # In the notify_connect(), the following methods would be called in order.
  #
  # - publishInterfaces()
  # - connectNext()
  # - subscribeInterfaces()
  # - updateConnectorProfile() 
  #
  # In the concrete Port, this method should be overridden. This method
  # processes the given ConnectorProfile argument and if the given parameter
  # is invalid, it would return error code of ReturnCode_t.
  # Usually, publishInterfaces() method should set interfaces information
  # owned by this Port, and publish it to the other Ports.
  # <br>
  # When this method is called, other Ports' interfaces information may not
  # be completed. Therefore, the process to obtain other Port's interfaces
  # information should be done in the subscribeInterfaces() method.
  # <br>
  # This operation should create the new connection for the new
  # connector_id, and should update the connection for the existing
  # connection_id.
  #
  # @param connector_profile The connection profile information
  # @return The return code of ReturnCode_t type.
  #
  #@endif
  def publishInterfaces(self, connector_profile):
    pass


  ##
  # @if jp
  #
  # @brief  Port Ф notify_connect() 򥳡뤹
  #
  # ConnectorProfile  port_ref ˳ǼƤ Port Υ֥
  # ե󥹤Υ󥹤椫顢Ȥ Port μ Port Ф
  # notify_connect() 򥳡뤹롣
  #
  # @param self
  # @param connector_profile ³˴ؤץե
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief Call notify_connect() of the next Port
  #
  # This operation calls the notify_connect() of the next Port's 
  # that stored in ConnectorProfile's port_ref sequence.
  #
  # @param connector_profile The connection profile information
  #
  # @return The return code of ReturnCode_t type.
  #
  # @endif
  # virtual ReturnCode_t connectNext(ConnectorProfile& connector_profile);
  def connectNext(self, connector_profile):
    index = OpenRTM_aist.CORBA_SeqUtil.find(connector_profile.ports,
                                            self.find_port_ref(self._profile.port_ref))
    if index < 0:
      return (RTC.BAD_PARAMETER, connector_profile)

    index += 1
    if index < len(connector_profile.ports):
      p = connector_profile.ports[index]
      return p.notify_connect(connector_profile)

    return (RTC.RTC_OK, connector_profile)


  ##
  # @if jp
  #
  # @brief  Port Ф notify_disconnect() 򥳡뤹
  #
  # ConnectorProfile  port_ref ˳ǼƤ Port Υ֥
  # ե󥹤Υ󥹤椫顢Ȥ Port μ Port Ф
  # notify_disconnect() 򥳡뤹롣
  #
  # @param self
  # @param connector_profile ³˴ؤץե
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief Call notify_disconnect() of the next Port
  #
  # This operation calls the notify_disconnect() of the next Port's 
  # that stored in ConnectorProfile's port_ref sequence.
  #
  # @param connector_profile The connection profile information
  #
  # @return The return code of ReturnCode_t type.
  #
  # @endif
  # virtual ReturnCode_t disconnectNext(ConnectorProfile& connector_profile);
  def disconnectNext(self, connector_profile):
    index = OpenRTM_aist.CORBA_SeqUtil.find(connector_profile.ports,
                                            self.find_port_ref(self._profile.port_ref))
    if index < 0:
      return RTC.BAD_PARAMETER

    if index == (len(connector_profile.ports) - 1):
      return RTC.RTC_OK

    index += 1

    while index < len(connector_profile.ports):
      p = connector_profile.ports[index]
      index += 1
      try:
        return p.notify_disconnect(connector_profile.connector_id)
      except:
        self._rtcout.RTC_WARN(OpenRTM_aist.Logger.print_exception())
        continue

    return RTC.RTC_ERROR


  ##
  # @if jp
  #
  # @brief Interface (֥饹)
  #
  # Υڥ졼ϡnotify_connect() 󥹤֤˥
  # ؿǤ롣
  # notify_connect() Ǥϡ
  #
  #  - publishInterfaces()
  #  - connectNext()
  #  - subscribeInterfaces()
  #  - updateConnectorProfile()
  #
  # ν protected ؿ뤵³Ԥ롣
  # <br>
  #  Port ǤϤΥڥ졼򥪡С饤ɤȤ
  # Ϳ줿 ConnectorProfile ˽Ԥѥ᡼Ŭ
  # ǤСRteurnCode_t Υ顼ɤ֤
  #  ConnectorProfile ˤ¾ Port  Interface ˴ؤ
  # ƴޤޤƤ롣
  # ̾ subscribeInterafaces() ˤƤϡ Port Ѥ
  # Interface ˴ؤ׵¦Υ󥿡եФ
  # ꤷʤФʤʤ
  # <br>
  # Υڥ졼ϡ connector_id ФƤ³
  # ¸ connector_id ФƤϹŬڤ˹Ԥɬפ롣<BR>
  # ֥饹Ǥμ
  #
  # @param self
  # @param connector_profile ³˴ؤץե
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  #
  # @brief Publish interface information
  #
  # This operation is pure virutal method that would be called at the
  # mid-flow of the notify_connect() process sequence.
  # In the notify_connect(), the following methods would be called in order.
  #
  #  - publishInterfaces()
  #  - connectNext()
  #  - subscribeInterfaces()
  #  - updateConnectorProfile()
  #
  # In the concrete Port, this method should be overridden. This method
  # processes the given ConnectorProfile argument and if the given parameter
  # is invalid, it would return error code of ReturnCode_t.
  # The given argument ConnectorProfile includes all the interfaces
  # information in it.
  # Usually, subscribeInterafaces() method obtains information of interfaces
  # from ConnectorProfile, and should set it to the interfaces that require
  # them.
  # <br>
  # This operation should create the new connection for the new
  # connector_id, and should update the connection for the existing
  # connection_id.
  #
  # @param connector_profile The connection profile information
  #
  # @return The return code of ReturnCode_t type.
  #
  #@endif
  def subscribeInterfaces(self, connector_profile):
    pass


  ##
  # @if jp
  #
  # @brief Interface ³(֥饹)
  #
  # Υڥ졼ϡnotify_disconnect() 󥹤ν˥
  # ؿǤ롣
  # notify_disconnect() Ǥϡ
  #  - disconnectNext()
  #  - unsubscribeInterfaces()
  #  - eraseConnectorProfile()
  # ν protected ؿ뤵³Ԥ롣
  # <br>
  #  Port ǤϤΥڥ졼򥪡С饤ɤȤ
  # Ϳ줿 ConnectorProfile ˽³Ԥ<BR>
  # ֥饹Ǥμ
  #
  # @param self
  # @param connector_profile ³˴ؤץե
  #
  # @else
  #
  # @brief Disconnect interface connection
  #
  # This operation is pure virutal method that would be called at the
  # end of the notify_disconnect() process sequence.
  # In the notify_disconnect(), the following methods would be called.
  #  - disconnectNext()
  #  - unsubscribeInterfaces()
  #  - eraseConnectorProfile() 
  # <br>
  # In the concrete Port, this method should be overridden. This method
  # processes the given ConnectorProfile argument and disconnect interface
  # connection.
  #
  # @param connector_profile The connection profile information
  #
  # @endif
  def unsubscribeInterfaces(self, connector_profile):
    pass


  ##
  # @if jp
  #
  # @brief ³κꤹ롣
  #
  # @param limit_value 
  #
  # @else
  #
  # @brief Set the maximum number of connections
  #
  #
  # @param limit_value The maximum number of connections
  #
  # @endif
  #
  # virtual void setConnectionLimit(int limit_value);
  def setConnectionLimit(self, limit_value):
    self._connectionLimit = limit_value
    return
    

  ##
  # @if jp
  # @brief Interface
  #
  # Interface롣
  #
  #  dataport.dataflow_type
  #
  # @return ReturnCode_t Υ꥿󥳡
  #
  # @else
  # @brief Publish interface information
  #
  # Publish interface information.
  #
  #
  # @return The return code of ReturnCode_t type
  #
  # @endif
  #
  # virtual ReturnCode_t _publishInterfaces(void);
  def _publishInterfaces(self):
    if not (self._connectionLimit < 0) :
      if self._connectionLimit <= len(self._profile.connector_profiles):
        self._rtcout.RTC_PARANOID("Connected number has reached the limitation.")
        self._rtcout.RTC_PARANOID("Can connect the port up to %d ports.",
                                  self._connectionLimit)
        self._rtcout.RTC_PARANOID("%d connectors are existing",
                                  len(self._profile.connector_profiles))
        return RTC.RTC_ERROR

    return RTC.RTC_OK


  ##
  # @if jp
  #
  # @brief ConnectorProfile  connector_id եɤɤȽ
  #
  # ꤵ줿 ConnectorProfile  connector_id Ǥ뤫ɤȽ
  # Ԥ
  #
  # @param self
  # @param connector_profile Ƚоݥͥץե
  #
  # @return Ϳ줿 ConnectorProfile  connector_id ǤС
  #         trueǤʤ false ֤
  #
  # @else
  #
  # @brief Whether connector_id of ConnectorProfile is empty
  #
  # @return If the given ConnectorProfile's connector_id is empty string,
  #         it returns true.
  #
  # @endif
  # bool isEmptyId(const ConnectorProfile& connector_profile) const;
  def isEmptyId(self, connector_profile):
    return connector_profile.connector_id == ""


  ##
  # @if jp
  #
  # @brief UUID
  #
  # Υڥ졼 UUID 롣
  #
  # @param self
  #
  # @return uuid
  #
  # @else
  #
  # @brief Get the UUID
  #
  # This operation generates UUID.
  #
  # @return uuid
  #
  # @endif
  # const std::string getUUID() const;
  def getUUID(self):
    return str(OpenRTM_aist.uuid1())


  ##
  # @if jp
  #
  # @brief UUID ConnectorProfile ˥åȤ
  #
  # Υڥ졼 UUID ConnectorProfile ˥åȤ롣
  #
  # @param self
  # @param connector_profile connector_id 򥻥åȤ ConnectorProfile
  #
  # @else
  #
  # @brief Create and set the UUID to the ConnectorProfile
  #
  # This operation generates and set UUID to the ConnectorProfile.
  #
  # @param connector_profile ConnectorProfile to be set connector_id
  #
  # @endif
  # void setUUID(ConnectorProfile& connector_profile) const;
  def setUUID(self, connector_profile):
    connector_profile.connector_id = self.getUUID()
    assert(connector_profile.connector_id != "")


  ##
  # @if jp
  #
  # @brief id ¸ ConnectorProfile ΤΤɤȽꤹ
  #
  # Υڥ졼Ϳ줿 ID ¸ ConnectorProfile Υꥹ
  # ¸ߤ뤫ɤȽꤹ롣
  #
  # @param self
  # @param id_ Ƚꤹ connector_id
  #
  # @return id ¸Ƚ
  #
  # @else
  #
  # @brief Whether the given id exists in stored ConnectorProfiles
  #
  # This operation returns boolean whether the given id exists in 
  # the Port's ConnectorProfiles.
  #
  # @param id connector_id to be find in Port's ConnectorProfiles
  #
  # @endif
  # bool isExistingConnId(const char* id);
  def isExistingConnId(self, id_):
    return OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                           self.find_conn_id(id_)) >= 0


  ##
  # @if jp
  #
  # @brief id  ConnectorProfile õ
  #
  # Υڥ졼Ϳ줿 ID  ConnectorProfile  Port 
  #  ConnectorProfile Υꥹ椫õ
  # ⤷Ʊ id  ConnectorProfile ʤС ConnectorProfile
  # ֤롣
  #
  # @param self
  # @param id_  connector_id
  #
  # @return connector_id  ConnectorProfile
  #
  # @else
  #
  # @brief Find ConnectorProfile with id
  #
  # This operation returns ConnectorProfile with the given id from Port's
  # ConnectorProfiles' list.
  # If the ConnectorProfile with connector id that is identical with the
  # given id does not exist, empty ConnectorProfile is returned.
  #
  # @param id the connector_id to be searched in Port's ConnectorProfiles
  #
  # @return CoonectorProfile with connector_id
  #
  # @endif
  # ConnectorProfile findConnProfile(const char* id);
  def findConnProfile(self, id_):
    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                            self.find_conn_id(id_))
    if index < 0 or index >= len(self._profile.connector_profiles):
      return RTC.ConnectorProfile("","",[],[])

    return self._profile.connector_profiles[index]


  ##
  # @if jp
  #
  # @brief id  ConnectorProfile õ
  #
  # Υڥ졼Ϳ줿 ID  ConnectorProfile  Port 
  #  ConnectorProfile Υꥹ椫õǥå֤
  # ⤷Ʊ id  ConnectorProfile ʤС-1 ֤
  #
  # @param self
  # @param id_  connector_id
  #
  # @return Port  ConnectorProfile ꥹȤΥǥå
  #
  # @else
  #
  # @brief Find ConnectorProfile with id
  #
  # This operation returns ConnectorProfile with the given id from Port's
  # ConnectorProfiles' list.
  # If the ConnectorProfile with connector id that is identical with the
  # given id does not exist, empty ConnectorProfile is returned.
  #
  # @param id the connector_id to be searched in Port's ConnectorProfiles
  #
  # @return The index of ConnectorProfile of the Port
  #
  # @endif
  # CORBA::Long findConnProfileIndex(const char* id);
  def findConnProfileIndex(self, id_):
    return OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                           self.find_conn_id(id_))


  ##
  # @if jp
  #
  # @brief ConnectorProfile ɲä⤷Ϲ
  #
  # Υڥ졼Ϳ줿Ϳ줿 ConnectorProfile 
  # Port ɲä⤷Ϲ¸롣
  # Ϳ줿 ConnectorProfile  connector_id Ʊ ID 
  # ConnectorProfile ꥹȤˤʤСꥹȤɲä
  # Ʊ ID ¸ߤ ConnectorProfile ¸롣
  #
  # @param self
  # @param connector_profile ɲä⤷Ϲ ConnectorProfile
  #
  # @else
  #
  # @brief Append or update the ConnectorProfile list
  #
  # This operation appends or updates ConnectorProfile of the Port
  # by the given ConnectorProfile.
  # If the connector_id of the given ConnectorProfile does not exist
  # in the Port's ConnectorProfile list, the given ConnectorProfile would be
  # append to the list. If the same id exists, the list would be updated.
  #
  # @param connector_profile the ConnectorProfile to be appended or updated
  #
  # @endif
  # void updateConnectorProfile(const ConnectorProfile& connector_profile);
  def updateConnectorProfile(self, connector_profile):
    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                            self.find_conn_id(connector_profile.connector_id))

    if index < 0:
      OpenRTM_aist.CORBA_SeqUtil.push_back(self._profile.connector_profiles,
                                           connector_profile)
    else:
      self._profile.connector_profiles[index] = connector_profile


  ##
  # @if jp
  #
  # @brief ConnectorProfile 
  #
  # Υڥ졼 Port  PortProfile ݻƤ
  # ConnectorProfileList ΤͿ줿 id  ConnectorProfile
  # 롣
  #
  # @param self
  # @param id_  ConnectorProfile  id
  #
  # @return ˺Ǥ true
  #         ꤷ ConnectorProfile Ĥʤ false ֤
  #
  # @else
  #
  # @brief Delete the ConnectorProfile
  #
  # This operation deletes a ConnectorProfile specified by id from
  # ConnectorProfileList owned by PortProfile of this Port.
  #
  # @param id The id of the ConnectorProfile to be deleted.
  #
  # @endif
  # bool eraseConnectorProfile(const char* id);
  def eraseConnectorProfile(self, id_):
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)

    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.connector_profiles,
                                            self.find_conn_id(id_))

    if index < 0:
      return False

    OpenRTM_aist.CORBA_SeqUtil.erase(self._profile.connector_profiles, index)

    return True


  ##
  # @if jp
  #
  # @brief PortInterfaceProfile  󥿡եϿ
  #
  # Υڥ졼 Port  PortProfile ΡPortInterfaceProfile
  # ˥󥿡եξɲä롣
  # ξϡget_port_profile() ä PortProfile Τ
  # PortInterfaceProfile ͤѹΤߤǤꡢºݤ˥󥿡ե
  # 󶡤׵ᤷꤹˤϡ֥饹ǡ publishInterface() ,
  #  subscribeInterface() δؿŬڤ˥С饤ɤ󥿡ե
  # 󶡡׵ԤʤФʤʤ
  #
  # 󥿡ե(Υ󥹥)̾ Port ǰդǤʤФʤʤ
  # Ʊ̾Υ󥿡եǤϿƤ硢δؿ false 
  # ֤
  #
  # @param self
  # @param instance_name 󥿡եΥ󥹥󥹤̾
  # @param type_name 󥿡եη̾
  # @param pol 󥿡ե° (RTC::PROVIDED ⤷ RTC:REQUIRED)
  #
  # @return 󥿡եϿ̡
  #         Ʊ̾Υ󥿡եϿƤ false ֤
  #
  # @else
  #
  # @brief Append an interface to the PortInterfaceProfile
  #
  # This operation appends interface information to the PortInterfaceProfile
  # that is owned by the Port.
  # The given interfaces information only updates PortInterfaceProfile of
  # PortProfile that is obtained through get_port_profile().
  # In order to provide and require interfaces, proper functions (for
  # example publishInterface(), subscribeInterface() and so on) should be
  # overridden in subclasses, and these functions provide concrete interface
  # connection and disconnection functionality.
  #
  # The interface (instance) name have to be unique in the Port.
  # If the given interface name is identical with stored interface name,
  # this function returns false.
  #
  # @param name The instance name of the interface.
  # @param type_name The type name of the interface.
  # @param pol The interface's polarity (RTC::PROVIDED or RTC:REQUIRED)
  #
  # @return false would be returned if the same name is already registered.
  #
  # @endif
  # bool appendInterface(const char* name, const char* type_name,
  #                      PortInterfacePolarity pol);
  def appendInterface(self, instance_name, type_name, pol):
    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.interfaces,
                                            self.find_interface(instance_name, pol))

    if index >= 0:
      return False

    # setup PortInterfaceProfile
    prof = RTC.PortInterfaceProfile(instance_name, type_name, pol)
    OpenRTM_aist.CORBA_SeqUtil.push_back(self._profile.interfaces, prof)

    return True


  ##
  # @if jp
  #
  # @brief PortInterfaceProfile 饤󥿡եϿ
  #
  # Υڥ졼 Port  PortProfile ΡPortInterfaceProfile
  # 饤󥿡եξ롣
  #
  # @param self
  # @param name 󥿡եΥ󥹥󥹤̾
  # @param pol 󥿡ե° (RTC::PROVIDED ⤷ RTC:REQUIRED)
  #
  # @return 󥿡ե̡
  #         󥿡եϿƤʤ false ֤
  #
  # @else
  #
  # @brief Delete an interface from the PortInterfaceProfile
  #
  # This operation deletes interface information from the
  # PortInterfaceProfile that is owned by the Port.
  #
  # @param name The instance name of the interface.
  # @param pol The interface's polarity (RTC::PROVIDED or RTC:REQUIRED)
  #
  # @return false would be returned if the given name is not registered.
  #
  # @endif
  # bool deleteInterface(const char* name, PortInterfacePolarity pol);
  def deleteInterface(self, name, pol):
    index = OpenRTM_aist.CORBA_SeqUtil.find(self._profile.interfaces,
                                            self.find_interface(name, pol))

    if index < 0:
      return False

    OpenRTM_aist.CORBA_SeqUtil.erase(self._profile.interfaces, index)
    return True


  ##
  # @if jp
  #
  # @brief PortProfile  properties  NameValue ͤɲä
  #
  # PortProfile  properties  NameValue ͤɲä롣
  # ɲäǡηValueTypeǻꤹ롣
  #
  # @param self
  # @param key properties  name
  # @param value properties  value
  #
  # @else
  #
  # @brief Add NameValue data to PortProfile's properties
  #
  # @param key The name of properties
  # @param value The value of properties
  #
  # @endif
  #  template <class ValueType>
  #  void addProperty(const char* key, ValueType value)
  def addProperty(self, key, value):
    OpenRTM_aist.CORBA_SeqUtil.push_back(self._profile.properties,
                                         OpenRTM_aist.NVUtil.newNV(key, value))

  ##
  # @if jp
  #
  # @brief PortProfile  properties  NameValue ͤǤɲä
  #
  # PortProfile  properties  NameValue ͤǤɲä롣
  #
  # @param key properties  name
  # @param value properties  value
  #
  # @else
  #
  # @brief Append NameValue data to PortProfile's properties
  #
  # Append NameValue data to PortProfile's properties.
  #
  # @param key The name of properties
  # @param value The value of properties
  #
  # @endif
  # void appendProperty(const char* key, const char* value)
  def appendProperty(self, key, value):
    OpenRTM_aist.NVUtil.appendStringValue(self._profile.properties, key, value)



  ##
  # @if jp
  #
  # @brief ¸ߤʤݡȤdisconnect롣
  #
  # @else
  #
  # @brief Disconnect ports that doesn't exist. 
  #
  # @endif
  # void updateConnectors()
  def updateConnectors(self):
    guard = OpenRTM_aist.ScopedLock(self._profile_mutex)
    
    connector_ids = []
    clist = self._profile.connector_profiles

    for cprof in clist:
      if not self.checkPorts(cprof.ports):
        connector_ids.append(cprof.connector_id)
        self._rtcout.RTC_WARN("Dead connection: %s", cprof.connector_id)

    for cid in connector_ids:
      self.disconnect(cid)

    return


  ##
  # @if jp
  #
  # @brief ݡȤ¸ߤǧ롣
  #
  # @param ports ǧݡ
  # @return true:¸ߤ,false:¸ߤʤ
  #
  # @else
  #
  # @brief Existence of ports
  #
  # @param ports Checked ports
  # @return true:existent,false:non existent
  #
  # @endif
  # bool checkPorts(::RTC::PortServiceList& ports)
  def checkPorts(self, ports):
    for port in ports:
      try:
        if port._non_existent():
          self._rtcout.RTC_WARN("Dead Port reference detected.")
          return False
      except:
        self._rtcout.RTC_WARN(OpenRTM_aist.Logger.print_exception())
        return False

    return True


  #inline void onNotifyConnect(const char* portname,
  #                            RTC::ConnectorProfile& profile)
  def onNotifyConnect(self, portname, profile):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectListenerType.ON_NOTIFY_CONNECT
      self._portconnListeners.portconnect_[type].notify(portname, profile)
    return


  #inline void onNotifyDisconnect(const char* portname,
  #                               RTC::ConnectorProfile& profile)
  def onNotifyDisconnect(self, portname, profile):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectListenerType.ON_NOTIFY_DISCONNECT
      self._portconnListeners.portconnect_[type].notify(portname, profile)
    return


  #inline void onUnsubscribeInterfaces(const char* portname,
  #                                    RTC::ConnectorProfile& profile)
  def onUnsubscribeInterfaces(self, portname, profile):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectListenerType.ON_UNSUBSCRIBE_INTERFACES
      self._portconnListeners.portconnect_[type].notify(portname, profile)
    return


  #inline void onPublishInterfaces(const char* portname,
  #                                RTC::ConnectorProfile& profile,
  #                                ReturnCode_t ret)
  def onPublishInterfaces(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_PUBLISH_INTERFACES
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return


  #inline void onConnectNextport(const char* portname,
  #                              RTC::ConnectorProfile& profile,
  #                              ReturnCode_t ret)
  def onConnectNextport(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_CONNECT_NEXTPORT
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return


  #inline void onSubscribeInterfaces(const char* portname,
  #                                  RTC::ConnectorProfile& profile,
  #                                  ReturnCode_t ret)
  def onSubscribeInterfaces(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_SUBSCRIBE_INTERFACES
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return


  #inline void onConnected(const char* portname,
  #                        RTC::ConnectorProfile& profile,
  #                        ReturnCode_t ret)
  def onConnected(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_CONNECTED
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return


  #inline void onDisconnectNextport(const char* portname,
  #                                 RTC::ConnectorProfile& profile,
  #                                 ReturnCode_t ret)
  def onDisconnectNextport(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_DISCONNECT_NEXT
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return


  #inline void onDisconnected(const char* portname,
  #                           RTC::ConnectorProfile& profile,
  #                           ReturnCode_t ret)
  def onDisconnected(self, portname, profile, ret):
    if self._portconnListeners != None:
      type = OpenRTM_aist.PortConnectRetListenerType.ON_DISCONNECTED
      self._portconnListeners.portconnret_[type].notify(portname, profile, ret)
    return



  #============================================================
  # Functor
  #============================================================

  ##
  # @if jp
  # @class if_name
  # @brief instance_name  PortInterfaceProfile õ Functor
  # @else
  # @brief A functor to find a PortInterfaceProfile named instance_name
  # @endif
  class if_name:
    def __init__(self, name):
      self._name = name

    def __call__(self, prof):
      return str(self._name) == str(prof.instance_name)
    

  ##
  # @if jp
  # @class find_conn_id
  # @brief id  ConnectorProfile õ Functor
  # @else
  # @brief A functor to find a ConnectorProfile named id
  # @endif
  class find_conn_id:
    def __init__(self, id_):
      """
       \param id_(string)
      """
      self._id = id_

    def __call__(self, cprof):
      """
       \param cprof(RTC.ConnectorProfile)
      """
      return str(self._id) == str(cprof.connector_id)

  ##
  # @if jp
  # @class find_port_ref
  # @brief 󥹥ȥ饯 port_ref Ʊ֥ȻȤõ Functor
  # @else
  # @brief A functor to find the object reference that is identical port_ref
  # @endif
  class find_port_ref:
    def __init__(self, port_ref):
      """
       \param port_ref(RTC.PortService)
      """
      self._port_ref = port_ref

    def __call__(self, port_ref):
      """
       \param port_ref(RTC.PortService)
      """
      return self._port_ref._is_equivalent(port_ref)

  ##
  # @if jp
  # @class connect_func
  # @brief Port ³Ԥ Functor
  # @else
  # @brief A functor to connect Ports
  # @endif
  class connect_func:
    def __init__(self, p, prof):
      """
       \param p(RTC.PortService)
       \param prof(RTC.ConnectorProfile)
      """
      self._port_ref = p
      self._connector_profile = prof
      self.return_code = RTC.RTC_OK

    def __call__(self, p):
      """
       \param p(RTC.PortService)
      """
      if not self._port_ref._is_equivalent(p):
        retval = p.notify_connect(self._connector_profile)
        if retval != RTC.RTC_OK:
          self.return_code = retval

  ##
  # @if jp
  # @class disconnect_func
  # @brief Port ³Ԥ Functor
  # @else
  # @brief A functor to disconnect Ports
  # @endif
  class disconnect_func:
    def __init__(self, p, prof):
      """
       \param p(RTC.PortService)
       \param prof(RTC.ConnectorProfile)
      """
      self._port_ref = p
      self._connector_profile = prof
      self.return_code = RTC.RTC_OK
      
    def __call__(self, p):
      """
       \param p(RTC.PortService)
      """
      if not self._port_ref._is_equivalent(p):
        retval = p.disconnect(self._connector_profile.connector_id)
        if retval != RTC.RTC_OK:
          self.return_code = retval

  ##
  # @if jp
  # @class disconnect_all_func
  # @brief Port ³Ԥ Functor
  # @else
  # @brief A functor to disconnect all Ports
  # @endif
  class disconnect_all_func:
    def __init__(self, p):
      """
       \param p(OpenRTM_aist.PortBase)
      """
      self.return_code = RTC.RTC_OK
      self._port = p

    def __call__(self, p):
      """
       \param p(RTC.ConnectorProfile)
      """
      retval = self._port.disconnect(p.connector_id)
      if retval != RTC.RTC_OK:
        self.return_code = retval

  ##
  # @if jp
  # @class find_interface
  # @brief name  polarity  interface õ Functor
  # @else
  # @brief A functor to find interface from name and polarity
  # @endif
  class find_interface:
    def __init__(self, name, pol):
      """
       \param name(string)
       \param pol(RTC.PortInterfacePolarity)
      """
      self._name = name
      self._pol = pol

    def __call__(self, prof):
      """
       \param prof(RTC.PortInterfaceProfile)
      """
      name = prof.instance_name
      return (str(self._name) == str(name)) and (self._pol == prof.polarity)
