/*
 * Copyright 2017 Fraunhofer Institute for Manufacturing Engineering and Automation (IPA)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0

 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/// @file Swissranger.h
/// Platform independent interface to MESA Swissranger camera. Implementation depends on
/// libusbSR library.
/// @author Jan Fischer
/// @date July 2008

#ifndef __IPA_SWISSRANGER_H__
#define __IPA_SWISSRANGER_H__

#include "StdAfx.h"

// Windows
#ifdef _WIN32
	// Windows with MinGW
	#ifdef __MINGW__
		typedef short __wchar_t;
	#endif
#endif

// Linux
#ifdef __LINUX__
typedef unsigned long DWORD;
#endif

#ifdef __LINUX__
	#include <cob_camera_sensors/AbstractRangeImagingSensor.h>
#else
	#include <cob_driver/cob_camera_sensors/common/include/cob_camera_sensors/AbstractRangeImagingSensor.h>
#endif

#include <stdio.h>
#include <math.h>
#include <sstream>
#include <assert.h>
#include <libMesaSR.h>

namespace ipa_CameraSensors {

// former SR31Consts.h entries
#define SAFE_FREE(p)       { if(p) { delete (p); (p)=0; } }
#define SWISSRANGER_COLUMNS 176
#define SWISSRANGER_ROWS 144

/// Callback function to catch annoying debug message from swissranger camera
/// @param srCam Swissranger camera instance
/// @param msg the received message of type CM_XXX
/// @param param is a message specific parameter
/// @param is a message specific pointer
/// @return
int LibMesaCallback(SRCAM srCam, unsigned int msg, unsigned int param, void* data);

/// @ingroup RangeCameraDriver
/// Interface class to SwissRanger camera SR-3000.
/// Platform independent interface to SwissRanger camera SR-3000. Implementation depends on
/// libusbSR library.
class __DLL_LIBCAMERASENSORS__ Swissranger : public AbstractRangeImagingSensor
{
public:

	Swissranger();
	~Swissranger();

	//*******************************************************************************
	// AbstractRangeImagingSensor interface implementation
	//*******************************************************************************

	unsigned long Init(std::string directory, int cameraIndex = 0);

	unsigned long Open();
	unsigned long Close();

	unsigned long SetProperty(t_cameraProperty* cameraProperty);
	unsigned long SetPropertyDefaults();
	unsigned long GetProperty(t_cameraProperty* cameraProperty);

	unsigned long AcquireImages(int widthStepRange, int widthStepGray, int widthStepCartesian, char* RangeImage=NULL, char* IntensityImage=NULL,
		char* cartesianImage=NULL, bool getLatestFrame=true, bool undistort=true,
		ipa_CameraSensors::t_ToFGrayImageType grayImageType = ipa_CameraSensors::INTENSITY_32F1);
	unsigned long AcquireImages(cv::Mat* rangeImage = 0, cv::Mat* grayImage = 0,
		cv::Mat* cartesianImage = 0, bool getLatestFrame = true, bool undistort = true,
		ipa_CameraSensors::t_ToFGrayImageType grayImageType = ipa_CameraSensors::INTENSITY_32F1);

	unsigned long SaveParameters(const char* filename);

	bool isInitialized() {return m_initialized;}
	bool isOpen() {return m_open;}

private:
	//*******************************************************************************
	// Camera specific members
	//*******************************************************************************

	unsigned long GetCalibratedZMatlab(int u, int v, float zRaw, float& zCalibrated);
	unsigned long GetCalibratedZSwissranger(int u, int v, int width, float& zCalibrated);
	unsigned long GetCalibratedXYMatlab(int u, int v, float z, float& x, float& y);
	unsigned long GetCalibratedXYSwissranger(int u, int v, int width, float& x, float& y);

	/// Load general Swissranger parameters and previously determined calibration parameters.
	/// @param filename Swissranger parameter path and file name.
	/// @param cameraIndex The index of the camera within the configuration file
	///		   i.e. SR_CAM_0 or SR_CAM_1
	/// @return Return value
	unsigned long LoadParameters(const char* filename, int cameraIndex);

	/// Parses the data extracted by <code>LoadParameters</code> and calls the
	/// corresponding <code>SetProperty</code> functions.
	/// @return Return code
	unsigned long SetParameters();

	SRCAM m_SRCam; 			 ///< Handle to USB SR3000 camera
	int m_NumOfImages;		 ///< Number of images the siwssranger returns (i.e. an intensity and a range image)
	ImgEntry* m_DataBuffer;  ///< Image array

	// Stores for cartesian data, when native swissranger calibration is used
	float m_X[SWISSRANGER_COLUMNS * SWISSRANGER_ROWS];
	float m_Y[SWISSRANGER_COLUMNS * SWISSRANGER_ROWS];
	float m_Z[SWISSRANGER_COLUMNS * SWISSRANGER_ROWS];

	bool m_CoeffsInitialized; ///< True, when m_CoeffsAx have been initialized
	bool m_GrayImageAcquireCalled; ///< Is false, when acquiring gray image has not been called, yet

	/// Given a 32 bit swissranger depth value, the real depth value in meteres is given by:
	/// z(u,v)=a0(u,v)+a1(u,v)*d(u,v)+a2(u,v)*d(u,v)^2
	///       +a3(u,v)*d(u,v)^3+a4(u,v)*d(u,v)^4+a5(u,v)*d(u,v)^5
	///       +a6(u,v)*d(u,v)^6;
	cv::Mat m_CoeffsA0; ///< a0 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA1; ///< a1 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA2; ///< a2 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA3; ///< a3 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA4; ///< a4 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA5; ///< a5 z-calibration parameters. One matrix entry corresponds to one pixel
	cv::Mat m_CoeffsA6; ///< a6 z-calibration parameters. One matrix entry corresponds to one pixel
};

/// Creates, intializes and returns a smart pointer object for the camera.
/// @return Smart pointer, refering to the generated object
__DLL_LIBCAMERASENSORS__ AbstractRangeImagingSensorPtr CreateRangeImagingSensor_Swissranger();

} // End namespace ipa_CameraSensors
#endif // __IPA_SWISSRANGER_H__


