// this is for emacs file handling -*- mode: c++; indent-tabs-mode: nil -*-

// -- BEGIN LICENSE BLOCK ----------------------------------------------
// This file is part of the GPU Voxels Software Library.
//
// This program is free software licensed under the CDDL
// (COMMON DEVELOPMENT AND DISTRIBUTION LICENSE Version 1.0).
// You can find a copy of this license in LICENSE.txt in the top
// directory of the source code.
//
// © Copyright 2014 FZI Forschungszentrum Informatik, Karlsruhe, Germany
//
// -- END LICENSE BLOCK ------------------------------------------------

//----------------------------------------------------------------------
/*!\file
 *
 * \author  Herbert Pietrzyk
 * \date    2018-08-02
 *
 *  This example uses helper functions to fill voxellists with Boxes, cylinders and spheres.
 */
//----------------------------------------------------------------------
#include <cstdlib>
#include <signal.h>

#include <gpu_voxels/GpuVoxels.h>
#include <gpu_voxels/helpers/MetaPointCloud.h>
#include <gpu_voxels/logging/logging_gpu_voxels.h>
#include <gpu_voxels/helpers/GeometryGeneration.h>

using namespace gpu_voxels;
GpuVoxelsSharedPtr gvl;

void ctrlchandler(int)
{
  gvl.reset();
  exit(EXIT_SUCCESS);
}
void killhandler(int)
{
  gvl.reset();
  exit(EXIT_SUCCESS);
}

int main(int argc, char* argv[])
{
  signal(SIGINT, ctrlchandler);
  signal(SIGTERM, killhandler);

  icl_core::logging::initialize(argc, argv);

  /*
   * First, we generate an API class, which defines the
   * volume of our space and the resolution.
   * Be careful here! The size is limited by the memory
   * of your GPU. Even if an empty Octree is small, a
   * Voxelmap will always require the full memory.
   */

  gvl = GpuVoxels::getInstance();
  gvl->initialize(200, 200, 200, 0.01);// ==> 200 Voxels, each one is 10 mm in size so the map represents 2x2x2 meter
  //gvl->initialize(64, 64, 64, 0.2);// ==> 200 Voxels, each one is 10 mm in size so the map represents 2x2x2 meter
  float side_length;
  gvl->getVoxelSideLength(side_length);
  float delta = side_length / 1.0f;
 
  gvl->addMap(MT_BITVECTOR_VOXELLIST, "myVoxelList");

  //the upper one is always the on generated by coordinates
  //box, by coordinates
  Vector3f center_min = Vector3f(0.9,0.9,0.9);
  Vector3f center_max = Vector3f(1.1,1.1,1.1);
  std::vector<Vector3ui> box_coordinates = geometry_generation::createBoxOfPoints(center_min, center_max, delta, side_length);
  gvl->getMap("myVoxelList")->insertCoordinateList(box_coordinates, eBVM_OCCUPIED);

  //box, by points
  center_min = Vector3f(0.9,0.9,0.5);
  center_max = Vector3f(1.1,1.1,0.7);
  std::vector<Vector3f> box_points = geometry_generation::createBoxOfPoints(center_min, center_max, delta);
  gvl->getMap("myVoxelList")->insertPointCloud(box_points, eBVM_OCCUPIED);

  //sphere, by coordinates
  Vector3f sphere_center = Vector3f(0.9,0.5,0.9);
  std::vector<Vector3ui> sphere_coordinates = geometry_generation::createSphereOfPoints(sphere_center, 0.1f, delta, side_length);
  gvl->getMap("myVoxelList")->insertCoordinateList(sphere_coordinates, eBVM_OCCUPIED);

  //sphere, by points
  sphere_center = Vector3f(0.9,0.5,0.5);
  std::vector<Vector3f> sphere_points = geometry_generation::createSphereOfPoints(sphere_center, 0.1f, delta);
  gvl->getMap("myVoxelList")->insertPointCloud(sphere_points, eBVM_OCCUPIED);

  //cylinder, by coordinates
  Vector3f cylinder_center = Vector3f(0.9,1.5,0.9);
  std::vector<Vector3ui> cylinder_coordinates = geometry_generation::createCylinderOfPoints(cylinder_center, 0.1f, 0.2f, delta, side_length);
  gvl->getMap("myVoxelList")->insertCoordinateList(cylinder_coordinates, eBVM_OCCUPIED);

  //cylinder, by points
  cylinder_center = Vector3f(0.9,1.5,0.5);
  std::vector<Vector3f> cylinder_points = geometry_generation::createCylinderOfPoints(cylinder_center, 0.1f, 0.2f, delta);
  gvl->getMap("myVoxelList")->insertPointCloud(cylinder_points, eBVM_OCCUPIED);

  gvl->visualizeMap("myVoxelList");

  std::cout << "Finished. busy waiting from here on." << std::endl;
  while(true)
  {}
}
