(defun calc-mass-force-offset-from-forces
  (sensor-rot-list sensor-forces-list)
  (let* ((g-vec (scale -1e-3 *g-vec*)) ;; [kg] -> [g]
         (N (length sensor-rot-list))
         (R-mat (make-matrix (* 3 N) 4))
         (fv (instantiate float-vector (* 3 N))))
    (dotimes (i N)
      (let ((tr (transform (transpose (elt sensor-rot-list i)) g-vec)))
        (dotimes (j 3)
          (setf (aref R-mat (+ j (* i 3)) j) 1.0)
          (setf (aref R-mat (+ j (* i 3)) 3) (elt tr j))
          (setf (elt fv (+ j (* i 3))) (elt (elt sensor-forces-list i) j)))
        ))
    (let ((ret (transform (pseudo-inverse R-mat) fv)))
      (warn ";; avg force error ~A[N]~%" (/ (norm (v- fv (transform R-mat ret))) N))
      ret)))

(defun calc-com-moment-offset-from-moments
  (sensor-rot-list sensor-moments-list estimated-mass)
  (let* ((g-vec (scale -1e-3 *g-vec*)) ;; [kg] -> [g]
         (N (length sensor-rot-list))
         (R-mat (make-matrix (* 3 N) 6))
         (mv (instantiate float-vector (* 3 N))))
    (dotimes (i N)
      (let ((tr (outer-product-matrix (scale (- estimated-mass) (transform (transpose (elt sensor-rot-list i)) g-vec)))))
        (dotimes (j 3)
          (setf (aref R-mat (+ j (* i 3)) j) 1.0)
          (dotimes (k 3)
            (setf (aref R-mat (+ j (* i 3)) (+ k 3)) (aref tr j k)))
          (setf (elt mv (+ j (* i 3))) (elt (elt sensor-moments-list i) j)))
        ))
    (let ((ret (transform (pseudo-inverse R-mat) mv)))
      (warn ";; avg moment error ~A[Nm]~%" (/ (norm (v- mv (transform R-mat ret))) N))
      ret)))

(defun forceCalib-for-limbs
  (robot
   &key (base-time 1000)
        (limbs '(:rarm :larm))
        (poses (make-default-ForceCalibPoses robot))
        (fname))
  (let ((pc (send robot :copy-worldcoords))
        (pav (send robot :angle-vector))
        rot-list force-list moment-list)
    (dolist (limb limbs) (send *ri* :zero-set-forcemoment-offset-param limb))
    (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
    (send robot :angle-vector (car poses))
    (send *ri* :angle-vector (send robot :angle-vector) (* base-time 5))
    (send *ri* :wait-interpolation)
    (dolist (apose poses)
      (send robot :angle-vector apose)
      (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
      (send *ri* :angle-vector (send robot :angle-vector) (* base-time 2.5))
      (send *ri* :wait-interpolation)
      (unix:usleep base-time)
      ;;
      (send *ri* :state)
      (push (mapcar #'(lambda (x) (send *ri* :force-vector x)) limbs) force-list)
      (push (mapcar #'(lambda (x) (send *ri* :moment-vector x)) limbs) moment-list)
      (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
      (push
       (mapcar #'(lambda (a)
                   (copy-object (send (car (send (send *ri* :robot) a :force-sensors)) :worldrot)))
               limbs)
       rot-list)
      )
    (send robot :angle-vector pav)
    (send robot :newcoords pc)
    (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
    (let* ((estimated-f-values
            (mapcar #'(lambda (a)
                        (let ((idx (position a limbs)))
                          (calc-mass-force-offset-from-forces
                           (mapcar #'(lambda (x) (elt x idx)) rot-list)
                           (mapcar #'(lambda (x) (elt x idx)) force-list))))
                    limbs))
           (estimated-m-values
            (mapcar #'(lambda (a em)
                        (let ((idx (position a limbs)))
                          (calc-com-moment-offset-from-moments
                           (mapcar #'(lambda (x) (elt x idx)) rot-list)
                           (mapcar #'(lambda (x) (elt x idx)) moment-list)
                           (elt em 3))))
                    limbs estimated-f-values))
           (ret (mapcar #'(lambda (ll ff mm)
                            (list ll
                                  :force-offset (subseq ff 0 3)
                                  :moment-offset (subseq mm 0 3)
                                  :link-offset-mass (elt ff 3)
                                  :link-offset-centroid (subseq mm 3 6)))
                        limbs
                        estimated-f-values
                        estimated-m-values)))
      (format t ";; offset result is~%")
      (mapcar #'(lambda (l)
                  (format t ";;   ~A ~%" l))
              ret)
      (when fname
        (mapcar #'(lambda (l)
                    (send* *ri* :set-forcemoment-offset-param
                           (car l)
                           (cdr l)))
                ret)
        (format t ";; result file => ~A_[robot name]_[date string] in robot's pc.~%" fname)
        (send *ri* :dump-forcemoment-offset-params fname))
      ret)))

;; example
(defun make-default-ForceCalibPoses
  (robot &key (pose-method :reset-manip-pose))
  (let ((pav (send robot :angle-vector))
        (pc (send robot :copy-worldcoords))
        poses)
    (send robot pose-method)
    (push (send robot :angle-vector) poses)
    (let ((ps (list -50 0 50))
          (ys (list -50 0 50)))
      (dolist (lp ps)
        (send robot :arms :wrist-p :joint-angle lp)
        (dolist (ly ys)
          (send robot :arms :wrist-y :joint-angle ly)
          (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
          (unless nil;;(send robot :self-collision-check)
            (push (send robot :angle-vector) poses))
          )
        ))
    (send robot :angle-vector pav)
    (send robot :newcoords pc)
    (reverse poses)))

(defun make-default-ForceCalibPosesForLegs
  (robot &key (pose-method :reset-manip-pose))
  (let ((pav (send robot :angle-vector))
        (pc (send robot :copy-worldcoords))
        poses)
    (send robot :reset-manip-pose)
    (let ((ps (list -30 0 30))
          (rs (list -18 0 18)))
      (dolist (lp ps)
        (send robot :legs :ankle-p :joint-angle lp)
        (dolist (lr rs)
          (send robot :lleg :ankle-r :joint-angle lr)
          (send robot :rleg :ankle-r :joint-angle (- lr))
          (if *viewer* (send (send *viewer* :get :pickviewer) :draw-objects))
          (unless nil;;(send robot :self-collision-check)
            (push (send robot :angle-vector) poses))
          )
        ))
    poses))
