/*******************************************************************************
 * Copyright (c) 2022 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "displaywidget.h"
#include "nvcom.h"

#include <QMouseEvent>

constexpr int DisplayWidget::IMAGE_PADDING;

DisplayWidget::DisplayWidget(QWidget * parent): QWidget(parent),
        resizeWidget(false), zoomPercent(100),
        mouseX(-1), mouseY(-1), mouseOnImageIndex(-1), numDisplayFrames(0),
        labelFont("Courier New", 12, QFont::Bold) {

    labelFont.setStyleHint(QFont::Monospace);
    labelFont.setWeight(2);
    setMouseTracking(true);

    setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    setMinimumSize(320,240);
}

void DisplayWidget::setNVCom(std::shared_ptr<NVCom> nvcom) {
    this->nvcom = nvcom;
}

void DisplayWidget::setDisplayFrame(const std::vector<cv::Mat_<cv::Vec3b>>& frames, int numFrames, bool resize) {
    for (int i=0; i<numFrames; ++i) {
        displayFrames[i] = frames[i];
    }
    numDisplayFrames = numFrames;
    if(resize) {
        resizeWidget = true;
    }
}

void DisplayWidget::setZoom(int percent) {
    zoomPercent = percent;
    resizeWidget = true;
}

void DisplayWidget::paintEvent(QPaintEvent *) {
    if(nvcom == nullptr) {
        // Should never happen
        return;
    }

    std::unique_lock<std::mutex> lock(nvcom->getDisplayMutex());
    if(numDisplayFrames==0 || displayFrames[0].data==nullptr) {
        return; // No frame available yet
    }

    const double scale = zoomPercent/100.0;

    // Resize widget if necessary
    if(resizeWidget) {
        int width = 0;
        for(int i=0; i<numDisplayFrames; i++) {
            if(i> 0) {
                width += IMAGE_PADDING;
            }
            width += displayFrames[i].cols*scale;
        }

        setFixedSize(width, displayFrames[0].rows*scale);
        resizeWidget = false;
    }

    painter.begin(this);
    painter.scale(scale, scale);

    // Display the received images
    int xpos = 0;
    for (int i=0; i<numDisplayFrames; ++i) {
        QImage img(displayFrames[i].data, displayFrames[i].cols, displayFrames[i].rows,
            displayFrames[i].step[0], QImage::Format_RGB888);
        painter.drawImage(xpos, 0, img);
        xpos += displayFrames[i].cols + IMAGE_PADDING/scale;
    }
    drawMouseLabel();

    painter.end();
}

void DisplayWidget::drawMouseLabel() {
    if(mouseX < 0 || mouseY < 0 || mouseOnImageIndex < 0) {
        return; // No position to draw
    }

    cv::Point3f point = nvcom->getDisparityMapPoint(mouseX, mouseY);
    if(point == cv::Point3f(0, 0, 0)) {
        // No point available
        return;
    }

    char labelStr[100];
    snprintf(labelStr, sizeof(labelStr), "x = %7.3lf m\ny = %7.3lf m\nz = %7.3lf m",
        point.x, point.y, point.z);

    const double scale = zoomPercent/100.0;

    int drawX = mouseX + mouseOnImageIndex * (displayFrames[0].cols + IMAGE_PADDING/scale);
    int drawY = mouseY;

    painter.setPen(QPen(QColor::fromRgb(255, 0, 0), 2));
    painter.drawEllipse(QPointF(drawX, mouseY), 6, 6);

    int offsetX = 25, offsetY = 0;
    painter.setPen(Qt::black);
    painter.setFont(labelFont);
    QRectF boundingBox;
    painter.drawText(QRectF(drawX + offsetX, drawY + offsetY, 1000, 1000), Qt::AlignLeft | Qt::TextDontPrint,
        labelStr, &boundingBox);

    if((mouseOnImageIndex==0 && boundingBox.x() + boundingBox.width() > displayFrames[0].cols) ||
            boundingBox.x() + boundingBox.width() > width()/scale) {
        // Display label to the left if there isn't enough space on the right
        boundingBox.translate(-boundingBox.width() - 2*offsetX, 0);
        drawX -= boundingBox.width();
        offsetX = -offsetX;
    }

    if(boundingBox.y() + boundingBox.height() > height()/scale) {
        // Display label on top if there isn't enough space below
        boundingBox.translate(0, -boundingBox.height() - 2*offsetY);
        drawY -= boundingBox.height();
        offsetY = -offsetY;
    }

    constexpr int border = 10;
    painter.fillRect(boundingBox.x() - border, boundingBox.y() - border,
        boundingBox.width() + 2*border, boundingBox.height() + 2*border, QColor::fromRgbF(1.0, 1.0, 1.0, 0.5));
    painter.drawText(QRectF(drawX + offsetX, drawY + offsetY, 1000, 1000), Qt::AlignLeft,
        labelStr);
}


void DisplayWidget::mouseMoveEvent(QMouseEvent *event) {
    if(event != nullptr) {
        double scale = zoomPercent/100.0;
        int x =      ((int) event->x() % (int) (displayFrames[0].cols*scale + IMAGE_PADDING)) / scale;
        int rawIdx = (int) event->x() / (int) (displayFrames[0].cols*scale + IMAGE_PADDING);
        if (rawIdx >=0 && rawIdx < numDisplayFrames) {
            mouseOnImageIndex = rawIdx;
        } else {
            mouseOnImageIndex = -1;
        }

        if (mouseOnImageIndex==2) {
            // Explicitly disable depth tooltip on right image in three-image mode
            mouseOnImageIndex = -1;
        }

        int y = event->y() / scale;

        if(rawIdx >=0 && rawIdx < numDisplayFrames) {
            mouseX = x;
            mouseY = y;
            repaint();
        }
    }
}
