/*******************************************************************************
 * Copyright (c) 2020 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_SENSORDATA_H
#define VISIONTRANSFER_SENSORDATA_H

namespace visiontransfer {

/**
 * \brief Base class for sensor records with timestamp and status (reliability) fields
 */
class SensorRecord {
protected:
    int timestampSec;
    int timestampUSec;
    unsigned char status;
public:
    SensorRecord(int timestampSec, int timestampUSec, unsigned char status): timestampSec(timestampSec), timestampUSec(timestampUSec), status(status) {}
    void getTimestamp(int& s, int& us) const { s = timestampSec; us = timestampUSec; }
    unsigned char getStatus() const { return status; }
};

/**
 * \brief Encapsulate a scalar sensor measurement, containing the value, as well as timestamp and status fields
 */
class TimestampedScalar: public SensorRecord {
public:
    double value() const { return valueIntl; }
    TimestampedScalar(int timestampSec, int timestampUSec, unsigned char status, double value): SensorRecord(timestampSec, timestampUSec, status), valueIntl(value) {}
    TimestampedScalar(): SensorRecord(0, 0, 0), valueIntl(0) { }
private:
    double valueIntl;
};


/**
 * \brief Encapsulate a 3D sensor report, containing X, Y, Z, as well as timestamp and status fields
 */
class TimestampedVector: public SensorRecord {
public:
    double x() const { return xIntl; }
    double y() const { return yIntl; }
    double z() const { return zIntl; }
    TimestampedVector(int timestampSec, int timestampUSec, unsigned char status, double x, double y, double z): SensorRecord(timestampSec, timestampUSec, status), xIntl(x), yIntl(y), zIntl(z) {}
    TimestampedVector(): SensorRecord(0, 0, 0), xIntl(0), yIntl(0), zIntl(0) { }
private:
    double xIntl, yIntl, zIntl;
};

/**
 * \brief Encapsulate a 4D (quaternion) sensor report, containing i, j, k, r, as well as timestamp and status fields and measurement accuracy
 *
 * Component r is the real part of the quaternion, also called w (ijkr corresponds to xyzw).
 */
class TimestampedQuaternion: public SensorRecord {
public:
    double i() const { return iIntl; }
    double j() const { return jIntl; }
    double k() const { return kIntl; }
    double r() const { return rIntl; }
    double accuracy() const { return accuracyIntl; }
    TimestampedQuaternion(int timestampSec, int timestampUSec, unsigned char status, double i, double j, double k, double r, double accuracy): SensorRecord(timestampSec, timestampUSec, status), iIntl(i), jIntl(j), kIntl(k), rIntl(r), accuracyIntl(accuracy) {}
    TimestampedQuaternion(): SensorRecord(0, 0, 0), iIntl(0), jIntl(0), kIntl(0), rIntl(0), accuracyIntl(0) { }
private:
    double iIntl, jIntl, kIntl, rIntl;
    double accuracyIntl;
};

} // namespace

#endif

