/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "imagequeue.h"
#include <functional>
#include <exception>

using namespace cv;
using namespace std;

template <class T>
ImageQueue<T>::ImageQueue(bool allowFrameSkip, int queueSize, bool multiThreaded)
    :allowFrameSkip(allowFrameSkip), queueSize(queueSize), queue(queueSize),
        readIndex(0), writeIndex(0), done(false), multiThreaded(multiThreaded),
        threadCreated(false) {
}

template <class T>
ImageQueue<T>::~ImageQueue() {
    stopThreads();
}

template <class T>
void ImageQueue<T>::stopThreads() {
    done = true;

    // Notify queue thread if blocked
    pushCond.notify_all();
    popCond.notify_all();

    if(threadCreated) {
        imageThread.join();
        threadCreated = false;
    }
}

template <class T>
std::shared_ptr<const T> ImageQueue<T>::pop(double timeout) {
    if(!multiThreaded) {
        // Synchroneously queue frame
        while(!done && queue[readIndex] == nullptr)
            queueFrame();
    // Start the thread if not yet running
    } else if(!done && !threadCreated) {
        imageThread = thread(bind(&ImageQueue::threadMain, this));
        threadCreated = true;
    }

    {
        unique_lock<mutex> lock(queueMutex);

        while(queue[readIndex] == nullptr) {
            if(done) {
                // No more images available. Return an empty object
                return std::shared_ptr<const T>();
            }
            else {
                // Queue underrun! Wait for more images
                if(timeout < 0) {
                    popCond.wait(lock);
                } else {
                    popCond.wait_for(lock, std::chrono::microseconds(static_cast<unsigned int>(timeout*1e6)));
                }
                if(timeout >= 0 && queue[readIndex] == nullptr) {
                    return std::shared_ptr<const T>();
                }
            }
        }

        // Get image from queue
        std::shared_ptr<const T> ret = queue[readIndex];
        // Clear queue slot
        queue[readIndex].reset();
        // Notify queue thread if blocked
        pushCond.notify_all();
        // Advance index
        readIndex = (readIndex + 1)%queueSize;

        return ret;
    }
}

template <class T>
void ImageQueue<T>::threadMain() {
    try {
        while(!done)
            queueFrame();
    } catch(const std::exception &e) {
        cerr << "Exception in image queueing thread: " << e.what() << endl;
    }
}

template <class T>
void ImageQueue<T>::waitForFreeSlot() {
    unique_lock<mutex> lock(queueMutex);
    if(queue[writeIndex] != nullptr)
        pushCond.wait(lock);
}

template <class T>
void ImageQueue<T>::push(const std::shared_ptr<const T>& frame) {
    unique_lock<mutex> lock(queueMutex);
    if(done) {
        return; // everything has already finished
    }

    if(queue[writeIndex] != nullptr) {
        // Queue is full
        if(!allowFrameSkip) {
            // We have to wait
            pushCond.wait(lock);
            if(done) {
                return;
            }
        } else {
            // We can discard a frame
            queue[writeIndex].reset();
        }
    }

    queue[writeIndex] = frame;

    // Advance Index
    writeIndex = (writeIndex + 1)%queueSize;
    // Notify waiting pop()
    popCond.notify_one();
}

template <class T>
void ImageQueue<T>::closeQueue() {
    done = true;
    // Notify waiting pop()
    popCond.notify_all();
}

// Explicitly instantiate templates
template class ImageQueue<MonoFrame<unsigned char>::Type>;
template class ImageQueue<StereoFrame<unsigned char>::Type>;
template class ImageQueue<DoubleStereoFrame<unsigned char>::Type>;
template class ImageQueue<MonoFrame<unsigned short>::Type>;
template class ImageQueue<StereoFrame<unsigned short>::Type>;
template class ImageQueue<DoubleStereoFrame<unsigned short>::Type>;
template class ImageQueue<MonoFrame<float>::Type>;
template class ImageQueue<StereoFrame<float>::Type>;
template class ImageQueue<DoubleStereoFrame<float>::Type>;
