/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "nvcom.h"
#include "displaywidget.h"
#include "connectiondialog.h"
#include "pointclouddialog.h"

#include <unistd.h>
#include <numeric>
#include <sstream>

#include <QFileDialog>
#include <QInputDialog>
#include <QLabel>
#include <QMessageBox>
#include <QFontDatabase>
#include <QComboBox>

#ifdef _WIN32
#include <windows.h>
#endif

using namespace std::chrono;
using namespace std;
using namespace cv;
using namespace visiontransfer;

MainWindow::MainWindow(QWidget *parent, QApplication& app): QMainWindow(parent),
    writeDirSelected(false), fpsLabel(nullptr), sizeLabel(nullptr), droppedLabel(nullptr),
    appSettings("nerian.com", "nvcom"), fpsTimer(this), colorCombo(nullptr), closeAfterSend(false),
    resizeWindow(false), lastDropped(0) {

    QObject::connect(&app, &QApplication::lastWindowClosed, this,
        &MainWindow::writeApplicationSettings);

    fpsTimer.setInterval(200);
    for(int i=0; i<6; i++) {
        fpsCounters.push_back(std::make_pair(0, steady_clock::now()));
    }
}

MainWindow::~MainWindow() {
    if(nvcom != nullptr) {
        nvcom->terminate();
    }
}

void MainWindow::writeApplicationSettings() {
    // Write settings before exit
    if(!fullscreen) {
        appSettings.setValue("geometry", saveGeometry());
        appSettings.setValue("state", saveState(UI_VERSION));
    }

    appSettings.setValue("write_raw", settings.writeRaw16Bit);
    appSettings.setValue("display_coordinate", settings.displayCoordinate);
    appSettings.setValue("write_point_cloud", settings.writePointCloud);
    appSettings.setValue("point_cloud_max_dist", settings.pointCloudMaxDist);
    appSettings.setValue("color_scheme", settings.colorScheme);
    appSettings.setValue("max_frame_rate", settings.maxFrameRate);
    appSettings.setValue("read_dir", settings.readDir.c_str());
    appSettings.setValue("write_dir", settings.writeDir.c_str());
    appSettings.setValue("zoom", settings.zoomPercent);
    appSettings.setValue("binary_point_cloud", settings.binaryPointCloud);
}

bool MainWindow::init(int argc, char** argv) {
    if(!parseOptions(argc, argv)) {
        return false;
    }

#ifdef _WIN32
    if(!settings.nonGraphical) {
        FreeConsole();
    }
#endif

    if(!settings.nonGraphical) {
        ui.reset(new Ui::MainWindow);
        ui->setupUi(this);
    }

    QObject::connect(this, &MainWindow::asyncDisplayException, this, [this](const QString& msg){
        displayException(msg.toStdString());
        nvcom->terminate();
        emit enableButtons(true, false);
    });

    QObject::connect(this, &MainWindow::updateStatusBar, this, [this](int dropped, int imgWidth, int imgHeight,
            int bits0, int bits1, int bits2){
        char str[30];
        std::stringstream ss;
        ss << imgWidth << " x " << imgHeight << " pixels; ";
        if (bits0 > 0) ss << bits0;
        if (bits1 > 0) ss << "/" << bits1;
        if (bits2 > 0) ss << "/" << bits2;
        ss << " bits";
        sizeLabel->setText(ss.str().c_str());
        snprintf(str, sizeof(str), "Dropped frames: %d", dropped);
        droppedLabel->setText(str);
    });

    QObject::connect(this, &MainWindow::repaintDisplayWidget, this, [this](){
        ui->displayWidget->repaint();
    });

    QObject::connect(this, &MainWindow::updateFpsLabel, this, [this](const QString& text){
        fpsLabel->setText(text);
    });

    // Initialize window
    if(!settings.nonGraphical) {
        initGui();
        if(fullscreen) {
            makeFullscreen();
        } else {
            // Restore window position
            restoreGeometry(appSettings.value("geometry").toByteArray());
            restoreState(appSettings.value("state").toByteArray(), UI_VERSION);
        }
        show();
    }

    // FPS display
    fpsTimer.start();
    QObject::connect(&fpsTimer, &QTimer::timeout, this, &MainWindow::displayFrameRate);

    if(settings.remoteHost == "") {
        showConnectionDialog();
    } else {
        reinitNVCom();
    }

    if(writeImages) {
        // Automatically start sequence grabbing if a write directory is provided
        if(nvcom != nullptr) {
            nvcom->setCaptureSequence(true);
        }
        if(!settings.nonGraphical) {
            ui->actionCapture_sequence->setChecked(true);
        }
    }

    return true;
}

void MainWindow::reinitNVCom() {
    emit enableButtons(false, false);

    // Make sure all sockets are closed before creating a new object
    if(nvcom != nullptr) {
        nvcom->terminate();
        nvcom.reset();
    }

    nvcom.reset(new NVCom(settings));
    nvcom->setFrameDisplayCallback([this](int origW, int origH, const std::vector<cv::Mat_<cv::Vec3b>>& images, int numActiveImages, bool resize){
        displayFrame(origW, origH, images, numActiveImages, resize);
    });
    nvcom->setExceptionCallback([this](const std::exception& ex){
        emit asyncDisplayException(ex.what());
    });
    nvcom->setSendCompleteCallback([this]() {
        settings.readImages = false;
        if(closeAfterSend) {
            QApplication::quit();
        } else if(!settings.nonGraphical) {
            ui->actionSend_images->setChecked(false);
        }
    });
    nvcom->setConnectedCallback([this]() {
        if(!settings.nonGraphical) {
            emit enableButtons(true, true);
        }
    });
    nvcom->setDisconnectCallback([this]() {
        emit ui->actionConnect->triggered();
    });

    if(!settings.nonGraphical) {
        ui->displayWidget->setNVCom(nvcom);
    }
    nvcom->connect();
}

void MainWindow::initGui() {
    initToolBar();
    initStatusBar();

    // Menu items
    QObject::connect(ui->actionChoose_capture_directory, &QAction::triggered,
        this, [this]{chooseWriteDirectory(true);});

    // Zooming
    zoomLabel = new QLabel("100%", this);
    zoomLabel->setFont(QFontDatabase::systemFont(QFontDatabase::FixedFont));
    ui->toolBar->insertWidget(ui->actionZoom_out, zoomLabel);
    QObject::connect(ui->actionZoom_in, &QAction::triggered,
        this, [this]{zoom(1);});
    QObject::connect(ui->actionZoom_out, &QAction::triggered,
        this, [this]{zoom(-1);});

    // Button enable / disable
    QObject::connect(this, &MainWindow::enableButtons, this, [this](bool nvcomReady, bool connected){
        if(fullscreen) {
            return; // No toolbar visible
        }

        ui->actionConnect->setEnabled(nvcomReady);
        ui->actionSend_images->setEnabled(connected);
        ui->actionCapture_single_frame->setEnabled(connected);
        ui->actionCapture_sequence->setEnabled(connected);
        ui->actionNo_color_coding->setEnabled(connected);
        ui->actionRed_blue_color_coding->setEnabled(connected);
        ui->actionRainbow_color_coding->setEnabled(connected);
        ui->actionCapture_16_bit->setEnabled(connected);
        ui->actionWrite_3D_point_clouds->setEnabled(connected);
        ui->actionChoose_capture_directory->setEnabled(connected);
        ui->actionZoom_in->setEnabled(connected);
        ui->actionZoom_out->setEnabled(connected);
        ui->actionDisplayCoord->setEnabled(connected);
        zoomLabel->setEnabled(connected);
    });

    // Update GUI status
    colorCodingChange(settings.colorScheme);
    ui->actionCapture_16_bit->setChecked(settings.writeRaw16Bit);
    ui->actionDisplayCoord->setChecked(settings.displayCoordinate);
    ui->actionWrite_3D_point_clouds->setChecked(settings.writePointCloud);
    ui->actionSend_images->setChecked(settings.readImages);
    zoom(0);
}

void MainWindow::makeFullscreen() {
    ui->menuBar->hide();
    delete ui->toolBar;
    ui->toolBar = nullptr;
    setWindowFlags(Qt::Window | Qt::FramelessWindowHint);
    showFullScreen();
}

void MainWindow::initStatusBar() {
    // Setup status bar
    droppedLabel = new QLabel(this);
    sizeLabel = new QLabel(this);
    fpsLabel = new QLabel(this);
    ui->statusBar->addPermanentWidget(droppedLabel);
    ui->statusBar->addPermanentWidget(sizeLabel);
    ui->statusBar->addPermanentWidget(fpsLabel);
    sizeLabel->setText("Waiting for data...");
}

void MainWindow::initToolBar() {
    // Create color coding combobox
    colorCombo = new QComboBox;
    ui->toolBar->insertWidget(ui->actionZoom_in, colorCombo);
    ui->toolBar->insertSeparator(ui->actionZoom_in);
    colorCombo->insertItem(0, "Raw");
    colorCombo->insertItem(1, "Red / Blue");
    colorCombo->insertItem(2, "Rainbow");
    colorCombo->setItemIcon(0, QIcon(":nvcom/icons/color_grey.png"));
    colorCombo->setItemIcon(1, QIcon(":nvcom/icons/color_red_blue.png"));
    colorCombo->setItemIcon(2, QIcon(":nvcom/icons/color_rainbow.png"));

    QObject::connect(colorCombo, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),
        [this](int index){colorCodingChange(static_cast<Settings::ColorScheme>(index));});

    QObject::connect(ui->actionNo_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_NONE);});
    QObject::connect(ui->actionRed_blue_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_RED_BLUE);});
    QObject::connect(ui->actionRainbow_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_RAINBOW);});

    QObject::connect(ui->actionCapture_16_bit, &QAction::triggered, this, [this]{
        if(nvcom != nullptr) {
            settings.writeRaw16Bit = ui->actionCapture_16_bit->isChecked();
            if(nvcom != nullptr) {
                nvcom->updateSettings(settings);
            }
        }
    });
    QObject::connect(ui->actionDisplayCoord, &QAction::triggered, this, [this]{
        if(nvcom != nullptr) {
            settings.displayCoordinate = ui->actionDisplayCoord->isChecked();
            if(nvcom != nullptr) {
                nvcom->updateSettings(settings);
            }
        }
    });
    QObject::connect(ui->actionWrite_3D_point_clouds, &QAction::triggered,
        this, [this]{
            settings.writePointCloud = false;
            if(ui->actionWrite_3D_point_clouds->isChecked()) {
                openPointCloudDialog();
            }
            ui->actionWrite_3D_point_clouds->setChecked(settings.writePointCloud);

            if(nvcom != nullptr) {
                nvcom->updateSettings(settings);
            }
        });

    QObject::connect(ui->actionCapture_single_frame, &QAction::triggered,
        this, [this]{
            if(chooseWriteDirectory(false) && nvcom != nullptr) {
                nvcom->captureSingleFrame();
            }
        });

    QObject::connect(ui->actionCapture_sequence, &QAction::triggered,
        this, [this]{
            if(chooseWriteDirectory(false) && nvcom != nullptr) {
                nvcom->setCaptureSequence(ui->actionCapture_sequence->isChecked());
            } else {
                ui->actionCapture_sequence->setChecked(false);
            }
        });

    QObject::connect(ui->actionQuit, &QAction::triggered,
        this, [this]{close();});

    QObject::connect(ui->actionConnect, &QAction::triggered,
        this, &MainWindow::showConnectionDialog);
    QObject::connect(ui->actionSend_images, &QAction::triggered,
        this, &MainWindow::transmitInputFolder);
}

void MainWindow::openPointCloudDialog() {
    PointCloudDialog diag(this, settings);
    diag.exec();
    settings = diag.getSettings();
}

void MainWindow::displayFrame(int origW, int origH, const std::vector<cv::Mat_<cv::Vec3b>>& frames, int numActiveImages, bool resize) {
    fpsCounters.back().first++;

    if(!settings.nonGraphical) {
        ui->displayWidget->setDisplayFrame(frames, numActiveImages, resize);
        emit repaintDisplayWidget();

        int dropped = nvcom->getNumDroppedFrames();

        if(resize || dropped != lastDropped) {
            lastDropped = dropped;
            int bitsArr[3];
            nvcom->getBitDepths(bitsArr);
            emit updateStatusBar(dropped, origW, origH, bitsArr[0], bitsArr[1], bitsArr[2]);
        }
    }
}

bool MainWindow::parseOptions(int argc, char** argv) {
    bool failed = false;
    writeImages = false;
    fullscreen = false;
    settings.readImages = false;
    settings.readDir = appSettings.value("read_dir", "").toString().toStdString();
    settings.writeDir = appSettings.value("write_dir", "").toString().toStdString();
    settings.maxFrameRate = appSettings.value("max_frame_rate", 30).toDouble();
    settings.nonGraphical = false;
    settings.remoteHost = "";
    settings.remotePort = 7681;
    settings.tcp = false;
    settings.writeRaw16Bit = appSettings.value("write_raw", false).toBool();
    settings.displayCoordinate = appSettings.value("display_coordinate", false).toBool();
    settings.displayImage = -1;
    settings.disableReception = false;
    settings.printTimestamps = false;
    settings.writePointCloud = appSettings.value("write_point_cloud", false).toBool();
    settings.pointCloudMaxDist = appSettings.value("point_cloud_max_dist", 10).toDouble();
    settings.colorScheme = static_cast<Settings::ColorScheme>(appSettings.value("color_scheme", 2).toInt());
    settings.zoomPercent = appSettings.value("zoom", 100).toInt();
    settings.binaryPointCloud = appSettings.value("binary_point_cloud", false).toBool();

    int c;
    while ((c = getopt(argc, argv, "-:c:f:w:s:nt:r:i:h:p:dT3:z:Fb:")) != -1) {
        switch(c) {
            case 'c': settings.colorScheme = static_cast<Settings::ColorScheme>(atoi(optarg)); break;
            case 'f': settings.maxFrameRate = atof(optarg); break;
            case 'w':
                settings.writeDir = optarg;
                writeImages = true;
                writeDirSelected = true;
                break;
            case 's':
                settings.readDir = optarg;
                settings.readImages = true;
                closeAfterSend = true;
                break;
            case 'n': settings.nonGraphical = true; break;
            case 'p': settings.remotePort = atoi(optarg); break;
            case 'h': settings.remoteHost = optarg; break;
            case 't': settings.tcp = (string(optarg) == "on"); break;
            case 'r': settings.writeRaw16Bit = (string(optarg) == "on"); break;
            case 'i': settings.displayImage = atoi(optarg); break;
            case 'd': settings.disableReception = true; break;
            case 'T': settings.printTimestamps = true; break;
            case '3': {
                double dist = atof(optarg);
                if(dist > 0) {
                    settings.pointCloudMaxDist = dist;
                    settings.writePointCloud = true;
                } else {
                    settings.writePointCloud = false;
                }
                break;
            }
            case 'z': settings.zoomPercent = atoi(optarg); break;
            case 'F': fullscreen = true; break;
            case 'b': settings.binaryPointCloud = (string(optarg) == "on"); break;
            case '-':  // --help
            default:
                settings.nonGraphical = true;
                failed = true;
                cout << endl;
        }
    }

    if(argc - optind != 0 || failed) {
        cerr << "Usage: " << endl
             << argv[0] << " [OPTIONS]" << endl << endl
             << "Options: " << endl
             << "--help     Show this information" << endl
             << "-c VAL     Select color coding scheme (0 = no color, 1 = red / blue," << endl
             << "           2 = rainbow)" << endl
             << "-s DIR     Send the images from the given directory" << endl
             << "-f FPS     Limit send frame rate to FPS" << endl
             << "-d         Disable image reception" << endl
             << "-n         Non-graphical" << endl
             << "-w DIR     Immediately write all images to DIR" << endl
             << "-3 VAL     Write a 3D point cloud with distances up to VAL (0 = off)" << endl
             << "-b on/off  Write point clouds in binary rather than text format" << endl
             << "-h HOST    Use the given remote hostname for communication" << endl
             << "-p PORT    Use the given remote port number for communication" << endl
             << "-t on/off  Activate / deactivate TCP transfers" << endl
             << "-r on/off  Activate / deactivate output of raw 16-bit disparity maps" << endl
             << "-i 0/1     Only displays the left (0) or right (1) image" << endl
             << "-T         Print frame timestamps" << endl
             << "-z VAL     Set zoom factor to VAL percent" << endl
             << "-F         Run in fullscreen mode" << endl;
        return false;
    }

    return true;
}

void MainWindow::colorCodingChange(Settings::ColorScheme newScheme) {
    colorCombo->setCurrentIndex(static_cast<int>(newScheme));

    // First uncheck
    switch(newScheme) {
        case Settings::COLOR_SCHEME_NONE:
            ui->actionNo_color_coding->setChecked(true);
            ui->actionRed_blue_color_coding->setChecked(false);
            ui->actionRainbow_color_coding->setChecked(false);
            break;
        case Settings::COLOR_SCHEME_RAINBOW:
            ui->actionRainbow_color_coding->setChecked(true);
            ui->actionNo_color_coding->setChecked(false);
            ui->actionRed_blue_color_coding->setChecked(false);
            break;
        case Settings::COLOR_SCHEME_RED_BLUE:
            ui->actionRed_blue_color_coding->setChecked(true);
            ui->actionNo_color_coding->setChecked(false);
            ui->actionRainbow_color_coding->setChecked(false);
            break;
    }

    settings.colorScheme = newScheme;

    if(nvcom != nullptr) {
        nvcom->updateSettings(settings);
    }
}

bool MainWindow::chooseWriteDirectory(bool forceChoice) {
    if(!forceChoice && writeDirSelected) {
        return true;
    }

    QString newDir = QFileDialog::getExistingDirectory(this,
        "Choose capture directory", settings.writeDir.c_str(),
         QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);

    if(newDir != "") {
#ifdef _WIN32
        settings.writeDir = newDir.toLocal8Bit().constData();
#else
        settings.writeDir = newDir.toUtf8().constData();
#endif
        writeDirSelected = true;
    }

    if(nvcom != nullptr) {
        nvcom->updateSettings(settings);
        nvcom->resetCaptureIndex();
    }

    return newDir != "";
}

void MainWindow::showConnectionDialog() {
    ConnectionDialog diag(this);
    if(diag.exec() == QDialog::Accepted && diag.getSelectedDevice().getIpAddress() != "") {
        settings.remoteHost = diag.getSelectedDevice().getIpAddress();
        settings.tcp = (diag.getSelectedDevice().getNetworkProtocol() == DeviceInfo::PROTOCOL_TCP);
        reinitNVCom();
    }
}

void MainWindow::displayException(const std::string& msg) {
    cerr << "Exception occurred: " << msg << endl;
    QMessageBox msgBox(QMessageBox::Critical, "NVCom Exception!", msg.c_str());
    msgBox.exec();
}

void MainWindow::displayFrameRate() {
    // Compute frame rate
    int framesCount = 0;
    for(auto counter: fpsCounters) {
        framesCount += counter.first;
    }

    int elapsedTime = duration_cast<microseconds>(
        steady_clock::now() - fpsCounters[0].second).count();
    double fps = framesCount / (elapsedTime*1.0e-6);

    // Drop one counter
    fpsCounters.pop_front();
    fpsCounters.push_back(std::make_pair(0, steady_clock::now()));

    // Update label
    char fpsStr[6];
    snprintf(fpsStr, sizeof(fpsStr), "%.2lf", fps);

    if(fpsLabel != nullptr) {
        emit updateFpsLabel(QString(fpsStr) +  " fps");
    }

    // Print console status messages at a lower update rate
    static int printCounter = 0;
    if((printCounter++)>5 && nvcom != nullptr) {
        cout << "Fps: " << fpsStr << "; output queue: " << nvcom->getWriteQueueSize() << endl;
        printCounter = 0;
    }
}

void MainWindow::transmitInputFolder() {
    if(!settings.readImages) {
        QString newReadDir= QFileDialog::getExistingDirectory(this,
            "Choose input directory", settings.readDir.c_str(),
            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
        if(newReadDir != "") {
#ifdef _WIN32
            settings.readDir = newReadDir.toLocal8Bit().constData();
#else
            settings.readDir = newReadDir.toUtf8().constData();
#endif
            settings.readImages = true;
        }
    } else {
        settings.readImages = false;
    }

    if(settings.readImages) {
        bool ok = false;
        settings.maxFrameRate = QInputDialog::getDouble(this, "Choose frame rate",
            "Send frame rate:", settings.maxFrameRate, 0.1, 100, 1, &ok);
        if(!ok) {
            settings.readImages = false;
        }
    }

    ui->actionSend_images->setChecked(settings.readImages);
    if(nvcom != nullptr) {
        nvcom->updateSettings(settings);
    }
}

void MainWindow::zoom(int direction) {
    if(direction != 0) {
        settings.zoomPercent =
            (settings.zoomPercent / 25)*25 + 25*direction;
    }

    if(settings.zoomPercent > 200) {
        settings.zoomPercent = 200;
    } else if(settings.zoomPercent < 25) {
        settings.zoomPercent = 25;
    }

    char labelText[5];
    snprintf(labelText, sizeof(labelText), "%3d%%", settings.zoomPercent);

    zoomLabel->setText(labelText);
    ui->displayWidget->setZoom(settings.zoomPercent);

    ui->actionZoom_in->setEnabled(settings.zoomPercent != 200);
    ui->actionZoom_out->setEnabled(settings.zoomPercent != 25);
}

int main(int argc, char** argv) {
    bool graphical = true;

    try {
        QApplication app(argc, argv);
        MainWindow win(nullptr, app);

        if(!win.init(argc, argv)) {
            return 1;
        }

        graphical = win.isGraphical();

        return app.exec();
    } catch(const std::exception& ex) {
        if(graphical) {
            QApplication app(argc, argv);
            MainWindow::displayException(ex.what());
        }
        return -1;
    }
}
