#!/usr/bin/env python

# Copyright (c) 2019 Nerian Vision GmbH
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

##############
#
# This script is called by CMake in order to generate C++ initialization
#   and update code for the parameter server and dynamic_reconfigure
#   (it reuses the .cfg file the dynamic_reconfigure build operates on).
#

import sys

CODE_TEMPLATE = '''// This file is AUTOGENERATED CODE produced by generate_nerian_config_cpp.py

#include <ros/ros.h>
#include <dynamic_reconfigure/server.h>
#include <nerian_stereo/NerianStereoConfig.h>
#include <visiontransfer/scenescanparameters.h>

#include "nerian_stereo_node_base.h"

using namespace visiontransfer;

namespace nerian_stereo {

// Callback that receives an updated config from ROS
void StereoNodeBase::autogen_dynamicReconfigureCallback(nerian_stereo::NerianStereoConfig &config, uint32_t level) {
    std::stringstream ss;
    // == START of autogenerated parameter blocks ==
%s
    // == END of autogenerated parameter blocks ==
}

// Obtain current parameter values from device and copy them to parameter server
void StereoNodeBase::autogen_updateParameterServerFromDevice(std::map<std::string, ParameterInfo>& cfg) {
    ROS_INFO("Setting initial parameters in the parameter server");
    std::string node_name = ros::this_node::getName();
    // Publish reboot flag to definitely be set to false in the parameter server
    getNH().setParam(node_name + "/reboot", false);
    // Publish the current config to the parameter server
    // == START of autogenerated parameter blocks ==
%s
    // == END of autogenerated parameter blocks ==
}

// Override the default parameter bounds with current (run-time) config
void StereoNodeBase::autogen_updateDynamicReconfigureFromDevice(std::map<std::string, ParameterInfo>& cfg) {
    nerian_stereo::NerianStereoConfig config_default, config_min, config_max;
    ROS_INFO("Updating dynamic_reconfigure defaults and limits");
    // Set defaults and min/max values according to SceneScan API
    // == START of autogenerated parameter blocks ==
%s

    // == END of autogenerated parameter blocks ==
    // Publish them
    dynReconfServer->setConfigMin(config_min);
    dynReconfServer->setConfigMax(config_max);
    dynReconfServer->setConfigDefault(config_default);
}

} // namespace
'''

TEMPLATE_PARAMETER_CHANGE = '''        if (config.{varname} != lastKnownConfig.{varname}) {{
            ROS_INFO("Request to set {varname} = %s", std::to_string(config.{varname}).c_str());
            sceneScanParameters->setNamedParameter("{varname}", config.{varname});
        }}'''

TEMPLATE_SETPARAM = '''    getNH().setParam(node_name + "/{varname}", cfg["{varname}"].getValue<{typ}>());'''

TEMPLATE_SETDEFAULTS = '''
    config_default.{varname} = cfg["{varname}"].getValue<{typ}>();
    config_min.{varname} = cfg["{varname}"].getMin<{typ}>();
    config_max.{varname} = cfg["{varname}"].getMax<{typ}>();'''


if __name__ == '__main__':
    # Default to filter
    infile = sys.stdin
    outfile = sys.stdout
    if len(sys.argv) >= 3:
        infile = open(sys.argv[1], 'r')
        outfile = open(sys.argv[2], 'w')

    # Parse cfg file contents and extract the parameter lines (name + type)
    varnames_and_types = []
    for line in infile.readlines():
        if line.startswith('gen.add('):
            varname = line.split('"')[1]
            typ = line.split(',')[1].strip().split('_')[0]
            varnames_and_types.append([varname, typ])

    # Dump code for each parameter. 'reboot' is handled specially (must not be True initially)
    paramchange = '\n'.join(TEMPLATE_PARAMETER_CHANGE.format(varname=vt[0]) for vt in varnames_and_types)
    setparam = '\n'.join(TEMPLATE_SETPARAM.format(varname=vt[0], typ=vt[1]) for vt in varnames_and_types if vt[0] != 'reboot')
    setdefaults = '\n'.join(TEMPLATE_SETDEFAULTS.format(varname=vt[0], typ=vt[1]) for vt in varnames_and_types if vt[0] != 'reboot')
    outfile.write(CODE_TEMPLATE % (paramchange, setparam, setdefaults))

