# Software License Agreement (BSD License)
#
# Copyright (c) 2012, Fraunhofer FKIE/US, Alexander Tiderko
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#  * Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above
#    copyright notice, this list of conditions and the following
#    disclaimer in the documentation and/or other materials provided
#    with the distribution.
#  * Neither the name of Fraunhofer nor the names of its
#    contributors may be used to endorse or promote products derived
#    from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
# COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

from python_qt_binding.QtCore import QFile, Qt, Signal
from python_qt_binding.QtGui import QIcon, QImage, QStandardItem, QStandardItemModel
import re
import roslib
import rospy
import traceback

from master_discovery_fkie.common import get_hostname, subdomain
from master_discovery_fkie.master_info import NodeInfo
from node_manager_fkie.common import lnamespace, namespace, normns, utf8
from node_manager_fkie.name_resolution import NameResolution
from parameter_handler import ParameterHandler
import node_manager_fkie as nm


class CellItem(QStandardItem):
    '''
    Item for a cell. References to a node item.
    '''
    ITEM_TYPE = Qt.UserRole + 41

    def __init__(self, name, item=None, parent=None):
        '''
        Initialize the CellItem object with given values.
        @param name: the name of the group
        @type name: C{str}
        @param parent: the parent item. In most cases this is the HostItem. The
        variable is used to determine the different columns of the NodeItem.
        @type parent: U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}
        '''
        QStandardItem.__init__(self)
        self.parent_item = parent
        self._name = name
        self.item = item

    @property
    def name(self):
        '''
        The name of this group.
        @rtype: C{str}
        '''
        return self._name


################################################################################
##############                  GrouptItem                        ##############
################################################################################
class GroupItem(QStandardItem):
    '''
    The GroupItem stores the information about a group of nodes.
    '''
    ITEM_TYPE = Qt.UserRole + 25

    def __init__(self, name, parent=None, has_remote_launched_nodes=False, is_group=False):
        '''
        Initialize the GroupItem object with given values.
        @param name: the name of the group
        @type name: C{str}
        @param parent: the parent item. In most cases this is the HostItem. The
        variable is used to determine the different columns of the NodeItem.
        @type parent: U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}
        '''
        dname = name
        if dname.rfind('@') <= 0:
            if is_group:
                dname = '{' + dname + '}'
            else:
                dname = dname + '/'
        QStandardItem.__init__(self, dname)
        self.parent_item = parent
        self._name = name
        self.setIcon(QIcon(':/icons/state_off.png'))
        self.descr_type = self.descr_name = self.descr = ''
        self.descr_images = []
        self._capcabilities = dict()
        self._has_remote_launched_nodes = has_remote_launched_nodes
        self._is_group = is_group
        self._remote_launched_nodes_updated = False
        self._state = NodeItem.STATE_OFF
        self.diagnostic_array = []
        self.is_system_group = name == 'SYSTEM'
        '''
     @ivar: dict(config : dict(namespace: dict(group:dict('type' : str, 'images' : [str], 'description' : str, 'nodes' : [str]))))
    '''
        self._re_cap_nodes = dict()

    @property
    def name(self):
        '''
        The name of this group.
        @rtype: C{str}
        '''
        return self._name

    @name.setter
    def name(self, new_name):
        '''
        Set the new name of this group and updates the displayed name of the item.
        @param new_name: The new name of the group. Used also to identify the group.
        @type new_name: C{str}
        '''
        self._name = new_name
        if self._is_group:
            self.setText('{' + self._name + '}')
        else:
            self.setText(self._name + '/')

    @property
    def state(self):
        '''
        The state of this group.
        @rtype: C{int}
        '''
        return self._state

    @property
    def is_group(self):
        return self._is_group

    @property
    def cfgs(self):
        lc, dc = self.get_configs()
        lc[len(lc):] = dc
        return lc

    def get_namespace(self):
        name = self._name
        if type(self) == HostItem:
            name = rospy.names.SEP
        elif type(self) == GroupItem and self._is_group:
            name = namespace(self._name)
        result = name
        if self.parent_item is not None:
            result = normns(self.parent_item.get_namespace() + rospy.names.SEP) + normns(result + rospy.names.SEP)
        return normns(result)

    def count_nodes(self):
        '''
        Returns count of nodes inside this group.
        '''
        result = 0
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                result += item.count_nodes()
            elif isinstance(item, NodeItem):
                result += 1
        return result

    def is_in_cap_group(self, nodename, config, ns, groupname):
        '''
        Returns `True` if the group contains the node.
        @param nodename: the name of the node to test
        @type nodename: str
        @param config: the configuration name
        @type config: str
        @param ns: namespace
        @type ns: str
        @param groupname: the group name
        @type groupname: str
        @return: `True`, if the nodename is in the group
        @rtype: bool
        '''
        try:
            if self._re_cap_nodes[(config, ns, groupname)].match(nodename):
                return True
        except:
            pass
        return False

    def _create_cap_nodes_pattern(self, config, cap):
        for ns, groups in cap.items():
            for groupname, descr in groups.items():
                try:
                    nodes = descr['nodes']
                    def_list = ['\A' + n.strip().replace('*', '.*') + '\Z' for n in nodes]
                    if def_list:
                        self._re_cap_nodes[(config, ns, groupname)] = re.compile('|'.join(def_list), re.I)
                    else:
                        self._re_cap_nodes[(config, ns, groupname)] = re.compile('\b', re.I)
                except:
                    rospy.logwarn("create_cap_nodes_pattern: %s" % traceback.format_exc(1))

    def addCapabilities(self, config, capabilities, masteruri):
        '''
        Add new capabilities. Based on this capabilities the node are grouped. The
        view will be updated.
        @param config: The name of the configuration containing this new capabilities.
        @type config: C{str}
        @param masteruri: The masteruri is used only used, if new nodes are created.
        @type masteruri: C{str}
        @param capabilities: The capabilities, which defines groups and containing nodes.
        @type capabilities: C{dict(namespace: dict(group:dict('type' : str, 'images' : [str], 'description' : str, 'nodes' : [str])))}
        '''
        self._capcabilities[config] = capabilities
        self._create_cap_nodes_pattern(config, capabilities)
        # update the view
        for ns, groups in capabilities.items():
            for group, descr in groups.items():
                group_changed = False
                # create nodes for each group
                nodes = descr['nodes']
                if nodes:
                    groupItem = self.getGroupItem(roslib.names.ns_join(ns, group))
                    groupItem.descr_name = group
                    if descr['type']:
                        groupItem.descr_type = descr['type']
                    if descr['description']:
                        groupItem.descr = descr['description']
                    if descr['images']:
                        groupItem.descr_images = list(descr['images'])
                    # move the nodes from host to the group
                    group_changed = self.move_nodes2group(groupItem, config, ns, group, self)
                    # create new or update existing items in the group
                    for node_name in nodes:
                        # do not add nodes with * in the name
                        if not re.search(r"\*", node_name):
                            items = groupItem.getNodeItemsByName(node_name)
                            if items:
                                for item in items:
                                    item.addConfig(config)
                                    group_changed = True
                            else:
                                items = self.getNodeItemsByName(node_name)
                                if items:
                                    # copy the state of the existing node
                                    groupItem.addNode(items[0].node_info, config)
                                elif config:
                                    groupItem.addNode(NodeInfo(node_name, masteruri), config)
                                group_changed = True
                    if group_changed:
                        groupItem.updateDisplayedConfig()
                        groupItem.updateIcon()

    def move_nodes2group(self, group_item, config, ns, groupname, host_item):
        '''
        Returns `True` if the group was changed by adding a new node.

        @param GroupItem group_item: item to parse the children for nodes.
        @param str config: the configuration name
        @param str ns: namespace
        @param str groupname: the group name
        @param HostItem host_item: the host item contain the capability groups
        @return: `True`, if the group was changed by adding a new node.
        @rtype: bool
        '''
        self_changed = False
        group_changed = False
        for i in reversed(range(self.rowCount())):
            item = self.child(i)
            if isinstance(item, NodeItem):
                if host_item.is_in_cap_group(item.name, config, ns, groupname):
                    row = self.takeRow(i)
                    group_item._addRow_sorted(row)
                    group_changed = True
            elif isinstance(item, GroupItem) and not item.is_group:
                group_changed = item.move_nodes2group(group_item, config, ns, groupname, host_item)
        if self_changed:
            self.update_displayed_config()
            self.updateIcon()
        return group_changed

    def remCapablities(self, config):
        '''
        Removes internal entry of the capability, so the new nodes are not grouped.
        To update view L{NodeTreeModel.removeConfigNodes()} and L{GroupItem.clearUp()}
        must be called.
        @param config: The name of the configuration containing this new capabilities.
        @type config: C{str}
        '''
        try:
            del self._capcabilities[config]
        except:
            pass
        else:
            # todo update view?
            pass

    def getCapabilityGroups(self, node_name):
        '''
        Returns the names of groups, which contains the given node.
        @param node_name: The name of the node
        @type node_name: C{str}
        @return: The name of the configuration containing this new capabilities.
        @rtype: C{dict(config : [str])}
        '''
        result = dict()  # dict(config : [group names])
        try:
            for cfg, cap in self._capcabilities.items():
                for ns, groups in cap.items():
                    for group, _ in groups.items():  # _:=decription
                        if self.is_in_cap_group(node_name, cfg, ns, group):
                            if cfg not in result:
                                result[cfg] = []
                            result[cfg].append(roslib.names.ns_join(ns, group))
        except:
            pass
#      import traceback
#      print traceback.format_exc(1)
        return result

    def getNodeItemsByName(self, node_name, recursive=True):
        '''
        Since the same node can be included by different groups, this method searches
        for all nodes with given name and returns these items.
        @param node_name: The name of the node
        @type node_name: C{str}
        @param recursive: Searches in (sub) groups
        @type recursive: C{bool}
        @return: The list with node items.
        @rtype: C{[U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}]}
        '''
        result = []
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                if recursive:
                    result[len(result):] = item.getNodeItemsByName(node_name)
            elif isinstance(item, NodeItem) and item == node_name:
                return [item]
        return result

    def getNodeItems(self, recursive=True):
        '''
        Returns all nodes in this group and subgroups.
        @param recursive: returns the nodes of the subgroups
        @type recursive: bool
        @return: The list with node items.
        @rtype: C{[U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}]}
        '''
        result = []
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                if recursive:
                    result[len(result):] = item.getNodeItems()
            elif isinstance(item, NodeItem):
                result.append(item)
        return result

    def getGroupItem(self, group_name, is_group=True, nocreate=False):
        '''
        Returns a GroupItem with given name. If no group with this name exists, a
        new one will be created.
        Assumption: No groups in group!!
        @param group_name: the name of the group
        @type group_name: C{str}
        @param nocreate: avoid creation of new group if not exists. (Default: False)
        @return: The group with given name
        @rtype: L{GroupItem}
        '''
        lns, rns = group_name, ''
        if nm.settings().group_nodes_by_namespace:
            lns, rns = lnamespace(group_name)
            if lns == rospy.names.SEP and type(self) == HostItem:
                lns, rns = lnamespace(rns)
        if lns == rospy.names.SEP:
            return self
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                if item == lns:
                    if rns:
                        return item.getGroupItem(rns, is_group)
                    return item
                elif item > lns and not nocreate:
                    items = []
                    newItem = GroupItem(lns, self, is_group=(is_group and not rns))
                    items.append(newItem)
                    cfgitem = CellItem(group_name, newItem)
                    items.append(cfgitem)
                    self.insertRow(i, items)
                    if rns:
                        return newItem.getGroupItem(rns, is_group)
                    return newItem
        if nocreate:
            return None
        items = []
        newItem = GroupItem(lns, self, is_group=(is_group and not rns))
        items.append(newItem)
        cfgitem = CellItem(group_name, newItem)
        items.append(cfgitem)
        self.appendRow(items)
        if rns:
            return newItem.getGroupItem(rns, is_group)
        return newItem

    def addNode(self, node, cfg=''):
        '''
        Adds a new node with given name.
        @param node: the NodeInfo of the node to create
        @type node: L{NodeInfo}
        @param cfg: The configuration, which describes the node
        @type cfg: C{str}
        '''
        groups = self.getCapabilityGroups(node.name)
        if groups:
            for _, group_list in groups.items():
                for group_name in group_list:
                    # insert in the group
                    groupItem = self.getGroupItem(group_name, True)
                    groupItem.addNode(node, cfg)
        else:
            group_item = self
            if nm.settings().group_nodes_by_namespace:
                if type(group_item) == HostItem:
                    # insert in the group
                    group_item = self.getGroupItem(namespace(node.name), False)
            # insert in order
            new_item_row = NodeItem.newNodeRow(node.name, node.masteruri)
            group_item._addRow_sorted(new_item_row)
            new_item_row[0].node_info = node
            if cfg or cfg == '':
                new_item_row[0].addConfig(cfg)
            group_item.updateIcon()

    def _addRow_sorted(self, row):
        for i in range(self.rowCount()):
            item = self.child(i)
            if item > row[0].name:
                self.insertRow(i, row)
                row[0].parent_item = self
                return
        self.appendRow(row)
        row[0].parent_item = self

    def clearUp(self, fixed_node_names=None):
        '''
        Removes not running and not configured nodes.
        @param fixed_node_names: If the list is not None, the node not in the list are
        set to not running!
        @type fixed_node_names: C{[str]}
        '''
        self._clearup(fixed_node_names)
        self._clearup_reinsert()
        self._clearup_riseup()

    def _clearup(self, fixed_node_names=None):
        '''
        Removes not running and not configured nodes.
        '''
        removed = False
        # move running nodes without configuration to the upper layer, remove not running and duplicate nodes
        for i in reversed(range(self.rowCount())):
            item = self.child(i)
            if isinstance(item, NodeItem):
                # set the running state of the node to None
                if fixed_node_names is not None and item.name not in fixed_node_names:
                    item.node_info = NodeInfo(item.name, item.node_info.masteruri)
                if not (item.has_configs() or item.is_running() or item.published or item.subscribed or item.services):
                    removed = True
                    self.removeRow(i)
                elif not isinstance(self, HostItem):
                    has_launches = NodeItem.has_launch_cfgs(item.cfgs)
                    has_defaults = NodeItem.has_default_cfgs(item.cfgs)
                    has_std_cfg = item.has_std_cfg()
                    if item.state == NodeItem.STATE_RUN and not (has_launches or has_defaults or has_std_cfg):
                        # if it is in a group, is running, but has no configuration, move it to the host
                        if self.parent_item is not None and isinstance(self.parent_item, HostItem):
                            items_in_host = self.parent_item.getNodeItemsByName(item.name, True)
                            if len(items_in_host) == 1:
                                row = self.takeRow(i)
                                self.parent_item._addRow_sorted(row)
                            else:
                                # remove item
                                removed = True
                                self.removeRow(i)
            else:  # if type(item) == GroupItem:
                removed = item._clearup(fixed_node_names) or removed
        if self.rowCount() == 0 and self.parent_item is not None:
            self.parent_item._remove_group(self.name)
        elif removed:
            self.updateIcon()
        return removed

    def _clearup_reinsert(self):
        inserted = False
        for i in reversed(range(self.rowCount())):
            item = self.child(i)
            if isinstance(item, NodeItem):
                if item.with_namespace:
                        group_item = self.getGroupItem(namespace(item.name), False, nocreate=True)
                        if group_item is not None and group_item != self:
                            inserted = True
                            row = self.takeRow(i)
                            group_item._addRow_sorted(row)
                            group_item.updateIcon()
            else:
                inserted = item._clearup_reinsert() or inserted
        return inserted

    def _clearup_riseup(self):
        changed = False
        for i in reversed(range(self.rowCount())):
            item = self.child(i)
            if isinstance(item, NodeItem):
                # remove group if only one node is inside
                if self.rowCount() == 1:
                    if not self.is_group and not isinstance(self, HostItem):
                        changed = True
                        row = self.takeRow(i)
                        if self.parent_item is not None:
                            self.parent_item._addRow_sorted(row)
                            self.parent_item._remove_group(self.name)
                            self.parent_item._clearup_riseup()
            else:
                changed = item._clearup_riseup() or changed
        return changed

    def _remove_group(self, name):
        for i in reversed(range(self.rowCount())):
            item = self.child(i)
            if type(item) == GroupItem and item == name and item.rowCount() == 0:
                self.removeRow(i)

    def reset_remote_launched_nodes(self):
        self._remote_launched_nodes_updated = False

    def remote_launched_nodes_updated(self):
        if self._has_remote_launched_nodes:
            return self._remote_launched_nodes_updated
        return True

    def updateRunningNodeState(self, nodes):
        '''
        Updates the running state of the nodes given in a dictionary.
        @param nodes: A dictionary with node names and their running state described by L{NodeInfo}.
        @type nodes: C{dict(str: U{master_discovery_fkie.NodeInfo<http://docs.ros.org/kinetic/api/master_discovery_fkie/html/modules.html#master_discovery_fkie.master_info.NodeInfo>})}
        '''
        for (name, node) in nodes.items():
            # get the node items
            items = self.getNodeItemsByName(name)
            if items:
                for item in items:
                    # update the node item
                    item.node_info = node
            else:
                # create the new node
                self.addNode(node)
            if self._has_remote_launched_nodes:
                self._remote_launched_nodes_updated = True
        self.clearUp(nodes.keys())

    def getRunningNodes(self):
        '''
        Returns the names of all running nodes. A running node is defined by his
        PID.
        @see: U{master_dicovery_fkie.NodeInfo<http://docs.ros.org/kinetic/api/master_discovery_fkie/html/modules.html#master_discovery_fkie.master_info.NodeInfo>}
        @return: A list with node names
        @rtype: C{[str]}
        '''
        result = []
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                result[len(result):] = item.getRunningNodes()
            elif isinstance(item, NodeItem) and item.node_info.pid is not None:
                result.append(item.name)
        return result

    def markNodesAsDuplicateOf(self, running_nodes, is_sync_running=False):
        '''
        While a synchronization same node on different hosts have the same name, the
        nodes with the same on other host are marked.
        @param running_nodes: The dictionary with names of running nodes and their masteruri
        @type running_nodes: C{dict(str:str)}
        @param is_sync_running: If the master_sync is running, the nodes are marked
          as ghost nodes. So they are handled as running nodes, but has not run
          informations. This nodes are running on remote host, but are not
          syncronized because of filter or errors.
        @type is_sync_running: bool
        '''
        ignore = ['/master_sync', '/master_discovery', '/node_manager']
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, GroupItem):
                item.markNodesAsDuplicateOf(running_nodes, is_sync_running)
            elif isinstance(item, NodeItem):
                if is_sync_running:
                    item.is_ghost = (item.node_info.uri is None and (item.name in running_nodes and running_nodes[item.name] == item.node_info.masteruri))
                    item.has_running = (item.node_info.uri is None and item.name not in ignore and (item.name in running_nodes and running_nodes[item.name] != item.node_info.masteruri))
                else:
                    if item.is_ghost:
                        item.is_ghost = False
                    item.has_running = (item.node_info.uri is None and item.name not in ignore and (item.name in running_nodes))

    def updateIcon(self):
        if isinstance(self, HostItem):
            # skip the icon update on a host item
            return
        has_running = False
        has_off = False
        has_duplicate = False
        has_ghosts = False
        diag_level = 0
        for i in range(self.rowCount()):
            item = self.child(i)
            if isinstance(item, (GroupItem, NodeItem)):
                if item.state == NodeItem.STATE_WARNING:
                    self.setIcon(QIcon(':/icons/crystal_clear_warning.png'))
                    return
                elif item.state == NodeItem.STATE_OFF:
                    has_off = True
                elif item.state == NodeItem.STATE_RUN:
                    has_running = True
                    if item.diagnostic_array and item.diagnostic_array[-1].level > 0:
                        if diag_level == 0:
                            diag_level = item.diagnostic_array[-1].level
                        elif item.diagnostic_array[-1].level == 2:
                            diag_level = 2
                        self.diagnostic_array = item.diagnostic_array
                elif item.state == NodeItem.STATE_GHOST:
                    has_ghosts = True
                elif item.state == NodeItem.STATE_DUPLICATE:
                    has_duplicate = True
                elif item.state == NodeItem.STATE_PARTS:
                    has_running = True
                    has_off = True
        diag_icon = None
        if diag_level > 0:
            diag_icon = NodeItem._diagnostic_level2icon(diag_level)
        if has_duplicate:
            self._state = NodeItem.STATE_DUPLICATE
            self.setIcon(QIcon(':/icons/imacadam_stop.png'))
        elif has_ghosts:
            self._state = NodeItem.STATE_GHOST
            self.setIcon(QIcon(':/icons/state_ghost.png'))
        elif has_running and has_off:
            if diag_icon is not None:
                self.setIcon(diag_icon)
            else:
                self._state = NodeItem.STATE_PARTS
                self.setIcon(QIcon(':/icons/state_part.png'))
        elif not has_running:
            self._state = NodeItem.STATE_OFF
            self.setIcon(QIcon(':/icons/state_off.png'))
        elif not has_off and has_running:
            if diag_icon is not None:
                self.setIcon(diag_icon)
            else:
                self._state = NodeItem.STATE_RUN
                self.setIcon(QIcon(':/icons/state_run.png'))
        if self.parent_item is not None:
            self.parent_item.updateIcon()

    def _create_html_list(self, title, items):
        result = ''
        if items:
            result += '<b><u>%s</u></b>' % title
            if len(items) > 1:
                result += ' <span style="color:gray;">[%d]</span>' % len(items)
            result += '<ul><span></span><br>'
            for i in items:
                result += '<a href="node://%s">%s</a><br>' % (i, i)
            result += '</ul>'
        return result

    def updateTooltip(self):
        '''
        Creates a tooltip description based on text set by L{updateDescription()}
        and all childs of this host with valid sensor description. The result is
        returned as a HTML part.
        @return: the tooltip description coded as a HTML part
        @rtype: C{str}
        '''
        tooltip = self.generateDescription(False)
        self.setToolTip(tooltip if tooltip else self.name)
        return tooltip

    def generateDescription(self, extended=True):
        tooltip = ''
        if self.descr_type or self.descr_name or self.descr:
            tooltip += '<h4>%s</h4><dl>' % self.descr_name
            if self.descr_type:
                tooltip += '<dt>Type: %s</dt></dl>' % self.descr_type
            if extended:
                try:
                    from docutils import examples
                    if self.descr:
                        tooltip += '<b><u>Detailed description:</u></b>'
                        tooltip += examples.html_body(utf8(self.descr))
                except:
                    rospy.logwarn("Error while generate description for a tooltip: %s", traceback.format_exc(1))
                    tooltip += '<br>'
            # get nodes
            nodes = []
            for j in range(self.rowCount()):
                nodes.append(self.child(j).name)
            if nodes:
                tooltip += self._create_html_list('Nodes:', nodes)
        return '<div>%s</div>' % tooltip

    def updateDescription(self, descr_type, descr_name, descr):
        '''
        Sets the description of the robot. To update the tooltip of the host item use L{updateTooltip()}.
        @param descr_type: the type of the robot
        @type descr_type: C{str}
        @param descr_name: the name of the robot
        @type descr_name: C{str}
        @param descr: the description of the robot as a U{http://docutils.sourceforge.net/rst.html|reStructuredText}
        @type descr: C{str}
        '''
        self.descr_type = descr_type
        self.descr_name = descr_name
        self.descr = descr

    def updateDisplayedConfig(self):
        '''
        Updates the configuration representation in other column.
        '''
        if self.parent_item is not None:
            # get nodes
            cfgs = []
            for j in range(self.rowCount()):
                if self.child(j).cfgs:
                    cfgs[len(cfgs):] = self.child(j).cfgs
            if cfgs:
                cfgs = list(set(cfgs))
            cfg_col = self.parent_item.child(self.row(), NodeItem.COL_CFG)
            if cfg_col is not None and isinstance(cfg_col, QStandardItem):
                cfg_col.setText('[%d]' % len(cfgs) if len(cfgs) > 1 else "")
                # set tooltip
                # removed for clarity !!!
#        tooltip = ''
#        if len(cfgs) > 0:
#          tooltip = ''
#          if len(cfgs) > 0:
#            tooltip = ''.join([tooltip, '<h4>', 'Configurations:', '</h4><dl>'])
#            for c in cfgs:
#              if NodeItem.is_default_cfg(c):
#                tooltip = ''.join([tooltip, '<dt>[default]', c[0], '</dt>'])
#              else:
#                tooltip = ''.join([tooltip, '<dt>', c, '</dt>'])
#            tooltip = ''.join([tooltip, '</dl>'])
#        cfg_col.setToolTip(''.join(['<div>', tooltip, '</div>']))
                # set icons
                has_launches = NodeItem.has_launch_cfgs(cfgs)
                has_defaults = NodeItem.has_default_cfgs(cfgs)
                if has_launches and has_defaults:
                    cfg_col.setIcon(QIcon(':/icons/crystal_clear_launch_file_def_cfg.png'))
                elif has_launches:
                    cfg_col.setIcon(QIcon(':/icons/crystal_clear_launch_file.png'))
                elif has_defaults:
                    cfg_col.setIcon(QIcon(':/icons/default_cfg.png'))
                else:
                    cfg_col.setIcon(QIcon())

    def get_configs(self):
        '''
        Returns a tuple with counts for launch and default configurations.
        '''
        cfgs = []
        for j in range(self.rowCount()):
            if isinstance(self.child(j), GroupItem):
                glcfgs, gdcfgs = self.child(j).get_configs()
                cfgs[len(cfgs):] = glcfgs
                cfgs[len(cfgs):] = gdcfgs
            elif self.child(j).cfgs:
                cfgs[len(cfgs):] = self.child(j).cfgs
        cfgs = list(set(cfgs))
        dccfgs = []
        lccfgs = []
        for c in cfgs:
            if NodeItem.is_default_cfg(c):
                dccfgs.append(c)
            else:
                lccfgs.append(c)
        return (lccfgs, dccfgs)

    def type(self):
        return GroupItem.ITEM_TYPE

    def __eq__(self, item):
        '''
        Compares the name of the group.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            return self.name.lower() == item.lower()
        elif not (item is None):
            return self.name.lower() == item.name.lower()
        return False

    def __ne__(self, item):
        return not (self == item)

    def __gt__(self, item):
        '''
        Compares the name of the group.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            # put the group with SYSTEM nodes at the end
            if self.is_system_group:
                if self.name.lower() != item.lower():
                    return True
            elif item.lower() == 'system':
                return False
            return self.name.lower() > item.lower()
        elif not (item is None):
            # put the group with SYSTEM nodes at the end
            if item.is_system_group:
                if self.name.lower() != item.lower():
                    return True
            elif self.is_syste_group:
                return False
            return self.name.lower() > item.name.lower()
        return False


################################################################################
##############                   HostItem                         ##############
################################################################################

class HostItem(GroupItem):
    '''
    The HostItem stores the information about a host.
    '''
    ITEM_TYPE = Qt.UserRole + 26

    def __init__(self, masteruri, address, local, parent=None):
        '''
        Initialize the HostItem object with given values.
        @param masteruri: URI of the ROS master assigned to the host
        @type masteruri: C{str}
        @param address: the address of the host
        @type address: C{str}
        @param local: is this host the localhost where the node_manager is running.
        @type local: C{bool}
        '''
        self._has_remote_launched_nodes = False
        name = self.create_host_description(masteruri, address)
        self._masteruri = masteruri
        self._host = address
        self._mastername = address
        self._local = local
        GroupItem.__init__(self, name, parent, has_remote_launched_nodes=self._has_remote_launched_nodes)
        image_file = nm.settings().robot_image_file(name)
        if QFile.exists(image_file):
            self.setIcon(QIcon(image_file))
        else:
            if local:
                self.setIcon(QIcon(':/icons/crystal_clear_miscellaneous.png'))
            else:
                self.setIcon(QIcon(':/icons/remote.png'))
        self.descr_type = self.descr_name = self.descr = ''

    @property
    def host(self):
        return self._host

    @property
    def hostname(self):
        return nm.nameres().hostname(self._host)

    @property
    def address(self):
        if NameResolution.is_legal_ip(self._host):
            return self._host
        else:
            result = nm.nameres().resolve_cached(self._host)
            if result:
                return result[0]
        return None

    @property
    def addresses(self):
        return nm.nameres().resolve_cached(self._host)

    @property
    def masteruri(self):
        return self._masteruri

    @property
    def mastername(self):
        result = nm.nameres().mastername(self._masteruri, self._host)
        if result is None or not result:
            result = self.hostname
        return result

    def create_host_description(self, masteruri, address):
        '''
        Returns the name generated from masteruri and address
        @param masteruri: URI of the ROS master assigned to the host
        @type masteruri: C{str}
        @param address: the address of the host
        @type address: C{str}
        '''
        name = nm.nameres().mastername(masteruri, address)
        if not name:
            name = address
        hostname = nm.nameres().hostname(address)
        if hostname is None:
            hostname = utf8(address)
        if not nm.settings().show_domain_suffix:
            name = subdomain(name)
        result = '%s@%s' % (name, hostname)
        maddr = get_hostname(masteruri)
        mname = nm.nameres().hostname(maddr)
        if mname is None:
            mname = utf8(maddr)
        if mname != hostname:
            result += '[%s]' % masteruri
            self._has_remote_launched_nodes = True
        return result

    def updateTooltip(self):
        '''
        Creates a tooltip description based on text set by L{updateDescription()}
        and all childs of this host with valid sensor description. The result is
        returned as a HTML part.
        @return: the tooltip description coded as a HTML part
        @rtype: C{str}
        '''
        tooltip = self.generateDescription(False)
        self.setToolTip(tooltip if tooltip else self.name)
        return tooltip

    def generateDescription(self, extended=True):
        from docutils import examples
        tooltip = ''
        if self.descr_type or self.descr_name or self.descr:
            tooltip += '<h4>%s</h4><dl>' % self.descr_name
            if self.descr_type:
                tooltip += '<dt>Type: %s</dt></dl>' % self.descr_type
            if extended:
                try:
                    if self.descr:
                        tooltip += '<b><u>Detailed description:</u></b>'
                        tooltip += examples.html_body(self.descr, input_encoding='utf8')
                except:
                    rospy.logwarn("Error while generate description for a tooltip: %s", traceback.format_exc(1))
                    tooltip += '<br>'
        tooltip += '<h3>%s</h3>' % self.mastername
        tooltip += '<font size="+1"><i>%s</i></font><br>' % self.masteruri
        tooltip += '<font size="+1">Host: <b>%s%s</b></font><br>' % (self.hostname, ' %s' % self.addresses if self.addresses else '')
        tooltip += '<a href="open-sync-dialog://%s">open sync dialog</a>' % (utf8(self.masteruri).replace('http://', ''))
        tooltip += '<p>'
        tooltip += '<a href="show-all-screens://%s">show all screens</a>' % (utf8(self.masteruri).replace('http://', ''))
        tooltip += '<p>'
        tooltip += '<a href="rosclean://%s" title="calls `rosclean purge` at `%s`">rosclean purge</a>' % (self.hostname, self.hostname)
        tooltip += '<p>'
        tooltip += '<a href="poweroff://%s" title="calls `sudo poweroff` at `%s` via SSH">poweroff `%s`</a>' % (self.hostname, self.hostname, self.hostname)
        tooltip += '<p>'
        tooltip += '<a href="remove-all-launch-server://%s">kill all launch server</a>' % utf8(self.masteruri).replace('http://', '')
        tooltip += '<p>'
        # get sensors
        capabilities = []
        for j in range(self.rowCount()):
            item = self.child(j)
            if isinstance(item, GroupItem):
                capabilities.append(item.name)
        if capabilities:
            tooltip += '<b><u>Capabilities:</u></b>'
            try:
                tooltip += examples.html_body('- %s' % ('\n- '.join(capabilities)), input_encoding='utf8')
            except:
                rospy.logwarn("Error while generate description for a tooltip: %s", traceback.format_exc(1))
        return '<div>%s</div>' % tooltip if tooltip else ''

    def type(self):
        return HostItem.ITEM_TYPE

    def __eq__(self, item):
        '''
        Compares the address of the masteruri.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            rospy.logwarn("compare HostItem with unicode depricated")
            return False
        elif isinstance(item, tuple):
            return self.masteruri == item[0] and self.host == item[1]
        elif isinstance(item, HostItem):
            return self.masteruri == item.masteruri and self.host == item.host
        return False

    def __gt__(self, item):
        '''
        Compares the address of the masteruri.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            rospy.logwarn("compare HostItem with unicode depricated")
            return False
        elif isinstance(item, tuple):
            return self.masteruri > item[0]
        elif isinstance(item, HostItem):
            return self.masteruri > item.masteruri
        return False


################################################################################
##############                   NodeItem                         ##############
################################################################################

class NodeItem(QStandardItem):
    '''
    The NodeItem stores the information about the node using the ExtendedNodeInfo
    class and represents it in a U{QTreeView<https://srinikom.github.io/pyside-docs/PySide/QtGui/QTreeView.html>} using the
    U{QStandardItemModel<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItemModel.html>}
    '''

    ITEM_TYPE = QStandardItem.UserType + 35
    NAME_ROLE = Qt.UserRole + 1
    COL_CFG = 1
#  COL_URI = 2

    STATE_OFF = 0
    STATE_RUN = 1
    STATE_WARNING = 2
    STATE_GHOST = 3
    STATE_DUPLICATE = 4
    STATE_PARTS = 5

    def __init__(self, node_info):
        '''
        Initialize the NodeItem instance.
        @param node_info: the node information
        @type node_info: U{master_discovery_fkie.NodeInfo<http://docs.ros.org/kinetic/api/master_discovery_fkie/html/modules.html#master_discovery_fkie.master_info.NodeInfo>}
        '''
        QStandardItem.__init__(self, node_info.name)
        self._parent_item = None
        self._node_info = node_info.copy()
#    self.ICONS = {'empty' : QIcon(),
#                  'run'    : QIcon(':/icons/state_run.png'),
#                  'off'     :QIcon(':/icons/state_off.png'),
#                  'warning' : QIcon(':/icons/crystal_clear_warning.png'),
#                  'stop'    : QIcon('icons/imacadam_stop.png'),
#                  'cfg+def' : QIcon(':/icons/crystal_clear_launch_file_def_cfg.png'),
#                  'cfg'     : QIcon(':/icons/crystal_clear_launch_file.png'),
#                  'default_cfg' : QIcon(':/icons/default_cfg.png')
#                  }
        self._cfgs = []
        self.launched_cfg = None  # is used to store the last configuration to launch the node
        self.next_start_cfg = None  # is used to set the configuration for next start of the node
        self._std_config = None  # it's config with empty name. for default proposals
        self._is_ghost = False
        self._has_running = False
        self.setIcon(QIcon(':/icons/state_off.png'))
        self._state = NodeItem.STATE_OFF
        self.diagnostic_array = []
        self.nodelet_mngr = ''
        self.nodelets = []
        self.has_screen = True
        self._with_namespace = rospy.names.SEP in node_info.name

    @property
    def state(self):
        return self._state

    @property
    def name(self):
        return self._node_info.name

    @name.setter
    def name(self, new_name):
        self.setText(new_name)

    @property
    def masteruri(self):
        return self._node_info.masteruri

    @property
    def published(self):
        return self._node_info.publishedTopics

    @property
    def subscribed(self):
        return self._node_info.subscribedTopics

    @property
    def services(self):
        return self._node_info.services

    @property
    def parent_item(self):
        return self._parent_item

    @parent_item.setter
    def parent_item(self, parent_item):
        self._parent_item = parent_item
        if parent_item is None:
            self.setText(self._node_info.name)
            self._with_namespace = rospy.names.SEP in self._node_info.name
        else:
            new_name = self._node_info.name.replace(parent_item.get_namespace(), '', 1)
            self.setText(new_name)
            self._with_namespace = rospy.names.SEP in new_name

    @property
    def node_info(self):
        '''
        Returns the NodeInfo instance of this node.
        @rtype: U{master_discovery_fkie.NodeInfo<http://docs.ros.org/kinetic/api/master_discovery_fkie/html/modules.html#master_discovery_fkie.master_info.NodeInfo>}
        '''
        return self._node_info

    @node_info.setter
    def node_info(self, node_info):
        '''
        Sets the NodeInfo and updates the view, if needed.
        '''
        abbos_changed = False
        run_changed = False
#     print "!!!", self.name
#     print "  subs: ", self._node_info.subscribedTopics, node_info.subscribedTopics
#     print "  pubs: ", self._node_info.publishedTopics, node_info.publishedTopics
#     print "  srvs: ", self._node_info.services, node_info.services
        if self._node_info.publishedTopics != node_info.publishedTopics:
            abbos_changed = True
            self._node_info._publishedTopics = list(node_info.publishedTopics)
        if self._node_info.subscribedTopics != node_info.subscribedTopics:
            abbos_changed = True
            self._node_info._subscribedTopics = list(node_info.subscribedTopics)
        if self._node_info.services != node_info.services:
            abbos_changed = True
            self._node_info._services = list(node_info.services)
        if self._node_info.pid != node_info.pid:
            self._node_info.pid = node_info.pid
            run_changed = True
        if self._node_info.uri != node_info.uri:
            self._node_info.uri = node_info.uri
            run_changed = True
        # update the tooltip and icon
        if run_changed and (self.is_running() or self.has_configs) or abbos_changed:
            self.has_screen = True
            self.updateDispayedName()
#      self.updateDisplayedURI()
            if self.parent_item is not None:
                self.parent_item.updateIcon()

    @property
    def uri(self):
        return self._node_info.uri

    @property
    def pid(self):
        return self._node_info.pid

    @property
    def has_running(self):
        '''
        Returns C{True}, if there are exists other nodes with the same name. This
        variable must be set manually!
        @rtype: C{bool}
        '''
        return self._has_running

    @has_running.setter
    def has_running(self, state):
        '''
        Sets however other node with the same name are running or not (on other hosts)
        and updates the view of this item.
        '''
        if self._has_running != state:
            self._has_running = state
            if self.has_configs() or self.is_running():
                self.updateDispayedName()
            if self.parent_item is not None:
                self.parent_item.updateIcon()

    @property
    def is_ghost(self):
        '''
        Returns C{True}, if there are exists other runnig nodes with the same name. This
        variable must be set manually!
        @rtype: C{bool}
        '''
        return self._is_ghost

    @is_ghost.setter
    def is_ghost(self, state):
        '''
        Sets however other node with the same name is running (on other hosts) and
        and the host showing this node the master_sync is running, but the node is
        not synchronized.
        '''
        if self._is_ghost != state:
            self._is_ghost = state
            if self.has_configs() or self.is_running():
                self.updateDispayedName()
            if self.parent_item is not None and not isinstance(self.parent_item, HostItem):
                self.parent_item.updateIcon()

    @property
    def with_namespace(self):
        '''
        Returns `True` if the node name contains a '/' in his name

        :rtype: bool
        '''
        return self._with_namespace

    def append_diagnostic_status(self, diagnostic_status):
        self.diagnostic_array.append(diagnostic_status)
        self.updateDispayedName()
        if self.parent_item is not None and not isinstance(self.parent_item, HostItem):
            self.parent_item.updateIcon()
        if len(self.diagnostic_array) > 15:
            del self.diagnostic_array[0]

    def data(self, role):
        if role == self.NAME_ROLE:
            return self.name
        else:
            return QStandardItem.data(self, role)

    @staticmethod
    def _diagnostic_level2icon(level):
        if level == 1:
            return QIcon(':/icons/state_diag_warn.png')
        elif level == 2:
            return QIcon(':/icons/state_diag_error.png')
        elif level == 3:
            return QIcon(':/icons/state_diag_stale.png')
        else:
            return QIcon(':/icons/state_diag_other.png')

    def updateDispayedName(self):
        '''
        Updates the name representation of the Item
        '''
        tooltip = '<h4>%s</h4><dl>' % self.node_info.name
        tooltip += '<dt><b>URI:</b> %s</dt>' % self.node_info.uri
        tooltip += '<dt><b>PID:</b> %s</dt>' % self.node_info.pid
        tooltip += '<dt><b>ORG.MASTERURI:</b> %s</dt></dl>' % self.node_info.masteruri
        master_discovered = nm.nameres().has_master(self.node_info.masteruri)
#    local = False
#    if not self.node_info.uri is None and not self.node_info.masteruri is None:
#      local = (get_hostname(self.node_info.uri) == get_hostname(self.node_info.masteruri))
        if self.node_info.pid is not None:
            self._state = NodeItem.STATE_RUN
            if self.diagnostic_array and self.diagnostic_array[-1].level > 0:
                level = self.diagnostic_array[-1].level
                self.setIcon(self._diagnostic_level2icon(level))
                self.setToolTip(self.diagnostic_array[-1].message)
            else:
                self.setIcon(QIcon(':/icons/state_run.png'))
                self.setToolTip('')
        elif self.node_info.uri is not None and not self.node_info.isLocal:
            self._state = NodeItem.STATE_RUN
            self.setIcon(QIcon(':/icons/state_unknown.png'))
            tooltip += '<dl><dt>(Remote nodes will not be ping, so they are always marked running)</dt></dl>'
            tooltip += '</dl>'
            self.setToolTip('<div>%s</div>' % tooltip)
#    elif not self.node_info.isLocal and not master_discovered and not self.node_info.uri is None:
# #    elif not local and not master_discovered and not self.node_info.uri is None:
#      self._state = NodeItem.STATE_RUN
#      self.setIcon(QIcon(':/icons/state_run.png'))
#      tooltip = ''.join([tooltip, '<dl><dt>(Remote nodes will not be ping, so they are always marked running)</dt></dl>'])
#      tooltip = ''.join([tooltip, '</dl>'])
#      self.setToolTip(''.join(['<div>', tooltip, '</div>']))
        elif self.node_info.pid is None and self.node_info.uri is None and (self.node_info.subscribedTopics or self.node_info.publishedTopics or self.node_info.services):
            self.setIcon(QIcon(':/icons/crystal_clear_warning.png'))
            self._state = NodeItem.STATE_WARNING
            tooltip += '<dl><dt>Can\'t get node contact information, but there exists publisher, subscriber or services of this node.</dt></dl>'
            tooltip += '</dl>'
            self.setToolTip('<div>%s</div>' % tooltip)
        elif self.node_info.uri is not None:
            self._state = NodeItem.STATE_WARNING
            self.setIcon(QIcon(':/icons/crystal_clear_warning.png'))
            if not self.node_info.isLocal and master_discovered:
                tooltip = '<h4>%s is not local, however the ROS master on this host is discovered, but no information about this node received!</h4>' % self.node_info.name
                self.setToolTip('<div>%s</div>' % tooltip)
        elif self.is_ghost:
            self._state = NodeItem.STATE_GHOST
            self.setIcon(QIcon(':/icons/state_ghost.png'))
            tooltip = '<h4>The node is running, but not synchronized because of filter or errors, see master_sync log.</h4>'
            self.setToolTip('<div>%s</div>' % tooltip)
        elif self.has_running:
            self._state = NodeItem.STATE_DUPLICATE
            self.setIcon(QIcon(':/icons/imacadam_stop.png'))
            tooltip = '<h4>There are nodes with the same name on remote hosts running. These will be terminated, if you run this node! (Only if master_sync is running or will be started somewhere!)</h4>'
            self.setToolTip('<div>%s</div>' % tooltip)
        else:
            self._state = NodeItem.STATE_OFF
            self.setIcon(QIcon(':/icons/state_off.png'))
            self.setToolTip('')
        # removed common tooltip for clarity !!!
#    self.setToolTip(''.join(['<div>', tooltip, '</div>']))

    def updateDisplayedURI(self):
        '''
        Updates the URI representation in other column.
        '''
        if self.parent_item is not None:
            uri_col = self.parent_item.child(self.row(), NodeItem.COL_URI)
            if uri_col is not None and isinstance(uri_col, QStandardItem):
                uri_col.setText(utf8(self.node_info.uri) if self.node_info.uri is not None else "")

    @property
    def cfgs(self):
        '''
        Returns the list with all launch configurations assigned to this item.
        @rtype: C{[str]}
        '''
        return self._cfgs

    def addConfig(self, cfg):
        '''
        Add the given configurations to the node.
        @param cfg: the loaded configuration, which contains this node.
        @type cfg: C{str}
        '''
        if cfg == '':
            self._std_config = cfg
        elif cfg and cfg not in self._cfgs:
            self._cfgs.append(cfg)
            self.updateDisplayedConfig()

    def remConfig(self, cfg):
        '''
        Remove the given configurations from the node.
        @param cfg: the loaded configuration, which contains this node.
        @type cfg: C{str}
        '''
        result = False
        if cfg == '':
            self._std_config = None
            result = True
        if cfg in self._cfgs:
            self._cfgs.remove(cfg)
            result = True
        if result and (self.has_configs() or self.is_running()):
            self.updateDisplayedConfig()
        return result

    def updateDisplayedConfig(self):
        '''
        Updates the configuration representation in other column.
        '''
        if self.parent_item is not None:
            cfg_col = self.parent_item.child(self.row(), NodeItem.COL_CFG)
            if cfg_col is not None and isinstance(cfg_col, QStandardItem):
                cfg_count = len(self._cfgs)
                cfg_col.setText(utf8(''.join(['[', utf8(cfg_count), ']'])) if cfg_count > 1 else "")
                # set tooltip
                # removed tooltip for clarity !!!
#        tooltip = ''
#        if len(self._cfgs) > 0:
#          tooltip = ''
#          if len(self._cfgs) > 0:
#            tooltip = ''.join([tooltip, '<h4>', 'Configurations:', '</h4><dl>'])
#            for c in self._cfgs:
#              if NodeItem.is_default_cfg(c):
#                tooltip = ''.join([tooltip, '<dt>[default]', c[0], '</dt>'])
#              else:
#                tooltip = ''.join([tooltip, '<dt>', c, '</dt>'])
#            tooltip = ''.join([tooltip, '</dl>'])
#        cfg_col.setToolTip(''.join(['<div>', tooltip, '</div>']))
                # set icons
                has_launches = NodeItem.has_launch_cfgs(self._cfgs)
                has_defaults = NodeItem.has_default_cfgs(self._cfgs)
                if has_launches and has_defaults:
                    cfg_col.setIcon(QIcon(':/icons/crystal_clear_launch_file_def_cfg.png'))
                elif has_launches:
                    cfg_col.setIcon(QIcon(':/icons/crystal_clear_launch_file.png'))
                elif has_defaults:
                    cfg_col.setIcon(QIcon(':/icons/default_cfg.png'))
                else:
                    cfg_col.setIcon(QIcon())
#      the update of the group will be perform in node_tree_model to reduce calls
#        if isinstance(self.parent_item, GroupItem):
#          self.parent_item.updateDisplayedConfig()

    def type(self):
        return NodeItem.ITEM_TYPE

    @classmethod
    def newNodeRow(self, name, masteruri):
        '''
        Creates a new node row and returns it as a list with items. This list is
        used for the visualization of node data as a table row.
        @param name: the node name
        @type name: C{str}
        @param masteruri: the URI or the ROS master assigned to this node.
        @type masteruri: C{str}
        @return: the list for the representation as a row
        @rtype: C{[L{NodeItem}, U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}(Cofigurations), U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}(Node URI)]}
        '''
        items = []
        item = NodeItem(NodeInfo(name, masteruri))
        items.append(item)
        cfgitem = CellItem(name, item)
        items.append(cfgitem)
#    uriitem = QStandardItem()
#    items.append(uriitem)
        return items

    def has_configs(self):
        return not (len(self._cfgs) == 0)  # and self._std_config is None)

    def is_running(self):
        return not (self._node_info.pid is None and self._node_info.uri is None)

    def has_std_cfg(self):
        return self._std_config == ''

    def count_launch_cfgs(self):
        result = 0
        for c in self.cfgs:
            if not self.is_default_cfg(c):
                result += 1
        return result

    def count_default_cfgs(self):
        result = 0
        for c in self.cfgs:
            if self.is_default_cfg(c):
                result += 1
        return result

    @classmethod
    def has_launch_cfgs(cls, cfgs):
        for c in cfgs:
            if not cls.is_default_cfg(c):
                return True
        return False

    @classmethod
    def has_default_cfgs(cls, cfgs):
        for c in cfgs:
            if cls.is_default_cfg(c):
                return True
        return False

    @classmethod
    def is_default_cfg(cls, cfg):
        return isinstance(cfg, tuple)

    def __eq__(self, item):
        '''
        Compares the name of the node.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            return self.name == item
        elif not (item is None):
            return self.name == item.name
        return False

    def __gt__(self, item):
        '''
        Compares the name of the node.
        '''
        if isinstance(item, str) or isinstance(item, unicode):
            return self.name > item
        elif not (item is None):
            return self.name > item.name
        return False


################################################################################
##############                NodeTreeModel                       ##############
################################################################################

class NodeTreeModel(QStandardItemModel):
    '''
    The model to show the nodes running in a ROS system or loaded by a launch
    configuration.
    '''
#  ICONS = {'default'        : QIcon(),
#           'run'            : QIcon(":/icons/state_run.png"),
#           'warning'        : QIcon(":/icons/crystal_clear_warning.png"),
#           'def_launch_cfg' : QIcon(":/icons/crystal_clear_launch_file_def_cfg.png"),
#           'launch_cfg'     : QIcon(":/icons/crystal_clear_launch_file.png"),
#           'def_cfg'        : QIcon(":/icons/default_cfg.png") }

    header = [('Name', 450),
              ('Info', -1)]
#            ('URI', -1)]

    hostInserted = Signal(HostItem)
    '''@ivar: the Qt signal, which is emitted, if a new host was inserted.
  Parameter: U{QtCore.QModelIndex<https://srinikom.github.io/pyside-docs/PySide/QtCore/QModelIndex.html>} of the inserted host item'''

    def __init__(self, host_address, masteruri, parent=None):
        '''
        Initialize the model.
        '''
        super(NodeTreeModel, self).__init__(parent)
        self.setColumnCount(len(NodeTreeModel.header))
        self.setHorizontalHeaderLabels([label for label, _ in NodeTreeModel.header])
        self._local_host_address = host_address
        self._local_masteruri = masteruri
        self._std_capabilities = {'': {'SYSTEM': {'images': [],
                                                  'nodes': ['/rosout',
                                                            '/master_discovery',
                                                            '/zeroconf',
                                                            '/master_sync',
                                                            '/node_manager',
                                                            '/dynamic_reconfigure/*'],
                                                  'type': '',
                                                  'description': 'This group contains the system management nodes.'}}}

        # create a handler to request the parameter
        self.parameterHandler = ParameterHandler()
#    self.parameterHandler.parameter_list_signal.connect(self._on_param_list)
        self.parameterHandler.parameter_values_signal.connect(self._on_param_values)
#    self.parameterHandler.delivery_result_signal.connect(self._on_delivered_values)

    @property
    def local_addr(self):
        return self._local_host_address

    def flags(self, index):
        if not index.isValid():
            return Qt.NoItemFlags
        return Qt.ItemIsEnabled | Qt.ItemIsSelectable

    def _set_std_capabilities(self, host_item):
        if host_item is not None:
            cap = self._std_capabilities
            mastername = roslib.names.SEP.join(['', host_item.mastername, '*', 'default_cfg'])
            if mastername not in cap['']['SYSTEM']['nodes']:
                cap['']['SYSTEM']['nodes'].append(mastername)
            host_item.addCapabilities('', cap, host_item.masteruri)
            return cap
        return dict(self._std_capabilities)

    def get_hostitem(self, masteruri, address):
        '''
        Searches for the host item in the model. If no item is found a new one will
        created and inserted in sorted order.
        @param masteruri: ROS master URI
        @type masteruri: C{str}
        @param address: the address of the host
        @type address: C{str}
        @return: the item associated with the given master
        @rtype: L{HostItem}
        '''
        if masteruri is None:
            return None
        resaddr = nm.nameres().hostname(address)
        host = (masteruri, resaddr)
        # [address] + nm.nameres().resolve_cached(address)
        local = (self.local_addr in [address] + nm.nameres().resolve_cached(address) and
                 self._local_masteruri == masteruri)
        # find the host item by address
        root = self.invisibleRootItem()
        for i in range(root.rowCount()):
            if root.child(i) == host:
                return root.child(i)
            elif root.child(i) > host:
                hostItem = HostItem(masteruri, resaddr, local)
                self.insertRow(i, hostItem)
                self.hostInserted.emit(hostItem)
                self._set_std_capabilities(hostItem)
                return hostItem
        hostItem = HostItem(masteruri, resaddr, local)
        self.appendRow(hostItem)
        self.hostInserted.emit(hostItem)
        self._set_std_capabilities(hostItem)
        return hostItem

    def updateModelData(self, nodes):
        '''
        Updates the model data.
        @param nodes: a dictionary with name and info objects of the nodes.
        @type nodes: C{dict(str:L{NodeInfo}, ...)}
        '''
        # separate into different hosts
        hosts = dict()
        addresses = []
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            host.reset_remote_launched_nodes()
        for (name, node) in nodes.items():
            addr = get_hostname(node.uri if node.uri is not None else node.masteruri)
            addresses.append(node.masteruri)
            host = (node.masteruri, addr)
            if host not in hosts:
                hosts[host] = dict()
            hosts[host][name] = node
        # update nodes for each host
        for ((masteruri, host), nodes_filtered) in hosts.items():
            hostItem = self.get_hostitem(masteruri, host)
            # rename the host item if needed
            if hostItem is not None:
                hostItem.updateRunningNodeState(nodes_filtered)
            # request for all nodes in host the parameter capability_group
            self._requestCapabilityGroupParameter(hostItem)
        # update nodes of the hosts, which are not more exists
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host.masteruri not in addresses:
                host.updateRunningNodeState({})
        self.removeEmptyHosts()
        # update the duplicate state
#    self.markNodesAsDuplicateOf(self.getRunningNodes())

    def _requestCapabilityGroupParameter(self, host_item):
        if host_item is not None:
            items = host_item.getNodeItems()
            params = [roslib.names.ns_join(item.name, 'capability_group') for item in items if not item.has_configs() and item.is_running() and not host_item.is_in_cap_group(item.name, '', '', 'SYSTEM')]
            if params:
                self.parameterHandler.requestParameterValues(host_item.masteruri, params)

    def _on_param_values(self, masteruri, code, msg, params):
        '''
        Updates the capability groups of nodes from ROS parameter server.
        @param masteruri: The URI of the ROS parameter server
        @type masteruri: C{str}
        @param code: The return code of the request. If not 1, the message is set and the list can be ignored.
        @type code: C{int}
        @param msg: The message of the result.
        @type msg: C{str}
        @param params: The dictionary the parameter names and request result.
        @type params: C{dict(paramName : (code, statusMessage, parameterValue))}
        '''
        host = nm.nameres().address(masteruri)
        if host is None:
            # try with hostname of the masteruri
            host = get_hostname(masteruri)
        hostItem = self.get_hostitem(masteruri, host)
        changed = False
        if hostItem is not None and code == 1:
            capabilities = self._set_std_capabilities(hostItem)
            available_ns = set([''])
            available_groups = set(['SYSTEM'])
            # assumption: all parameter are 'capability_group' parameter
            for p, (code_n, _, val) in params.items():  # _:=msg_n
                nodename = roslib.names.namespace(p).rstrip(roslib.names.SEP)
                ns = roslib.names.namespace(nodename).rstrip(roslib.names.SEP)
                if not ns:
                    ns = roslib.names.SEP
                available_ns.add(ns)
                if code_n == 1:
                    # add group
                    if val:
                        available_groups.add(val)
                        if ns not in capabilities:
                            capabilities[ns] = dict()
                        if val not in capabilities[ns]:
                            capabilities[ns][val] = {'images': [], 'nodes': [], 'type': '', 'description': 'This group is created from `capability_group` parameter of the node defined in ROS parameter server.'}
                        if nodename not in capabilities[ns][val]['nodes']:
                            capabilities[ns][val]['nodes'].append(nodename)
                            changed = True
                else:
                    try:
                        for group, _ in capabilities[ns].items():
                            try:
                                # remove the config from item, if parameter was not foun on the ROS parameter server
                                groupItem = hostItem.getGroupItem(roslib.names.ns_join(ns, group))
                                if groupItem is not None:
                                    nodeItems = groupItem.getNodeItemsByName(nodename, True)
                                    for item in nodeItems:
                                        item.remConfig('')
                                capabilities[ns][group]['nodes'].remove(nodename)
                                # remove the group, if empty
                                if not capabilities[ns][group]['nodes']:
                                    del capabilities[ns][group]
                                    if not capabilities[ns]:
                                        del capabilities[ns]
                                groupItem.updateDisplayedConfig()
                                changed = True
                            except:
                                pass
                    except:
                        pass
            # clearup namespaces to remove empty groups
            for ns in capabilities.keys():
                if ns and ns not in available_ns:
                    del capabilities[ns]
                    changed = True
                else:
                    for group in capabilities[ns].keys():
                        if group and group not in available_groups:
                            del capabilities[ns][group]
                            changed = True
            # update the capabilities and the view
            if changed:
                if capabilities:
                    hostItem.addCapabilities('', capabilities, hostItem.masteruri)
                hostItem.clearUp()
        else:
            rospy.logwarn("Error on retrieve \'capability group\' parameter from %s: %s", utf8(masteruri), msg)

    def set_std_capablilities(self, capabilities):
        '''
        Sets the default capabilities description, which is assigned to each new
        host.
        @param capabilities: the structure for capabilities
        @type capabilities: C{dict(namespace: dict(group:dict('type' : str, 'description' : str, 'nodes' : [str])))}
        '''
        self._std_capabilities = capabilities

    def addCapabilities(self, masteruri, host_address, cfg, capabilities):
        '''
        Adds groups to the model
        @param masteruri: ROS master URI
        @type masteruri: C{str}
        @param host_address: the address the host
        @type host_address: C{str}
        @param cfg: the configuration name (launch file name or tupel for default configuration)
        @type cfg: C{str or (str, str))}
        @param capabilities: the structure for capabilities
        @type capabilities: C{dict(namespace: dict(group:dict('type' : str, 'description' : str, 'nodes' : [str])))}
        '''
        hostItem = self.get_hostitem(masteruri, host_address)
        if hostItem is not None:
            # add new capabilities
            hostItem.addCapabilities(cfg, capabilities, hostItem.masteruri)
        self.removeEmptyHosts()

    def appendConfigNodes(self, masteruri, host_address, nodes):
        '''
        Adds nodes to the model. If the node is already in the model, only his
        configuration list will be extended.
        @param masteruri: ROS master URI
        @type masteruri: C{str}
        @param host_address: the address the host
        @type host_address: C{str}
        @param nodes: a dictionary with node names and their configurations
        @type nodes: C{dict(str : str)}
        '''
        hostItem = self.get_hostitem(masteruri, host_address)
        if hostItem is not None:
            groups = set()
            for (name, cfg) in nodes.items():
                items = hostItem.getNodeItemsByName(name)
                for item in items:
                    if item.parent_item is not None:
                        groups.add(item.parent_item)
                        # only added the config to the node, if the node is in the same group
                        if isinstance(item.parent_item, HostItem):
                            item.addConfig(cfg)
                        elif hostItem.is_in_cap_group(item.name, cfg, rospy.names.namespace(item.name).rstrip(rospy.names.SEP), item.parent_item.name):
                            item.addConfig(cfg)
                        # test for default group
                        elif hostItem.is_in_cap_group(item.name, '', '', item.parent_item.name):
                            item.addConfig(cfg)
                    else:
                        item.addConfig(cfg)
                if not items:
                    # create the new node
                    node_info = NodeInfo(name, masteruri)
                    hostItem.addNode(node_info, cfg)
                    # get the group of the added node to be able to update the group view, if needed
                    items = hostItem.getNodeItemsByName(name)
                    for item in items:
                        if item.parent_item is not None:
                            groups.add(item.parent_item)
            # update the changed groups
            for g in groups:
                g.updateDisplayedConfig()
            hostItem.clearUp()
        self.removeEmptyHosts()
        # update the duplicate state
#    self.markNodesAsDuplicateOf(self.getRunningNodes())

    def removeConfigNodes(self, cfg):
        '''
        Removes nodes from the model. If node is running or containing in other
        launch or default configurations , only his configuration list will be
        reduced.
        @param cfg: the name of the confugration to close
        @type cfg: C{str}
        '''
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            items = host.getNodeItems()
            groups = set()
            for item in items:
                removed = item.remConfig(cfg)
                if removed and item.parent_item is not None:
                    groups.add(item.parent_item)
            for g in groups:
                g.updateDisplayedConfig()
            host.remCapablities(cfg)
            host.clearUp()
            if host.rowCount() == 0:
                self.invisibleRootItem().removeRow(i)
            elif groups:
                # request for all nodes in host the parameter capability_group
                self._requestCapabilityGroupParameter(host)

    def removeEmptyHosts(self):
        # remove empty hosts
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host.rowCount() == 0 or not host.remote_launched_nodes_updated():
                self.invisibleRootItem().removeRow(i)

    def isDuplicateNode(self, node_name):
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host is not None:  # should not occur
                nodes = host.getNodeItemsByName(node_name)
                for n in nodes:
                    if n.has_running:
                        return True
        return False

    def getNode(self, node_name, masteruri):
        '''
        Since the same node can be included by different groups, this method searches
        for all nodes with given name and returns these items.
        @param node_name: The name of the node
        @type node_name: C{str}
        @return: The list with node items.
        @rtype: C{[U{QtGui.QStandardItem<https://srinikom.github.io/pyside-docs/PySide/QtGui/QStandardItem.html>}]}
        '''
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host is not None and (masteruri is None or host.masteruri == masteruri):
                res = host.getNodeItemsByName(node_name)
                if res:
                    return res
        return []

    def getRunningNodes(self):
        '''
        Returns a list with all known running nodes.
        @rtype: C{[str]}
        '''
        running_nodes = list()
        # # determine all running nodes
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host is not None:  # should not occur
                running_nodes[len(running_nodes):] = host.getRunningNodes()
        return running_nodes

    def markNodesAsDuplicateOf(self, running_nodes, is_sync_running=False):
        '''
        If there are a synchronization running, you have to avoid to running the
        node with the same name on different hosts. This method helps to find the
        nodes with same name running on other hosts and loaded by a configuration.
        The nodes loaded by a configuration will be inform about a currently running
        nodes, so a warning can be displayed!
        @param running_nodes: The dictionary with names of running nodes and their masteruri
        @type running_nodes: C{dict(str:str)}
        @param is_sync_running: If the master_sync is running, the nodes are marked
          as ghost nodes. So they are handled as running nodes, but has not run
          informations. This nodes are running on remote host, but are not
          syncronized because of filter or errors.
        @type is_sync_running: bool
        '''
        for i in reversed(range(self.invisibleRootItem().rowCount())):
            host = self.invisibleRootItem().child(i)
            if host is not None:  # should not occur
                host.markNodesAsDuplicateOf(running_nodes, is_sync_running)

    def updateHostDescription(self, masteruri, host, descr_type, descr_name, descr):
        '''
        Updates the description of a host.
        @param masteruri: ROS master URI of the host to update
        @type masteruri: C{str}
        @param host: host to update
        @type host: C{str}
        @param descr_type: the type of the robot
        @type descr_type: C{str}
        @param descr_name: the name of the robot
        @type descr_name: C{str}
        @param descr: the description of the robot as a U{http://docutils.sourceforge.net/rst.html|reStructuredText}
        @type descr: C{str}
        '''
        root = self.invisibleRootItem()
        for i in range(root.rowCount()):
            if root.child(i) == (utf8(masteruri), utf8(host)):
                h = root.child(i)
                h.updateDescription(descr_type, descr_name, descr)
                return h.updateTooltip()
