// -*- C++ -*-
/*!
 * @file OutPortPushConnector.h
 * @brief Push type connector class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2009
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_OUTPORTPUSHCONNECTOR_H
#define RTC_OUTPORTPUSHCONNECTOR_H

#include <rtm/OutPortConnector.h>
#include <rtm/InPortConsumer.h>
#include <rtm/PublisherBase.h>

namespace RTC
{
  class ConnectorListeners;

  /*!
   * @if jp
   * @class OutPortPushConnector
   * @brief OutPortPushConnector 饹
   *
   * OutPort  push ǡեΤ Connector 饹Υ
   * Ȥϡ³ dataflow_type  push ꤵ줿硢
   * OutPort ˤäͭ졢InPortPushConnector Фˤʤäơ
   * ǡݡȤ push Υǡե¸롣Ĥ³Фơ
   * ĤΥǡȥ꡼󶡤ͣ Connector б롣
   * Connector  ³ UUID  ID ˤ̤롣
   *
   * OutPortPushConnector ϰʲλĤΥ֥Ȥͭ롣
   *
   * - InPortConsumer
   * - Buffer
   * - Publisher
   *
   * OutPort ˽񤭹ޤ줿ǡ OutPortPushConnector::write() 
   * 졢Connector  Publisher ˥ǡ񤭹ࡣPublisher Ϥ
   * ˽äƥǡ Buffer  InPortConsumer Ф
   * push 뤳Ȥ InPort ˥ǡž롣
   *
   * @since 1.0.0
   *
   * @else
   * @class OutPortPushConnector
   * @brief OutPortPushConnector class
   *
   * Connector class of OutPort for push type dataflow.  When "push"
   * is specified as dataflow_type at the time of establishing
   * connection, this object is generated and owned by the OutPort.
   * This connector and InPortPushConnector make a pair and realize
   * push type dataflow of data ports.  One connector corresponds to
   * one connection which provides a data stream.  Connector is
   * distinguished by ID of the UUID that is generated at establishing
   * connection.
   *
   * OutPortPushConnector owns and manages the following objects.
   *
   * - InPortConsumer
   * - Buffer
   * - Publisher
   *
   * @since 1.0.0
   *
   * Data written into the OutPort is passed to
   * OutPortPushConnector::write(), and the connector writes into the
   * publisher.  The publisher gets data from the buffer based on the
   * policy and it is transferred to InPort by pushing it into the
   * InPortConsumer.
   *
   * @endif
   */
  class OutPortPushConnector
    : public OutPortConnector
  {
  public:
    DATAPORTSTATUS_ENUM

    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * OutPortPushConnector Υ󥹥ȥ饯ϥ֥˲
     * ˤȤ롣ConnectorInfo ³ޤߡξ˽ѥ
     * åХåե롣InPort 󥿡եФ
     * 󥷥塼ޥ֥ȤؤΥݥ󥿤ꡢͭĤΤǡ
     * OutPortPushConnector  InPortConsumer βǤġƼ磻
     * ٥ȤФ륳Хå󶡤 ConnectorListeners 
     * Ŭڤʥߥ󥰤ǥХåƤӽФǡХåե
     *  OutPortBase 󶡤ϤΥݥ󥿤롣
     *
     * @param info ConnectorInfo
     * @param consumer InPortConsumer
     * @param listeners ConnectorListeners Υꥹʥ֥ȥꥹ
     * @param buffer CdrBufferBase ΥХåե
     *
     * @else
     * @brief Constructor
     *
     * OutPortPushConnector's constructor is given the following
     * arguments.  According to ConnectorInfo which includes
     * connection information, a publisher and a buffer are created.
     * It is also given a pointer to the consumer object for the
     * InPort interface.  The owner-ship of the pointer is owned by
     * this OutPortPushConnector, it has responsibility to destruct
     * the InPortConsumer.  OutPortPushConnector also has
     * ConnectorListeners to provide event callback mechanisms, and
     * they would be called at the proper timing.  If data buffer is
     * given by OutPortBase, the pointer to the buffer is also given
     * as arguments.
     *
     * @param info ConnectorInfo
     * @param consumer InPortConsumer
     * @param listeners ConnectorListeners type lsitener object list
     * @param buffer CdrBufferBase type buffer
     *
     * @endif
     */
    OutPortPushConnector(ConnectorInfo info,
                         InPortConsumer* consumer,
                         ConnectorListeners& listeners,
                         CdrBufferBase* buffer = 0);

    /*!
     * @if jp
     * @brief ǥȥ饯
     *
     * disconnect() ƤФ졢consumer, publisher, buffer Ρ롣
     *
     * @else
     *
     * @brief Destructor
     *
     * This operation calls disconnect(), which destructs and deletes
     * the consumer, the publisher and the buffer.
     *
     * @endif
     */
    virtual ~OutPortPushConnector();

    /*!
     * @if jp
     * @brief ǡν񤭹
     *
     * PublisherФƥǡ񤭹ߡˤбInPortإǡ
     * ž롣ｪλ PORT_OK ֤롣ʳ
     * 硢顼ͤȤơCONNECTION_LOST, BUFFER_FULL,
     * BUFFER_ERROR, PORT_ERROR, BUFFER_TIMEOUT, PRECONDITION_NO_MET 
     * ֤롣
     *
     * @return PORT_OK              ｪλ
     *         CONNECTION_LOST      ³Ȥ
     *         BUFFER_FULL          ХåեդǤ
     *         BUFFER_ERROR         Хåե顼
     *         BUFFER_TIMEOUT       Хåեؤν񤭹ߤॢȤ
     *         PRECONDITION_NOT_MET ʤ
     *         PORT_ERROR           ¾Υ顼
     *
     * @else
     *
     * @brief Writing data
     *
     * This operation writes data into publisher and then the data
     * will be transferred to correspondent InPort. If data is written
     * properly, this function will return PORT_OK return code. Except
     * normal return, CONNECTION_LOST, BUFFER_FULL, BUFFER_ERROR,
     * PORT_ERROR, BUFFER_TIMEOUT and PRECONDITION_NO_MET will be
     * returned as error codes.
     *  
     * @return PORT_OK              Normal return
     *         CONNECTION_LOST      Connectin lost
     *         BUFFER_FULL          Buffer full
     *         BUFFER_ERROR         Buffer error
     *         BUFFER_TIMEOUT       Timeout
     *         PRECONDITION_NOT_MET Precondition not met
     *         PORT_ERROR           Other error
     *
     * @endif
     */
    virtual ReturnCode write(const cdrMemoryStream& data);

    /*!
     * @if jp
     * @brief ³
     *
     * consumer, publisher, buffer Ρ롣
     *
     * @else
     *
     * @brief disconnect
     *
     * This operation destruct and delete the consumer, the publisher
     * and the buffer.
     *
     * @endif
     */
    virtual ReturnCode disconnect();

    /*!
     * @if jp
     * @brief ƥֲ
     *
     * Υͥ򥢥ƥֲ
     *
     * @else
     *
     * @brief Connector activation
     *
     * This operation activates this connector
     *
     * @endif
     */
    virtual void activate();

    /*!
     * @if jp
     * @brief 󥢥ƥֲ
     *
     * Υͥ󥢥ƥֲ
     *
     * @else
     *
     * @brief Connector deactivation
     *
     * This operation deactivates this connector
     *
     * @endif
     */
    virtual void deactivate();

    /*!
     * @if jp
     * @brief Buffer 
     *
     * Connector ݻƤ Buffer ֤
     *
     * @else
     * @brief Getting Buffer
     *
     * This operation returns this connector's buffer
     *
     * @endif
     */
    virtual CdrBufferBase* getBuffer();

  protected:
    /*!
     * @if jp
     * @brief Publisher
     *
     * Ϳ줿³˴Ťѥ֥å롣
     *
     * @param info ³
     * @return ѥ֥åؤΥݥ
     *
     * @else
     * @brief create buffer
     *
     * This function creates a publisher based on given information.
     *
     * @param info Connector information
     * @return The poitner to the publisher
     *
     * @endif
     */
    virtual PublisherBase* createPublisher(ConnectorInfo& info);

    /*!
     * @if jp
     * @brief Buffer
     *
     * Ϳ줿³˴ŤХåե롣
     *
     * @param info ³
     * @return ХåեؤΥݥ
     *
     * @else
     * @brief create buffer
     *
     * This function creates a buffer based on given information.
     *
     * @param info Connector information
     * @return The poitner to the buffer
     *
     * @endif
     */
    virtual CdrBufferBase* createBuffer(ConnectorInfo& info);

    /*!
     * @if jp
     * @brief ³Ω˥ХåƤ
     * @else
     * @brief Invoke callback when connection is established
     * @endif
     */
    void onConnect();

    /*!
     * @if jp
     * @brief ³ǻ˥ХåƤ
     * @else
     * @brief Invoke callback when connection is destroied
     * @endif
     */
    void onDisconnect();

  private:
    /*!
     * @if jp
     * @brief InPortConsumer ؤΥݥ
     * @else
     * @brief A pointer to an InPortConsumer
     * @endif
     */
    InPortConsumer* m_consumer;

    /*!
     * @if jp
     * @brief Publisher ؤΥݥ
     * @else
     * @brief A pointer to a publisher
     * @endif
     */
    PublisherBase* m_publisher;

    /*!
     * @if jp
     * @brief ConnectorListenrs ؤλ
     * @else
     * @brief A reference to a ConnectorListener
     * @endif
     */
    ConnectorListeners& m_listeners;

    /*!
     * @if jp
     * @brief Buffer ؤΥݥ
     * @else
     * @brief A pointer to a buffer
     * @endif
     */
    CdrBufferBase* m_buffer;

  };
}; // namespace RTC

#endif  // RTC_PUSH_CONNECTOR_H
