// -*- C++ -*-
/*!
 * @file SdoConfiguration.h
 * @brief RT component base class
 * @date $Date: 2008-01-14 07:49:34 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: SdoConfiguration.h 2177 2011-06-06 01:01:47Z n-ando $
 *
 */

#ifndef RTC_SDOCONFIGURATION_H
#define RTC_SDOCONFIGURATION_H

#include <string>

// CORBA header include
#include <rtm/RTC.h>
#include <coil/Mutex.h>
#include <coil/Guard.h>

// local includes
#include <rtm/idl/SDOPackageSkel.h>
#include <rtm/ConfigAdmin.h>
#include <rtm/SystemLogger.h>
#include <rtm/SdoServiceAdmin.h>

// SdoConfiguration with SeqEx 159120
// SdoConfiguration with SeqUtil 114504 114224

/*!
 * @if jp
 * @namespace SDOPackage
 *
 * @brief SDO ѥå
 *
 * @else
 *
 * @namespace SDOPackage
 *
 * @brief SDO Package
 *
 * @endif
 */

#ifdef WIN32
#pragma warning( disable : 4290 )
#endif

namespace SDOPackage
{
  /*!
   * @if jp
   *
   * @class Configuration_impl
   * @brief SDO Configuration 饹
   *
   * Configuration interface  Resource Data Model 줿ǡ
   * ɲáԤΥ󥿡եǤ롣
   * DeviceProfile, ServiceProfile, ConfigurationProfile  Organization
   * ѹԤΥڥ졼Ƥ롣SDO λͤǤϥ
   * ӥƥ˴ؤܺ٤ˤĤƤϵꤷƤʤ
   * 
   * ʣ (Configuration) ݻ뤳Ȥˤꡢưפ᤯
   * ȿǤ뤳ȤǤ롣줿ʣ ConfigurationSets
   *  configuration profile Ȥݻ뤳ȤǤ롣ҤȤĤ
   * ConfigurationSet ˴Ϣդ줿ץѥƥͤΥꥹȤ
   * ˡIDܺ٤ȤȤ˻äƤ롣ˤꡢܤξܺ٤򵭽Ҥ
   * ̤뤳ȤǤ롣Configuration interface Υڥ졼Ϥ
   * ConfiguratioinSets δٱ礹롣
   *
   *
   * - ConfigurationSet: id, description, NVList 鹽1åȤ
   * - ConfigurationSetList: ConfigurationSet Υꥹ
   * - Parameter: name, type, allowed_values 鹽ѥ᡼
   * - ActiveConfigurationSet: ͭʥե졼1åȡ
   *
   * ʲSDOͤƤʤ⤷ϲ᤬狼ʤȼ
   *
   * ʲδؿ ParameterList ФƽԤ
   * - get_configuration_parameters()
   *
   * ʲδؿϥƥ֤ConfigurationSetФԤ
   * - get_configuration_parameter_values()
   * - get_configuration_parameter_value()
   * - set_configuration_parameter()
   *
   * ʲδؿConfigurationSetListФƽԤ
   * - get_configuration_sets()
   * - get_configuration_set()
   * - set_configuration_set_values()
   * - get_active_configuration_set()
   * - add_configuration_set()
   * - remove_configuration_set()
   * - activate_configuration_set()
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class Configuration_impl
   * @brief Configuration implementation class
   *
   * Configuration interface provides operations to add or remove data
   * specified in resource data model. These operations provide functions to
   * change DeviceProfile, ServiceProfile, ConfigurationProfile, and
   * Organization. This SDO specification does not address access control or
   * security aspects.
   *
   * Different configurations can be stored for simple and quick activation.
   * Different predefined configurations are stored as different
   * ConfigurationSets or configuration profile. A ConfigurationSet stores the
   * value of all properties assigned for the particular configuration along
   * with its unique id and description to identify and describe the
   * configuration respectively. Operations in the configuration interface
   * help manage these ConfigurationSets.
   *
   *
   * - ConfigurationSet: id, description, one configuration set to consist 
   *                     of NVList 
   * - ConfigurationSetList: List of ConfigurationSet
   * - Parameter: Parameter definition consist of name, type and allowed_values
   * - ActiveConfigurationSet: One set of configuration set that is valid.
   *
   * The following functions do processing to ParameterList. 
   * - get_configuration_parameters()
   *
   * The following functions do processing to active ConfigurationSet
   * - get_configuration_parameter_values()
   * - get_configuration_parameter_value()
   * - set_configuration_parameter()
   *
   * The following functions do processing to ConfigurationSetList
   * - get_configuration_sets()
   * - get_configuration_set()
   * - set_configuration_set_values()
   * - get_active_configuration_set()
   * - add_configuration_set()
   * - remove_configuration_set()
   * - activate_configuration_set()
   *
   * @since 0.4.0
   *
   * @endif
   */
  class Configuration_impl
    : public virtual POA_SDOPackage::Configuration,
      public virtual PortableServer::RefCountServantBase
  {
    typedef coil::Mutex Mutex;
    typedef coil::Guard<Mutex> Guard;
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     *
     * @param configAdmin ConfigurationSetList
     * 
     * @else
     *
     * @brief Constructor
     * 
     * Constructor
     *
     * @param configAdmin ConfigurationSetList
     *
     * @endif
     */
    Configuration_impl(RTC::ConfigAdmin& configAdmin,
                       RTC::SdoServiceAdmin& sdoServiceAdmin);
    
    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     * 
     * @else
     *
     * @brief Virtual destractor
     *
     * Virtual destractor.
     
     * @endif
     */
    virtual ~Configuration_impl(void);
    
    //============================================================
    //
    // <<< CORBA interfaces >>>
    //
    //============================================================
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] SDO  DeviceProfile Υå
     *
     * Υڥ졼 SDO  DeviceProfile 򥻥åȤ롣SDO 
     * DeviceProfile ݻƤʤϿ DeviceProfile 
     * DeviceProfile 򤹤ǤݻƤϴ¸ΤΤ֤롣
     *
     * @param dProfile SDO ˴Ϣդ DeviceProfile
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InvalidParameter  "dProfile"  null Ǥ롣
     * @exception InternalError Ū顼ȯ
     * 
     * @else
     *
     * @brief [CORBA interface] Set DeviceProfile of SDO
     *
     * This operation sets the DeviceProfile of an SDO. If the SDO does not
     * have DeviceProfile, the operation will create a new DeviceProfile,
     * otherwise it will replace the existing DeviceProfile.
     *
     * @param dProfile The device profile that is to be assigned to this SDO.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InvalidParameter The argument "dProfile" is null.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean set_device_profile(const DeviceProfile& dProfile)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] SDO  ServiceProfile Υå
     *
     * Υڥ졼Ϥ Configuration interface ͭо SDO 
     * ServiceProfile ɲä롣⤷ ServiceProfile  id Ǥ
     *  ID 줽 ServiceProfile Ǽ롣⤷ id 
     * ʤСSDO Ʊ id  ServiceProfile 򸡺롣
     * Ʊ id ¸ߤʤФ ServiceProfile ɲäid ¸ߤ
     * 񤭤򤹤롣<br>
     * (աǿСǤϥڥ졼̾add_service_profileѹ)
     *
     * @param sProfile ɲä ServiceProfile
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "sProfile"  nullǤ롣
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Set SDO's ServiceProfile
     *
     * This operation adds ServiceProfile to the target SDO that navigates this
     * Configuration interface. If the id in argument ServiceProfile is null,
     * new id is created and the ServiceProfile is stored. If the id is not
     * null, the target SDO searches for ServiceProfile in it with the same id.
     * It adds the ServiceProfile if not exist, or overwrites if exist.
     *
     * @param sProfile ServiceProfile to be added.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InvalidParameter The argument "sProfile" is null.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean add_service_profile(const ServiceProfile& sProfile)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] Organization ɲ
     *
     * Υڥ졼 Organization object Υե󥹤ɲä롣
     *
     * @param org ɲä Organization
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InvalidParameter  "organization"  null Ǥ롣
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Add Organization
     *
     * This operation adds reference of an Organization object.
     *
     * @param org Organization to be added.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InvalidParameter The argument organization is null.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean add_organization(Organization_ptr org)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ServiceProfile κ
     *
     * Υڥ졼Ϥ Configuration interface  SDO 
     * Service  ServiceProfile 롣 ServiceProfile
     * ϰˤꤵ롣
     *
     * @param id  ServiceProfile  serviceID
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "id"  null Ǥ롣⤷ "id" 
     *            Ϣդ줿 ServiceProfile ¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Remove ServiceProfile
     *
     * This operation removes ServiceProfile object to the SDO that has this
     * Configuration interface. The ServiceProfile object to be removed is
     * specified by argument.
     *
     * @param id serviceID of a ServiceProfile to be removed.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The argument "sProfile" is null, or if the
     *                             object that is specified by argument
     *                             "sProfile" does not exist.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean remove_service_profile(const char* id)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] Organization λȤκ
     *
     * Υڥ졼 Organization λȤ롣
     *
     * @param organization_id  Organization ΰդ id
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "organization_id"  null Ǥ롣
     *                             ⤷ "organization_id" ˴Ϣդ줿 
     *                             OrganizationProfile ¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Remove the reference of Organization 
     *
     * This operation removes the reference of an Organization object.
     *
     * @param organization_id Unique id of the organization to be removed.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The argument "organizationID" is null,
     *                             or the object which is specified by argument
     *                             "organizationID" does not exist.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean remove_organization(const char* organization_id)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ѥ᡼ΥꥹȤμ
     *
     * Υڥ졼 configuration parameter ΥꥹȤ֤
     * SDO ǽʥѥ᡼ʤжΥꥹȤ֤
     *
     * @return ħդѥ᡼Υꥹȡ
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get a list of configuration parameters
     *
     * This operation returns a list of Parameters. An empty list is returned
     * if the SDO does not have any configurable parameter.
     *
     * @return The list with definitions of parameters characterizing the
     *          configuration.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual ParameterList* get_configuration_parameters()
      throw (CORBA::SystemException,
	     NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] Configuration parameter ͤΥꥹȤμ
     *
     * Υڥ졼Ƥ configuration ѥ᡼֤ͤ
     *
     * @return Ƥ configuration ѥ᡼ͤΥꥹȡ
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get a list of the value of configuration 
     *                          parameters
     *
     * This operation returns all configuration parameters and their values.
     *
     * @return List of all configuration parameters and their values.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual NVList* get_configuration_parameter_values()
      throw (CORBA::SystemException,
	     NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] Configuration parameter ͤμ
     *
     * Υڥ졼ϰ "name" ǻꤵ줿ѥ᡼֤ͤ
     *
     * @param name ͤ׵᤹ѥ᡼̾
     *
     * @return ꤵ줿ѥ᡼͡
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "name"  null Ǥ롣
     *            ⤷ "name" ˴Ϣդ줿ѥ᡼¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get the value of configuration parameter
     *
     * This operation returns a value of parameter that is specified by
     * argument "name."
     *
     * @param name Name of the parameter whose value is requested.
     *
     * @return The value of the specified parameter.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The value of the argument "name" is
     *                             empty String, or null, or the parameter
     *                             that is specified by argument "name"
     *                             does not exist.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Any* get_configuration_parameter_value(const char* name)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] Configuration ѥ᡼ѹ
     *
     * Υڥ졼 "name" ǻꤷѥ᡼ͤ "value" 
     * ѹ롣
     *
     * @param name ѹоݥѥ᡼̾
     * @param value ѹоݥѥ᡼ο͡
     *
     * @return ڥ졼ɤ֤
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter ( "name"⤷"value")  null Ǥ롣
     *            ⤷ "name" ˴Ϣդ줿ѥ᡼¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Modify the configuration parameter value
     *
     * This operation sets a parameter to a value that is specified by argument
     * "value." The parameter to be modified is specified by argument " name."
     *
     * @param name The name of parameter to be modified.
     * @param value New value of the specified parameter.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The arguments ("name" and/or "value") is
     *                             null, or the parameter that is specified by
     *                             the argument "name" does not exist.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean set_configuration_parameter(const char* name,
						       const CORBA::Any& value)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
        
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet ꥹȤμ 
     *
     * Υڥ졼 ConfigurationProfile  ConfigurationSet 
     * ꥹȤ֤ SDO  ConfigurationSet ʤжΥꥹȤ֤
     *
     * @return ݻƤ ConfigurationSet ΥꥹȤθ͡
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get a list of ConfigurationSet
     *
     * This operation returns a list of ConfigurationSets that the
     * ConfigurationProfile has. An empty list is returned if the SDO does not
     * have any ConfigurationSets.
     * This operation returns a list of all ConfigurationSets of the SDO.
     * If no predefined ConfigurationSets exist, then empty list is returned.
     *
     * @return The list of stored configuration with their current values.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual ConfigurationSetList* get_configuration_sets()
      throw (CORBA::SystemException,
	     NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet μ
     *
     * Υڥ졼ϰǻꤵ줿 ConfigurationSet  ID ˴Ϣ
     * դ줿 ConfigurationSet ֤
     *
     * @param config_id ConfigurationSet μ̻ҡ
     *
     * @return ˤꤵ줿 ConfigurationSet
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter "config_id"  null ꤵ줿
     *            ConfigurationSet ¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get a ConfigurationSet
     *
     * This operation returns the ConfigurationSet specified by the parameter
     * configurationSetID.
     *
     * @param config_id Identifier of ConfigurationSet requested.
     *
     * @return The configuration set specified by the parameter config_id.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The parameter 'config_id' is null or 
     *                             there are no ConfigurationSets stored with
     *                             such id.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual ConfigurationSet* get_configuration_set(const char* config_id)
      throw (CORBA::SystemException,
	     NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ƥ֤ ConfigurationSet 
     *
     * Υڥ졼SDOθߥƥ֤ ConfigurationSet ֤
     * (⤷SDOθߤ꤬ͽ줿 ConfigurationSet ˤꤵ
     * ʤС)
     * ConfigurationSet ϰʲξˤϥƥ֤ǤϤʤΤȤߤʤ롣
     *
     * - ߤ꤬ͽ줿 ConfigurationSet ˤꥻåȤƤʤ
     * - SDO ꤬ƥ֤ˤʤäѹ줿
     * - SDO ꤹ ConfigurationSet ѹ줿
     * 
     * ξˤϡ ConfigurationSet ֤롣
     *
     * @return ߥƥ֤ ConfigurationSet
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Get active ConfigurationSet
     *
     * This operation returns the current active ConfigurationSet of an
     * SDO (i.e., if the current configuration of the SDO was set using
     * predefined configuration set).
     * ConfigurationSet cannot be considered active if the:
     *
     * - Current configuration of the SDO was not set using any predefined
     *   ConfigurationSet, or
     * - Configuration of the SDO was changed after it has been active, or
     * - ConfigurationSet that was used to configure the SDO was modified.
     *
     * Empty ConfigurationSet is returned in these cases.
     *
     * @return The active ConfigurationSet.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual ConfigurationSet* get_active_configuration_set()
      throw (CORBA::SystemException,
	     NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet ɲä
     *
     * ConfigurationProfile  ConfigurationSet ɲä륪ڥ졼
     *
     * @param configuration_set ɲä ConfigurationSet
     *
     * @return ڥ졼ɤ
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter "configurationSet"  null 
     *            "configurationSet"줿°ΣĤ
     *            ꤵ줿 configurationSet ID¸ߤ롣
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Add ConfigurationSet
     *
     * This operation adds a ConfigurationSet to the ConfigurationProfile.
     *
     * @param configuration_set The ConfigurationSet that is added.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The argument "configurationSet" is null,
     *                             or one of the attributes defining
     *                             "configurationSet" is invalid, or the 
     *                             specified identifier of the configuration
     *                             set already exists.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean
    add_configuration_set(const ConfigurationSet& configuration_set)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet 򥻥åȤ
     *
     * Υڥ졼ϻꤵ줿 id  ConfigurationSet 򹹿롣
     *
     * @param config_id ѹ ConfigurationSet  ID
     * @param configuration_set ѹ ConfigurationSet ΤΡ
     *
     * @return ConfigurationSet ˹Ǥ true
     *         Ǥʤ false ֤
     *
     * @exception InvalidParameter config_id  null 
     *                             ꤵ줿 id ǳǼ줿 ConfigurationSet
     *                             ¸ߤʤꤵ줿 configuration_set
     *                             °ΣĤ
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Set ConfigurationSet
     *
     * This operation modifies the specified ConfigurationSet of an SDO.
     *
     * Note: The number of parameters differs between spec and IDL!!
     *
     * @param config_id The ID of ConfigurationSet to be modified.
     * @param configuration_set ConfigurationSet to be replaced.
     *
     * @return A flag indicating if the ConfigurationSet was modified 
     *         successfully. "true" - The ConfigurationSet was modified
     *         successfully. "false" - The ConfigurationSet could not be
     *         modified successfully.
     *
     * @exception InvalidParameter The parameter 'configurationSetID' is null
     *                             or there is no ConfigurationSet stored with 
     *                             such id.
     *                             This exception is also raised if one of the 
     *                             attributes defining ConfigurationSet is not 
     *                             valid.
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean
    set_configuration_set_values(const ConfigurationSet& configuration_set)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet 
     *
     * ConfigurationProfile  ConfigurationSet 롣
     *
     * @param config_id  ConfigurationSet  id
     *
     * @return ڥ졼ɤ
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "configurationSetID"  null Ǥ롢
     *            ⤷ϡǻꤵ줿 ConfigurationSet ¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Remove ConfigurationSet
     *
     * This operation removes a ConfigurationSet from the ConfigurationProfile.
     *
     * @param config_id The id of ConfigurationSet which is removed.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The arguments "configurationSetID" is null,
     *                             or the object specified by the argument
     *                             "configurationSetID" does not exist.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean remove_configuration_set(const char* config_id)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    /*!
     * @if jp
     * 
     * @brief [CORBA interface] ConfigurationSet Υƥֲ
     *
     * ConfigurationProfile ˳Ǽ줿 ConfigurationSet ΤĤ
     * ƥ֤ˤ롣
     * Υڥ졼 ConfigurationSet 򥢥ƥ֤ˤ롣
     * ʤSDO Υե졼󡦥ץѥƥγǼƤ
     * ConfigurationSet ˤꤵץѥƥͤѹ롣
     * ꤵ줿 ConfigurationSet ͤƥ֡ե졼
     * ˥ԡȤȤ̣롣
     *
     * @param config_id ƥֲ ConfigurationSet  id
     *
     * @return ڥ졼ɤ
     *
     * @exception SDONotExists åȤSDO¸ߤʤ(㳰ϡCORBAɸ
     *                         ƥ㳰OBJECT_NOT_EXIST˥ޥåԥ󥰤)
     * @exception InvalidParameter  "config_id"  null Ǥ롢⤷
     *            ǻꤵ줿 ConfigurationSet ¸ߤʤ
     * @exception NotAvailable SDO¸ߤ뤬ʤ
     * @exception InternalError Ū顼ȯ
     * @else
     *
     * @brief [CORBA interface] Activate ConfigurationSet
     *
     * This operation activates one of the stored ConfigurationSets in the
     * ConfigurationProfile.
     * This operation activates the specified stored ConfigurationSets.
     * This means that the configuration properties of the SDO are changed as
     * the values of these properties specified in the stored ConfigurationSet.
     * In other words, values of the specified ConfigurationSet are now copied
     * to the active configuration.
     *
     * @param config_id Identifier of ConfigurationSet to be activated.
     *
     * @return If the operation was successfully completed.
     *
     * @exception SDONotExists The target SDO does not exist.(This exception 
     *                         is mapped to CORBA standard system exception
     *                         OBJECT_NOT_EXIST.)
     * @exception InvalidParameter The argument ("configID") is null or
     *                             there is no configuration set with identifier
     *                             specified by the argument.
     * @exception NotAvailable The target SDO is reachable but cannot respond.
     * @exception InternalError The target SDO cannot execute the operation
     *                          completely due to some internal error.
     * @endif
     */
    virtual CORBA::Boolean activate_configuration_set(const char* config_id)
      throw (CORBA::SystemException,
	     InvalidParameter, NotAvailable, InternalError);
    
    // end of CORBA interface definition
    //============================================================
    
    /*!
     * @if jp
     *
     * @brief ֥ȡե󥹤
     * 
     * оݤΥ֥ȥե󥹤
     * 
     * @return ֥ȥե
     * 
     * @else
     *
     * @brief Get object reference
     * 
     * Get the target object reference.
     * 
     * @return Object reference
     * 
     * @endif
     */
    Configuration_ptr getObjRef();
    
    /*!
     * @if jp
     *
     * @brief SDO  DeviceProfile 
     * 
     * SDO  DeviceProfile 
     * 
     * @return SDO  DeviceProfile
     * 
     * @else
     *
     * @brief Get the DeviceProfile of SDO
     * 
     * Get the DeviceProfile of SDO.
     * 
     * @return DeviceProfile of SDO
     * 
     * @endif
     */
    const DeviceProfile getDeviceProfile();
    
    /*!
     * @if jp
     *
     * @brief SDO  Organization ꥹȤ
     * 
     * SDO  Organization ꥹȤ
     * 
     * @return SDO  Organization ꥹ
     * 
     * @else
     *
     * @brief Get a list of Organization of SDO
     * 
     * Get a list of Organization of SDO.
     * 
     * @return List of Organization of SDO
     * 
     * @endif
     */
    const OrganizationList getOrganizations();
    
  protected:
    ::RTC::Logger rtclog;
    /*!
     * @if jp
     *
     * @brief UUID
     * 
     * UUID
     * 
     * @return UUID
     * 
     * @else
     *
     * @brief Generate UUID
     * 
     * Generate UUID.
     * 
     * @return UUID that has been generated
     * 
     * @endif
     */
    const std::string getUUID() const;
    
    /*!
     * @if jp
     * @brief CORBA ֥Ȥؤλ
     * @else
     * @brief The reference to CORBA object
     * @endif
     */
    Configuration_var m_objref;
    
    /*!
     * @if jp
     * @brief Lock դ SDO DeviceProfile
     * @else
     * @brief SDO DeviceProfile with mutex lock
     * @endif
     */
    DeviceProfile m_deviceProfile;
    Mutex m_dprofile_mutex;
    
    /*!
     * @if jp
     * @brief SDO Parameter
     * 
     * Ѥ¸ѿ(ѥ᡼)ǡ¤
     * ѥ᡼¤ϡѿ̾ȷ롣
     * Ƥ°ϰʲΤȤꡣ
     *  - name : ѥ᡼̾
     *  - type : ѥ᡼η̾ѥ᡼ǡΥꥸʥϰϤ
     *           ° allowedValues Ǹꤹ뤳ȤǤ롣
     * - allowedValues : ѥ᡼뤳ȤΤǤ͡
     *                   ѥ᡼˸ͭꤹɬפΤ
     *                   °ϻѤ롣㤨Сʸѥ᡼˵
     *                   ͤˤäƸꤷꡢͷѥ᡼˵
     *                   ͤϰϤˤäƸꤷꤹ롣ѥ᡼˵
     *                   ͤϡϰϤޤϥ󥿡Х빽¤
     *                   뤳ȤǤ롣
     *                   ⤷ѥ᡼Ф󤬤ʤϡallowedValues
     *                   °nullȤʤ롣ʤѥ᡼˸ͭϰ
     *                   ǤФɤΤ褦ͤ뤳ȤǤ롣
     * 
     * @else
     * @brief SDO Parameter
     * 
     * Data structure to define a variable (parameter) independently 
     * of implementation technologies. The Parameter structure defines 
     * the name and type of a variable.
     * Attributes defined in Parameter.
     *  - name : Parameters name.
     *  - type : Name of parameter's type. The original value scope of 
     *           parameter data type can be constrained by definitions 
     *           allocated in the attribute allowedValues.
     * - allowedValues : Values that the parameter can accept.
     *           This attribute is used only when the value scope
     *           inherent to the parameter type must be constrained. For
     *           example, the values allowed for a string parameter may
     *           be constrained by an enumeration, or the values
     *           allowed for a numeric parameter may be constrained by
     *           a range. The values allowed for a parameter can be
     *           defined in enumeration, range, or interval structures.
     *           The value of attribute allowedValues is null if there is
     *           no constraint on a parameter value, that is, any value
     *           can be assigned to the parameter as far as it follows the
     *           value scope inherent to the parameters type.
     * @endif
     *
     *    struct Parameter
     *    {
     *      string         name;
     *      TypeCode  type;
     *      AllowedValues allowed_values;
     *    };
     */

    /*!
     * @if jp
     * @brief Lock դ SDO ParameterList
     * @else
     * @brief SDO ParameterList with mutex lock
     * @endif
     */
    ParameterList m_parameters;
    Mutex m_params_mutex;
    
    /*!
     * @if jp
     * @brief Lock դ SDO ConfigurationSetList
     * @else
     * @brief SDO ConfigurationSetList with mutex lock
     * @endif
     */
    /*
      struct ConfigurationSet
      {
      string id;
      string description;
      NVList configuration_data;
      };
    */
    RTC::ConfigAdmin& m_configsets;
    Mutex m_config_mutex;

    /*!
     * @if jp
     * @brief Lock դ SDO Service ֥
     * @else
     * @brief SDO Service admin object with mutex lock
     * @endif
     */
    RTC::SdoServiceAdmin& m_sdoservice;
    Mutex m_sdoservice_mutex;
    
    /*!
     * @if jp
     * @brief  Lock դ SDO OrganizationList
     * @else
     * @brief SDO OrganizationList with mutex lock
     * @endif
     */
    OrganizationList m_organizations;
    Mutex m_org_mutex;
    
    /*!
     * @if jp
     * @brief  NVListfunctor
     * @else
     * @brief  Functor for NVList
     * @endif
     */
    struct nv_name
    {
      nv_name(const char* name) : m_name(name) {};
      bool operator()(const NameValue& nv)
      {
	return m_name == std::string(nv.name);
      }
      std::string m_name;
    };
    
    /*!
     * @if jp
     * @brief  Organizationfunctor
     * @else
     * @brief  Functor for Organization
     * @endif
     */
    struct org_id
    {
      org_id(const char* id) : m_id(id) {};
      bool operator()(const Organization_ptr& o)
      {
	CORBA::String_var id(o->get_organization_id());
	return m_id == (const char *)id;
      }
      const std::string m_id;
    };
    
    /*!
     * @if jp
     * @brief  ConfigurationSetfunctor
     * @else
     * @brief  Functor for ConfigurationSet
     * @endif
     */
    struct config_id
    {
      config_id(const char* id) :  m_id(id) {};
      bool operator()(const ConfigurationSet& c)
      {
	std::string id(c.id);
	return m_id == id;
      }
      const std::string m_id;
    };
  };  // class Configuration_impl
}; // namespace SDOPackage

#ifdef WIN32
#pragma warning( default : 4290 )
#endif

#endif // RTC_SDOCONFIGURATION_H
