// -*- C++ -*-
/*!
 * @file   ConfigAdminTests.cpp
 * @brief  ConfigAdmin test class
 * @date   $Date: 2008/05/12 01:47:32 $
 *
 * $Id: ConfigAdminTests.cpp,v 1.1 2008/05/12 01:47:32 arafune Exp $
 *
 */

/*
 * $Log: ConfigAdminTests.cpp,v $
 * Revision 1.1  2008/05/12 01:47:32  arafune
 * The first commitment.
 *
 *
 */

#ifndef ConfigAdmin_cpp
#define ConfigAdmin_cpp

#include <cppunit/ui/text/TestRunner.h>
#include <cppunit/TextOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>
#include <strstream>
#include <rtm/ConfigAdmin.h>

/*!
 * @class ConfigTests class
 * @brief Config test
 */
namespace Config
{
  class ConfigTests
    : public CppUnit::TestFixture
  {
    CPPUNIT_TEST_SUITE(ConfigTests);
    CPPUNIT_TEST(test_constructor);
    CPPUNIT_TEST(test_update);
    CPPUNIT_TEST(test_update_with_illegal_value);
    CPPUNIT_TEST(test_update_with_illegal_default_value);
    CPPUNIT_TEST_SUITE_END();
		
  private:
	
	
  public:
    /*!
     * @brief Constructor
     */
    ConfigTests()
    {
    }
		
    /*!
     * @brief Destructor
     */
    ~ConfigTests()
    {
    }
		
    /*!
     * @brief Test initialization
     */
    virtual void setUp()
    {
    }
		
    /*!
     * @brief Test finalization
     */
    virtual void tearDown()
    { 
    }
		
    /*!
     * @brief 󥹥ȥ饯Υƥ
     */
    void test_constructor()
    {
      std::string name = "name of parameter";
      double value = 0.0;
      std::string default_value = "3.14159";
			
      RTC::Config<double> config(name.c_str(), value, default_value.c_str());
			
      // ꤷ̾ΤꤵƤ뤫
      CPPUNIT_ASSERT_EQUAL(name, std::string(config.name));
			
      // ꤷǥե͡ʤʸɽˤꤵƤ뤫
      CPPUNIT_ASSERT_EQUAL(default_value, std::string(config.default_value));
    }
		
    /*!
     * @brief update()᥽åɤΥƥ
     * 
     * - Хɤѿupdate()ǻꤵ줿ͤ뤫
     */
    void test_update()
    {
      std::string name = "name of parameter";
      double value = 0.0;
      std::string default_value = "3.14159";
			
      RTC::Config<double> config(name.c_str(), value, default_value.c_str());
			
      // update()ϡѿ֤ͤΤޤޤΤϤ
      CPPUNIT_ASSERT_EQUAL(0.0, value);
			
      // update()
      CPPUNIT_ASSERT_EQUAL(true, config.update("1.41421356"));
			
      // Хɤѿupdate()ǻꤵ줿ͤ뤫
      CPPUNIT_ASSERT_EQUAL(1.41421356, value);
    }
		
    /*!
     * @brief update()᥽åɤΥƥ
     * 
     * - ηѴǤʤƤupdate()ƽФտޤɤ꼺Ԥ뤫
     * - Хɤѿͤ󥹥ȥ饯ǻꤷǥեͤ뤫
     */
    void test_update_with_illegal_value()
    {
      std::string name = "name of parameter";
      double value = 0.0;
      std::string default_value = "3.14159";
			
      RTC::Config<double> config(name.c_str(), value, default_value.c_str());
			
      // update()ϡѿ֤ͤΤޤޤΤϤ
      CPPUNIT_ASSERT_EQUAL(0.0, value);
			
      // ưͤѴǤʤƤupdate()ƽФտޤɤ꼺Ԥ뤫
      CPPUNIT_ASSERT_EQUAL(false, config.update("Not float value"));
			
      // Хɤѿͤ󥹥ȥ饯ǻꤷǥեͤ뤫
      CPPUNIT_ASSERT_EQUAL(3.14159, value);
    }
		
    /*!
     * @brief update()᥽åɤΥƥ
     * 
     * - ưͤѴǤʤǥեͤꤵ졢ưͤѴǤʤƤupdate()ƽФ硢
     * Хɤѿͤϡ뤳Ȥʤͤαޤ뤫
     */
    void test_update_with_illegal_default_value()
    {
      std::string name = "name of parameter";
      double value = 0.0;
      std::string illegal_default_value = "Not float value";
			
      RTC::Config<double> config(name.c_str(), value, illegal_default_value.c_str());

      // update()ϡѿ֤ͤΤޤޤΤϤ
      CPPUNIT_ASSERT_EQUAL(0.0, value);
			
      // ưͤѴǤʤǥեͤꤵ졢
      // ưͤѴǤʤƤupdate()ƽФ硢
      // Хɤѿͤϡ뤳Ȥʤͤαޤ뤫
      CPPUNIT_ASSERT_EQUAL(false, config.update("Not float value too"));
      CPPUNIT_ASSERT_EQUAL(0.0, value);
    }
		
  };
}; // namespace Config

/*!
 * @class ConfigAdminTests class
 * @brief ConfigAdmin test
 */
namespace ConfigAdmin
{
  class OnUpdateCallbackMock : public RTC::OnUpdateCallback
  {
  public:
    OnUpdateCallbackMock(void) : result(false) {}
    virtual ~OnUpdateCallbackMock(void){}
    void operator()(const char* config_set)
      {
        // νϤƤФƤ
//      std::cout << "OnUpdateCallbackMock1 config_set=" << config_set << std::endl;
        result = true;
      }
    bool result;
  };

  class OnUpdateParamCallbackMock : public RTC::OnUpdateParamCallback
  {
  public:
    OnUpdateParamCallbackMock(void) : result(false) {}
    virtual ~OnUpdateParamCallbackMock(void){}
    void operator()(const char* config_set, const char* config_param)
      {
//      std::cout << "OnUpdateParamCallbackMock2 config_set=" << config_set << std::endl;
//      std::cout << "OnUpdateParamCallbackMock2 config_param=" << config_param << std::endl;
        result = true;
      }
    bool result;
  };

  class OnSetConfigurationSetCallbackMock : public RTC::OnSetConfigurationSetCallback
  {
  public:
    OnSetConfigurationSetCallbackMock(void) : result(false) {}
    virtual ~OnSetConfigurationSetCallbackMock(void){}
    void operator()(const coil::Properties& config_set)
      {
//      std::cout << "OnSetConfigurationSetCallbackMock3 config_set=" << std::endl << config_set << std::endl;
        result = true;
      }
    bool result;
  };

  class OnAddConfigurationAddCallbackMock : public RTC::OnAddConfigurationAddCallback
  {
  public:
    OnAddConfigurationAddCallbackMock(void) : result(false) {}
    virtual ~OnAddConfigurationAddCallbackMock(void){}
    void operator()(const coil::Properties& config_set)
      {
//      std::cout << "OnAddConfigurationAddCallbackMock4 config_set=" << std::endl << config_set << std::endl;
        result = true;
      }
    bool result;
  };

  class OnRemoveConfigurationSetCallbackMock : public RTC::OnRemoveConfigurationSetCallback
  {
  public:
    OnRemoveConfigurationSetCallbackMock(void) : result(false) {}
    virtual ~OnRemoveConfigurationSetCallbackMock(void){}
    void operator()(const char* config_set)
      {
//      std::cout << "OnRemoveConfigurationSetCallbackMock5 config_set=" << config_set << std::endl;
        result = true;
      }
    bool result;
  };

  class OnActivateSetCallbackMock : public RTC::OnActivateSetCallback
  {
  public:
    OnActivateSetCallbackMock(void) : result(false) {}
    virtual ~OnActivateSetCallbackMock(void){}
    void operator()(const char* config_id)
      {
//      std::cout << "OnActivateSetCallbackMock6 config_id=" << config_id << std::endl;
        result = true;
      }
    bool result;
  };

  // ConfigAdmin Ѿơprotected: ؿ򥪡Сɤ
  class ConfigAdminMock : public RTC::ConfigAdmin
  {
  public:
    ConfigAdminMock(coil::Properties& configsets)
      : RTC::ConfigAdmin(configsets) {}
    virtual ~ConfigAdminMock(void){}

    void onUpdateMock(const char* config_set)
      {
//      std::cout << "ConfigAdmin::onUpdate() 1 before" << std::endl;
        RTC::ConfigAdmin::onUpdate(config_set);
//      std::cout << "ConfigAdmin::onUpdate() 1 after" << std::endl;
      }
    void onUpdateParamMock(const char* config_set, const char* config_param)
      {
//      std::cout << "ConfigAdmin::onUpdateParam() 2 before" << std::endl;
        RTC::ConfigAdmin::onUpdateParam(config_set, config_param);
//      std::cout << "ConfigAdmin::onUpdateParam() 2 after" << std::endl;
      }
    void onSetConfigurationSetMock(const coil::Properties& config_set)
      {
//      std::cout << "ConfigAdmin::onSetConfigurationSet() 3 before" << std::endl;
        RTC::ConfigAdmin::onSetConfigurationSet(config_set);
//      std::cout << "ConfigAdmin::onSetConfigurationSet() 3 after" << std::endl;
      }
    void onAddConfigurationSetMock(const coil::Properties& config_set)
      {
//      std::cout << "ConfigAdmin::onAddConfigurationSet() 4 before" << std::endl;
        RTC::ConfigAdmin::onAddConfigurationSet(config_set);
//      std::cout << "ConfigAdmin::onAddConfigurationSet() 4 after" << std::endl;
      }
    void onRemoveConfigurationSetMock(const char* config_id)
      {
//      std::cout << "ConfigAdmin::onRemoveConfigurationSet() 5 before" << std::endl;
        RTC::ConfigAdmin::onRemoveConfigurationSet(config_id);
//      std::cout << "ConfigAdmin::onRemoveConfigurationSet() 5 after" << std::endl;
      }
    void onActivateSetMock(const char* config_id)
      {
//      std::cout << "ConfigAdmin::onActivateSet() 6 before" << std::endl;
        RTC::ConfigAdmin::onActivateSet(config_id);
//      std::cout << "ConfigAdmin::onActivateSet() 6 after" << std::endl;
      }
  };


  class ConfigAdminTests
    : public CppUnit::TestFixture
  {
    CPPUNIT_TEST_SUITE(ConfigAdminTests);

    CPPUNIT_TEST(test_setOnUpdate);
    CPPUNIT_TEST(test_setOnUpdateParam);
    CPPUNIT_TEST(test_setOnSetConfigurationSet);
    CPPUNIT_TEST(test_setOnAddConfigurationSet);
    CPPUNIT_TEST(test_setOnRemoveConfigurationSet);
    CPPUNIT_TEST(test_setOnActivateSet);
    CPPUNIT_TEST(test_constructor);
    CPPUNIT_TEST(test_bindParameter);
    CPPUNIT_TEST(test_bindParameter_already_exist);
    CPPUNIT_TEST(test_bindParameter_illegal_default_value);
    CPPUNIT_TEST(test_update);
    CPPUNIT_TEST(test_update_by_inexist_configuration_set);
    CPPUNIT_TEST(test_update_with_specified_parameter_name);
    CPPUNIT_TEST(test_update_by_active_configuration_set);
    CPPUNIT_TEST(test_isExist);
    CPPUNIT_TEST(test_isChanged_on_addConfigurationSet);
    CPPUNIT_TEST(test_isChanged_on_removeConfigurationSet);
    CPPUNIT_TEST(test_isChanged_on_activateConfigurationSet);
    CPPUNIT_TEST(test_getActiveId);
    CPPUNIT_TEST(test_haveConfig);
    CPPUNIT_TEST(test_isActive_on_addConfigurationSet);
    CPPUNIT_TEST(test_isActive_on_removeConfigurationSet);
    CPPUNIT_TEST(test_getConfigurationSets);
    CPPUNIT_TEST(test_addConfigurationSet_and_getConfigurationSet);
    CPPUNIT_TEST(test_setConfigurationSetValues);
    CPPUNIT_TEST(test_setConfigurationSetValues_with_inexist_configuration_set);
    CPPUNIT_TEST(test_getActiveConfigurationSet);
    CPPUNIT_TEST(test_removeConfigurationSet);
    CPPUNIT_TEST(test_removeConfigurationSet_with_inexist_configuration_id);

    CPPUNIT_TEST_SUITE_END();
		
  private:
    // helper
    const coil::Properties* getPropertiesBy(
					   const std::string& name,
					   const std::vector<coil::Properties*>& propertiesSet) const
    {
      for (int i = 0; i < (int) propertiesSet.size(); ++i)
	{
	  if (std::string(propertiesSet[i]->getName()) == name)
	    {
	      return propertiesSet[i];
	    }
	}
			
      return 0; // not found
    }
	
  public:
    /*!
     * @brief Constructor
     */
    ConfigAdminTests()
    {
    }
		
    /*!
     * @brief Destructor
     */
    ~ConfigAdminTests()
    {
    }
		
    /*!
     * @brief Test initialization
     */
    virtual void setUp()
    {
    }
		
    /*!
     * @brief Test finalization
     */
    virtual void tearDown()
    { 
    }
		
    /*!
     * @brief 󥹥ȥ饯Υƥ
     * 
     * - ǻꤷե졼󥻥åȤꤵ뤫
     */
    void test_constructor()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
			
      RTC::ConfigAdmin configAdmin(configSet);
			
      // ǻꤷե졼󥻥åȤꤵ뤫
      const coil::Properties& activeConfigSet = configAdmin.getConfigurationSet("config_id");
      CPPUNIT_ASSERT_EQUAL(std::string("value"), activeConfigSet.getProperty("key"));
    }
		
    /*!
     * @brief bindParameter()᥽åɤΥƥ
     * 
     * - bindParameter()ǡѿХɤǤ뤫
     * - ХɤѿϡꤷǥեͤƤ뤫
     */		
    void test_bindParameter()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // bindParameter()ǡѿХɤǤ뤫
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName, var, default_value));
			
      // ХɤѿϡꤷǥեͤƤ뤫
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
    }
		
    /*!
     * @brief bindParameter()᥽åɤΥƥ
     * 
     * - Ʊ̾ΤѿХɤߤơտޤɤ꼺Ԥ뤫
     */
    void test_bindParameter_already_exist()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ޤϡΥХɤԤ
      const char* varName1 = "name";
      double var1 = 0.0;
      const char* default_value1 = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName1, var1, default_value1));
      CPPUNIT_ASSERT_EQUAL(3.14159, var1);
			
      // Ʊ̾ΤѿХɤߤơտޤɤ꼺Ԥ뤫
      const char* varName2 = varName1;
      double var2 = 1.0;
      const char* default_value2 = "1.41421356";
      CPPUNIT_ASSERT_EQUAL(
			   false, configAdmin.bindParameter(varName2, var2, default_value2));
				
      // ХѿͤϹ뤳ȤʤݻƤ뤫
      CPPUNIT_ASSERT_EQUAL(3.14159, var1);
    }
		
    /*!
     * @brief bindParameter()᥽åɤΥƥ
     * 
     * - ΥǡѴǤʤǥեͤꤷơտޤɤ꼺Ԥ뤫
     */
    void test_bindParameter_illegal_default_value()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ΥǡѴǤʤǥեͤꤷơտޤɤ꼺Ԥ뤫
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "Illegal default value";
      CPPUNIT_ASSERT_EQUAL(
			   false, configAdmin.bindParameter(varName, var, default_value));
    }
		
    /*!
     * @brief update()᥽åɤΥƥ
     * 
     * - ե졼󥻥åȤꤷupdate()ƤǥХѿͤ뤫
     */
    void test_update()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȣ
      coil::Properties configSet1("set 1");
      configSet1.setProperty("name", "1.41421356");
			
      // ե졼󥻥åȣ
      coil::Properties configSet2("set 2");
      configSet2.setProperty("name", "1.7320508");
			
      // ĤΥե졼󥻥åȤɲäƤ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // ѿХɤ
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName, var, default_value));
			
      // update()ϡޤѿǥեͤΤޤޤǤ뤳Ȥǧ
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
			
      // ե졼󥻥åȣꤷupdate()ƤǥХѿͤ뤫
      configAdmin.update("set 1");
      CPPUNIT_ASSERT_EQUAL(1.41421356, var);

      // ե졼󥻥åȣꤷupdate()ƤǥХѿͤ뤫
      configAdmin.update("set 2");
      CPPUNIT_ASSERT_EQUAL(1.7320508, var);
    }
		
    /*!
     * @brief update()᥽åɤΥƥ
     * 
     * - ¸ߤʤե졼IDꤷupdate()ƽФˡ
     * Хѿ줺ˡΤޤݻ뤫
     */
    void test_update_by_inexist_configuration_set()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet("set");
      configSet.setProperty("name", "1.41421356");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));
			
      // ѿХɤ
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName, var, default_value));
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
			
      // ¸ߤʤե졼IDꤷupdate()ƽФˡ
      // Хѿ줺ˡΤޤݻ뤫
      configAdmin.update("inexist set");
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
    }
		
    /*!
     * @brief update()᥽åɡ̾λˤΥƥ
     * 
     * - ꤷե졼󥻥åȤλꤷ̾Τѿ뤫
     */
    void test_update_with_specified_parameter_name()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ĤΥѥ᡼ͤޤॳե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      configSet1.setProperty("name 1", "1.41421356");
      configSet1.setProperty("name 2", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));

      coil::Properties configSet2("set 2");
      configSet2.setProperty("name 1", "3.14159");
      configSet2.setProperty("name 2", "2.71828");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // ĤѿХɤ
      const char* varName1 = "name 1";
      double var1 = 0.0;
      const char* default_value1 = "2.23620679";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName1, var1, default_value1));
      CPPUNIT_ASSERT_EQUAL(2.23620679, var1);
			
      const char* varName2 = "name 2";
      double var2 = 0.0;
      const char* default_value2 = "2.4494897";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName2, var2, default_value2));
      CPPUNIT_ASSERT_EQUAL(2.4494897, var2);
			
      // ĤΤѿΤߤ̾λꤷupdate()Ԥꤷѿ뤫
      configAdmin.update("set 2", "name 1");
      CPPUNIT_ASSERT_EQUAL(3.14159, var1);
      CPPUNIT_ASSERT_EQUAL(2.4494897, var2);
    }
		
    /*!
     * @brief update()᥽åɡʥƥ֥ե졼󥻥åȡˤΥƥ
     * 
     * - update()ƽФˤꡢХѿƥ֤ʥե졼󥻥åȤͤǹ뤫
     */
    void test_update_by_active_configuration_set()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      configSet1.setProperty("name", "1.41421356");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));

      coil::Properties configSet2("set 2");
      configSet2.setProperty("name", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // ѿХɤ
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName, var, default_value));
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
			
      // "set 1"Τۤ򥢥ƥ֤ˤ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 1"));
			
      // ƥ֤ˤǤϡޤХѿϹƤʤϤ
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
			
      // update()ƽФˤꡢХѿƥ֤ʥե졼󥻥åȤͤǹ뤫
      configAdmin.update();
      CPPUNIT_ASSERT_EQUAL(1.41421356, var);

      // "set 2"Τۤ򥢥ƥ֤ˤ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 2"));
			
      // ƥ֤ˤǤϡޤХѿϹƤʤϤ
      CPPUNIT_ASSERT_EQUAL(1.41421356, var);

      // update()ƽФˤꡢХѿƥ֤ʥե졼󥻥åȤͤǹ뤫
      configAdmin.update();
      CPPUNIT_ASSERT_EQUAL(1.7320508, var);
    }
		
    /*!
     * @brief isExist()᥽åɤΥƥ
     * 
     * - Хɤѿ̾ΤisExist()ƽФͤ뤫
     * - ХɤƤʤ̾ΤisExist()ƽФͤ뤫
     */
    void test_isExist()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ѿХɤ
      const char* varName = "name";
      double var = 0.0;
      const char* default_value = "3.14159";
      CPPUNIT_ASSERT_EQUAL(
			   true, configAdmin.bindParameter(varName, var, default_value));
      CPPUNIT_ASSERT_EQUAL(3.14159, var);
			
      // Хɤѿ̾ΤisExist()ƽФͤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.isExist("name"));
			
      // ХɤƤʤ̾ΤisExist()ƽФͤ뤫
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.isExist("inexist name"));
    }
		
    /*!
     * @brief isChanged()᥽åɤΥƥ
     * 
     * - addConfigurationSet()ƽиϡisChanged()ϿͤȤʤ뤫
     */
    void test_isChanged_on_addConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // update()ƽФƥХѿƱԤisChanged()Ȥʤ֤ˤ
      configAdmin.update();
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.isChanged());
			
      // addConfigurationSet()ƽФ
      coil::Properties configSet("id");
      configSet.setProperty("key", "value");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));

      // addConfigurationSet()ƽиϡisChanged()ϿͤȤʤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.isChanged());
    }
		
    /*!
     * @brief isChanged()᥽åɤΥƥ
     * 
     * - removeConfigurationSet()ƽиϡisChanged()ϿͤȤʤ뤫
     */
    void test_isChanged_on_removeConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // addConfigurationSet()ƽФ
      coil::Properties configSet("id");
      configSet.setProperty("key", "value");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));

      // update()ƽФƥХѿƱԤisChanged()Ȥʤ֤ˤ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("id"));
      configAdmin.update();
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.isChanged());
			
      // removeConfigurationSet()θƽиϡisChanged()ϿͤȤʤ뤫
      //CPPUNIT_ASSERT_EQUAL(true, configAdmin.removeConfigurationSet("id"));
      //CPPUNIT_ASSERT_EQUAL(true, configAdmin.isChanged());
    }
		
    /*!
     * @brief isChanged()᥽åɤΥƥ
     * 
     * - activateConfigurationSet()ƽиϡisChanged()ϿͤȤʤ뤫
     */
    void test_isChanged_on_activateConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));

      coil::Properties configSet2("set 2");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // "set 1"Τۤ򥢥ƥ֤ˤ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 1"));
			
      // update()ԤisChanged()ͤξ֤ˤƤ
      configAdmin.update();
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.isChanged());

      // "set 2"Τۤ򥢥ƥ֤ˤȡisChanged()ͤȤʤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 2"));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.isChanged());
    }
		
    void test_isChanged_on_setConfigurationSetValues()
    {
      // TODO 뤳
    }
		
    /*!
     * @brief getActiveId()᥽åɤΥƥ
     * 
     * - ƥֲե졼󥻥åȤIDǤ뤫
     */
    void test_getActiveId()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));

      coil::Properties configSet2("set 2");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // ֤ǤϡƥID"default"ΤϤ
      CPPUNIT_ASSERT_EQUAL(std::string("default"), std::string(configAdmin.getActiveId()));
			
      // "set 1"򥢥ƥ֤ˤ塢տޤɤΥƥIDǤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 1"));
      CPPUNIT_ASSERT_EQUAL(std::string("set 1"), std::string(configAdmin.getActiveId()));
			
      // "set 2"򥢥ƥ֤ˤ塢տޤɤΥƥIDǤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 2"));
      CPPUNIT_ASSERT_EQUAL(std::string("set 2"), std::string(configAdmin.getActiveId()));
    }
		
    /*!
     * @brief haveConfig()᥽åɤΥƥ
     * 
     * - ¸ߤ륳ե졼󥻥åIDꤷˡhaveConfig()ͤ뤫
     * - ¸ߤʤե졼󥻥åIDꤷˡhaveConfig()ͤ뤫
     */
    void test_haveConfig()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet1("id");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
			
      // ¸ߤ륳ե졼󥻥åIDꤷˡhaveConfig()ͤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.haveConfig("id"));
			
      // ¸ߤʤե졼󥻥åIDꤷˡhaveConfig()ͤ뤫
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.haveConfig("inexist id"));
    }
		
    /*!
     * @brief isActive()᥽åɤΥƥ
     * 
     * - addConfigurationSet()ƽиϡisActive()ͤ뤫
     * - activateConfigurationSet()ǥƥֲϡisActive()ͤ뤫
     */
    void test_isActive_on_addConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // addConfigurationSet()ƽиϡisActive()ͤ뤫
      coil::Properties configSet("id");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.isActive());
			
      // activateConfigurationSet()ǥƥֲϡisActive()ͤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("id"));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.isActive());
    }
		
    /*!
     * @brief isActive()᥽åɤΥƥ
     * 
     * - removeConfigurationSet()ƽиϡisActive()ͤ뤫
     */
    void test_isActive_on_removeConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);

      // ե졼󥻥åȤɲäƥƥֲƤ
      coil::Properties configSet("id");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("id"));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.isActive());
			
      // ϿƤ륳ե졼󥻥åȤremoveConfigurationSet()ǲ硢
      // isActive()ͤ뤫
      //CPPUNIT_ASSERT_EQUAL(true, configAdmin.removeConfigurationSet("id"));
      //CPPUNIT_ASSERT_EQUAL(false, configAdmin.isActive());
    }
		
    /*!
     * @brief isActive()᥽åɤΥƥ
     * 
     * - activateConfigurationSet()ǥƥֲϡisActive()ͤ뤫
     */
    void test_isActive_on_activateConfigurationSet()
    {
      // test_isActive_on_addConfigurationSet()Ƿ
    }
		
    void test_isActive_on_setConfigurationSetValues()
    {
      // TODO 뤳
    }
		
    /*!
     * @brief getConfigurationSets()᥽åɤΥƥ
     * 
     * - ϿƤƤΥե졼󥻥åȤǤ뤫
     */
    void test_getConfigurationSets()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ĤΥѥ᡼ͤޤॳե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      configSet1.setProperty("name 1", "1.41421356");
      configSet1.setProperty("name 2", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
			
      coil::Properties configSet2("set 2");
      configSet2.setProperty("name 1", "3.14159");
      configSet2.setProperty("name 2", "2.71828");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // getConfigurationSets()ϿƤƤΥե졼󥻥åȤ
      // ϿΤȰפƤ뤳Ȥǧ
      std::vector<coil::Properties*> expectedConfigSets;
      expectedConfigSets.push_back(&configSet1);
      expectedConfigSets.push_back(&configSet2);
			
      const std::vector<coil::Properties*>& configSets
	= configAdmin.getConfigurationSets();
			
      CPPUNIT_ASSERT_EQUAL(
			   std::string("1.41421356"),
			   getPropertiesBy("set 1", configSets)->getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(
			   std::string("1.7320508"),
			   getPropertiesBy("set 1", configSets)->getProperty("name 2"));

      CPPUNIT_ASSERT_EQUAL(
			   std::string("3.14159"),
			   getPropertiesBy("set 2", configSets)->getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(
			   std::string("2.71828"),
			   getPropertiesBy("set 2", configSets)->getProperty("name 2"));
    }
		
    /*!
     * addConfigurationSet()᥽åɤgetConfigurationSet()᥽åɤΥƥ
     * 
     * - addConfigurationSet()ǡե졼󥻥åȤɲäǤ뤫
     * - getConfigurationSet()ǡɲäե졼󥻥åȤǤ뤫
     */
    void test_addConfigurationSet_and_getConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // addConfigurationSet()ǡե졼󥻥åȤɲäǤ뤫
      coil::Properties configSet("id");
      configSet.setProperty("key", "value");
      configAdmin.addConfigurationSet(configSet);
			
      // getConfigurationSet()ǡɲäե졼󥻥åȤǤ뤫
      const coil::Properties& configSetRet = configAdmin.getConfigurationSet("id");
      CPPUNIT_ASSERT_EQUAL(std::string("value"), configSetRet.getProperty("key"));
    }
		
    /*!
     * setConfigurationSetValues()᥽åɤΥƥ
     * 
     * - ꤷץѥƥꤷե졼󥻥åȤɲä뤫
     */
    void test_setConfigurationSetValues()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      coil::Properties configSet1("id");
      configSet1.setProperty("name 1", "1.41421356");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
			
      // ϿѤߤΥե졼󥻥åȤФơץѥƥɲä
      coil::Properties configSet2("id");
      configSet2.setProperty("name 2", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.setConfigurationSetValues(configSet2));
			
      // ե졼󥻥åȤơץѥƥտޤɤɲäƤ뤳Ȥǧ
      const coil::Properties& configSetRet = configAdmin.getConfigurationSet("id");
      CPPUNIT_ASSERT_EQUAL(std::string("1.41421356"), configSetRet.getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(std::string("1.7320508"), configSetRet.getProperty("name 2"));
    }
		
    /*!
     * @brief setConfigurationSetValues()᥽åɤΥƥ
     * 
     * - ¸ߤʤե졼󥻥åȤФƥץѥƥɲäߤơտޤɤ꼺Ԥ뤫
     * - ԸˡϿѤߤΥե졼󥻥åȤƶƤʤ
     */
    void test_setConfigurationSetValues_with_inexist_configuration_set()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      coil::Properties configSet1("id");
      configSet1.setProperty("name 1", "1.41421356");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
			
      // ¸ߤʤե졼󥻥åȤФƥץѥƥɲäߤơտޤɤ꼺Ԥ뤫
      coil::Properties configSet2("inexist id");
      configSet2.setProperty("name 2", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.setConfigurationSetValues(configSet2));
			
      // ԸˡϿѤߤΥե졼󥻥åȤƶƤʤ
      const coil::Properties& configSetRet = configAdmin.getConfigurationSet("id");
      CPPUNIT_ASSERT_EQUAL(std::string("1.41421356"), configSetRet.getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(std::string(""), configSetRet.getProperty("name 2"));
    }
		
    /*!
     * @brief getActiveConfigurationSet()᥽åɤΥƥ
     * 
     * - ƥ֥ե졼󥻥åȤǤ뤫
     */
    void test_getActiveConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet1("set 1");
      configSet1.setProperty("name 1", "1.41421356");
      configSet1.setProperty("name 2", "1.7320508");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet1));
			
      coil::Properties configSet2("set 2");
      configSet2.setProperty("name 1", "3.14159");
      configSet2.setProperty("name 2", "2.71828");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet2));
			
      // "set 1"򥢥ƥֲ塢ƥ֥ե졼󥻥åȤȤǤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 1"));
      const coil::Properties& activeConfigSet1 = configAdmin.getActiveConfigurationSet();
      CPPUNIT_ASSERT_EQUAL(std::string("1.41421356"), activeConfigSet1.getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(std::string("1.7320508"), activeConfigSet1.getProperty("name 2"));

      // "set 2"򥢥ƥֲ塢ƥ֥ե졼󥻥åȤȤǤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.activateConfigurationSet("set 2"));
      const coil::Properties& activeConfigSet2 = configAdmin.getActiveConfigurationSet();
      CPPUNIT_ASSERT_EQUAL(std::string("3.14159"), activeConfigSet2.getProperty("name 1"));
      CPPUNIT_ASSERT_EQUAL(std::string("2.71828"), activeConfigSet2.getProperty("name 2"));
    }
		
    /*!
     * @brief removeConfigurationSet()Υƥ
     * 
     * - ϿƤ륳ե졼󥻥åȤϿǤ뤫
     */
    void test_removeConfigurationSet()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet("id");
      configSet.setProperty("key", "value");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.haveConfig("id"));
			
      // äϿե졼󥻥åȤϿ
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.removeConfigurationSet("id"));
			
      // ե졼󥻥åȤ¸ߤʤȤǧ
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.haveConfig("id"));
    }
		
    /*!
     * @brief removeConfigurationSet()Υƥ
     * 
     * - ¸ߤʤե졼󥻥åIDꤷˡտޤɤ˼Ԥ뤫
     */
    void test_removeConfigurationSet_with_inexist_configuration_id()
    {
      coil::Properties nullProp;
      RTC::ConfigAdmin configAdmin(nullProp);
			
      // ե졼󥻥åȤɲäƤ
      coil::Properties configSet("id");
      configSet.setProperty("key", "value");
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.addConfigurationSet(configSet));
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.haveConfig("id"));
			
      // ¸ߤʤե졼󥻥åIDꤷˡտޤɤ˼Ԥ뤫
      CPPUNIT_ASSERT_EQUAL(false, configAdmin.removeConfigurationSet("inexist id"));
			
      // ϿƤ륳ե졼󥻥åȤϡɤ¸ߤƤ뤫
      CPPUNIT_ASSERT_EQUAL(true, configAdmin.haveConfig("id"));
    }
		
    void test_activateConfigurationSet()
    {
      // test_getActiveConfigurationSet()Ƿͤ
    }
		
    /*!
     * @brief setOnUpdate()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnUpdate()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);

      OnUpdateCallbackMock* cdm1 = new OnUpdateCallbackMock();
      OnUpdateCallbackMock* cdm2 = new OnUpdateCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnUpdate(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onUpdateMock("config_id");
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnUpdate(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onUpdateMock("config_id");
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		
    /*!
     * @brief setOnUpdateParam()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnUpdateParam()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);

      OnUpdateParamCallbackMock* cdm1 = new OnUpdateParamCallbackMock();
      OnUpdateParamCallbackMock* cdm2 = new OnUpdateParamCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnUpdateParam(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onUpdateParamMock("config_id", "param1");
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnUpdateParam(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onUpdateParamMock("config_id", "param2");
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		
    /*!
     * @brief setOnSetConfigurationSet()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnSetConfigurationSet()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);
      coil::Properties configSet2("config_id2");
      configSet2.setProperty("config_id2.key", "value2");

      OnSetConfigurationSetCallbackMock* cdm1 = new OnSetConfigurationSetCallbackMock();
      OnSetConfigurationSetCallbackMock* cdm2 = new OnSetConfigurationSetCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnSetConfigurationSet(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onSetConfigurationSetMock(configSet);
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnSetConfigurationSet(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onSetConfigurationSetMock(configSet2);
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		
    /*!
     * @brief setOnAddConfigurationSet()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnAddConfigurationSet()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);
      coil::Properties configSet2("config_id2");
      configSet2.setProperty("config_id2.key", "value2");

      OnAddConfigurationAddCallbackMock* cdm1 = new OnAddConfigurationAddCallbackMock();
      OnAddConfigurationAddCallbackMock* cdm2 = new OnAddConfigurationAddCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnAddConfigurationSet(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onAddConfigurationSetMock(configSet);
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnAddConfigurationSet(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onAddConfigurationSetMock(configSet2);
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		
    /*!
     * @brief setOnRemoveConfigurationSet()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnRemoveConfigurationSet()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);

      OnRemoveConfigurationSetCallbackMock* cdm1 = new OnRemoveConfigurationSetCallbackMock();
      OnRemoveConfigurationSetCallbackMock* cdm2 = new OnRemoveConfigurationSetCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnRemoveConfigurationSet(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onRemoveConfigurationSetMock("config_id");
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnRemoveConfigurationSet(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onRemoveConfigurationSetMock("config_id2");
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		
    /*!
     * @brief setOnActivateSet()᥽åɤΥƥ
     * 
     * - 
     */
    void test_setOnActivateSet()
    {
      coil::Properties configSet("config_id");
      configSet.setProperty("config_id.key", "value");
      ConfigAdminMock configAdmin(configSet);

      OnActivateSetCallbackMock* cdm1 = new OnActivateSetCallbackMock();
      OnActivateSetCallbackMock* cdm2 = new OnActivateSetCallbackMock();

      // 1ܤsetǥСѿ
      configAdmin.setOnActivateSet(cdm1);
      CPPUNIT_ASSERT(!cdm1->result);
      configAdmin.onActivateSetMock("config_id");
      CPPUNIT_ASSERT(cdm1->result);

      // 2ܤsetdeleteƤФ
      configAdmin.setOnActivateSet(cdm2);
      CPPUNIT_ASSERT(!cdm2->result);
      configAdmin.onActivateSetMock("config_id2");
      CPPUNIT_ASSERT(cdm2->result);

      // delete cdm1; delete cdm2 פǤ
      // ConfigAdmin::setOn*()ǡdelete m_* ¹ԤƤޤ
    }
		

  };
}; // namespace ConfigAdmin


/*
 * Register test suite
 */
CPPUNIT_TEST_SUITE_REGISTRATION(Config::ConfigTests);
CPPUNIT_TEST_SUITE_REGISTRATION(ConfigAdmin::ConfigAdminTests);

#ifdef LOCAL_MAIN
int main(int argc, char* argv[])
{

  FORMAT format = TEXT_OUT;
  int target = 0;
  std::string xsl;
  std::string ns;
  std::string fname;
  std::ofstream ofs;

  int i(1);
  while (i < argc)
    {
      std::string arg(argv[i]);
      std::string next_arg;
      if (i + 1 < argc) next_arg = argv[i + 1];
      else              next_arg = "";

      if (arg == "--text") { format = TEXT_OUT; break; }
      if (arg == "--xml")
	{
	  if (next_arg == "")
	    {
	      fname = argv[0];
	      fname += ".xml";
	    }
	  else
	    {
	      fname = next_arg;
	    }
	  format = XML_OUT;
	  ofs.open(fname.c_str());
	}
      if ( arg == "--compiler"  ) { format = COMPILER_OUT; break; }
      if ( arg == "--cerr"      ) { target = 1; break; }
      if ( arg == "--xsl"       )
	{
	  if (next_arg == "") xsl = "default.xsl"; 
	  else                xsl = next_arg;
	}
      if ( arg == "--namespace" )
	{
	  if (next_arg == "")
	    {
	      std::cerr << "no namespace specified" << std::endl;
	      exit(1); 
	    }
	  else
	    {
	      xsl = next_arg;
	    }
	}
      ++i;
    }
  CppUnit::TextUi::TestRunner runner;
  if ( ns.empty() )
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry().makeTest());
  else
    runner.addTest(CppUnit::TestFactoryRegistry::getRegistry(ns).makeTest());
  CppUnit::Outputter* outputter = 0;
  std::ostream* stream = target ? &std::cerr : &std::cout;
  switch ( format )
    {
    case TEXT_OUT :
      outputter = new CppUnit::TextOutputter(&runner.result(),*stream);
      break;
    case XML_OUT :
      std::cout << "XML_OUT" << std::endl;
      outputter = new CppUnit::XmlOutputter(&runner.result(),
					    ofs, "shift_jis");
      static_cast<CppUnit::XmlOutputter*>(outputter)->setStyleSheet(xsl);
      break;
    case COMPILER_OUT :
      outputter = new CppUnit::CompilerOutputter(&runner.result(),*stream);
      break;
    }
  runner.setOutputter(outputter);
  runner.run();
  return 0; // runner.run() ? 0 : 1;
}
#endif // MAIN
#endif // ConfigAdmin_cpp
