#!/usr/bin/env python

'''
    Copyright (C) 2014 Parrot SA

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the 
      distribution.
    * Neither the name of Parrot nor the names
      of its contributors may be used to endorse or promote products
      derived from this software without specific prior written
      permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
    FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
    COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
    OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
    AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
    OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
    OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
'''

import sys
import os
import re
import arsdkparser

MYDIR=os.path.abspath(os.path.dirname(__file__))
LIBARCONTROLLER_DIR=os.path.realpath(os.path.join(MYDIR, ".."))
PACKAGES_DIR=os.path.realpath(os.path.join(MYDIR, "../.."))
sys.path.append('%(PACKAGES_DIR)s/ARSDKBuildUtils/Utils/Python' % locals())
sys.path.append('%(PACKAGES_DIR)s/libARCommands/Tools' % locals())

from ARFuncs import *
from libARCommandsgen import *
from ARControllerUtils import *
from arsdkparser import *

DEVICE_CONTROLLER_FILE_NAME = 'deviceControllers.xml'
DEVICE_CONTROLLER_FILE = PACKAGES_DIR+'/libARController/Xml/'+DEVICE_CONTROLLER_FILE_NAME

bref='Device controller allow to drive a device.'

CTRL_DEVICE_H_NAME = 'ARCONTROLLER_Device.h'
CTRL_DEVICE_PRIV_H_NAME = 'ARCONTROLLER_Device.h'
CTRL_DEVICE_C_NAME = 'ARCONTROLLER_Device.c'
CTRL_DEVICE_JAVA_NAME = 'ARDeviceController.java'
CTRL_DEVICE_JNI_C_NAME = 'ARCONTROLLER_JNI_Device.c'

def generateDeviceControllers (ctx, SRC_DIR, INC_DIR):
    
    deviceControllers = parseDeviceControllersXml (DEVICE_CONTROLLER_FILE, ctx)
    
    #check deviceController list
    if not deviceControllers:
        exit (1)

    ARPrint ('generateDeviceControllers ...')

    #########################################
    # Write Device controller header file   #
    #########################################

    includeDefine = '_' + MODULE_DEVICE + '_H_'
    
    className = ARTypeName (MODULE_ARCONTROLLER, 'device', '')
    classPrivateName = ARTypeName (MODULE_ARCONTROLLER, 'device', 'private')

    headerFileName = CTRL_DEVICE_H_NAME
    filepath = INC_DIR + headerFileName
    hfile = open (filepath, 'w')

    hfile.write ('/**********************************************************\n')
    hfile.write (' *            AUTOGENERATED FILE                          *\n')
    hfile.write (' *             DO NOT MODIFY IT                           *\n')
    hfile.write (' *                                                        *\n')
    hfile.write (' * To add new commands :                                  *\n')
    hfile.write (' *  - Modify ../Xml/commands.xml file                     *\n')
    hfile.write (' *  - Re-run generateDeviceControllers.py script          *\n')
    hfile.write (' *                                                        *\n')
    hfile.write (' **********************************************************/\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write ('* @file '+headerFileName+'\n')
    hfile.write ('* @brief '+bref+'\n')
    hfile.write ('*/\n')
    hfile.write ('\n')

    hfile.write ('#ifndef '+includeDefine+'\n')
    hfile.write ('#define '+includeDefine+'\n')
    hfile.write ('\n')

    hfile.write ('#include <stdlib.h>\n')
    hfile.write ('\n')
    hfile.write ('#include <libARSAL/ARSAL_Print.h>\n')
    hfile.write ('#include <libARSAL/ARSAL_Mutex.h>\n')
    hfile.write ('#include <uthash/uthash.h>\n')
    hfile.write ('\n')
    hfile.write ('#include <libARController/ARCONTROLLER_Error.h>\n')
    hfile.write ('#include <libARController/ARCONTROLLER_Dictionary.h>\n')
    hfile.write ('#include <libARController/ARCONTROLLER_Feature.h>\n')
    hfile.write ('#include <libARController/ARCONTROLLER_Stream.h>\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * Enum characterizing the states of the device controller\n')
    hfile.write (' */\n')
    hfile.write ('typedef enum\n')
    hfile.write ('{\n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_STOPPED = 0, /**< device controller is stopped */\n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_STARTING, /**< device controller is starting */\n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_RUNNING, /**< device controller is running */\n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_PAUSED, /**< device controller is paused */\n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_STOPPING, /**< device controller is stopping */\n')
    hfile.write ('    \n')
    hfile.write ('    ARCONTROLLER_DEVICE_STATE_MAX /**< Max of the enumeration */\n')
    hfile.write ('}eARCONTROLLER_DEVICE_STATE;\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Callback used when the state of the device Controller is changed.\n')
    hfile.write (' * @param[in] newState The new state of the Device Contoller\n')
    hfile.write (' * @param[in] error Error causing this new state.\n')
    hfile.write (' * @param[in] customData Custom Data set by the register\n')
    hfile.write (' */\n')
    hfile.write ('typedef void (*'+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+') (eARCONTROLLER_DEVICE_STATE newState, eARCONTROLLER_ERROR error, void *customData);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Callback used when the extension state of the device Controller is changed.\n')
    hfile.write (' * @param[in] newState The new state of the extension of the Device Contoller\n')
    hfile.write (' * @param[in] product the type of the extension.\n')
    hfile.write (' * @param[in] name the name of the extension.\n')
    hfile.write (' * @param[in] error Error causing this new state.\n')
    hfile.write (' * @param[in] customData Custom Data set by the register\n')
    hfile.write (' */\n')
    hfile.write ('typedef void (*'+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+') (eARCONTROLLER_DEVICE_STATE newState, eARDISCOVERY_PRODUCT product, const char *name, eARCONTROLLER_ERROR error, void *customData);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Private part of the Device controller.\n')
    hfile.write (' */\n')
    hfile.write ('typedef struct '+classPrivateName+' '+classPrivateName+';\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Device controller allow to drive a device.\n')
    hfile.write (' */\n')
    hfile.write ('typedef struct\n')
    hfile.write ('{\n')
    for feature in ctx.features:
        hfile.write ('    '+ARTypeName (MODULE_FEATURE, get_ftr_old_name(feature), '')+' *'+ARUncapitalize(get_ftr_old_name(feature))+'; /**< */\n')
    hfile.write ('    '+classPrivateName+' *privatePart; /**< private part of the deviceController */\n')
    hfile.write ('}'+className+';\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Create a new Device Controller\n')
    hfile.write (' * @warning This function allocate memory\n')
    hfile.write (' * @post ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Delete')+'() must be called to delete the Device Controller and free the memory allocated.\n')
    hfile.write (' * @param[in] discoveryDevice The device to drive ; must be not NULL.\n')
    hfile.write (' * @param[out] error executing error.\n')
    hfile.write (' * @return the new Device Controller\n')
    hfile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Delete')+'\n')
    hfile.write (' */\n')
    hfile.write (''+className+' *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'New')+' (ARDISCOVERY_Device_t *discoveryDevice, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Delete the Device Controller\n')
    hfile.write (' * @warning This function free memory\n')
    hfile.write (' * @param device The device controller to delete\n')
    hfile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'New')+'\n')
    hfile.write (' */\n')
    hfile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Delete')+' ('+className+' **deviceController);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Start the Device Controller.\n')
    hfile.write (' * @post ARCONTROLLER_Device_Stop() must be called to stop the Device Controller.\n')
    hfile.write (' * @param deviceController The device controller to start.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' * @see ARCONTROLLER_Device_Stop\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Start')+' ('+className+' *deviceController);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Stop the Device Controller.\n')
    hfile.write (' * @param deviceController The device controller to Stop.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' * @see ARCONTROLLER_Device_Start\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Stop')+' ('+className+' *deviceController);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Set callback to receive the video stream.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param decoderConfigCallback callback to configure the stream decoder.\n')
    hfile.write (' * @param receiveFrameCallback The callback when a frame is received.\n')
    hfile.write (' * @param timeoutFrameCallback The callback when timeout on receive.\n')
    hfile.write (' * @param[in] customData custom data given as parameter to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetVideoStreamCallbacks')+' ('+className+' *deviceController, ARCONTROLLER_Stream_DecoderConfigCallback_t decoderConfigCallback, ARCONTROLLER_Stream_DidReceiveFrameCallback_t receiveFrameCallback, ARCONTROLLER_Stream_TimeoutFrameCallback_t timeoutFrameCallback, void *customData);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Set video stream compliant with the mp4 format.\n')
    hfile.write (' * @note Must be set for decoding H264 stream by the iOS hardware decoder.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param isMP4Compliant 1 if the video stream must be compliant with mp4 format ; otherwide 0.\n')
    hfile.write (' * @return Executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ARCONTROLLER_Device_SetVideoStreamMP4Compliant (ARCONTROLLER_Device_t *deviceController, int isMP4Compliant);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Set callback to receive the audio stream.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param decoderConfigCallback callback to configure the stream decoder.\n')
    hfile.write (' * @param receiveFrameCallback The callback when a frame is received.\n')
    hfile.write (' * @param timeoutFrameCallback The callback when timeout on receive.\n')
    hfile.write (' * @param[in] customData custom data given as parameter to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetAudioStreamCallbacks')+' ('+className+' *deviceController, ARCONTROLLER_Stream_DecoderConfigCallback_t decoderConfigCallback, ARCONTROLLER_Stream_DidReceiveFrameCallback_t receiveFrameCallback, ARCONTROLLER_Stream_TimeoutFrameCallback_t timeoutFrameCallback, void *customData);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Add callback to be informed when a commands is received.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param commandReceivedCallback The callback when a commands is received.\n')
    hfile.write (' * @param[in] customData custom data given as parameter to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddCommandReceivedCallback')+' ('+className+' *deviceController, ARCONTROLLER_DICTIONARY_CALLBACK_t commandReceivedCallback, void *customData);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Remove callback of command received information.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param commandReceivedCallback The callback to remove.\n')
    hfile.write (' * @param[in] customData custom data associated to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveCommandReceivedCallback')+' ('+className+' *deviceController, ARCONTROLLER_DICTIONARY_CALLBACK_t commandReceivedCallback, void *customData);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Add callback of Device Controller state changed information.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param StateChangedCallback The callback to add.\n')
    hfile.write (' * @param[in] customData custom data given as parameter to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' stateChangedCallback, void *customData);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Add callback of Device Controller extension state changed information.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param extensionStateChangedCallback The callback to add.\n')
    hfile.write (' * @param[in] customData custom data given as parameter to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddExtensionStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' extensionStateChangedCallback, void *customData);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Remove callback of Device Controller state changed information.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param StateChangedCallback The callback to remove.\n')
    hfile.write (' * @param[in] customData custom data associated to the callback.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' stateChangedCallback, void *customData);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Send audio stream frame through the network.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param[in] data The data to send.\n')
    hfile.write (' * @param[in] dataSize The data size.\n')
    hfile.write (' * @return executing error.\n')
    hfile.write (' *\n')
    hfile.write (' * @note Data format expected is PCM16LE with sample rate at 8000 hz; mono channel.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SendStreamFrame')+' ('+className+' *deviceController, uint8_t *data, int dataSize);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Get Element of a command received.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param[in] commandKey The key of the command.\n')
    hfile.write (' * @param[out] error Executing error.\n')
    hfile.write (' * @return Element dictionary of the command\n')
    hfile.write (' */\n')
    hfile.write ('ARCONTROLLER_DICTIONARY_ELEMENT_t *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetCommandElements')+' ('+className+' *deviceController, '+defineNotificationDef()+' commandKey, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Get the Device Controller state.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return The state of the Device Controller.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_DEVICE_STATE ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetState')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Get the Device Controller extension state.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return The extension state of the Device Controller.\n')
    hfile.write (' */\n')
    hfile.write ('eARCONTROLLER_DEVICE_STATE ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionState')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Get the Device Controller extension name.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @param[out] buffer The buffer to receive the name.\n')
    hfile.write (' * @param[in] bufferSize The allocated size of the given buffer.\n')
    hfile.write (' */\n')
    hfile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionName')+' ('+className+' *deviceController, char *buffer, int bufferSize, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')

    hfile.write ('/**\n')
    hfile.write (' * @brief Get the Device Controller extension product type.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return The product of the extension of the Device Controller.\n')
    hfile.write (' */\n')
    hfile.write ('eARDISCOVERY_PRODUCT ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionProduct')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Check if the device controller has outupt video stream.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return 1 if the device controller can recieved video stream, otherwide 0.\n')
    hfile.write (' */\n')
    hfile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasOutputVideoStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Check if the device controller has output audio stream.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return 1 if the device controller can recieved audio stream, otherwide 0.\n')
    hfile.write (' */\n')
    hfile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasOutputAudioStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('/**\n')
    hfile.write (' * @brief Check if the device controller has input audio stream.\n')
    hfile.write (' * @param deviceController The device controller.\n')
    hfile.write (' * @return 1 if the device controller can send audio stream, otherwide 0.\n')
    hfile.write (' */\n')
    hfile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasInputAudioStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error);\n')
    hfile.write ('\n')
    
    hfile.write ('#endif /* '+includeDefine+' */\n')
    hfile.write ('\n')
    hfile.write ('// END GENERATED CODE\n')
    hfile.close ()
    
    #################################################
    # Write Device controller private header file   #
    #################################################

    includeDefine = '_' + MODULE_DEVICE + '_PRIVATE_H_'

    headerPrivateFileName = CTRL_DEVICE_PRIV_H_NAME
    filepath = SRC_DIR + headerPrivateFileName
    hPrivFile = open (filepath, 'w')

    hPrivFile.write ('/**********************************************************\n')
    hPrivFile.write (' *            AUTOGENERATED FILE                          *\n')
    hPrivFile.write (' *             DO NOT MODIFY IT                           *\n')
    hPrivFile.write (' *                                                        *\n')
    hPrivFile.write (' * To add new commands :                                  *\n')
    hPrivFile.write (' *  - Modify ../Xml/commands.xml file                     *\n')
    hPrivFile.write (' *  - Re-run generateDeviceControllers.py script          *\n')
    hPrivFile.write (' *                                                        *\n')
    hPrivFile.write (' **********************************************************/\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write ('* @file '+headerPrivateFileName+'\n')
    hPrivFile.write ('* @brief '+bref+'\n')
    hPrivFile.write ('*/\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('#ifndef '+includeDefine+'\n')
    hPrivFile.write ('#define '+includeDefine+'\n')
    hPrivFile.write ('\n')
    hPrivFile.write ('#include <libARSAL/ARSAL.h>\n')
    hPrivFile.write ('#include <libARController/ARCONTROLLER_DICTIONARY_Key.h>\n')
    hPrivFile.write ('#include <libARCommands/ARCommands.h>\n')
    hPrivFile.write ('#include <libARController/ARCONTROLLER_Feature.h>\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('#define '+ARMacroName (MODULE_ARCONTROLLER, 'Device', 'DEFAULT_LOOPER_CMD_BUFFER_SIZE')+' 1024\n')
    hPrivFile.write ('\n')
    
    #TODO add commentary
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief \n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('typedef struct '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+';\n')

    hPrivFile.write ('\n')
    hPrivFile.write ('struct '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+'\n')
    hPrivFile.write ('{\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' callback; /**< callback used when the state is changed. */\n')
    hPrivFile.write ('    void *customData;  /**< custom data given as parameter to the callback. */\n')
    hPrivFile.write ('    \n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *next;\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *prev;\n')
    hPrivFile.write ('};\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('typedef struct '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+';\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('struct '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+'\n')
    hPrivFile.write ('{\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' callback; /**< callback used when the extension state is changed. */\n')
    hPrivFile.write ('    void *customData;  /**< custom data given as parameter to the callback. */\n')
    hPrivFile.write ('    \n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *next;\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *prev;\n')
    hPrivFile.write ('};\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Device controller allow to drive a device.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('struct '+classPrivateName+'\n')
    hPrivFile.write ('{\n')
    hPrivFile.write ('    ARDISCOVERY_Device_t *discoveryDevice; /**< the device to drive */\n')
    hPrivFile.write ('    ARDISCOVERY_NetworkConfiguration_t networkConfiguration; /**< the networkConfiguration of the device*/\n')
    hPrivFile.write ('    ARCONTROLLER_Network_t *networkController; /**< the networkController to used to send and receive  */\n')
    hPrivFile.write ('    ARSAL_Sem_t initSem; /**< semaphore used for the initialisation */\n')
    hPrivFile.write ('    ARSAL_Mutex_t mutex; /**< Mutex for multithreading */\n')
    hPrivFile.write ('    eARCONTROLLER_DEVICE_STATE state; /**< state of the deviceController*/\n')
    hPrivFile.write ('    ARSAL_Thread_t controllerLooperThread; /**< looper threads used to send data in loop */\n')
    hPrivFile.write ('    ARCONTROLLER_DICTIONARY_CALLBAK_LIST_ELEMENT_t *commandCallbacks; /**< Array storing callbacks to use when a command is received. */\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+ ' *stateChangedCallbacks; /**< Array storing callbacks to use when a state is changed. */\n')
    hPrivFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+ ' *extensionStateChangedCallbacks; /**< Array storing callbacks to use when a state of the extension is changed. */\n')
    hPrivFile.write ('    int startCancelled; /**< 1 if the start is canceled*/\n')
    hPrivFile.write ('    //video part\n')
    hPrivFile.write ('    int hasVideo; /**< 0 if the device has not Video stream ; otherwide 1 */\n')
    hPrivFile.write ('    int videoIsMP4Compliant; /**< 1 if the video is mp4 fomat compliant ; otherwide 0 */\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_DecoderConfigCallback_t videoDecoderConfigCallback;\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_DidReceiveFrameCallback_t videoReceiveCallback;\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_TimeoutFrameCallback_t videoTimeoutCallback;\n')
    hPrivFile.write ('    void *videoReceiveCustomData;\n')
    hPrivFile.write ('    //audio part\n')
    hPrivFile.write ('    int hasAudio; /**< 0 if the device has not Audio stream ; otherwide 1 */\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_DecoderConfigCallback_t audioDecoderConfigCallback;\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_DidReceiveFrameCallback_t audioReceiveCallback;\n')
    hPrivFile.write ('    ARCONTROLLER_Stream_TimeoutFrameCallback_t audioTimeoutCallback;\n')
    hPrivFile.write ('    void *audioReceiveCustomData;\n')
    hPrivFile.write ('    //extension part\n')
    hPrivFile.write ('    eARCONTROLLER_DEVICE_STATE extensionState; /**< extension state of the deviceController*/\n')
    hPrivFile.write ('    char *extensionName;\n')
    hPrivFile.write ('    eARDISCOVERY_PRODUCT extensionProduct;\n')
    hPrivFile.write ('};\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Start the Network.\n')
    hPrivFile.write (' * @post ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopNetwork')+' must be call to stop the device Controller before destroy it.\n')
    hPrivFile.write (' * @warning This is a blocking function.\n')
    hPrivFile.write (' * @param deviceController The device controller to start its network.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopNetwork')+'.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartNetwork')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Stop the Network.\n')
    hPrivFile.write (' * @param deviceController The device controller to stop its network.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartNetwork')+'.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopNetwork')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called on start of the device.\n')
    hPrivFile.write (' * @param deviceController The device controller on start.\n')
    hPrivFile.write (' * @param[in] isExtensionDevice 1 if the device started is the extension device, 0 otherwise.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnStart')+' ('+className+' *deviceController, int isExtensionDevice);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Set the network controller of the device controller to its features.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] specificFeature The feature on which the network will be set. If null, set network on all features.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetNetworkControllerToFeatures')+' ('+className+' *deviceController, void* specificFeature);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Register callback for each command received.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] specificFeature The feature to register. If null, register callbacks of all features.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RegisterCallbacks')+' ('+className+' *deviceController, void* specificFeature);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Unregister callback for each command received.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] specificFeature The feature to unregister. If null, unregister callbacks of all features.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'UnregisterCallbacks')+' ('+className+' *deviceController, void* specificFeature);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Start the looper thread sending pcmd commands.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopControllerLooperThread')+'.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartControllerLooperThread')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Stop the looper thread sending pcmd commands.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartControllerLooperThread')+'.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopControllerLooperThread')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Run of the looper thread.\n')
    hPrivFile.write (' * @param data The device controller.\n')
    hPrivFile.write (' * @return NULL.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartControllerLooperThread')+'.\n')
    hPrivFile.write (' * @see ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopControllerLooperThread')+'.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ControllerLooperThread')+' (void *data);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Set the Initial Date for the device.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetInitialDate')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Get the Initial Settings of the device.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] onExtensionDevice 1 if the device started is the extension device, 0 otherwise.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetInitialSettings')+' ('+className+' *deviceController, int onExtensionDevice);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Get the Initial States of the device.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] onExtensionDevice 1 if the device started is the extension device, 0 otherwise.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetInitialStates')+' ('+className+' *deviceController, int onExtensionDevice);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Callback used when the settings or states of the device are changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @return executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DictionaryChangedCallback')+' ('+defineNotificationDef()+' commandKey, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary, void *customData); // TODO int -> ARCommands Big \n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when all initials states of the device are received.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllStatesEnd')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when all initials settings of the device are received.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllSettingsEnd')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when an extension connection to the SkyController has changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnSkyControllerConnectionChangedReceived')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Callback used to add a json part during the device connection.\n')
    hPrivFile.write (' * @param jsonObj The json in which to add.\n')
    hPrivFile.write (' * @param customData The device controller.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARDISCOVERY_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SendJsonCallback')+' (json_object *jsonObj, void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when the ARDrone3 video stream state has changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param elementDictionary command element dictionary.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnARDrone3VideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when the Jumping sumo video stream state has changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param elementDictionary command element dictionary.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnJumpingSumoVideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when the PowerUp video stream state has changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param elementDictionary command element dictionary.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnPowerUpVideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Function called when the audio stream state has changed.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param elementDictionary command element dictionary.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAudioStreamStateChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Callback used to receive a json part during the device connection.\n')
    hPrivFile.write (' * @param jsonObj The json in which to read.\n')
    hPrivFile.write (' * @param customData The device controller.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write  ('eARDISCOVERY_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ReceiveJsonCallback')+' (json_object *jsonObj, void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Callback used when the Network Controller is Disconnected.\n')
    hPrivFile.write (' * @param customData The device controller on diconnection.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write  ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnDisconnectCallback')+' (void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Start thread run .\n')
    hPrivFile.write (' * @param data The device controller on diconnection.\n')
    hPrivFile.write (' * @return NULL.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartRun')+' (void *data);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Stop thread run .\n')
    hPrivFile.write (' * @param data The device controller on diconnection.\n')
    hPrivFile.write (' * @return NULL.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopRun')+' (void *data);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Start extension thread run .\n')
    hPrivFile.write (' * @param data The device controller.\n')
    hPrivFile.write (' * @return NULL.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ExtensionStartRun')+' (void *data);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Stop extension thread run .\n')
    hPrivFile.write (' * @param data The device controller.\n')
    hPrivFile.write (' * @return NULL.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ExtensionStopRun')+' (void *data);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Add a callback in a list.\n')
    hPrivFile.write (' * @param callbackList The list of callbacks.\n')
    hPrivFile.write (' * @param[in] callback The callback to add.\n')
    hPrivFile.write (' * @param[in] customData Data given as parameter to the callback.\n')
    hPrivFile.write (' * @return Executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' callback, void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Remove a callback from a list.\n')
    hPrivFile.write (' * @param callbackList The list of callbacks.\n')
    hPrivFile.write (' * @param[in] callback The callback to remove.\n')
    hPrivFile.write (' * @param[in] customData Data given at the adding of the callback.\n')
    hPrivFile.write (' * @return Executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveCallbackFromList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' callback, void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write (' /**\n')
    hPrivFile.write ('  * @brief Delete all callback of a list.\n')
    hPrivFile.write ('  * @param callbackList The list of callbacks.\n')
    hPrivFile.write ('  */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteCallbackList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Notify all callback of a list.\n')
    hPrivFile.write (' * @param callbackList The list of callbacks.\n')
    hPrivFile.write (' * @param[in] state The state of the device Controller ; given as parameter to the callback.\n')
    hPrivFile.write (' * @param[in] error Error causing this new state.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, eARCONTROLLER_DEVICE_STATE state, eARCONTROLLER_ERROR error);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Add an extension callback in a list.\n')
    hPrivFile.write (' * @param callbackList The list of extension callbacks.\n')
    hPrivFile.write (' * @param[in] callback The callback to add.\n')
    hPrivFile.write (' * @param[in] customData Data given as parameter to the callback.\n')
    hPrivFile.write (' * @return Executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddExtensionCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' callback, void *customData);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Remove an extension callback from a list.\n')
    hPrivFile.write (' * @param callbackList The list of extension callbacks.\n')
    hPrivFile.write (' * @param[in] callback The callback to remove.\n')
    hPrivFile.write (' * @param[in] customData Data given at the adding of the callback.\n')
    hPrivFile.write (' * @return Executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveExtensionCallbackFromList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' callback, void *customData);\n')
    hPrivFile.write ('\n')

    hPrivFile.write (' /**\n')
    hPrivFile.write ('  * @brief Delete all extension callbacks of a list.\n')
    hPrivFile.write ('  * @param callbackList The list of callbacks.\n')
    hPrivFile.write ('  */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteExtensionCallbackList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Notify all extension callback of a list.\n')
    hPrivFile.write (' * @param callbackList The list of callbacks.\n')
    hPrivFile.write (' * @param[in] state The state of the extension device Controller ; given as parameter to the callback.\n')
    hPrivFile.write (' * @param[in] product The product type of the extension ; given as parameter to the callback.\n')
    hPrivFile.write (' * @param[in] name The name of the extension ; given as parameter to the callback.\n')
    hPrivFile.write (' * @param[in] error Error causing this new state.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllExtensionCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, eARCONTROLLER_DEVICE_STATE state, eARDISCOVERY_PRODUCT product, const char *name, eARCONTROLLER_ERROR error);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Set the Device Controller state and notify all listeners.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] state The new state to set.\n')
    hPrivFile.write (' * @param[in] error Error causing this new state.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' ('+className+' *deviceController, eARCONTROLLER_DEVICE_STATE state, eARCONTROLLER_ERROR error);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Set the Device Controller extension state and notify all listeners.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @param[in] state The new extension state to set.\n')
    hPrivFile.write (' * @param[in] error Error causing this new state.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetExtensionState')+' ('+className+' *deviceController, eARCONTROLLER_DEVICE_STATE state, eARCONTROLLER_ERROR error);\n')
    hPrivFile.write ('\n')

    hPrivFile.write ('/**\n')
    hPrivFile.write (' * @brief Delete the extension device.\n')
    hPrivFile.write (' * @param deviceController The device controller.\n')
    hPrivFile.write (' * @return Executing error.\n')
    hPrivFile.write (' */\n')
    hPrivFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteExtension')+' ('+className+' *deviceController);\n')
    hPrivFile.write ('\n')
    
    hPrivFile.write ('#endif /* '+includeDefine+' */\n')
    hPrivFile.write ('\n')
    hPrivFile.write ('// END GENERATED CODE\n')
    hPrivFile.close ()
    
    #################################################
    # Write Device controller c file                #
    #################################################
    
    classTag = 'ARCONTROLLER_Device'
    className = 'ARCONTROLLER_Device_t'
    
    cFileName = CTRL_DEVICE_C_NAME
    filepath = SRC_DIR + cFileName
    cFile = open (filepath, 'w')

    cFile.write ('/**********************************************************\n')
    cFile.write (' *            AUTOGENERATED FILE                          *\n')
    cFile.write (' *             DO NOT MODIFY IT                           *\n')
    cFile.write (' *                                                        *\n')
    cFile.write (' * To add new commands :                                  *\n')
    cFile.write (' *  - Modify ../Xml/commands.xml file                     *\n')
    cFile.write (' *  - Re-run generateDeviceControllers.py script          *\n')
    cFile.write (' *                                                        *\n')
    cFile.write (' **********************************************************/\n')
    cFile.write ('\n')

    cFile.write ('/**\n')
    cFile.write ('* @file '+cFileName+'\n')
    cFile.write ('* @brief '+bref+'\n')
    cFile.write ('*/\n')
    cFile.write ('\n')

    cFile.write ('#ifndef _GNU_SOURCE\n')
    cFile.write ('#define _GNU_SOURCE\n')
    cFile.write ('#endif\n')
    cFile.write ('\n')

    cFile.write ('#include <stdio.h>\n')
    cFile.write ('#include <time.h>\n')
    cFile.write ('#include <json/json.h>\n')
    cFile.write ('\n')
    
    cFile.write ('#include <libARSAL/ARSAL_Mutex.h>\n')
    cFile.write ('#include <libARSAL/ARSAL_Sem.h>\n')
    cFile.write ('#include <libARSAL/ARSAL_Time.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Network.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Feature.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_DICTIONARY_Key.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Dictionary.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Frame.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Stream.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Device.h>\n')
    cFile.write ('#include <ARCONTROLLER_NAckCbs.h>\n')
    cFile.write ('\n')
    cFile.write ('#include "ARCONTROLLER_Device.h"\n')
    cFile.write ('\n')
    
    cFile.write ('#define '+MODULE_DEVICE+'_TAG "'+classTag+'"\n')
    cFile.write ('\n')
    
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             local header:\n')
    cFile.write (' *\n')
    cFile.write (' ****************************************/\n')
    cFile.write ('\n')

    cFile.write ('static int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ElementCompare')+'('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *a, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *b);\n')
    cFile.write ('\n')
    
    cFile.write ('static int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ExtensionElementCompare')+'('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *a, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *b);\n')
    cFile.write ('\n')
    
    
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             implementation:\n')
    cFile.write (' *\n')
    cFile.write (' ****************************************/\n')
    cFile.write ('\n')
    
    cFile.write (''+className+' *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'New')+' (ARDISCOVERY_Device_t *discoveryDevice, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Create a new Device Controller --\n')
    cFile.write ('    \n')
    
    cFile.write ('    //local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    eARDISCOVERY_ERROR dicoveryError = ARDISCOVERY_OK;\n')
    cFile.write ('    '+className+' *deviceController =  NULL;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (discoveryDevice == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Allocate the Device Controller\n')
    cFile.write ('    deviceController = malloc (sizeof('+className+'));\n')
    cFile.write ('    if (deviceController != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        //initialization of the device controller\n')
    for feature in ctx.features:
        cFile.write ('        deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' = NULL;\n')
    cFile.write ('        deviceController->privatePart = NULL;\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Allocate the private part of the Device Controller\n')
    cFile.write ('        deviceController->privatePart = malloc (sizeof('+classPrivateName+'));\n')
    cFile.write ('        if (deviceController->privatePart != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            //initialization of the private part of the device controller\n')
    cFile.write ('            deviceController->privatePart->networkController = NULL;\n')
    cFile.write ('            deviceController->privatePart->state = ARCONTROLLER_DEVICE_STATE_STOPPED;\n')
    cFile.write ('            deviceController->privatePart->controllerLooperThread = NULL;\n')
    cFile.write ('            deviceController->privatePart->commandCallbacks = NULL;\n')
    cFile.write ('            deviceController->privatePart->stateChangedCallbacks = NULL;\n')
    cFile.write ('            deviceController->privatePart->startCancelled = 0;\n')
    cFile.write ('            // Video Part\n')
    cFile.write ('            deviceController->privatePart->hasVideo = 0;\n')
    cFile.write ('            deviceController->privatePart->videoIsMP4Compliant = 0;\n')
    cFile.write ('            deviceController->privatePart->videoDecoderConfigCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->videoReceiveCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->videoTimeoutCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->videoReceiveCustomData = NULL;\n')
    cFile.write ('            // Audio Part\n')
    cFile.write ('            deviceController->privatePart->hasAudio = 0;\n')
    cFile.write ('            deviceController->privatePart->audioDecoderConfigCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->audioReceiveCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->audioTimeoutCallback = NULL;\n')
    cFile.write ('            deviceController->privatePart->audioReceiveCustomData = NULL;\n')
    cFile.write ('            // Extension part\n')
    cFile.write ('            deviceController->privatePart->extensionState = ARCONTROLLER_DEVICE_STATE_STOPPED;\n')
    cFile.write ('            deviceController->privatePart->extensionStateChangedCallbacks = NULL;\n')
    cFile.write ('            deviceController->privatePart->extensionName = NULL;\n')
    cFile.write ('            deviceController->privatePart->extensionProduct = ARDISCOVERY_PRODUCT_MAX;\n')
    cFile.write ('            \n')
    
    cFile.write ('            // Create the mutex/condition \n')
    cFile.write ('            if ((ARSAL_Mutex_Init (&(deviceController->privatePart->mutex)) != 0) ||\n')
    cFile.write ('               (ARSAL_Sem_Init (&(deviceController->privatePart->initSem), 0, 0) != 0))\n')
    cFile.write ('            {\n')
    cFile.write ('                localError = ARCONTROLLER_ERROR_INIT_MUTEX;\n')
    cFile.write ('            }\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Copy the device\n')
    cFile.write ('        \n')
    cFile.write ('        deviceController->privatePart->discoveryDevice = ARDISCOVERY_Device_NewByCopy (discoveryDevice, &dicoveryError);\n')
    cFile.write ('        if (dicoveryError != ARDISCOVERY_OK)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_INIT_DEVICE_COPY;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Get networkConfiguration of the device\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        dicoveryError = ARDISCOVERY_Device_InitNetworkConfiguration (deviceController->privatePart->discoveryDevice, &(deviceController->privatePart->networkConfiguration));\n')
    cFile.write ('        if (dicoveryError != ARDISCOVERY_OK)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_INIT_DEVICE_GET_NETWORK_CONFIG;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Check if the device has video\n')
    cFile.write ('        if (deviceController->privatePart->networkConfiguration.deviceToControllerARStreamData != -1)\n')
    cFile.write ('        {\n')
    cFile.write ('            deviceController->privatePart->hasVideo = 1;\n')
    cFile.write ('        }\n')
    cFile.write ('        //NO else ; device has not video\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')

    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Check if the device has audio\n')
    cFile.write ('        if (deviceController->privatePart->networkConfiguration.deviceToControllerARStreamAudioData != -1)\n')
    cFile.write ('        {\n')
    cFile.write ('            deviceController->privatePart->hasAudio = 1;\n')
    cFile.write ('        }\n')
    cFile.write ('        //NO else ; device has not audio\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')

    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Creation of the features:\n')
    cFile.write ('        switch (discoveryDevice->productID)\n')
    cFile.write ('        {\n')
    for deviceController in deviceControllers:
        cFile.write ('            case '+discoveryProduct (deviceController.product)+':\n')
        for featureName in deviceController.features:
            cFile.write ('                if (localError == ARCONTROLLER_OK)\n')
            cFile.write ('                {\n')
            cFile.write ('                    deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+' = ' + ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'New')+' (deviceController->privatePart->networkController, &localError);\n')
            cFile.write ('                }\n')
            cFile.write ('                \n')
        cFile.write ('                break;\n')
        cFile.write ('            \n')
        
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "device : %d not known", discoveryDevice->productID);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')

    cFile.write ('    // delete the Device Controller if an error occurred\n')
    cFile.write ('    if (localError != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Delete')+' (&deviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped no error \n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('    \n')
    
    cFile.write ('    return deviceController;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Delete')+' ('+className+' **deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Delete the Device Controller --\n')
    cFile.write ('    \n')

    cFile.write ('    if (deviceController != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        if ((*deviceController) != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            if ((*deviceController)->privatePart != NULL)\n')
    cFile.write ('            {\n')
    cFile.write ('                // delete all extension related vars\n')
    cFile.write ('                ARCONTROLLER_Device_DeleteExtension(*deviceController);\n')
    cFile.write ('                \n')
    
    cFile.write ('                ARSAL_Mutex_Destroy (&((*deviceController)->privatePart->mutex));\n')
    cFile.write ('                \n')
    
    cFile.write ('                ARSAL_Sem_Destroy (&((*deviceController)->privatePart->initSem));\n')
    cFile.write ('                \n')
    
    cFile.write ('                if ((*deviceController)->privatePart->commandCallbacks != NULL)\n')
    cFile.write ('                {\n')
    cFile.write ('                    // -- Delete all callback in array --\n')
    cFile.write ('                    ARCONTROLLER_Dictionary_DeleteCallbackList(&((*deviceController)->privatePart->commandCallbacks));\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')

    cFile.write ('                if ((*deviceController)->privatePart->stateChangedCallbacks != NULL)\n')
    cFile.write ('                {\n')
    cFile.write ('                    // -- Delete all callback in array --\n')
    cFile.write ('                    ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteCallbackList')+'(&((*deviceController)->privatePart->stateChangedCallbacks));\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')
    
    cFile.write ('                // Delete features:\n')
    cFile.write ('                switch ((*deviceController)->privatePart->discoveryDevice->productID)\n')
    cFile.write ('                {\n')
    for deviceController in deviceControllers:
        cFile.write ('                    case '+discoveryProduct (deviceController.product)+':\n')
        for featureName in deviceController.features:
            cFile.write ('                        ' + ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'Delete')+' (&((*deviceController)->'+ARUncapitalize(ftr_new_to_old_name(featureName))+'));\n')
            cFile.write ('                        \n')
        cFile.write ('                        break;\n')
        cFile.write ('                    \n')
    
    cFile.write ('                    default:\n')
    cFile.write ('                        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "device : %d not known", (*deviceController)->privatePart->discoveryDevice->productID);\n')
    cFile.write ('                        break;\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')
    
    cFile.write ('                ARDISCOVERY_Device_Delete (&((*deviceController)->privatePart->discoveryDevice));\n')
    cFile.write ('                \n')
    
    cFile.write ('                // free the private part of the Device Controller\n')
    cFile.write ('                free ((*deviceController)->privatePart);\n')
    cFile.write ('                (*deviceController)->privatePart = NULL;\n')
    cFile.write ('            }\n')
    cFile.write ('            \n')
    
    cFile.write ('            // free the Device Controller\n')
    cFile.write ('            free (*deviceController);\n')
    cFile.write ('            (*deviceController) = NULL;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteExtension')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    if (deviceController != NULL && deviceController->privatePart != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->stateChangedCallbacks != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            // -- Delete all callback in array --\n')
    cFile.write ('            ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteExtensionCallbackList')+'(&(deviceController->privatePart->extensionStateChangedCallbacks));\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        // Delete extension feature\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        switch (deviceController->privatePart->extensionProduct)\n')
    cFile.write ('        {\n')
    # For each device controller that can be an extension
    for deviceController in [x for x in deviceControllers if x.flags.can_be_extension]:
        cFile.write ('            case '+discoveryProduct (deviceController.product)+':\n')
        cFile.write ('                switch (deviceController->privatePart->discoveryDevice->productID)\n')
        cFile.write ('                {\n')
        # For each device controller that can have an extension
        for extensionDeviceController in [y for y in deviceControllers if y.flags.can_have_extension]:
            cFile.write ('                    case '+discoveryProduct (extensionDeviceController.product)+':\n')

            # Remove all features of the extension that are not in the list of features of the extender
            for featureName in [z for z in deviceController.features if z not in extensionDeviceController.features]:
                cFile.write ('                        ' + ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'Delete') + '(&(deviceController->' + ARUncapitalize(ftr_new_to_old_name(featureName))+'));\n')
            cFile.write ('                         break;\n')
        cFile.write ('                    default:\n')
        cFile.write ('                        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "device : %d can not have an extension", deviceController->privatePart->discoveryDevice->productID);\n')
        cFile.write ('                }\n')
        cFile.write ('                break;\n')
    cFile.write ('            case ARDISCOVERY_PRODUCT_MAX:\n')
    cFile.write ('                // not connected to an extension device, so don\'t do anything\n')
    cFile.write ('                break;\n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "device : %d can not be an extension", deviceController->privatePart->extensionProduct);\n')
    cFile.write ('                error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('        ARSAL_Mutex_Unlock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        \n')
    cFile.write ('        if (deviceController->privatePart->extensionName != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            free(deviceController->privatePart->extensionName);\n')
    cFile.write ('            deviceController->privatePart->extensionName = NULL;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RegisterCallbacks')+' ('+className+' *deviceController, void* specificFeature)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Register the Callbacks --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        return ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    for feature in ctx.features:
        cFile.write ('    if ((deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL) && ((specificFeature == NULL) || (specificFeature == deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+')))\n')
        cFile.write ('    {\n')
        for evt in feature.evts:
            cFile.write ('        if (error == ARCONTROLLER_OK)\n')
            cFile.write ('        {\n')
            cFile.write ('            error = '+ARFunctionName(MODULE_FEATURE, get_ftr_old_name(feature), 'addCallback')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+', '+defineNotification(feature, evt)+', ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DictionaryChangedCallback')+', deviceController);\n')
            cFile.write ('        }\n')
            cFile.write ('        \n')
        cFile.write ('    }\n')
        cFile.write ('    \n')
        
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'UnregisterCallbacks')+' ('+className+' *deviceController, void *specificFeature)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Unregister the Callbacks --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    eARCONTROLLER_ERROR removingError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        return ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    
    for feature in ctx.features:
        cFile.write ('        if ((deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL) && ((specificFeature == NULL) || (specificFeature == deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+')))\n')
        cFile.write ('        {\n')
        
        for evt in feature.evts:
            cFile.write ('            removingError = '+ARFunctionName(MODULE_FEATURE, get_ftr_old_name(feature), 'removeCallback')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+', '+defineNotification(feature, evt)+', ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DictionaryChangedCallback')+', deviceController);\n')
            cFile.write ('            if (error != ARCONTROLLER_OK)\n')
            cFile.write ('            {\n')
            cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error occured durring removing of the callback for '+defineNotification(feature, evt)+'; error :%s", ARCONTROLLER_Error_ToString (removingError));\n')
            cFile.write ('            }\n')
            cFile.write ('            \n')
        cFile.write ('        }\n')
        cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
            
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Start')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Start the Device Controller --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARSAL_Thread_t startingThread = NULL;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK) \n')
    cFile.write ('    {\n')

    cFile.write ('        if (deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_STOPPED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_STARTING, ARCONTROLLER_OK);\n')
    
    cFile.write ('            if (ARSAL_Thread_Create (&startingThread, ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartRun')+', deviceController) != 0)\n')
    cFile.write ('            {\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Creation of Starting thread failed.");\n')
    cFile.write ('                error = ARCONTROLLER_ERROR_INIT_THREAD;\n')
    cFile.write ('            }\n')
    cFile.write ('            else\n')
    cFile.write ('            {\n')
    cFile.write ('                ARSAL_Thread_Destroy (&startingThread);\n')
    cFile.write ('                startingThread = NULL;\n')
    cFile.write ('            }\n')
    cFile.write ('            \n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Stop')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Stop the Device Controller --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARSAL_Thread_t stoppingThread = NULL;\n')
    cFile.write ('    \n')
        
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    
    cFile.write ('    if (error == ARCONTROLLER_OK) \n')
    cFile.write ('    {\n')

    cFile.write ('        if (deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_RUNNING)\n')
    cFile.write ('        {\n')
    cFile.write ('            ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_STOPPING, ARCONTROLLER_OK);\n')
    
    cFile.write ('            if (ARSAL_Thread_Create (&stoppingThread, ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopRun')+', deviceController) != 0)\n')
    cFile.write ('            {\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Creation of Stopping thread failed.");\n')
    cFile.write ('                error = ARCONTROLLER_ERROR_INIT_THREAD;\n')
    cFile.write ('            }\n')
    cFile.write ('            else\n')
    cFile.write ('            {\n')
    cFile.write ('                ARSAL_Thread_Destroy (&stoppingThread);\n')
    cFile.write ('                stoppingThread = NULL;\n')
    cFile.write ('            }\n')
    cFile.write ('            \n')
    cFile.write ('        }\n')
    cFile.write ('        else if ((deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_STARTING) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('        {\n')
    cFile.write ('            // Cancel the connection\n')
    cFile.write ('            ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_STOPPING, ARCONTROLLER_OK);\n')
    
    cFile.write ('            ARSAL_Mutex_Lock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('            deviceController->privatePart->startCancelled = 1;\n')
    cFile.write ('            ARSAL_Sem_Post (&(deviceController->privatePart->initSem));\n')
    cFile.write ('            ARSAL_Sem_Post (&(deviceController->privatePart->initSem));\n')
    cFile.write ('            ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetVideoStreamCallbacks')+' ('+className+' *deviceController, ARCONTROLLER_Stream_DecoderConfigCallback_t decoderConfigCallback, ARCONTROLLER_Stream_DidReceiveFrameCallback_t receiveFrameCallback, ARCONTROLLER_Stream_TimeoutFrameCallback_t timeoutFrameCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set video stream callbacks --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->hasVideo)\n')
    cFile.write ('        {\n')
    cFile.write ('            deviceController->privatePart->videoDecoderConfigCallback = decoderConfigCallback;\n')
    cFile.write ('            deviceController->privatePart->videoReceiveCallback = receiveFrameCallback;\n')
    cFile.write ('            deviceController->privatePart->videoTimeoutCallback = timeoutFrameCallback;\n')
    cFile.write ('            deviceController->privatePart->videoReceiveCustomData = customData;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_VIDEO;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_Device_SetVideoStreamMP4Compliant (ARCONTROLLER_Device_t *deviceController, int isMP4Compliant)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set video stream compliant with the mp4 format. --\n')
    cFile.write ('\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('\n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARCONTROLLER_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->hasVideo)\n')
    cFile.write ('        {\n')
    cFile.write ('            deviceController->privatePart->videoIsMP4Compliant = isMP4Compliant;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_VIDEO;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetAudioStreamCallbacks')+' ('+className+' *deviceController, ARCONTROLLER_Stream_DecoderConfigCallback_t decoderConfigCallback, ARCONTROLLER_Stream_DidReceiveFrameCallback_t receiveFrameCallback, ARCONTROLLER_Stream_TimeoutFrameCallback_t timeoutFrameCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set audio stream callbacks --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->hasAudio)\n')
    cFile.write ('        {\n')
    cFile.write ('            deviceController->privatePart->audioDecoderConfigCallback = decoderConfigCallback;\n')
    cFile.write ('            deviceController->privatePart->audioReceiveCallback = receiveFrameCallback;\n')
    cFile.write ('            deviceController->privatePart->audioTimeoutCallback = timeoutFrameCallback;\n')
    cFile.write ('            deviceController->privatePart->audioReceiveCustomData = customData;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_AUDIO;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddCommandReceivedCallback')+' ('+className+' *deviceController, ARCONTROLLER_DICTIONARY_CALLBACK_t commandReceivedCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Add Command received callback --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Dictionary_AddCallbackInList (&(deviceController->privatePart->commandCallbacks), commandReceivedCallback, customData);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveCommandReceivedCallback')+' ('+className+' *deviceController, ARCONTROLLER_DICTIONARY_CALLBACK_t commandReceivedCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Remove Command received callback --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
        
    cFile.write ('        error = ARCONTROLLER_Dictionary_RemoveCallbackFromList (&(deviceController->privatePart->commandCallbacks), commandReceivedCallback, customData);\n')
    
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('ARCONTROLLER_DICTIONARY_ELEMENT_t *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetCommandElements')+' ('+className+' *deviceController, '+defineNotificationDef()+' commandKey, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get Command Elements --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *elements = NULL;\n')
    cFile.write ('    '+defineNotificationDef()+' featureKey = ' + ARFunctionName (MODULE_DICTIONARY, 'Key', 'GetFeatureFromCommandKey')+' (commandKey);\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (featureKey)\n')
    cFile.write ('        {\n')
    for feature in ctx.features:
        cFile.write ('            case '+defineNotification(feature)+':\n')
        cFile.write ('                elements = ' + ARFunctionName (MODULE_ARCONTROLLER, get_ftr_old_name(feature), 'GetCommandElements')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+', commandKey, &localError);\n')
        cFile.write ('                \n')
        cFile.write ('                break;\n')
        cFile.write ('            \n')
    
    cFile.write ('            default :\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error commandKey :%d not known", commandKey);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('    \n')
    
    cFile.write ('    return elements;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' stateChangedCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Add State Changed callback --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = '+ ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddCallbackInList')+' (&(deviceController->privatePart->stateChangedCallbacks), stateChangedCallback, customData);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddExtensionStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' extensionStateChangedCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Add Extension State Changed callback --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = '+ ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddExtensionCallbackInList')+' (&(deviceController->privatePart->extensionStateChangedCallbacks), extensionStateChangedCallback, customData);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveStateChangedCallback')+' ('+className+' *deviceController, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' stateChangedCallback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Remove State Changed Callback --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveCallbackFromList')+' (&(deviceController->privatePart->stateChangedCallbacks), stateChangedCallback, customData);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SendStreamFrame')+' ('+className+' *deviceController, uint8_t *data, int dataSize)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Send Stream Frame --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int locked = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL) ||\n')
    cFile.write ('        (data == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 1;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Network_SendAudioFrame (deviceController->privatePart->networkController, data, dataSize);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (locked)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('        locked = 0;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_DEVICE_STATE ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetState')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get State --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_DEVICE_STATE state = ARCONTROLLER_DEVICE_STATE_MAX;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        \n')
    
    cFile.write ('        state = deviceController->privatePart->state;\n')
    cFile.write ('        \n')
    
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('    \n')
    
    cFile.write ('    return state;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_DEVICE_STATE ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionState')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get Extension State --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_DEVICE_STATE extensionState = ARCONTROLLER_DEVICE_STATE_MAX;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        \n')
    
    cFile.write ('        extensionState = deviceController->privatePart->extensionState;\n')
    cFile.write ('        \n')
    
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('    \n')
    
    cFile.write ('    return extensionState;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionName')+' ('+className+' *deviceController, char *buffer, int bufferSize, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get Extension Name --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL) ||\n')
    cFile.write ('        (buffer == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        \n')
    
    cFile.write ('        if (deviceController->privatePart->extensionName != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            int desiredChars = snprintf(buffer, bufferSize, "%s", deviceController->privatePart->extensionName);\n')
    cFile.write ('            if (desiredChars >= bufferSize)\n')
    cFile.write ('            {\n')
    cFile.write ('                localError = ARCONTROLLER_ERROR_BUFFER_SIZE;\n')
    cFile.write ('            }\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_STATE;\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')

    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARDISCOVERY_PRODUCT ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetExtensionProduct')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get Extension Product --\n')
    cFile.write ('    \n')

    cFile.write ('    eARDISCOVERY_PRODUCT extensionProduct = ARDISCOVERY_PRODUCT_MAX;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        \n')
    
    cFile.write ('        extensionProduct = deviceController->privatePart->extensionProduct;\n')
    cFile.write ('        \n')
    
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('    \n')
    
    cFile.write ('    return extensionProduct;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasOutputVideoStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Check if the device controller has outupt video stream --\n')
    cFile.write ('\n')

    cFile.write ('    int hasVideoStreamOutput = 0;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('\n')

    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('\n')

    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('\n')
    cFile.write ('        hasVideoStreamOutput = deviceController->privatePart->hasVideo;\n')
    cFile.write ('\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('\n')

    cFile.write ('    // Return the error\n')
    cFile.write ('    if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = localError;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: error is not returned \n')
    cFile.write ('\n')

    cFile.write ('    return hasVideoStreamOutput;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasOutputAudioStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Check if the device controller has output audio stream --\n')
    cFile.write ('\n')

    cFile.write ('    int hasAudioStreamOutput = 0;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('\n')

    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('\n')

    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('\n')
    cFile.write ('        hasAudioStreamOutput = (deviceController->privatePart->networkConfiguration.controllerToDeviceARStreamAudioData != -1);\n')
    cFile.write ('\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('\n')

    cFile.write ('    return hasAudioStreamOutput;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'hasInputAudioStream')+' ('+className+' *deviceController, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Check if the device controller has input audio stream --\n')
    cFile.write ('\n')

    cFile.write ('    int hasAudioStreamInput = 0;\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('\n')

    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets localError to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('\n')

    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('\n')
    cFile.write ('        hasAudioStreamInput = (deviceController->privatePart->networkConfiguration.deviceToControllerARStreamAudioData != -1);\n')
    cFile.write ('\n')
    cFile.write ('        ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    }\n')
    cFile.write ('\n')

    cFile.write ('    return hasAudioStreamInput;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('\n')
    cFile.write ('/********************\n')
    cFile.write ('* Private Functions *\n')
    cFile.write ('*********************/\n')
    cFile.write ('\n')
    
    cFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartRun')+' (void *data)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Thread Run of Start --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+className+' *deviceController = ('+className+' *) data;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        return NULL;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')    
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartNetwork')+' (deviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartControllerLooperThread')+' (deviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RegisterCallbacks')+' (deviceController, NULL);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnStart')+' (deviceController, 0);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_RUNNING, ARCONTROLLER_OK);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        // Failed or canceled : Stop the DeviceController.\n')
    cFile.write ('        \n')
    cFile.write ('        if (deviceController->privatePart->startCancelled)\n')
    cFile.write ('        {\n')
    cFile.write ('            // DeviceController start canceled \n')
    cFile.write ('            error = ARCONTROLLER_ERROR_CANCELED;\n')
    cFile.write ('        }\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_INFO, '+MODULE_DEVICE+'_TAG, "Start failed or canceled.");\n')
    cFile.write ('        ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_STOPPING, error);\n')
    cFile.write ('        ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopRun')+' (deviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Print error\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Start fail error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return NULL;\n')
    cFile.write ('}\n')
    
    cFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopRun')+' (void *data)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Thread Run of Stop --\n')
    cFile.write ('    \n')

    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+className+' *deviceController = ('+className+' *) data;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')

    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL) {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "'+ ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopRun')+' failed with error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('        return NULL;\n')
    cFile.write ('    }\n\n')

    cFile.write ('    error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'UnregisterCallbacks')+' (deviceController, NULL);\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "UnregisterCallback failed with error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')

    cFile.write ('    error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopControllerLooperThread')+' (deviceController);\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "StopControllerLooperThread failed with error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')

    cFile.write ('    error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopNetwork')+' (deviceController);\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "StopNetwork failed with error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')

    cFile.write ('    ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' (deviceController, ARCONTROLLER_DEVICE_STATE_STOPPED, ARCONTROLLER_OK);\n')
    cFile.write ('    \n')
    
    cFile.write ('    return NULL;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('void *ARCONTROLLER_Device_ExtensionStopRun (void *data)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Thread Run of Stop --\n')
    cFile.write ('    \n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    ARCONTROLLER_Device_t *deviceController = (ARCONTROLLER_Device_t *) data;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Stop fail error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('        return NULL;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')

    cFile.write ('    switch (deviceController->privatePart->extensionProduct)\n')
    cFile.write ('    {\n')
    # For each device controller that can be an extension
    for deviceController in [x for x in deviceControllers if x.flags.can_be_extension]:
        cFile.write ('        // If the extension is a ' + discoveryProduct (deviceController.product) + '\n')
        cFile.write ('        case '+discoveryProduct (deviceController.product)+':\n')
        cFile.write ('            switch (deviceController->privatePart->discoveryDevice->productID)\n')
        cFile.write ('            {\n')
        # For each device controller that can have an extension
        for extensionDeviceController in [y for y in deviceControllers if y.flags.can_have_extension]:
            cFile.write ('                // If the remote controller on which we are connected is a ' + discoveryProduct (extensionDeviceController.product) + '\n')
            cFile.write ('                case '+discoveryProduct (extensionDeviceController.product)+':\n')

            # for all features of the extension that are not in the list of features of the extender
            for featureName in [z for z in deviceController.features if z not in extensionDeviceController.features]:
                cFile.write ('                    if (error == ARCONTROLLER_OK && deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+' != NULL)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        error = '+ ARFunctionName (MODULE_ARCONTROLLER, 'device', 'UnregisterCallbacks') +' (deviceController, deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+');\n')
                cFile.write ('                    }\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK && deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+' != NULL)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        error = '+ ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'UnregisterARCommands') +'(deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+');\n')
                cFile.write ('                    }\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
                cFile.write ('                        '+ ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'Delete') +' (&(deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+'));\n')
                cFile.write ('                        ARSAL_Mutex_Unlock(&(deviceController->privatePart->mutex));\n')
                cFile.write ('                    }\n')
            cFile.write ('                    break;\n')
        cFile.write ('                default:\n')
        cFile.write ('                    error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
        cFile.write ('            }\n')
        cFile.write ('            break;\n')

    cFile.write ('        default:\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
    cFile.write ('    }\n')

    cFile.write ('    // reset extension values\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->extensionName != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            free(deviceController->privatePart->extensionName);\n')
    cFile.write ('            deviceController->privatePart->extensionName = NULL;\n')
    cFile.write ('        }\n')
    cFile.write ('        deviceController->privatePart->extensionProduct = ARDISCOVERY_PRODUCT_MAX;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')

    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARCONTROLLER_Device_SetExtensionState (deviceController, ARCONTROLLER_DEVICE_STATE_STOPPED, ARCONTROLLER_OK);\n')
    cFile.write ('    }\n')
    cFile.write ('    //else TODO: see what to do in case of error here\n')
    cFile.write ('    \n')
    cFile.write ('    // Print error\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Stop fail error :%s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return NULL;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartNetwork')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Start the Network --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Create the Network Controller\n')
    cFile.write ('        deviceController->privatePart->networkController = ARCONTROLLER_Network_New (deviceController->privatePart->discoveryDevice, ARCONTROLLER_Device_OnDisconnectCallback, ARCONTROLLER_Device_SendJsonCallback, ARCONTROLLER_Device_ReceiveJsonCallback, deviceController, &error);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetNetworkControllerToFeatures')+' (deviceController, NULL);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')

    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    for feature in ctx.features:
        cFile.write ('        if (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL)\n')
        cFile.write ('        {\n')
        cFile.write ('            error = '+ARFunctionName (MODULE_FEATURE, get_ftr_old_name(feature), 'RegisterARCommands')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+');\n')
        cFile.write ('            if (error != ARCONTROLLER_OK)\n')
        cFile.write ('            {\n')
        cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error occured durring registering ARCommands to the feature '+defineNotification(feature)+'; error :%s", ARCONTROLLER_Error_ToString (error));\n')
        cFile.write ('            }\n')
        cFile.write ('        }\n')
        cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
        
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // If device has video\n')
    cFile.write ('        if (deviceController->privatePart->hasVideo)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_Network_SetVideoReceiveCallback (deviceController->privatePart->networkController, deviceController->privatePart->videoDecoderConfigCallback, deviceController->privatePart->videoReceiveCallback, deviceController->privatePart->videoTimeoutCallback, deviceController->privatePart->videoReceiveCustomData);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')

    cFile.write ('    if ((error == ARCONTROLLER_OK) && (deviceController->privatePart->hasVideo))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Network_SetVideoStreamMP4Compliant (deviceController->privatePart->networkController, deviceController->privatePart->videoIsMP4Compliant);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('\n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // If device has audio\n')
    cFile.write ('        if (deviceController->privatePart->hasAudio)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_Network_SetAudioReceiveCallback (deviceController->privatePart->networkController, deviceController->privatePart->audioDecoderConfigCallback, deviceController->privatePart->audioReceiveCallback, deviceController->privatePart->audioTimeoutCallback, deviceController->privatePart->audioReceiveCustomData);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopNetwork')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Stop the Network --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    for feature in ctx.features:
        cFile.write ('        if (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL)\n')
        cFile.write ('        {\n')
        cFile.write ('            error = '+ARFunctionName (MODULE_FEATURE, get_ftr_old_name(feature), 'UnregisterARCommands')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+');\n')
        cFile.write ('            if (error != ARCONTROLLER_OK)\n')
        cFile.write ('            {\n')
        cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error occured durring unregistering ARCommands to the feature '+defineNotification(feature)+'; error :%s", ARCONTROLLER_Error_ToString (error));\n')
        cFile.write ('            }\n')
        cFile.write ('        }\n')
        cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Delete network:\n')
    cFile.write ('        ' + ARFunctionName (MODULE_ARCONTROLLER, "Network", 'Delete') + ' (&(deviceController->privatePart->networkController)); //TODO read error !!!!!!!! \n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetNetworkControllerToFeatures')+' ('+className+' *deviceController, void* specificFeature)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set the Network Controoler to the Features --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    for feature in ctx.features:
        cFile.write ('    if ((error == ARCONTROLLER_OK) && (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL) && ((specificFeature == NULL) || (specificFeature == deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+')))\n')
        cFile.write ('    {\n')
        cFile.write ('        error = '+ARFunctionName(MODULE_FEATURE, get_ftr_old_name(feature), 'SetNetworkController')+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+', deviceController->privatePart->networkController);\n')
        cFile.write ('        if (error != ARCONTROLLER_OK)\n')
        cFile.write ('        {\n')
        cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error occured durring setting the network Controller to the feature of the callback for '+defineNotification(feature)+'; error :%s", ARCONTROLLER_Error_ToString (error));\n')
        cFile.write ('        }\n')
        cFile.write ('        \n')
        cFile.write ('    }\n')
        cFile.write ('    \n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StartControllerLooperThread')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Create the Sending Looper Thread --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (ARSAL_Thread_Create (&(deviceController->privatePart->controllerLooperThread), ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ControllerLooperThread')+', deviceController) != 0)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Creation of Controller Looper thread failed.");\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_INIT_THREAD;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'StopControllerLooperThread')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Create the Sending Looper Thread --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (deviceController->privatePart->controllerLooperThread != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_Thread_Join (deviceController->privatePart->controllerLooperThread, NULL);\n')
    cFile.write ('            ARSAL_Thread_Destroy (&(deviceController->privatePart->controllerLooperThread));\n')
    cFile.write ('            deviceController->privatePart->controllerLooperThread = NULL;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnStart')+' ('+className+' *deviceController, int isExtensionDevice)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Device On Start --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetInitialDate')+' (deviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetInitialSettings')+' (deviceController, isExtensionDevice);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (!deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetInitialStates')+' (deviceController, isExtensionDevice);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Manage cancel\n')
    cFile.write ('    if ((error == ARCONTROLLER_OK) && (deviceController->privatePart->startCancelled))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_CANCELED;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error or not canceled\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetInitialDate')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set Initial Date --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    time_t currentTime = (time_t)-1;\n')
    cFile.write ('    struct tm ts;\n')
    cFile.write ('    char data[20];\n')
    cFile.write ('    int dataSize = 0;\n')
    cFile.write ('    char strTime[20];\n')
    cFile.write ('    int strTimeSize = 0;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        currentTime = time(NULL);\n')
    cFile.write ('        if (currentTime == ((time_t)-1))\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_INIT_GET_DATE;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ts = *localtime(&currentTime);\n')
    cFile.write ('        dataSize = strftime(data, sizeof(data), "%Y-%m-%d", &ts);\n')
    cFile.write ('        if (dataSize == 0)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_INIT_GET_DATE;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = deviceController->common->sendCommonCurrentDate (deviceController->common, data);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ts = *localtime(&currentTime);\n')
    cFile.write ('        strTimeSize = strftime(strTime, sizeof(data), "T%H%M%S%z", &ts);\n')
    cFile.write ('        if (strTimeSize == 0)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_INIT_GET_DATE;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = deviceController->common->sendCommonCurrentTime (deviceController->common, strTime);\n')
    cFile.write ('    }\n')
    cFile.write ('    // No else: skipped by an error\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'GetInitialSettings')+' ('+className+' *deviceController, int onExtensionDevice)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get initial all Settings --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (!onExtensionDevice && deviceController->skyController != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = deviceController->skyController->sendSettingsAllSettings (deviceController->skyController);\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = deviceController->common->sendSettingsAllSettings (deviceController->common);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Sem_Wait (&(deviceController->privatePart->initSem));\n') # TODO manage error
    cFile.write ('    }\n')
    cFile.write ('    \n')
            
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'getInitialStates')+' ('+className+' *deviceController, int onExtensionDevice)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Get initial all States --\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if (deviceController == NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')

    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (!onExtensionDevice && deviceController->skyController != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = deviceController->skyController->sendCommonAllStates (deviceController->skyController);\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = deviceController->common->sendCommonAllStates (deviceController->common);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Sem_Wait (&(deviceController->privatePart->initSem));\n') # TODO manage error !!!!!!!!!!!!!!!!!!!
    cFile.write ('    }\n')
    cFile.write ('    \n')
            
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DictionaryChangedCallback')+' ('+defineNotificationDef()+' commandKey, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Callback of changing of a dictionary --\n')
    cFile.write ('    \n')
    
    cFile.write ('    '+className+' *deviceController = ('+className+' *) customData;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Notify All Callback\n')
    cFile.write ('        ARCONTROLLER_DICTIONARY_NotifyAllCallbackInList (&(deviceController->privatePart->commandCallbacks), commandKey, elementDictionary);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (commandKey)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_COMMON_COMMONSTATE_ALLSTATESCHANGED:\n')
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_COMMONSTATE_ALLSTATESCHANGED:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllStatesEnd')+' (deviceController);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_COMMON_SETTINGSSTATE_ALLSETTINGSCHANGED:\n')
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_SETTINGSSTATE_ALLSETTINGSCHANGED:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllSettingsEnd')+' (deviceController);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')

    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED:\n')
    cFile.write ('                ARSAL_PRINT (ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Connexion changed received");\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnSkyControllerConnectionChangedReceived')+' (deviceController);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnARDrone3VideoEnableChanged')+' (deviceController, elementDictionary);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    
    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnJumpingSumoVideoEnableChanged')+' (deviceController, elementDictionary);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')

    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnPowerUpVideoEnableChanged')+' (deviceController, elementDictionary);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')

    cFile.write ('            case ARCONTROLLER_DICTIONARY_KEY_COMMON_AUDIOSTATE_AUDIOSTREAMINGRUNNING:\n')
    cFile.write ('                ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAudioStreamStateChanged')+' (deviceController, elementDictionary);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')

    cFile.write ('            default :\n')
    cFile.write ('                //Do Nothing\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllStatesEnd')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    
    cFile.write ('    // -- End of receiving of initial all States --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Sem_Post (&(deviceController->privatePart->initSem));\n') # TODO manage error
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAllSettingsEnd')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- End of receiving of initial all Settings --\n')
    cFile.write ('    \n')
    
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_Sem_Post (&(deviceController->privatePart->initSem));\n') # TODO manage error
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnSkyControllerConnectionChangedReceived')+' ('+className+' *deviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *element = NULL;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ARG_t *arg = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    eARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS connectionStatus = ARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        element = ARCONTROLLER_Device_GetCommandElements(deviceController, ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED, &error);\n')
    cFile.write ('    }\n')
    cFile.write ('    // read the connection status from the command\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS, arg);\n')
    cFile.write ('        if (arg != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            connectionStatus = arg->value.U8;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "ConnectionStatus : %i", connectionStatus);\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        // if not all args are valid, generate an error\n')
    cFile.write ('        if (connectionStatus == ARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS_MAX)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (connectionStatus)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS_CONNECTED:\n')
    cFile.write ('            {\n')
    cFile.write ('                ARSAL_Thread_t startingThread = NULL;\n')
    cFile.write ('                char *productName = NULL;\n')
    cFile.write ('                eARDISCOVERY_PRODUCT product = ARDISCOVERY_PRODUCT_MAX;\n')
    cFile.write ('                \n')
    cFile.write ('                // read product and name\n')
    cFile.write ('                if (error == ARCONTROLLER_OK)\n')
    cFile.write ('                {\n')
    cFile.write ('                    HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_DEVICENAME, arg);\n')
    cFile.write ('                    if (arg != NULL)\n')
    cFile.write ('                    {\n')
    cFile.write ('                        productName = arg->value.String;\n')
    cFile.write ('                    }\n')
    cFile.write ('                    \n')
    cFile.write ('                    HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_DEVICEPRODUCTID, arg);\n')
    cFile.write ('                    if (arg != NULL)\n')
    cFile.write ('                    {\n')
    cFile.write ('                        product = ARDISCOVERY_getProductFromProductID(arg->value.U16);\n')
    cFile.write ('                    }\n')
    cFile.write ('                    \n')
    cFile.write ('                    // if not all args are valid, generate an error\n')
    cFile.write ('                    if ((productName == NULL) ||\n')
    cFile.write ('                        (product == ARDISCOVERY_PRODUCT_MAX))\n')
    cFile.write ('                    {\n')
    cFile.write ('                        error = ARCONTROLLER_ERROR;\n')
    cFile.write ('                    }\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')
    cFile.write ('                // product should be one of the product that can be an extension\n')
    cFile.write ('                if (error == ARCONTROLLER_OK)\n')
    cFile.write ('                {\n')
    cFile.write ('                    if (\n')
    isFirst = True
    for deviceController in [x for x in deviceControllers if x.flags.can_be_extension]:
        if isFirst:
            isFirst = False
        else:
            cFile.write (' &&\n')
        cFile.write ('                        (product != '+discoveryProduct(deviceController.product)+')')
    cFile.write (')\n')

    cFile.write ('                    {\n')
    cFile.write ('                        error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
    cFile.write ('                    }\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')
    cFile.write ('                if (error == ARCONTROLLER_OK)\n')
    cFile.write ('                {\n')
    cFile.write ('                    deviceController->privatePart->extensionProduct = product;\n')
    cFile.write ('                    if (asprintf((&(deviceController->privatePart->extensionName)), "%s", productName) <= 0)\n')
    cFile.write ('                    {\n')
    cFile.write ('                        error = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('                    }\n')
    cFile.write ('                }\n')
    cFile.write ('                \n')
    cFile.write ('                if (error == ARCONTROLLER_OK)\n')
    cFile.write ('                {\n')
    cFile.write ('                    ARCONTROLLER_Device_SetExtensionState(deviceController, ARCONTROLLER_DEVICE_STATE_STARTING, error);\n')
    cFile.write ('                    if (ARSAL_Thread_Create (&startingThread, ARCONTROLLER_Device_ExtensionStartRun, deviceController) != 0)\n')
    cFile.write ('                    {\n')
    cFile.write ('                        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Creation of Starting thread failed.");\n')
    cFile.write ('                        error = ARCONTROLLER_ERROR_INIT_THREAD;\n')
    cFile.write ('                    }\n')
    cFile.write ('                    else\n')
    cFile.write ('                    {\n')
    cFile.write ('                        // Destroy the pointer because we don\'t care about it. The thread will run anyway\n')
    cFile.write ('                        ARSAL_Thread_Destroy (&startingThread);\n')
    cFile.write ('                        startingThread = NULL;\n')
    cFile.write ('                        \n')
    cFile.write ('                        // TODO: what to do ???\n')
    cFile.write ('                    }\n')
    cFile.write ('                }\n')
    cFile.write ('            }\n')
    cFile.write ('            break;\n')
    cFile.write ('            case ARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS_DISCONNECTING:\n')
    cFile.write ('            \n')
    cFile.write ('                ARCONTROLLER_Device_SetExtensionState(deviceController, ARCONTROLLER_DEVICE_STATE_STOPPING, error);\n')
    cFile.write ('                \n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    cFile.write ('            case ARCOMMANDS_SKYCONTROLLER_DEVICESTATE_CONNEXIONCHANGED_STATUS_NOTCONNECTED:\n')
    cFile.write ('            {\n')
    cFile.write ('                if (deviceController->privatePart->extensionProduct != ARDISCOVERY_PRODUCT_MAX)\n')
    cFile.write ('                {\n')
    cFile.write ('                    ARSAL_Thread_t stoppingThread = NULL;\n')
    cFile.write ('                    \n')
    cFile.write ('                    if (ARSAL_Thread_Create (&stoppingThread, ARCONTROLLER_Device_ExtensionStopRun, deviceController) != 0)\n')
    cFile.write ('                    {\n')
    cFile.write ('                        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Creation of Stopping thread failed.");\n')
    cFile.write ('                        error = ARCONTROLLER_ERROR_INIT_THREAD;\n')
    cFile.write ('                    }\n')
    cFile.write ('                    else\n')
    cFile.write ('                    {\n')
    cFile.write ('                        // Destroy the pointer because we don\'t care about it. The thread will run anyway\n')
    cFile.write ('                        ARSAL_Thread_Destroy (&stoppingThread);\n')
    cFile.write ('                        stoppingThread = NULL;\n')
    cFile.write ('                        \n')
    cFile.write ('                        // TODO: what to do ???\n')
    cFile.write ('                    }\n')
    cFile.write ('                    \n')
    cFile.write ('                }\n')
    cFile.write ('            }\n')
    cFile.write ('            break;\n')
    cFile.write ('            \n')
    cFile.write ('            default:\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Error ARController_Device_OnSkyControllerConnectionChangedReceived : %s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')

    cFile.write ('void *ARCONTROLLER_Device_ExtensionStartRun (void *data)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Start the Thread of the Extension --\n')
    cFile.write ('\n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    ARCONTROLLER_Device_t *deviceController = (ARCONTROLLER_Device_t *) data;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('\n')

    cFile.write ('    ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('    switch (deviceController->privatePart->extensionProduct)\n')
    cFile.write ('    {\n')
    # For each device controller that can be an extension
    for deviceController in [x for x in deviceControllers if x.flags.can_be_extension]:
        cFile.write ('        // If the extension is a ' + discoveryProduct (deviceController.product) + '\n')
        cFile.write ('        case '+discoveryProduct (deviceController.product)+':\n')
        cFile.write ('            switch (deviceController->privatePart->discoveryDevice->productID)\n')
        cFile.write ('            {\n')
        # For each device controller that can have an extension
        for extensionDeviceController in [y for y in deviceControllers if y.flags.can_have_extension]:
            cFile.write ('                // If the remote controller on which we are connected is a ' + discoveryProduct (extensionDeviceController.product) + '\n')
            cFile.write ('                case '+discoveryProduct (extensionDeviceController.product)+':\n')

            # for all features of the extension that are not in the list of features of the extender
            for featureName in [z for z in deviceController.features if z not in extensionDeviceController.features]:
                cFile.write ('                    // register the feature ' + ARUncapitalize(ftr_new_to_old_name(featureName)) + '\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+' = ' + ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'New')+' (deviceController->privatePart->networkController, &error);\n')
                cFile.write ('                    }\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        error = ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetNetworkControllerToFeatures') + '(deviceController, deviceController->' + ARUncapitalize(ftr_new_to_old_name(featureName)) + ');\n')
                cFile.write ('                    }\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        error = '+ARFunctionName (MODULE_FEATURE, ftr_new_to_old_name(featureName), 'RegisterARCommands')+' (deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+');\n')
                cFile.write ('                    }\n')
                cFile.write ('                    if (error == ARCONTROLLER_OK)\n')
                cFile.write ('                    {\n')
                cFile.write ('                        error = '+ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RegisterCallbacks')+' (deviceController, deviceController->'+ARUncapitalize(ftr_new_to_old_name(featureName))+');\n')
                cFile.write ('                    }\n')
            cFile.write ('                    break;\n')
        cFile.write ('                default:\n')
        cFile.write ('                    error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
        cFile.write ('            }\n')
        cFile.write ('            break;\n')
    cFile.write ('        default:\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_EXTENSION_PRODUCT_NOT_VALID;\n')
    cFile.write ('    }\n')
    cFile.write ('    ARSAL_Mutex_Unlock(&(deviceController->privatePart->mutex));\n')

    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_OnStart (deviceController, 1);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "Error ExtensionStartRun : %s", ARCONTROLLER_Error_ToString (error));\n')
    cFile.write ('    }\n')
    cFile.write ('    ARCONTROLLER_Device_SetExtensionState (deviceController, ARCONTROLLER_DEVICE_STATE_RUNNING, error);\n')
    cFile.write ('    \n')
    cFile.write ('    return NULL;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnARDrone3VideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- ARDrone3 video enable changed --\n')
    cFile.write ('\n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ARG_t *arg = NULL;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *element = NULL;\n')
    cFile.write ('    eARCOMMANDS_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED videoState = ARCOMMANDS_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL)||\n')
    cFile.write ('        (elementDictionary == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the command received in the device controller\n')
    cFile.write ('        HASH_FIND_STR (elementDictionary, ARCONTROLLER_DICTIONARY_SINGLE_KEY, element);\n')
    cFile.write ('        \n')
    cFile.write ('        if (element == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ELEMENT;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "element is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the value\n')
    cFile.write ('        HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED, arg);\n')
    cFile.write ('        \n')
    cFile.write ('        if (arg != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            videoState = arg->value.I32;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ARGUMENTS;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "argument is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (videoState)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCOMMANDS_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ENABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StartVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_DISABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StopVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_ARDRONE3_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ERROR:\n')
    cFile.write ('                //Do nothing\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "videoState unknown :%d ", videoState);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnJumpingSumoVideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Jumping Sumo video enable changed --\n')
    cFile.write ('\n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ARG_t *arg = NULL;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *element = NULL;\n')
    cFile.write ('    eARCOMMANDS_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED videoState = ARCOMMANDS_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL)||\n')
    cFile.write ('        (elementDictionary == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the command received in the device controller\n')
    cFile.write ('        HASH_FIND_STR (elementDictionary, ARCONTROLLER_DICTIONARY_SINGLE_KEY, element);\n')
    cFile.write ('        \n')
    cFile.write ('        if (element == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ELEMENT;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "element is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the value\n')
    cFile.write ('        HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED, arg);\n')
    cFile.write ('        \n')
    cFile.write ('        if (arg != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            videoState = arg->value.I32;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ARGUMENTS;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "argument is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (videoState)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCOMMANDS_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ENABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StartVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_DISABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StopVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_JUMPINGSUMO_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ERROR:\n')
    cFile.write ('                //Do nothing\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "videoState unknown :%d ", videoState);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnPowerUpVideoEnableChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- PowerUp video enable changed --\n')
    cFile.write ('\n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ARG_t *arg = NULL;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *element = NULL;\n')
    cFile.write ('    eARCOMMANDS_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED videoState = ARCOMMANDS_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL)||\n')
    cFile.write ('        (elementDictionary == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the command received in the device controller\n')
    cFile.write ('        HASH_FIND_STR (elementDictionary, ARCONTROLLER_DICTIONARY_SINGLE_KEY, element);\n')
    cFile.write ('        \n')
    cFile.write ('        if (element == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ELEMENT;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "element is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the value\n')
    cFile.write ('        HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED, arg);\n')
    cFile.write ('        \n')
    cFile.write ('        if (arg != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            videoState = arg->value.I32;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ARGUMENTS;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "argument is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch (videoState)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCOMMANDS_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ENABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StartVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_DISABLED:\n')
    cFile.write ('                ARCONTROLLER_Network_StopVideoStream(deviceController->privatePart->networkController);\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            case ARCOMMANDS_POWERUP_MEDIASTREAMINGSTATE_VIDEOENABLECHANGED_ENABLED_ERROR:\n')
    cFile.write ('                //Do nothing\n')
    cFile.write ('                break;\n')
    cFile.write ('                \n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "videoState unknown :%d ", videoState);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnAudioStreamStateChanged')+' (ARCONTROLLER_Device_t *deviceController, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Audio stream state changed --\n')
    cFile.write ('\n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ARG_t *arg = NULL;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *element = NULL;\n')
    cFile.write ('    uint8_t state = 0;\n')
    cFile.write ('    \n')
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL)||\n')
    cFile.write ('        (elementDictionary == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the command received in the device controller\n')
    cFile.write ('        HASH_FIND_STR (elementDictionary, ARCONTROLLER_DICTIONARY_SINGLE_KEY, element);\n')
    cFile.write ('        \n')
    cFile.write ('        if (element == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ELEMENT;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "element is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the value\n')
    cFile.write ('        HASH_FIND_STR (element->arguments, ARCONTROLLER_DICTIONARY_KEY_COMMON_AUDIOSTATE_AUDIOSTREAMINGRUNNING_RUNNING, arg);\n')
    cFile.write ('        \n')
    cFile.write ('        if (arg != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            state = arg->value.U8;\n')
    cFile.write ('        }\n')
    cFile.write ('        else\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR_NO_ARGUMENTS;\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_ERROR, ARCONTROLLER_DEVICE_TAG, "argument is NULL");\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        ARCONTROLLER_Network_StopAudioStream(deviceController->privatePart->networkController);\n')
    cFile.write ('\n')
    cFile.write ('        if (state != 0)\n')
    cFile.write ('        {\n')
    cFile.write ('                ARCONTROLLER_Network_StartAudioStream(deviceController->privatePart->networkController);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARDISCOVERY_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SendJsonCallback')+' (json_object *jsonObj, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Connection callback to receive the Json --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+className+' *deviceController = customData;\n')
    cFile.write ('    eARDISCOVERY_ERROR error = ARDISCOVERY_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    json_object *valueJsonObj = NULL;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((jsonObj == NULL) ||\n')
    cFile.write ('        (deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARDISCOVERY_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARDISCOVERY_OK)\n')
    cFile.write ('    {\n')
    
    cFile.write ('        // add ARDISCOVERY_CONNECTION_JSON_CONTROLLER_NAME_KEY\n')
    cFile.write ('        valueJsonObj = json_object_new_string ("DEFAULT_SDK_CONTROLLER"); //TODO get from the device controller\n')
    cFile.write ('        json_object_object_add (jsonObj, ARDISCOVERY_CONNECTION_JSON_CONTROLLER_NAME_KEY, valueJsonObj);\n')
    cFile.write ('        \n')
    
    cFile.write ('        // add ARDISCOVERY_CONNECTION_JSON_CONTROLLER_TYPE_KEY\n')
    cFile.write ('        valueJsonObj = json_object_new_string ("DEFAULT_SDK_TYPE"); //TODO get from the device controller\n')
    cFile.write ('        json_object_object_add (jsonObj, ARDISCOVERY_CONNECTION_JSON_CONTROLLER_TYPE_KEY, valueJsonObj);\n')
    cFile.write ('        \n')
    
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARDISCOVERY_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ReceiveJsonCallback')+' (json_object *jsonObj, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('     // -- Connection callback to receive the Json --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+className+' *deviceController = customData;\n')
    cFile.write ('    eARDISCOVERY_ERROR error = ARDISCOVERY_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    if ((jsonObj == NULL) ||\n')
    cFile.write ('        (deviceController == NULL) ||\n')
    cFile.write ('        (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARDISCOVERY_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARDISCOVERY_OK)\n')
    cFile.write ('    {\n')
    #TODO see what to do or remove function !!!!!!!!!!!!!!!!!!!!!!
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write  ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'OnDisconnectCallback')+' (void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('     // -- Disconnection callback --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+className+' *deviceController = ('+className+' *) customData;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error =' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'Stop')+' (deviceController);\n')
    cFile.write ('    }\n')
    
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void *' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ControllerLooperThread')+' (void *data)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Sending Looper Thread --\n')
    cFile.write ('    \n')

    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    '+className+' *deviceController = data;\n')
    cFile.write ('    u_int8_t cmdBuffer['+ARMacroName (MODULE_ARCONTROLLER, 'Device', 'DEFAULT_LOOPER_CMD_BUFFER_SIZE')+'];\n')
    cFile.write ('    int controllerLoopIntervalUs = 0;\n')
    cFile.write ('    int mustBeSent = 0;\n')
    cFile.write ('    \n')

    cFile.write ('    // Check parameters\n')
    cFile.write ('    if ((deviceController == NULL) || (deviceController->privatePart == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    // No Else: the checking parameters sets error to ARNETWORK_ERROR_BAD_PARAMETER and stop the processing\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        controllerLoopIntervalUs = MSEC_TO_USEC (deviceController->privatePart->networkConfiguration.controllerLoopIntervalMs);\n')
    cFile.write ('        if (!(controllerLoopIntervalUs > 0))\n')
    cFile.write ('        {\n')
    cFile.write ('            error = ARCONTROLLER_ERROR;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        while ((deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_RUNNING) ||\n')
    cFile.write ('               (deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_STARTING) ||\n')
    cFile.write ('               (deviceController->privatePart->state == ARCONTROLLER_DEVICE_STATE_PAUSED))\n')
    cFile.write ('        {\n')
    cFile.write ('            usleep (controllerLoopIntervalUs);\n')
    cFile.write ('\n')
    cFile.write ('            if (deviceController->privatePart->state != ARCONTROLLER_DEVICE_STATE_RUNNING)\n')
    cFile.write ('                continue;\n')
    cFile.write ('\n')
    cFile.write ('            ARSAL_Mutex_Lock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('\n')

    for feature in ctx.features:
        #if there are NON_ACK cmd
        if [cmd for cmd in feature.cmds if cmd.bufferType == ArCmdBufferType.NON_ACK]:
            cFile.write ('            if (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+' != NULL)\n')
            cFile.write ('            {\n')
            for cmd in feature.cmds:
                if cmd.bufferType == ArCmdBufferType.NON_ACK:
                    cFile.write ('                mustBeSent = '+nAckCbMustBeSent (feature, cmd)+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+');\n')
                    cFile.write ('                if (mustBeSent)\n')
                    cFile.write ('                {\n')
                    cFile.write ('                    error = '+ sendNAckFunctionName (feature, cmd)+' (deviceController->'+ARUncapitalize(get_ftr_old_name(feature))+', cmdBuffer, '+ARMacroName (MODULE_ARCONTROLLER, 'Device', 'DEFAULT_LOOPER_CMD_BUFFER_SIZE')+');\n')
                    cFile.write ('                    if (error != ARCONTROLLER_OK)\n')
                    cFile.write ('                    {\n')
                    cFile.write ('                        ARSAL_PRINT (ARSAL_PRINT_ERROR, '+MODULE_DEVICE+'_TAG, "Error occured while send '+cmd.name+' : %s", ARCONTROLLER_Error_ToString (error));\n')
                    cFile.write ('                    }\n')
                    cFile.write ('                }\n')
                    cFile.write ('\n')
            cFile.write ('            }\n')
            cFile.write ('\n')
    cFile.write ('            ARSAL_Mutex_Unlock(&(deviceController->privatePart->mutex));\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    
    cFile.write ('    return NULL;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' callback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Add callback in array --\n')
    cFile.write ('    \n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *newElement = NULL;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Add the callback\n')
    cFile.write ('    newElement = malloc (sizeof('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+'));\n')
    cFile.write ('    if (newElement != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        newElement->callback = callback;\n')
    cFile.write ('        newElement->customData = customData;\n')
    cFile.write ('        DL_APPEND ((*callbackList), newElement);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'AddExtensionCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' callback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Add extension callback in array --\n')
    cFile.write ('    \n')
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *newElement = NULL;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Add the callback\n')
    cFile.write ('    newElement = malloc (sizeof('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+'));\n')
    cFile.write ('    if (newElement != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        newElement->callback = callback;\n')
    cFile.write ('        newElement->customData = customData;\n')
    cFile.write ('        DL_APPEND ((*callbackList), newElement);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveCallbackFromList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'StateChangedCallback')+' callback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Remove callback from array --\n')
    cFile.write ('\n')
        
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *elementFind = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' likeElement;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Element to find\n')
    cFile.write ('    likeElement.callback = callback;\n')
    cFile.write ('    likeElement.customData = customData;\n')
    cFile.write ('\n')
        
    cFile.write ('    DL_SEARCH ((*callbackList), elementFind, &likeElement, ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ElementCompare')+');\n')
    cFile.write ('    if (elementFind != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        DL_DELETE ((*callbackList), elementFind);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_COMMAND_CALLBACK_NOT_REGISTERED;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'RemoveExtensionCallbackFromList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'ExtensionStateChangedCallback')+' callback, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Remove callback from array --\n')
    cFile.write ('\n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *elementFind = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' likeElement;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Element to find\n')
    cFile.write ('    likeElement.callback = callback;\n')
    cFile.write ('    likeElement.customData = customData;\n')
    cFile.write ('\n')
    
    cFile.write ('    DL_SEARCH ((*callbackList), elementFind, &likeElement, ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ExtensionElementCompare')+');\n')
    cFile.write ('    if (elementFind != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        DL_DELETE ((*callbackList), elementFind);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_ERROR_COMMAND_CALLBACK_NOT_REGISTERED;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return error;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteCallbackList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Delete all callback in array --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *element = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *elementTmp = NULL;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Delete each element, use the safe iterator\n')
    cFile.write ('    DL_FOREACH_SAFE ((*callbackList), element, elementTmp)\n')
    cFile.write ('    {\n')
    cFile.write ('        DL_DELETE ((*callbackList), element);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'DeleteExtensionCallbackList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Delete all callback in array --\n')
    cFile.write ('    \n')
    
    cFile.write ('    // Local declarations\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *element = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *elementTmp = NULL;\n')
    cFile.write ('\n')
    
    cFile.write ('    // Delete each element, use the safe iterator\n')
    cFile.write ('    DL_FOREACH_SAFE ((*callbackList), element, elementTmp)\n')
    cFile.write ('    {\n')
    cFile.write ('        DL_DELETE ((*callbackList), element);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, eARCONTROLLER_DEVICE_STATE state, eARCONTROLLER_ERROR error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Notify all listeners --\n')
    cFile.write ('    \n')
        
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *callbackElement = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *callbackElementTmp = NULL;\n')
    cFile.write ('\n')
       
    cFile.write ('    // for each callback\n')
    cFile.write ('    DL_FOREACH_SAFE ((*callbackList), callbackElement, callbackElementTmp)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (callbackElement->callback != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            callbackElement->callback (state, error, callbackElement->customData);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllExtensionCallbackInList')+' ('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' **callbackList, eARCONTROLLER_DEVICE_STATE state, eARDISCOVERY_PRODUCT product, const char *name, eARCONTROLLER_ERROR error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Notify all listeners --\n')
    cFile.write ('    \n')
    
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *callbackElement = NULL;\n')
    cFile.write ('    '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *callbackElementTmp = NULL;\n')
    cFile.write ('\n')
    
    cFile.write ('    // for each callback\n')
    cFile.write ('    DL_FOREACH_SAFE ((*callbackList), callbackElement, callbackElementTmp)\n')
    cFile.write ('    {\n')
    cFile.write ('        if (callbackElement->callback != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            callbackElement->callback (state, product, name, error, callbackElement->customData);\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetState')+' ('+className+' *deviceController, eARCONTROLLER_DEVICE_STATE state,  eARCONTROLLER_ERROR error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set the Device Controller State and notify all listeners.\n')
    cFile.write ('    \n')
    
    cFile.write ('    ARSAL_Mutex_Lock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    deviceController->privatePart->state = state;\n')
    cFile.write ('    ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllCallbackInList')+' (&(deviceController->privatePart->stateChangedCallbacks), deviceController->privatePart->state, error);\n')
    cFile.write ('    \n')
    
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('void ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'SetExtensionState')+' ('+className+' *deviceController, eARCONTROLLER_DEVICE_STATE state,  eARCONTROLLER_ERROR error)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Set the Device Controller Extension State and notify all listeners.\n')
    cFile.write ('    \n')
    
    cFile.write ('    ARSAL_Mutex_Lock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    deviceController->privatePart->extensionState = state;\n')
    cFile.write ('    ARSAL_Mutex_Unlock (&(deviceController->privatePart->mutex));\n')
    cFile.write ('    ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'NotifyAllExtensionCallbackInList')+' (&(deviceController->privatePart->extensionStateChangedCallbacks), deviceController->privatePart->extensionState, deviceController->privatePart->extensionProduct, deviceController->privatePart->extensionName, error);\n')
    cFile.write ('    \n')
    
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             local implementation:\n')
    cFile.write (' *\n')
    cFile.write (' ****************************************/\n')
    cFile.write ('\n')

    cFile.write ('static int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ElementCompare')+'('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *a, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'STATE_CHANGED_CALLBACK_ELEMENT')+' *b)\n')
    cFile.write ('{\n')
    cFile.write ('    return !((a->callback == b->callback) && (a->customData == b->customData));\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('static int ' + ARFunctionName (MODULE_ARCONTROLLER, 'device', 'ExtensionElementCompare')+'('+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *a, '+ARTypeName (MODULE_ARCONTROLLER, 'device', 'EXTENSION_STATE_CHANGED_CALLBACK_ELEMENT')+' *b)\n')
    cFile.write ('{\n')
    cFile.write ('    return !((a->callback == b->callback) && (a->customData == b->customData));\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    
    cFile.close ()

def generateControllersJava (ctx, JNI_JAVA_DIR):
    
    #########################################
    # Write Device controller header file   #
    #########################################
    
    className = 'ARDeviceController'
    classPrivateName = ARTypeName (MODULE_ARCONTROLLER, 'device', 'private')

    fileName = CTRL_DEVICE_JAVA_NAME
    filepath = JNI_JAVA_DIR + fileName
    jfile = open (filepath, 'w')

    jfile.write ('/**********************************************************\n')
    jfile.write (' *            AUTOGENERATED FILE                          *\n')
    jfile.write (' *             DO NOT MODIFY IT                           *\n')
    jfile.write (' *                                                        *\n')
    jfile.write (' * To add new commands :                                  *\n')
    jfile.write (' *  - Modify ../Xml/commands.xml file                     *\n')
    jfile.write (' *  - Re-run generateDeviceControllers.py script          *\n')
    jfile.write (' *                                                        *\n')
    jfile.write (' **********************************************************/\n')
    jfile.write ('\n')

    jfile.write ('/**\n')
    jfile.write ('* @file '+fileName+'\n')
    jfile.write ('* @brief '+bref+'\n')
    jfile.write ('*/\n')
    jfile.write ('\n')
    
    jfile.write ('package com.parrot.arsdk.arcontroller;\n')
    jfile.write ('\n')
    jfile.write ('import com.parrot.arsdk.ardiscovery.ARDISCOVERY_PRODUCT_ENUM;\n')
    jfile.write ('import com.parrot.arsdk.arsal.ARSALPrint;\n')
    jfile.write ('import com.parrot.arsdk.arcommands.*;\n')
    jfile.write ('import com.parrot.arsdk.ardiscovery.ARDiscoveryDevice;\n')
    jfile.write ('\n')
    jfile.write ('import com.parrot.arsdk.arsal.ARNativeData;\n')
    jfile.write ('\n')
    jfile.write ('import java.util.List;\n')
    jfile.write ('import java.util.ArrayList;\n')
    jfile.write ('\n')
    jfile.write ('public class ARDeviceController\n')
    jfile.write ('{\n')
    jfile.write ('    private static String TAG = "ARDeviceController";\n')
    jfile.write ('    \n')
    jfile.write ('    private static native void nativeStaticInit ();\n')
    jfile.write ('    \n')
    jfile.write ('    private native long nativeNew(long jdevice) throws ARControllerException;\n')
    jfile.write ('    private native void nativeDelete(long jARDeviceController);\n')
    jfile.write ('    \n')
    jfile.write ('    private native int nativeStart (long jDeviceController);\n')
    jfile.write ('    private native int nativeStop (long jDeviceController);\n')
    jfile.write ('    \n')
    
    for feature in ctx.features:
        jfile.write ('    private native long '+nativeGetFeature(feature)+' (long jDeviceController);\n')
        
    jfile.write ('    private native int nativeGetState (long jDeviceController) throws ARControllerException;\n')
    jfile.write ('    private native int nativeGetExtensionState (long jDeviceController);\n')
    jfile.write ('    private native String nativeGetExtensionName (long jDeviceController);\n')
    jfile.write ('    private native int nativeGetExtensionProduct (long jDeviceController);\n')
    jfile.write ('    private native long nativeGetCommandElements(long jDeviceController, int commandKey) throws ARControllerException;\n')
    jfile.write ('    private native int nativeSendStreamFrame (long jDeviceController, long data, int dataSize);\n')
    jfile.write ('    private native int nativeHasOutputVideoStream (long deviceController) throws ARControllerException;\n')
    jfile.write ('    private native int nativeHasOutputAudioStream (long deviceController) throws ARControllerException;\n')
    jfile.write ('    private native int nativeHasInputAudioStream (long deviceControlle) throws ARControllerException;\n')


    jfile.write ('\n')
    jfile.write ('    private long jniDeviceController;\n')
    jfile.write ('    private boolean initOk;\n')
    jfile.write ('    \n')
    jfile.write ('    private List<ARDeviceControllerListener> listeners;\n')
    jfile.write ('    private List<ARDeviceControllerStreamListener> streamlisteners;\n')
    jfile.write ('    private List<ARDeviceControllerStreamListener> audioStreamlisteners;\n')

    
    for feature in ctx.features:
        jfile.write ('    '+javaFeatureClassName(feature)+' '+javaFeatureName(feature)+';\n')
    
    
    jfile.write ('    \n')
    jfile.write ('    static\n')
    jfile.write ('    {\n')
    jfile.write ('        nativeStaticInit();\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    /**\n')
    jfile.write ('     * Constructor\n')
    jfile.write ('     */\n')
    jfile.write ('    public ARDeviceController (ARDiscoveryDevice device) throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        initOk = false;\n')
    jfile.write ('        ARCONTROLLER_ERROR_ENUM error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK;\n')
    jfile.write ('        \n')
    jfile.write ('        if (device != null)\n')
    jfile.write ('        {\n')
    jfile.write ('            jniDeviceController = nativeNew(device.getNativeDevice());\n')
    jfile.write ('        }\n')
    jfile.write ('        else\n')
    jfile.write ('        {\n')
    jfile.write ('            error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        if (jniDeviceController != 0)\n')
    jfile.write ('        {\n')
    jfile.write ('            listeners = new ArrayList<ARDeviceControllerListener>();\n')
    jfile.write ('            streamlisteners = new ArrayList<ARDeviceControllerStreamListener>();\n')
    jfile.write ('            audioStreamlisteners = new ArrayList<ARDeviceControllerStreamListener>();\n')
    jfile.write ('            initOk = true;\n')
    jfile.write ('            \n')

    jfile.write ('            reloadFeatures();\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        if (error != ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK)\n')
    jfile.write ('        {\n')
    jfile.write ('            throw new ARControllerException (error);\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    /**\n')
    jfile.write ('     * Dispose\n')
    jfile.write ('     */\n')
    jfile.write ('    public void dispose()\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_ERROR_ENUM error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                nativeDelete(jniDeviceController);\n')
    jfile.write ('                jniDeviceController = 0;\n')
    jfile.write ('                initOk = false;\n')
    jfile.write ('                \n')
    for feature in ctx.features:
        jfile.write ('                if ('+javaFeatureName(feature)+' != null)\n')
        jfile.write ('                {\n')
        jfile.write ('                    '+javaFeatureName(feature)+'.dispose();\n')
        jfile.write ('                    '+javaFeatureName(feature)+' = null;\n')
        jfile.write ('                }\n')
        jfile.write ('                \n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    /**\n')
    jfile.write ('     * Destructor\n')
    jfile.write ('     */\n')
    jfile.write ('    public void finalize () throws Throwable\n')
    jfile.write ('    {\n')
    jfile.write ('        try\n')
    jfile.write ('        {\n')
    jfile.write ('            dispose ();\n')
    jfile.write ('        }\n')
    jfile.write ('        finally\n')
    jfile.write ('        {\n')
    jfile.write ('            super.finalize ();\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    public ARCONTROLLER_ERROR_ENUM start ()\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_ERROR_ENUM error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeError = nativeStart(jniDeviceController);\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.getFromValue(nativeError);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return error;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    public ARCONTROLLER_ERROR_ENUM stop ()\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_ERROR_ENUM error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeError = nativeStop(jniDeviceController);\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.getFromValue(nativeError);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return error;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    
    for feature in ctx.features:
        jfile.write ('    public '+javaFeatureClassName(feature)+' getFeature'+ ARCapitalize(get_ftr_old_name(feature))+' ()\n')
        jfile.write ('    {\n')
        jfile.write ('        return '+javaFeatureName(feature)+';\n')
        jfile.write ('    }\n')
        jfile.write ('    \n')
    
    jfile.write ('    public ARControllerDictionary getCommandElements (ARCONTROLLER_DICTIONARY_KEY_ENUM commandKey) throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        ARControllerDictionary elementDictionary = null;\n')
    jfile.write ('        \n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                if(commandKey != null)\n')
    jfile.write ('                {\n')
    jfile.write ('                    long nativeElementDictionary = nativeGetCommandElements(jniDeviceController, commandKey.getValue());\n')
    jfile.write ('                    \n')
    jfile.write ('                    elementDictionary = new ARControllerDictionary(nativeElementDictionary);\n')
    jfile.write ('                }\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return elementDictionary;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    
    jfile.write ('    public ARCONTROLLER_ERROR_ENUM sendStreamingFrame (ARNativeData data)\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_ERROR_ENUM error = ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK;\n')
    jfile.write ('        \n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeError = nativeSendStreamFrame(jniDeviceController, data.getData(), data.getDataSize());\n')
    jfile.write ('                error = ARCONTROLLER_ERROR_ENUM.getFromValue(nativeError);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return error;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    
    jfile.write ('    public ARCONTROLLER_DEVICE_STATE_ENUM getState () throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_DEVICE_STATE_ENUM state = ARCONTROLLER_DEVICE_STATE_ENUM.ARCONTROLLER_DEVICE_STATE_MAX;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeState = nativeGetState(jniDeviceController);\n')
    jfile.write ('                state = ARCONTROLLER_DEVICE_STATE_ENUM.getFromValue(nativeState);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                throw new ARControllerException(ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return state;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    public ARCONTROLLER_DEVICE_STATE_ENUM getExtensionState ()\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_DEVICE_STATE_ENUM extensionState = ARCONTROLLER_DEVICE_STATE_ENUM.ARCONTROLLER_DEVICE_STATE_MAX;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeExtensionState = nativeGetExtensionState(jniDeviceController);\n')
    jfile.write ('                extensionState = ARCONTROLLER_DEVICE_STATE_ENUM.getFromValue(nativeExtensionState);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return extensionState;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    public boolean hasOutputVideoStream () throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean res = false;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeRes = nativeHasOutputVideoStream(jniDeviceController);\n')
    jfile.write ('                res = (nativeRes != 0);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                throw new ARControllerException(ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        return res;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    public boolean hasOutputAudioStream () throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean res = false;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeRes = nativeHasOutputAudioStream(jniDeviceController);\n')
    jfile.write ('                res = (nativeRes != 0);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                throw new ARControllerException(ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        return res;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    public boolean hasInputAudioStream () throws ARControllerException\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean res = false;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeRes = nativeHasInputAudioStream(jniDeviceController);\n')
    jfile.write ('                res = (nativeRes != 0);\n')
    jfile.write ('            }\n')
    jfile.write ('            else\n')
    jfile.write ('            {\n')
    jfile.write ('                throw new ARControllerException(ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR_JNI_INIT);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        return res;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    public String getExtensionName ()\n')
    jfile.write ('    {\n')
    jfile.write ('        String extensionName = null;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                extensionName = nativeGetExtensionName(jniDeviceController);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return extensionName;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    public ARDISCOVERY_PRODUCT_ENUM getExtensionProduct ()\n')
    jfile.write ('    {\n')
    jfile.write ('        ARDISCOVERY_PRODUCT_ENUM extensionProduct = ARDISCOVERY_PRODUCT_ENUM.ARDISCOVERY_PRODUCT_MAX;\n')
    jfile.write ('        synchronized (this)\n')
    jfile.write ('        {\n')
    jfile.write ('            if(initOk == true)\n')
    jfile.write ('            {\n')
    jfile.write ('                int nativeExtensionProduct = nativeGetExtensionProduct(jniDeviceController);\n')
    jfile.write ('                extensionProduct = ARDISCOVERY_PRODUCT_ENUM.getFromValue(nativeExtensionProduct);\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        return extensionProduct;\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
                 
    jfile.write ('    public synchronized void addListener (ARDeviceControllerListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('       if (! listeners.contains(listener))\n')
    jfile.write ('       {\n')
    jfile.write ('           listeners.add (listener);\n')
    jfile.write ('       }\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    public synchronized void removeListener (ARDeviceControllerListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('        listeners.remove (listener);\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    public synchronized void addStreamListener (ARDeviceControllerStreamListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('       if (! streamlisteners.contains(listener))\n')
    jfile.write ('       {\n')
    jfile.write ('           streamlisteners.add (listener);\n')
    jfile.write ('       }\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    public synchronized void removeStreamListener (ARDeviceControllerStreamListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('        streamlisteners.remove (listener);\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    public synchronized void addAudioStreamListener (ARDeviceControllerStreamListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('        if (! audioStreamlisteners.contains(listener))\n')
    jfile.write ('        {\n')
    jfile.write ('            audioStreamlisteners.add (listener);\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    public synchronized void removeAudioStreamListener (ARDeviceControllerStreamListener listener)\n')
    jfile.write ('    {\n')
    jfile.write ('        audioStreamlisteners.remove (listener);\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    private void reloadFeatures()\n')
    jfile.write ('    {\n')
    
    for feature in ctx.features:
        jfile.write ('        long '+nativeFeatureName(feature)+' = '+nativeGetFeature(feature)+' (jniDeviceController);\n')
        jfile.write ('        if (('+javaFeatureName(feature)+' == null) && ('+nativeFeatureName(feature)+' != 0))\n')
        jfile.write ('        {\n')
        jfile.write ('            '+javaFeatureName(feature)+' = new '+javaFeatureClassName(feature)+'('+nativeFeatureName(feature)+');\n')
        jfile.write ('        }\n')
        jfile.write ('        else\n')
        jfile.write ('        {\n')
        jfile.write ('            '+javaFeatureName(feature)+' = null;\n')
        jfile.write ('        }\n')
        jfile.write ('        \n')

    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    private void onStateChanged (int newState, int error)\n')
    jfile.write ('    {\n')
    jfile.write ('        for (ARDeviceControllerListener l : listeners)\n')
    jfile.write ('        {\n')
    jfile.write ('            l.onStateChanged (this, ARCONTROLLER_DEVICE_STATE_ENUM.getFromValue(newState), ARCONTROLLER_ERROR_ENUM.getFromValue(error));\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    private void onExtensionStateChanged (int newState, int product, String name, int error)\n')
    jfile.write ('    {\n')
    jfile.write ('        // reload the features because they might have changed\n')
    jfile.write ('        reloadFeatures();\n')
    jfile.write ('        for (ARDeviceControllerListener l : listeners)\n')
    jfile.write ('        {\n')
    jfile.write ('            l.onExtensionStateChanged (this, ARCONTROLLER_DEVICE_STATE_ENUM.getFromValue(newState), ARDISCOVERY_PRODUCT_ENUM.getFromValue(product), name, ARCONTROLLER_ERROR_ENUM.getFromValue(error));\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')

    jfile.write ('    private void onCommandReceived (int nativeCommandKey, long elementDictionary)\n')
    jfile.write ('    {\n')
    jfile.write ('        ARCONTROLLER_DICTIONARY_KEY_ENUM commandKey = ARCONTROLLER_DICTIONARY_KEY_ENUM.getFromValue(nativeCommandKey);\n')
    jfile.write ('        \n')
    jfile.write ('        ARControllerDictionary dictionary = new ARControllerDictionary(elementDictionary);\n')
    jfile.write ('        \n')
    jfile.write ('        for (ARDeviceControllerListener l : listeners)\n')
    jfile.write ('        {\n')
    jfile.write ('            l.onCommandReceived (this, ARCONTROLLER_DICTIONARY_KEY_ENUM.getFromValue(nativeCommandKey), dictionary);\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    private int decoderConfigCallback (ARControllerCodec codec)\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean failed = false;\n')
    jfile.write ('        \n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : streamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            ARCONTROLLER_ERROR_ENUM error = l.configureDecoder(this, codec);\n')
    jfile.write ('            \n')
    jfile.write ('            if (error != ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK)\n')
    jfile.write ('            {\n')
    jfile.write ('                failed = true;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        codec.dispose();\n')
    jfile.write ('        \n')
    jfile.write ('        return (failed) ? ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR.getValue() : ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK.getValue();\n')
    jfile.write ('    }\n')
    jfile.write ('    \n')
    jfile.write ('    private int didReceiveFrameCallback (long data, int dataCapacity, int dataSize, int nativeIsIFrame, int missed, long timestamp, long metadata, int metadataSize)\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean failed = false;\n')
    jfile.write ('        boolean isIFrame = (nativeIsIFrame != 0);\n')
    jfile.write ('        ARFrame frame = new ARFrame (data, dataCapacity, dataSize, isIFrame, missed, timestamp, metadata, metadataSize);\n')
    jfile.write ('        \n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : streamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            ARCONTROLLER_ERROR_ENUM error = l.onFrameReceived (this, frame);\n')
    jfile.write ('            \n')
    jfile.write ('            if (error != ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK)\n')
    jfile.write ('            {\n')
    jfile.write ('                failed = true;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('        \n')
    jfile.write ('        frame.dispose();\n')
    jfile.write ('        \n')
    jfile.write ('        return (failed) ? ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR.getValue() : ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK.getValue();\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    private void timeoutFrameCallback ()\n')
    jfile.write ('    {\n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : streamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            l.onFrameTimeout (this);\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    private int decoderAudioConfigCallback (ARControllerCodec codec)\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean failed = false;\n')
    jfile.write ('\n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : audioStreamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            ARCONTROLLER_ERROR_ENUM error = l.configureDecoder(this, codec);\n')
    jfile.write ('\n')
    jfile.write ('            if (error != ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK)\n')
    jfile.write ('            {\n')
    jfile.write ('                failed = true;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        codec.dispose();\n')
    jfile.write ('\n')
    jfile.write ('        return (failed) ? ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR.getValue() : ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK.getValue();\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    private int didReceiveAudioFrameCallback (long data, int dataCapacity, int dataSize, int nativeIsIFrame, int missed, long timestamp, long metadata, int metadataSize)\n')
    jfile.write ('    {\n')
    jfile.write ('        boolean failed = false;\n')
    jfile.write ('        boolean isIFrame = (nativeIsIFrame != 0);\n')
    jfile.write ('        ARFrame frame = new ARFrame (data, dataCapacity, dataSize, isIFrame, missed, timestamp, metadata, metadataSize);\n')
    jfile.write ('\n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : audioStreamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            ARCONTROLLER_ERROR_ENUM error = l.onFrameReceived (this, frame);\n')
    jfile.write ('\n')
    jfile.write ('            if (error != ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK)\n')
    jfile.write ('            {\n')
    jfile.write ('                failed = true;\n')
    jfile.write ('            }\n')
    jfile.write ('        }\n')
    jfile.write ('\n')
    jfile.write ('        frame.dispose();\n')
    jfile.write ('\n')
    jfile.write ('        return (failed) ? ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_ERROR.getValue() : ARCONTROLLER_ERROR_ENUM.ARCONTROLLER_OK.getValue();\n')
    jfile.write ('    }\n')
    jfile.write ('\n')
    jfile.write ('    private void  timeoutAudioFrameCallback ()\n')
    jfile.write ('    {\n')
    jfile.write ('        for (ARDeviceControllerStreamListener l : audioStreamlisteners)\n')
    jfile.write ('        {\n')
    jfile.write ('            l.onFrameTimeout (this);\n')
    jfile.write ('        }\n')
    jfile.write ('    }\n')
    jfile.write ('}\n')
    jfile.write ('\n')
    jfile.close ()
    
def generateControllersJNI (ctx, JNI_C_DIR):
    #################################################
    # Write Device controller JNI c file           #
    #################################################

    jniClassName = MODULE_ARCONTROLLER + '_JNI_Device'
    className = 'ARCONTROLLER_JNIDeviceController_t'
    classTag = 'ARCONTROLLER_JNIDEVICE_TAG'

    filepath = JNI_C_DIR + CTRL_DEVICE_JNI_C_NAME
    cFile = open (filepath, 'w')

    cFile.write ('/**********************************************************\n')
    cFile.write (' *            AUTOGENERATED FILE                          *\n')
    cFile.write (' *             DO NOT MODIFY IT                           *\n')
    cFile.write (' *                                                        *\n')
    cFile.write (' * To add new commands :                                  *\n')
    cFile.write (' *  - Modify ../Xml/commands.xml file                     *\n')
    cFile.write (' *  - Re-run generateFeatureControllers.py script         *\n')
    cFile.write (' *                                                        *\n')
    cFile.write (' **********************************************************/\n')
    cFile.write ('\n')

    cFile.write ('/**\n')
    cFile.write ('* @file '+jniClassName+'\n')
    cFile.write ('* @brief ARDeviceController JNI device c file.\n')
    cFile.write ('*/\n')
    cFile.write ('\n')
    
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             include file :\n')
    cFile.write (' *\n')
    cFile.write (' *****************************************/\n')
    cFile.write ('\n')
    cFile.write ('#include <jni.h>\n')
    cFile.write ('#include <stdlib.h>\n')
    cFile.write ('\n')
    cFile.write ('#include <libARSAL/ARSAL_Print.h>\n')
    cFile.write ('\n')
    cFile.write ('#include <libARDiscovery/ARDISCOVERY_Device.h>\n')
    cFile.write ('\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Error.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Stream.h>\n')
    cFile.write ('#include <libARController/ARCONTROLLER_Device.h>\n')
    cFile.write ('\n')
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             define :\n')
    cFile.write (' *\n')
    cFile.write (' *****************************************/\n')
    cFile.write ('\n')
    cFile.write ('#define ARCONTROLLER_JNIDEVICE_TAG "JNIControllerDevice"\n')
    cFile.write ('\n')
    cFile.write ('/**\n')
    cFile.write (' * @brief \n')
    cFile.write (' */\n')
    cFile.write ('typedef struct\n')
    cFile.write ('{\n')
    cFile.write ('    ARCONTROLLER_Device_t *nativeDeviceController; /**< native Device Controller*/\n')
    cFile.write ('    jobject jDeviceController; /**< java DeviceController*/\n')
    cFile.write ('}ARCONTROLLER_JNIDeviceController_t;\n')
    cFile.write ('\n')
    cFile.write ('static JavaVM *ARCONTROLLER_JNIDEVICE_VM; /**< reference to the java virtual machine */\n')
    cFile.write ('\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_ON_STATE_CHANGED;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_ON_EXTENSION_STATE_CHANGED;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_ON_COMMAND_RECEIVED;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_DECODER_CONFIG_CALLBACK;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_FRAME_CALLBACK;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_FRAME_CALLBACK;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_DECODER_AUDIO_CONFIG_CALLBACK;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_AUDIO_FRAME_CALLBACK;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_AUDIO_FRAME_CALLBACK;\n')
    cFile.write ('\n')
    cFile.write ('static jclass jARControllerCodecH264Cls;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_H264;\n')
    cFile.write ('\n')
    cFile.write ('static jclass jARControllerCodecMJPEGCls;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_MJPEG;\n')
    cFile.write ('\n')
    cFile.write ('static jclass jARControllerCodecPCM16LECls;\n')
    cFile.write ('static jmethodID ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_PCM16LE;\n')
    cFile.write ('\n')
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             private header:\n')
    cFile.write (' *\n')
    cFile.write (' *****************************************/\n')
    cFile.write ('ARCONTROLLER_JNIDeviceController_t *ARCONTROLLER_JNI_Device_NewJNIDeviceController (JNIEnv *env, jobject thizz, ARDISCOVERY_Device_t *discoveryDevice, eARCONTROLLER_ERROR *error);\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_DeleteJNIDeviceController (JNIEnv *env, ARCONTROLLER_JNIDeviceController_t **jniDeviceController);\n')
    cFile.write ('\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_StateChanged (eARCONTROLLER_DEVICE_STATE newState, eARCONTROLLER_ERROR error, void *customData);\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_ExtensionStateChanged (eARCONTROLLER_DEVICE_STATE newState, eARDISCOVERY_PRODUCT product, const char *name, eARCONTROLLER_ERROR error, void *customData);\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_CommandReceived (eARCONTROLLER_DICTIONARY_KEY commandKey, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary, void *customData);\n')
    cFile.write ('jobject ARCONTROLLER_JNI_Device_NewH264Codec (JNIEnv *env, ARCONTROLLER_Stream_Codec_t codec);\n')
    cFile.write ('jobject ARCONTROLLER_JNI_Device_NewMJPEGCodec (JNIEnv *env, ARCONTROLLER_Stream_Codec_t codec);\n')
    cFile.write ('\n')
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DecoderConfigCallback (ARCONTROLLER_Stream_Codec_t codec, void *customData);\n')
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DidReceiveFrameCallback (ARCONTROLLER_Frame_t *frame, void *customData);\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_TimeoutFrameCallback (void *customData);\n')
    cFile.write ('\n')
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DecoderAudioConfigCallback (ARCONTROLLER_Stream_Codec_t codec, void *customData);\n')
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DidReceiveAudioFrameCallback (ARCONTROLLER_Frame_t *frame, void *customData);\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_TimeoutAudioFrameCallback (void *customData);\n')
    cFile.write ('\n')
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             implementation :\n')
    cFile.write (' *\n')
    cFile.write (' *****************************************/\n')
    cFile.write ('\n')
    cFile.write ('static JavaVM* ARCONTROLLER_JNIDEVICE_VM = NULL; /** reference to the java virtual machine */\n')
    cFile.write ('\n')
    cFile.write ('/**\n')
    cFile.write (' * @brief save the reference to the java virtual machine\n')
    cFile.write (' * @note this function is automatically call on the JNI startup\n')
    cFile.write (' * @param[in] VM reference to the java virtual machine\n')
    cFile.write (' * @param[in] reserved data reserved\n')
    cFile.write (' * @return JNI version\n')
    cFile.write (' */\n')
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('JNI_OnLoad(JavaVM *VM, void *reserved)\n')
    cFile.write ('{\n')
    cFile.write ('    ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "Library has been loaded");\n')
    cFile.write ('\n')
    cFile.write ('    /* Saving the reference to the java virtual machine */\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_VM = VM;\n')
    cFile.write ('\n')
    cFile.write ('    /* Return the JNI version */\n')
    cFile.write ('    return JNI_VERSION_1_6;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('JNIEXPORT void JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeStaticInit (JNIEnv *env, jclass class)\n')
    cFile.write ('{    \n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    jclass jARDeviceControllerCls = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    // get ARDeviceController\n')
    cFile.write ('    jARDeviceControllerCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARDeviceController");\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_ON_STATE_CHANGED = (*env)->GetMethodID (env, jARDeviceControllerCls, "onStateChanged", "(II)V");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_ON_EXTENSION_STATE_CHANGED = (*env)->GetMethodID (env, jARDeviceControllerCls, "onExtensionStateChanged", "(IILjava/lang/String;I)V");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_DECODER_CONFIG_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "decoderConfigCallback", "(Lcom/parrot/arsdk/arcontroller/ARControllerCodec;)I");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_FRAME_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "didReceiveFrameCallback", "(JIIIIJJI)I");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_FRAME_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "timeoutFrameCallback", "()V");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_DECODER_AUDIO_CONFIG_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "decoderAudioConfigCallback", "(Lcom/parrot/arsdk/arcontroller/ARControllerCodec;)I");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_AUDIO_FRAME_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "didReceiveAudioFrameCallback", "(JIIIIJJI)I");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_AUDIO_FRAME_CALLBACK = (*env)->GetMethodID (env, jARDeviceControllerCls, "timeoutAudioFrameCallback", "()V");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_ON_COMMAND_RECEIVED = (*env)->GetMethodID (env, jARDeviceControllerCls, "onCommandReceived", "(IJ)V");\n')
    cFile.write ('    \n')
    cFile.write ('    // cleanup\n')
    cFile.write ('    (*env)->DeleteLocalRef (env, jARDeviceControllerCls);\n')
    cFile.write ('    \n')
    cFile.write ('    //Get jARControllerCodecH264Cls\n')
    cFile.write ('    jARControllerCodecH264Cls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerCodec$H264");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_H264 = (*env)->GetMethodID(env, jARControllerCodecH264Cls, "<init>", "(JIJI)V");\n')
    cFile.write ('    jARControllerCodecH264Cls = (*env)->NewGlobalRef (env, jARControllerCodecH264Cls);\n')
    cFile.write ('    \n')

    cFile.write ('    //Get jARControllerCodecMJPEGCls\n')
    cFile.write ('    jARControllerCodecMJPEGCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerCodec$Mjpeg");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_MJPEG = (*env)->GetMethodID(env, jARControllerCodecMJPEGCls, "<init>", "()V");\n')
    cFile.write ('    jARControllerCodecMJPEGCls = (*env)->NewGlobalRef (env, jARControllerCodecMJPEGCls);\n')
    cFile.write ('    \n')

    cFile.write ('    //Get jARControllerCodecPCM16LECls\n')
    cFile.write ('    jARControllerCodecPCM16LECls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerCodec$PCM16LE");\n')
    cFile.write ('    ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_PCM16LE = (*env)->GetMethodID(env, jARControllerCodecPCM16LECls, "<init>", "(II)V");\n')
    cFile.write ('    jARControllerCodecPCM16LECls = (*env)->NewGlobalRef (env, jARControllerCodecPCM16LECls);\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('/**\n')
    cFile.write (' * @brief Create and initialize a new device\n')
    cFile.write (' * @param env reference to the java environment\n')
    cFile.write (' * @param thizz reference to the object calling this function\n')
    cFile.write (' * @return new jni device object\n')
    cFile.write (' */\n')
    cFile.write ('JNIEXPORT jlong JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeNew (JNIEnv *env, jobject thizz, jlong jDevice)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Create the Device --\n')
    cFile.write ('\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    //ARCONTROLLER_Device_t *deviceController = NULL;\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = NULL;\n')
    cFile.write ('    ARDISCOVERY_Device_t *device = (ARDISCOVERY_Device_t*) (intptr_t) jDevice;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    // allocate the JNI Device Controller\n')
    cFile.write ('    jniDeviceController = ARCONTROLLER_JNI_Device_NewJNIDeviceController (env, thizz, device, &error);\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_AddStateChangedCallback (jniDeviceController->nativeDeviceController, ARCONTROLLER_JNI_Device_StateChanged, jniDeviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_AddExtensionStateChangedCallback (jniDeviceController->nativeDeviceController, ARCONTROLLER_JNI_Device_ExtensionStateChanged, jniDeviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_AddCommandReceivedCallback (jniDeviceController->nativeDeviceController, ARCONTROLLER_JNI_Device_CommandReceived, jniDeviceController);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_SetVideoStreamCallbacks (jniDeviceController->nativeDeviceController, ARCONTROLLER_JNI_Device_DecoderConfigCallback, ARCONTROLLER_JNI_Device_DidReceiveFrameCallback, ARCONTROLLER_JNI_Device_TimeoutFrameCallback , jniDeviceController);\n')
    cFile.write ('        if(error == ARCONTROLLER_ERROR_NO_VIDEO)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_INFO, ARCONTROLLER_JNIDEVICE_TAG, "This device has no video stream");\n')
    cFile.write ('            error = ARCONTROLLER_OK;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (error == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        error = ARCONTROLLER_Device_SetAudioStreamCallbacks (jniDeviceController->nativeDeviceController, ARCONTROLLER_JNI_Device_DecoderAudioConfigCallback, ARCONTROLLER_JNI_Device_DidReceiveAudioFrameCallback, ARCONTROLLER_JNI_Device_TimeoutAudioFrameCallback, jniDeviceController);\n')
    cFile.write ('        if(error == ARCONTROLLER_ERROR_NO_AUDIO)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_INFO, ARCONTROLLER_JNIDEVICE_TAG, "This device has no audio stream");\n')
    cFile.write ('            error = ARCONTROLLER_OK;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Delete the JNI device controller\n')
    cFile.write ('        ARCONTROLLER_JNI_Device_DeleteJNIDeviceController (env, &jniDeviceController);\n')
    cFile.write ('        \n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return (long) jniDeviceController;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('/**\n')
    cFile.write (' * @brief delete the Device\n')
    cFile.write (' * @param env reference to the java environment\n')
    cFile.write (' * @param thizz reference to the object calling this function\n')
    cFile.write (' * @param jDevice the ARCONTROLLER_Device_t to delete\n')
    cFile.write (' */\n')
    cFile.write ('JNIEXPORT void JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeDelete (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // -- Delete the DeviceController --\n')
    cFile.write ('\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('\n')
    cFile.write ('    ARCONTROLLER_JNI_Device_DeleteJNIDeviceController (env, &jniDeviceController);\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeStart (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    \n')
    cFile.write ('    return ARCONTROLLER_Device_Start (jniDeviceController->nativeDeviceController);\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeStop (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('\n')
    cFile.write ('    return ARCONTROLLER_Device_Stop (jniDeviceController->nativeDeviceController);\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    for feature in ctx.features:
        cFile.write ('JNIEXPORT jlong JNICALL\n')
        cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_'+nativeGetFeature(feature)+' (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
        cFile.write ('{\n')
        cFile.write ('    // local declarations\n')
        cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
        cFile.write ('\n')
        cFile.write ('    return (long) jniDeviceController->nativeDeviceController->'+ARUncapitalize(get_ftr_old_name(feature))+';\n')
        cFile.write ('}\n')
        cFile.write ('\n')
        
    cFile.write ('JNIEXPORT jlong JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeGetCommandElements (JNIEnv *env, jobject thizz, jlong jDeviceController, jint commandKey)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    ARCONTROLLER_DICTIONARY_ELEMENT_t *elements = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    elements = ARCONTROLLER_Device_GetCommandElements (jniDeviceController->nativeDeviceController, commandKey, &error);\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return (long) elements;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeGetState (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    eARCONTROLLER_DEVICE_STATE state = ARCONTROLLER_DEVICE_STATE_MAX;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    state = ARCONTROLLER_Device_GetState (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return state;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeGetExtensionState (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    eARCONTROLLER_DEVICE_STATE extensionState = ARCONTROLLER_DEVICE_STATE_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    extensionState = ARCONTROLLER_Device_GetExtensionState (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('    \n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return extensionState;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('JNIEXPORT jstring JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeGetExtensionName (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    jstring extensionNameStr = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    char extensionNameBuffer[128];\n')
    cFile.write ('    ARCONTROLLER_Device_GetExtensionName (jniDeviceController->nativeDeviceController, extensionNameBuffer, 128, &error);\n')
    cFile.write ('    \n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('    else\n')
    cFile.write ('    {\n')
    cFile.write ('        extensionNameStr = (*env)->NewStringUTF(env, extensionNameBuffer);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return extensionNameStr;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeGetExtensionProduct (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    eARDISCOVERY_PRODUCT extensionProduct = ARDISCOVERY_PRODUCT_MAX;\n')
    cFile.write ('    \n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    extensionProduct = ARCONTROLLER_Device_GetExtensionProduct (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('    \n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return extensionProduct;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeSendStreamFrame (JNIEnv *env, jobject thizz, jlong jDeviceController, jlong data, jint dataSize)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('\n')
    cFile.write ('    return ARCONTROLLER_Device_SendStreamFrame (jniDeviceController->nativeDeviceController, (uint8_t *) (intptr_t) data, dataSize);\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeHasOutputVideoStream (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int hasOutputVideoStream = 0;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    hasOutputVideoStream = ARCONTROLLER_Device_HasOutputVideoStream (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return hasOutputVideoStream;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeHasOutputAudioStream (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int hasOutputAudioStream = 0;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    hasOutputAudioStream = ARCONTROLLER_Device_HasOutputAudioStream (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return hasOutputAudioStream;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('JNIEXPORT jint JNICALL\n')
    cFile.write ('Java_com_parrot_arsdk_arcontroller_ARDeviceController_nativeHasInputAudioStream (JNIEnv *env, jobject thizz, jlong jDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) jDeviceController;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    int hasInputAudioStream = 0;\n')
    cFile.write ('\n')
    cFile.write ('    jclass exceptionCls = NULL;\n')
    cFile.write ('    jmethodID exceptionMethodInit = NULL;\n')
    cFile.write ('    jthrowable exception = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    hasInputAudioStream = ARCONTROLLER_Device_HasInputAudioStream (jniDeviceController->nativeDeviceController, &error);\n')
    cFile.write ('\n')
    cFile.write ('    if (error != ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // throw the exception\n')
    cFile.write ('        exceptionCls = (*env)->FindClass(env, "com/parrot/arsdk/arcontroller/ARControllerException");\n')
    cFile.write ('        exceptionMethodInit = (*env)->GetMethodID(env, exceptionCls, "<init>", "(I)V");\n')
    cFile.write ('        exception = (*env)->NewObject(env, exceptionCls, exceptionMethodInit, error);\n')
    cFile.write ('        (*env)->Throw(env, exception);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return hasInputAudioStream;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('/*****************************************\n')
    cFile.write (' *\n')
    cFile.write (' *             private implementation:\n')
    cFile.write (' *\n')
    cFile.write (' *****************************************/\n')
    cFile.write ('\n')
    cFile.write ('ARCONTROLLER_JNIDeviceController_t *ARCONTROLLER_JNI_Device_NewJNIDeviceController (JNIEnv *env, jobject thizz, ARDISCOVERY_Device_t *discoveryDevice, eARCONTROLLER_ERROR *error)\n')
    cFile.write ('{\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = malloc (sizeof (ARCONTROLLER_JNIDeviceController_t));\n')
    cFile.write ('    \n')
    cFile.write ('    if (jniDeviceController != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        // Create a new native deviceController\n')
    cFile.write ('        jniDeviceController->nativeDeviceController = ARCONTROLLER_Device_New (discoveryDevice, error);\n')
    cFile.write ('        \n')
    cFile.write ('        // create a global reference of the java object\n')
    cFile.write ('        jniDeviceController->jDeviceController = (*env)->NewGlobalRef (env, thizz);\n')
    cFile.write ('    }\n')
    cFile.write ('    else if (error != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        *error = ARCONTROLLER_ERROR_ALLOC;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return jniDeviceController;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_DeleteJNIDeviceController (JNIEnv *env, ARCONTROLLER_JNIDeviceController_t **jniDeviceController)\n')
    cFile.write ('{\n')
    cFile.write ('    if (jniDeviceController != NULL)\n')
    cFile.write ('    {\n')
    cFile.write ('        if ((*jniDeviceController) != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            // Delete native deviceController\n')
    cFile.write ('            ARCONTROLLER_Device_Delete (&((*jniDeviceController)->nativeDeviceController));\n')
    cFile.write ('            \n')
    cFile.write ('            // Delete global references\n')
    cFile.write ('            (*env)->DeleteGlobalRef (env, (*jniDeviceController)->jDeviceController);\n')
    cFile.write ('            (*jniDeviceController)->jDeviceController = NULL;\n')
    cFile.write ('            \n')
    cFile.write ('            free (*jniDeviceController);\n')
    cFile.write ('            (*jniDeviceController) = NULL;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('// called when the state of the device controller has changed\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_StateChanged (eARCONTROLLER_DEVICE_STATE newState, eARCONTROLLER_ERROR error, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('        \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // java onStateChanged callback\n')
    cFile.write ('        (*env)->CallVoidMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_ON_STATE_CHANGED, newState, error);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('// called when the state of the extension has changed\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_ExtensionStateChanged (eARCONTROLLER_DEVICE_STATE newState, eARDISCOVERY_PRODUCT product, const char *name, eARCONTROLLER_ERROR error, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    jstring extensionNameStr = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('    \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('        \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('        \n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        extensionNameStr = (*env)->NewStringUTF(env, name);\n')
    cFile.write ('        // java onStateChanged callback\n')
    cFile.write ('        (*env)->CallVoidMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_ON_EXTENSION_STATE_CHANGED, newState, product, extensionNameStr, error);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('// called when a command has been received from the drone\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_CommandReceived (eARCONTROLLER_DICTIONARY_KEY commandKey, ARCONTROLLER_DICTIONARY_ELEMENT_t *elementDictionary, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('    eARCONTROLLER_ERROR error = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // java onCommandReceived callback\n')
    cFile.write ('        (*env)->CallVoidMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_ON_COMMAND_RECEIVED, commandKey, (long) elementDictionary);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DecoderConfigCallback (ARCONTROLLER_Stream_Codec_t codec, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    \n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    jobject jCodec = NULL;\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR callbackError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('    \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('        \n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch(codec.type)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCONTROLLER_STREAM_CODEC_TYPE_H264:\n')
    cFile.write ('                jCodec = ARCONTROLLER_JNI_Device_NewH264Codec(env, codec);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    cFile.write ('            case ARCONTROLLER_STREAM_CODEC_TYPE_MJPEG:\n')
    cFile.write ('                jCodec = ARCONTROLLER_JNI_Device_NewMJPEGCodec(env, codec);\n')
    cFile.write ('                break;\n')
    cFile.write ('            \n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_WARNING, ARCONTROLLER_JNIDEVICE_TAG, "Unknown codec: %d", codec.type);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        // java decoderConfigCallback callback\n')
    cFile.write ('        callbackError = (*env)->CallIntMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_DECODER_CONFIG_CALLBACK, jCodec);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return callbackError;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('jobject ARCONTROLLER_JNI_Device_NewH264Codec (JNIEnv *env, ARCONTROLLER_Stream_Codec_t codec)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    jlong spsBuffer = (intptr_t) codec.parameters.h264parameters.spsBuffer;\n')
    cFile.write ('    jint spsSize =  codec.parameters.h264parameters.spsSize;\n')
    cFile.write ('    \n')
    cFile.write ('    jlong ppsBuffer = (intptr_t) codec.parameters.h264parameters.ppsBuffer;\n')
    cFile.write ('    jint ppsSize =  codec.parameters.h264parameters.ppsSize;\n')
    cFile.write ('    \n')
    cFile.write ('    jobject jCodecH264 = (*env)->NewObject(env, jARControllerCodecH264Cls, ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_H264, spsBuffer, spsSize, ppsBuffer, ppsSize);\n')
    cFile.write ('    \n')
    cFile.write ('    return jCodecH264;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('jobject ARCONTROLLER_JNI_Device_NewMJPEGCodec (JNIEnv *env, ARCONTROLLER_Stream_Codec_t codec)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    cFile.write ('    jobject jCodecMJPEG = (*env)->NewObject(env, jARControllerCodecMJPEGCls, ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_MJPEG);\n')
    cFile.write ('    \n')
    cFile.write ('    return jCodecMJPEG;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('jobject ARCONTROLLER_JNI_Device_NewPCM16LECodec (JNIEnv *env, ARCONTROLLER_Stream_Codec_t codec)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    jint sampleRate = codec.parameters.pcm16leParameters.sampleRate;\n')
    cFile.write ('    jint channel = codec.parameters.pcm16leParameters.channel;\n')
    cFile.write ('    \n')
    cFile.write ('\n')
    cFile.write ('    jobject jCodecPCM16LE = (*env)->NewObject(env, jARControllerCodecPCM16LECls, ARCONTROLLER_JNIDEVICE_METHOD_NEW_CODEC_PCM16LE, sampleRate, channel);\n')
    cFile.write ('\n')
    cFile.write ('    return jCodecPCM16LE;\n')
    cFile.write ('}\n')
    cFile.write ('\n')

    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DidReceiveFrameCallback (ARCONTROLLER_Frame_t *frame, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    \n')
    cFile.write ('    jlong data = 0;\n')
    cFile.write ('    jint dataCapacity = 0;\n')
    cFile.write ('    jint dataSize = 0;\n')
    cFile.write ('    jint isIFrame = 0;\n')
    cFile.write ('    jint missed = 0;\n')
    cFile.write ('    jlong timestamp = 0;\n')
    cFile.write ('    jlong metadata = 0;\n')
    cFile.write ('    jint metadataSize = 0;\n')
    cFile.write ('    \n')
    cFile.write ('    eARCONTROLLER_ERROR callbackError = ARCONTROLLER_OK;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('        \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        \n')
    cFile.write ('        if (frame != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            data = (long)frame->data;\n')
    cFile.write ('            dataCapacity = frame->capacity;\n')
    cFile.write ('            dataSize = frame->used;\n')
    cFile.write ('            isIFrame = frame->isIFrame;\n')
    cFile.write ('            missed = frame->missed;\n')
    cFile.write ('            timestamp = frame->timestamp;\n')
    cFile.write ('            metadata = (jlong)(intptr_t)frame->metadata;\n')
    cFile.write ('            metadataSize = frame->metadataSize;\n')
    cFile.write ('        }\n')
    cFile.write ('        \n')
    cFile.write ('        // java did receive frame callback\n')
    cFile.write ('        callbackError = (*env)->CallIntMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_FRAME_CALLBACK, data, dataCapacity, dataSize, isIFrame, missed, timestamp, metadata, metadataSize);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    return callbackError;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_TimeoutFrameCallback (void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    \n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('        \n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // java onStateChanged callback\n')
    cFile.write ('        (*env)->CallVoidMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_FRAME_CALLBACK);\n')
    cFile.write ('    }\n')
    cFile.write ('    \n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DecoderAudioConfigCallback (ARCONTROLLER_Stream_Codec_t codec, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('    jobject jCodec = NULL;\n')
    cFile.write ('\n')
    cFile.write ('    eARCONTROLLER_ERROR callbackError = ARCONTROLLER_OK;\n')
    cFile.write ('\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('\n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        switch(codec.type)\n')
    cFile.write ('        {\n')
    cFile.write ('            case ARCONTROLLER_STREAM_CODEC_TYPE_H264:\n')
    cFile.write ('                jCodec = ARCONTROLLER_JNI_Device_NewH264Codec(env, codec);\n')
    cFile.write ('                break;\n')
    cFile.write ('\n')
    cFile.write ('            case ARCONTROLLER_STREAM_CODEC_TYPE_MJPEG:\n')
    cFile.write ('                jCodec = ARCONTROLLER_JNI_Device_NewMJPEGCodec(env, codec);\n')
    cFile.write ('                break;\n')
    cFile.write ('\n')
    cFile.write ('            case ARCONTROLLER_STREAM_CODEC_TYPE_PCM16LE:\n')
    cFile.write ('                jCodec = ARCONTROLLER_JNI_Device_NewPCM16LECodec(env, codec);\n')
    cFile.write ('                break;\n')
    cFile.write ('            default:\n')
    cFile.write ('                ARSAL_PRINT(ARSAL_PRINT_WARNING, ARCONTROLLER_JNIDEVICE_TAG, "Unknown codec: %d", codec.type);\n')
    cFile.write ('                break;\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        // java decoderConfigCallback callback\n')
    cFile.write ('        callbackError = (*env)->CallIntMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_DECODER_AUDIO_CONFIG_CALLBACK, jCodec);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return callbackError;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('eARCONTROLLER_ERROR ARCONTROLLER_JNI_Device_DidReceiveAudioFrameCallback (ARCONTROLLER_Frame_t *frame, void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('\n')
    cFile.write ('    jlong data = 0;\n')
    cFile.write ('    jint dataCapacity = 0;\n')
    cFile.write ('    jint dataSize = 0;\n')
    cFile.write ('    jint isIFrame = 0;\n')
    cFile.write ('    jint missed = 0;\n')
    cFile.write ('    jlong timestamp = 0;\n')
    cFile.write ('    jlong metadata = 0;\n')
    cFile.write ('    jint metadataSize = 0;\n')
    cFile.write ('\n')
    cFile.write ('    eARCONTROLLER_ERROR callbackError = ARCONTROLLER_OK;\n')
    cFile.write ('\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('\n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('\n')
    cFile.write ('        if (frame != NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            data = (long)frame->data;\n')
    cFile.write ('            dataCapacity = frame->capacity;\n')
    cFile.write ('            dataSize = frame->used;\n')
    cFile.write ('            isIFrame = frame->isIFrame;\n')
    cFile.write ('            missed = frame->missed;\n')
    cFile.write ('            timestamp = frame->timestamp;\n')
    cFile.write ('            metadata = (jlong)(intptr_t)frame->metadata;\n')
    cFile.write ('            metadataSize = frame->metadataSize;\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        // java did receive frame callback\n')
    cFile.write ('        callbackError = (*env)->CallIntMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_DID_RECEIVE_AUDIO_FRAME_CALLBACK, data, dataCapacity, dataSize, isIFrame, missed, timestamp, metadata, metadataSize);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    return callbackError;\n')
    cFile.write ('}\n')
    cFile.write ('\n')
    cFile.write ('void ARCONTROLLER_JNI_Device_TimeoutAudioFrameCallback (void *customData)\n')
    cFile.write ('{\n')
    cFile.write ('    // local declarations\n')
    cFile.write ('    eARCONTROLLER_ERROR localError = ARCONTROLLER_OK;\n')
    cFile.write ('    JNIEnv* env = NULL;\n')
    cFile.write ('    jint getEnvResult = JNI_OK;\n')
    cFile.write ('    jint attachResult = 1;\n')
    cFile.write ('\n')
    cFile.write ('    ARCONTROLLER_JNIDeviceController_t *jniDeviceController = (ARCONTROLLER_JNIDeviceController_t*) (intptr_t) customData;\n')
    cFile.write ('\n')
    cFile.write ('    if ((jniDeviceController == NULL) ||\n')
    cFile.write ('        (jniDeviceController->jDeviceController == NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        localError = ARCONTROLLER_ERROR_BAD_PARAMETER;\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // get the environment\n')
    cFile.write ('        getEnvResult = (*ARCONTROLLER_JNIDEVICE_VM)->GetEnv(ARCONTROLLER_JNIDEVICE_VM, (void **) &env, JNI_VERSION_1_6);\n')
    cFile.write ('\n')
    cFile.write ('        // if no environment then attach the thread to the virtual machine\n')
    cFile.write ('        if (getEnvResult == JNI_EDETACHED)\n')
    cFile.write ('        {\n')
    cFile.write ('            ARSAL_PRINT(ARSAL_PRINT_DEBUG, ARCONTROLLER_JNIDEVICE_TAG, "attach the thread to the virtual machine ...");\n')
    cFile.write ('            attachResult = (*ARCONTROLLER_JNIDEVICE_VM)->AttachCurrentThread(ARCONTROLLER_JNIDEVICE_VM, &env, NULL);\n')
    cFile.write ('        }\n')
    cFile.write ('\n')
    cFile.write ('        if (env == NULL)\n')
    cFile.write ('        {\n')
    cFile.write ('            localError = ARCONTROLLER_ERROR_JNI_ENV;\n')
    cFile.write ('        }\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    if (localError == ARCONTROLLER_OK)\n')
    cFile.write ('    {\n')
    cFile.write ('        // java onStateChanged callback\n')
    cFile.write ('        (*env)->CallVoidMethod(env, jniDeviceController->jDeviceController, ARCONTROLLER_JNIDEVICE_METHOD_TIMEOUT_AUDIO_FRAME_CALLBACK);\n')
    cFile.write ('    }\n')
    cFile.write ('\n')
    cFile.write ('    // if the thread has been attached then detach the thread from the virtual machine\n')
    cFile.write ('    if ((getEnvResult == JNI_EDETACHED) && (env != NULL))\n')
    cFile.write ('    {\n')
    cFile.write ('        (*ARCONTROLLER_JNIDEVICE_VM)->DetachCurrentThread(ARCONTROLLER_JNIDEVICE_VM);\n')
    cFile.write ('    }\n')
    cFile.write ('}\n')
    cFile.write ('\n')

def list_files_deviceCtrls (ctx, SRC_DIR, INC_DIR):
    ''' Print device controllers generated files '''
    print INC_DIR + CTRL_DEVICE_H_NAME
    print SRC_DIR + CTRL_DEVICE_PRIV_H_NAME
    print SRC_DIR + CTRL_DEVICE_C_NAME

def list_files_deviceCtrls_java (ctx, JNI_JAVA_DIR):
    ''' Print device controllers generated files '''
    print JNI_JAVA_DIR + CTRL_DEVICE_JAVA_NAME

def list_files_deviceCtrls_jni (ctx, JNI_C_DIR):
    ''' Print device controllers generated files '''
    print JNI_C_DIR + CTRL_DEVICE_JNI_C_NAME
