/** @file     plot.h
 *  @author   Adi Singh, Sebastian Dengler
 *  @date     February 2019
 *  @brief    Visualizes published TsScans as native Rviz markers.
 *  Subscribes to a topic publishing TsScans and converts incoming
 *  data into Rviz Markers. Location, color and scale of a marker
 *  depend on payload of their corresponding TsPoint. The lifetime
 *  and scale of markers can be altered via dynamic reconfigure.
 */

#ifndef PLOT_H
#define PLOT_H

#include <deque>
#include <map>

#include <dynamic_reconfigure/server.h>
#include <gtest/gtest.h>
#include <message_filters/subscriber.h>
#include <ros/ros.h>
#include <rviz_visual_tools/rviz_visual_tools.h>
#include <tf2_geometry_msgs/tf2_geometry_msgs.h>
#include <tf2_ros/message_filter.h>
#include <tf2_ros/transform_listener.h>

#include <toposens_driver/sensor.h>
#include <toposens_markers/TsMarkersConfig.h>
#include <toposens_msgs/TsScan.h>

namespace toposens_markers
{
static const std::string kMarkersTopic = "ts_markers";
/**< Rviz namespace for plotted points.*/
static const std::string kMarkersNs = "TsMarkers";
/**< Rviz namespace for sensor mesh.*/
static const std::string kMeshNs = "TsSensor";

/** Handles lifetime management and realtime plotting of TsPoints on
 *   Rviz.
 *   Visual characteristics of a marker are defined as follows:
 *   @n Location - 3D coordinates of TsPoint relative to origin, which
 *   coincides with the sensor position.
 *   @n Color - Relative z-distance from the sensor, green being the
 *   closest points and red being the farthest.
 *   @n Scale - Product of the base scale (hard-coded, same for all markers),
 *   the user-defined global scale (dynamic, affects all markers equally)
 *   and the intensity of an individual TsPoint.
 */
class Plot
{
  /** Universal scale affecting all markers equally.*/
  static const auto _baseScale = rviz_visual_tools::scales::LARGE;

public:
  /** Subscribes to a TsScans topic and initializes visual tools for
   *  rviz plotting.
   *  @param nh Public nodehandle for pub-sub ops on ROS topics.
   *  @param private_nh Private nodehandle for accessing launch parameters.
   */
  Plot(ros::NodeHandle nh, ros::NodeHandle private_nh);
  ~Plot();

  /** Plots the sensor graphics
   */
  void publishSensorMeshes();

  /** Plots the default sensor
   */
  void publishDefaultSensorMesh();

private:
  /** Structure generated from cfg file for storing local copy of marker parameters.*/
  typedef dynamic_reconfigure::Server<TsMarkersConfig> Cfg;

  /** Callback triggered when a parameter is altered on the dynamic
   *  reconfigure server.
   *  @param cfg Structure holding updated values of parameters on server.
   *  @param level Indicates parameter that triggered the callback.
   */
  void _reconfig(TsMarkersConfig &cfg, uint32_t level);

  /** Converts point coordinates from local sensor coordinates to global
   *  coordinates and plots the points.
   *  @param  msg  Pointer to an incoming TsScan message.
   */
  void _plot(const toposens_msgs::TsScan::ConstPtr &msg);

  /**
   * Callback which receives the static transform messages containing the
   * positions of the sensors. These are stored in a dictionary. Unique
   * point colors for each sensor postion are also stored in a dictionary.
   * @param tf The transform message.
   */
  void _staticTFCallback(const tf2_msgs::TFMessage::ConstPtr &tf);

  std::string _frame_id;     /**< Frame ID assigned to rviz Marker messages.*/
  double _lifetime;          /**< Maintains current value of marker lifetime.*/
  int _color_mode;           /**< Maintains current value of marker color mode.*/
  float _color_range;        /**< Maintains current value over which the color is scaled.*/
  float _half_color_range;   /**< Maintains current value of the color range offset.*/
  std::shared_ptr<Cfg> _srv; /**< Pointer to config server*/

  std::deque<toposens_msgs::TsScan> _scans;    /**< Data structure for storing incoming scans.*/
  rviz_visual_tools::RvizVisualToolsPtr _rviz; /**< Helper for plotting markers.*/

  std::string _target_frame; /**< Target frame for scan transformations.*/
  message_filters::Subscriber<toposens_msgs::TsScan>
      _tf2_scans_sub;                        /**< Handler for subscribing to TsScans.*/
  tf2_ros::Buffer _tf2_buffer;               /**< Stores known frame mapping.*/
  tf2_ros::TransformListener *_tf2_listener; /**< Listener for frame mapping.*/
  tf2_ros::MessageFilter<toposens_msgs::TsScan> *_tf2_filter; /**< Reference to the tf2 filter.*/

  ros::Subscriber _scans_sub;     /**< Handler for subscribing to TsScans.*/
  std::string _sensor_mesh;       /**< File path to the sensor which is to be displayed. */
  ros::Subscriber _static_tf_sub; /**< Subscriber to get the positions of the sensors */
  std::map<std::string, geometry_msgs::Pose> _pose_map;  /**< Dictionary of sensor poses */
  std::map<std::string, std_msgs::ColorRGBA> _color_map; /**< Dictionary of sensor point colors */
  std::map<std::string, rviz_visual_tools::colors> _mesh_color_map;
};

}  // namespace toposens_markers

#endif
