#include "toposens_pointcloud/logging.h"

namespace toposens_pointcloud
{
/**
 * Maintains a persistent PointCloud structure holding the cumulative
 * of all scans received during a run. This cumulative data is saved
 * periodically as a PCD file in the package folder for
 * subsequent use with other PCL tools (like pcl_octree_viewer).
 */
Logging::Logging(ros::NodeHandle nh, ros::NodeHandle private_nh)
{
  int pcd_save_interval = 0;
  std::string target_frame;
  private_nh.param<int>("pcd_save_interval", pcd_save_interval, 0);
  private_nh.param<std::string>("target_frame", target_frame, "toposens");
  private_nh.param<std::string>("pcd_path", pcd_path_, "");

  /** If no file is specified as a launch argument, the default path
   *  "/home/$USER/.ros/toposens.pcd" is used to store the PCD.
   */
  if (pcd_path_ == "") pcd_path_ = std::string(get_current_dir_name()) + "/" + "toposens.pcd";

  cloud_sub_ = nh.subscribe(kPointCloudTopic, 100, &Logging::_accumulate, this);

  timer_ = nh.createTimer(ros::Duration(pcd_save_interval), &Logging::save,
                          this);  // continuous timer

  store_ = boost::make_shared<XYZINCloud>();
  pcl_conversions::toPCL(ros::Time::now(), store_->header.stamp);
  store_->header.frame_id = target_frame;
  store_->height = 1;
}

Logging::~Logging() { timer_.stop(); };

/**
 * Filename is provided as a launch argument. File is saved as a .pcd.
 */
void Logging::save(const ros::TimerEvent& event)
{
  boost::mutex::scoped_lock lock(store_mutex_);

  if (!store_->width)
  {
    ROS_WARN("No pointcloud data to save.");
    return;
  }
  pcl_conversions::toPCL(ros::Time::now(), store_->header.stamp);
  try
  {
    if (pcl::io::savePCDFile(pcd_path_, *store_) == 0)
    {
      ROS_INFO("Saved latest point cloud data (%s)", pcd_path_.c_str());
    }
  }
  catch (pcl::io::IOException& e)
  {
    ROS_ERROR("%s", e.what());
  }
}

/**
 * Waits for a pointcloud updates to become available and accumulates them
 * in a persistent pointcloud structure.
 */
void Logging::_accumulate(const XYZINCloud::ConstPtr& msg)
{
  boost::mutex::scoped_lock lock(store_mutex_);

  for (auto it = msg->points.begin(); it != msg->points.end(); ++it)
  {
    pcl::PointXYZINormal pt = *it;
    store_->points.push_back(pt);
  }

  store_->width = store_->points.size();
}

}  // namespace toposens_pointcloud
