#include "topo_isotp.h"
#include "isotp.h"
#include "stdint.h"
#include "isotp_defines.h"

#include "toposens/debug.h"

#ifdef TOPO_SENSOR
#include "sensor.h"
#include "comms_manager.h"
#include "stm32h7xx_hal.h"
#endif

#define ISOTP_BUFSIZE 4096

/* Alloc IsoTpLink statically in RAM */
static IsoTpLink g_link;

/* Alloc send and receive buffer statically in RAM */
static uint8_t g_isotpRecvBuf[ISOTP_BUFSIZE];
static uint8_t g_isotpSendBuf[ISOTP_BUFSIZE];

TopoError_t
topo_isotp_init(void* context)
{
#ifdef TOPO_SENSOR
    Debug_ASSERT(context != NULL, "NULL pointer");
#endif

    isotp_init_link(
        &g_link,
        0x7FF,
        g_isotpSendBuf,
        sizeof(g_isotpSendBuf),
        g_isotpRecvBuf,
        sizeof(g_isotpRecvBuf),
		context);

    return TOPO_OK;
}

TopoError_t
topo_isotp_poll()
{
    isotp_poll(&g_link);
    return TOPO_OK;
}

bool
topo_isotp_is_cantp_frame(uint8_t *buffer, uint16_t size)
{
    Debug_ASSERT(buffer != NULL, "NULL pointer");

    // The first nibble of the first byte contains the CAN-TP frame type.
    // The other values are used by the normal CAN protocol.
    int frame_type = buffer[0] >> 4 ;

    if( (frame_type == ISOTP_PCI_TYPE_SINGLE) ||
        (frame_type == ISOTP_PCI_TYPE_FIRST_FRAME) ||
        (frame_type == TSOTP_PCI_TYPE_CONSECUTIVE_FRAME) ||
        (frame_type == ISOTP_PCI_TYPE_FLOW_CONTROL_FRAME))
    {
        return true;
    }

    return false;
}

TopoError_t
topo_isotp_frame_received(uint8_t *buffer, uint16_t size)
{
    Debug_ASSERT(buffer != NULL, "NULL pointer");

    isotp_on_can_message(&g_link, buffer, size);

    return TOPO_OK;
}

TopoError_t
topo_isotp_send(uint8_t *buffer, uint16_t size)
{
    Debug_ASSERT(buffer != NULL, "NULL pointer");

    int ret;

    ret = isotp_send(&g_link, buffer, size);

    if (ISOTP_RET_OK == ret)
    {
        return TOPO_OK;
    }
    else
    {
        return TOPO_ERROR_GENERIC;
    }
}

TopoError_t
topo_isotp_receive(uint8_t *buffer, uint16_t size, uint16_t* actual_size)
{
    Debug_ASSERT(buffer != NULL, "NULL pointer");

    int ret;

    ret = isotp_receive(&g_link, buffer, size, actual_size);

    if (ISOTP_RET_OK == ret)
    {
    	return TOPO_OK;
    }
    else
    {
       return TOPO_ERROR_NODATA;
    }
}

#define PAYLOAD_LEN_TO_CAN_FRAME_LEN(X) ((X) << 16)

int
isotp_user_send_can(
    const uint32_t arbitration_id,
    const uint8_t* data,
    const uint8_t  size,
    void* context)
{
#ifdef TOPO_SENSOR
    SensorSettings_t* settings = (SensorSettings_t*)context;

    settings->can_tx_header.DataLength = PAYLOAD_LEN_TO_CAN_FRAME_LEN(size);

    COMMS_WaitForFrameToBeSent(settings);

    HAL_FDCAN_AddMessageToTxFifoQ(
        settings->can_handle,
        &settings->can_tx_header,
        (void*)data);
#endif
    return 0;
}

uint32_t
isotp_user_get_ms(void)
{
#ifdef TOPO_SENSOR
    return HAL_GetTick();
#endif
    return 0;
}

void
isotp_user_debug(const char* message, ...)
{
}
