/** **************************************************************************************
 * @file       n_frames_single_shot_example_uart.c
 * @ingroup    Toposens Sensor Library
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 ******************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <termios.h>  //Used for UART
#include <unistd.h>
#include "example_common.h"
#include "toposens/sensor_lib.h"

#define PORT     "/dev/ttyUSB0"
#define BAUDRATE B115200

static void
LogMsgCallback(uint16_t SenderId_u16, uint8_t* ReceivedPayload_pu8);

static void
SingleShotMeasurement()
{
    SetParameterSystemSensorMode(SENSOR_MODE_SINGLE_SHOT_TRANSMIT_LISTEN);

    RequestMeasurement();
    uint16_t current_sensing_sensor = WaitForSessionStart();
    uint16_t sender                 = WaitForSessionEnd();
    Sensor_Session_t* session_data  = GetSessionData(sender);
    PrintSessionData(session_data);
}

int
main(int argc, char** argv)
{
    int num_of_frames = 0;
    int c;

    if (argc < 2)
    {
        fprintf(stderr, "Pass -n argument.\n");
        return -1;
    }

    while ((c = getopt(argc, argv, "n:")) != -1)
        switch (c)
        {
            case 'n':
                num_of_frames = atoi(optarg);
                if (num_of_frames < 0)
                    num_of_frames = 0;
                break;
            case '?':
                if (optopt == 'n')
                    fprintf(stderr, "Option -%c requires an argument.\n", optopt);
                else
                    fprintf(stderr, "Unknown option character `\\x%x'.\n", optopt);
                return 1;
            default:;
        }

    RegisterLogMsgCallback(LogMsgCallback);
    InitInterface(PORT, BAUDRATE, IF_UART);
    if (RequestReboot())
    {
        printf("Blocking Reboot Request worked\n");
    }
    else
    {
        printf("Blocking Reboot Request worked but did not got the right ack\n");
    }
    WaitForReady();

    printf("Starting measurement\n");
    int i;
    for (i = 0; i < num_of_frames; i++)
    {
        SingleShotMeasurement();
    }

    DeinitInterface(IF_UART);
    return 0;
}

static void
LogMsgCallback(uint16_t SenderId_u16, uint8_t* ReceivedPayload_pu8)
{
    PrintLogMessage(SenderId_u16, ReceivedPayload_pu8);
}
