// -*- C++ -*-
/*!
 * @file ConfigurationListener.h
 * @brief Configuration related event listener classes
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2011
 *     Noriaki Ando
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_CONFIGURATIONLISTENER_H
#define RTC_CONFIGURATIONLISTENER_H

#include <vector>
#include <coil/Properties.h>
#include <coil/Mutex.h>
#include <coil/Guard.h>

namespace RTC
{
  //============================================================
  /*!
   * @if jp
   * @brief ConfigurationParamListener Υ
   *
   * - ON_UPDATE_CONFIG_PARAM,
   *
   * @else
   * @brief The types of ConnectorDataListener
   * 
   * - ON_UPDATE_CONFIG_PARAM,
   *
   * @endif
   */
  enum ConfigurationParamListenerType
    {
      ON_UPDATE_CONFIG_PARAM,
      CONFIG_PARAM_LISTENER_NUM
    };


  /*!
   * @if jp
   * @class ConfigurationParamListener 饹
   * @brief ConfigurationParamListener 饹
   *
   * Configuration ѥ᡼ѹ˴ؤꥹʥ饹
   * ʲΥ٥ȤФƥХå롣
   *
   * - ON_UPDATE_CONFIG_PARAM
   *
   * @else
   * @class ConfigurationParamListener class
   * @brief ConfigurationParamListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events for Configuration parameter.
   * The listener will be called on the following event.
   *
   * - ON_UPDATE_CONFIG_PARAM
   *
   * @endif
   */
  class ConfigurationParamListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief ConfigurationParamListenerType ʸѴ
     *
     * ConfigurationParamListenerType ʸѴ
     *
     * @param type Ѵо ConfigurationParamListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert ConfigurationParamListenerType into the string.
     *
     * Convert ConfigurationParamListenerType into the string.
     *
     * @param type The target ConfigurationParamListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(ConfigurationParamListenerType type)
    {
      static const char* typeString[] =
        {
          "ON_UPDATE_CONFIG_PARAM",
          "CONFIG_PARAM_LISTENER_NUM"
        };
      if (type < CONFIG_PARAM_LISTENER_NUM)
        {
          return typeString[type];
        }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationParamListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * ConfigurationParamListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for ConfigurationParamListener.
     *
     * @endif
     */
    virtual void operator()(const char* config_set_name,
                            const char* config_param_name) = 0;
  };


  //============================================================
  /*!
   * @if jp
   * @brief ConfigurationSetListener Υ
   *
   * - ON_SET_CONFIG_SET: ConfigurationSet ñ̤ͤåȤ줿
   * - ON_ADD_CONFIG_SET: ConfigurationSet ɲä줿
   *
   * @else
   * @brief The types of ConfigurationSetListener
   * 
   * - ON_SET_CONFIG_SET: Value list has been set as a configuration set 
   * - ON_ADD_CONFIG_SET: A new configuration set has been added
   *
   * @endif
   */

  enum ConfigurationSetListenerType
    {
      ON_SET_CONFIG_SET,
      ON_ADD_CONFIG_SET,
      CONFIG_SET_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class ConfigurationSetListener 饹
   * @brief ConfigurationSetListener 饹
   *
   * ConfigurationåȤѹ줿ɲä줿˸ƤӽФꥹʥ饹
   * ʲConfigurationåȤ˴Ϣ륤٥ȤФꥹʡ
   *
   * - ON_SET_CONFIG_SET: ConfigurationSet ñ̤ͤåȤ줿
   * - ON_ADD_CONFIG_SET: ConfigurationSet ɲä줿
   *
   * @else
   * @class ConfigurationSetListener class
   * @brief ConfigurationSetListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for configuration set's related events.
   *
   * - ON_SET_CONFIG_SET: Value list has been set as a configuration set 
   * - ON_ADD_CONFIG_SET: A new configuration set has been added
   *
   * @endif
   */
  class ConfigurationSetListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief ConfigurationSetNameListenerType ʸѴ
     *
     * ConfigurationSetNameListenerType ʸѴ
     *
     * @param type Ѵо ConfigurationSetNameListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert ConfigurationSetNameListenerType into the string.
     *
     * Convert ConfigurationSetNameListenerType into the string.
     *
     * @param type The target ConfigurationSetNameListenerType for
     *             transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(ConfigurationSetListenerType type)
    {
      static const char* typeString[] =
        {
          "ON_SET_CONFIG_SET",
          "ON_ADD_CONFIG_SET",
          "CONFIG_SET_LISTENER_NUM"
        };
      if (type < CONFIG_SET_LISTENER_NUM) { return typeString[type]; }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationSetListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * ConfigurationSetListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for ConfigurationSetListener
     *
     * @endif
     */
    virtual void operator()(const coil::Properties& config_set) = 0;
  };


  //============================================================
  /*!
   * @if jp
   * @brief ConfigurationSetNameListenerType
   *
   *
   * @else
   * @brief The types of ConfigurationSetNameListener
   * 
   * @endif
   */
  enum ConfigurationSetNameListenerType
    {
      ON_UPDATE_CONFIG_SET,
      ON_REMOVE_CONFIG_SET,
      ON_ACTIVATE_CONFIG_SET,
      CONFIG_SET_NAME_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class ConfigurationSetNameListener 饹
   * @brief ConfigurationSetNameListener 饹
   *
   * ConfigurationSet˴ؤ륤٥Ȥ˴ؤꥹʡ饹
   *
   * - ON_UPDATE_CONFIG_SET:
   * - ON_REMOVE_CONFIG_SET:
   * - ON_ACTIVATE_CONFIG_SET:
   *
   * @else
   * @class ConfigurationSetNameListener class
   * @brief ConfigurationSetNameListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events for ConfigurationSet.
   *
   * - ON_UPDATE_CONFIG_SET:
   * - ON_REMOVE_CONFIG_SET:
   * - ON_ACTIVATE_CONFIG_SET:
   *
   * @endif
   */
  class ConfigurationSetNameListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief ConfigurationSetNameListenerType ʸѴ
     *
     * ConfigurationSetNameListenerType ʸѴ
     *
     * @param type Ѵо ConfigurationSetNameListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert ConfigurationSetNameListenerType into the string.
     *
     * Convert ConfigurationSetNameListenerType into the string.
     *
     * @param type The target ConfigurationSetNameListenerType for
     *             transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(ConfigurationSetNameListenerType type)
    {
      static const char* typeString[] =
        {
          "ON_UPDATE_CONFIG_SET",
          "ON_REMOVE_CONFIG_SET",
          "ON_ACTIVATE_CONFIG_SET",
          "CONFIG_SET_NAME_LISTENER_NUM"
        };
      if (type < CONFIG_SET_NAME_LISTENER_NUM) { return typeString[type]; }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationSetNameListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * ConfigurationSetNameListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for ConfigurationSetNameListener.
     *
     * @endif
     */
    virtual void operator()(const char* config_set_name) = 0;
  };


  /*!
   * @if jp
   * @class ConfigurationParamListenerHolder
   * @brief ConfigurationParamListener ۥ饹
   *
   * ʣ ConfigurationParamListener ݻ륯饹
   *
   * @else
   * @class ConfigurationParamListenerHolder
   * @brief ConfigurationParamListener holder class
   *
   * This class manages one ore more instances of
   * ConfigurationParamListener class.
   *
   * @endif
   */
  class ConfigurationParamListenerHolder
  {
    typedef std::pair<ConfigurationParamListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    ConfigurationParamListenerHolder();
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationParamListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(ConfigurationParamListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(ConfigurationParamListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(const char* config_set_name, const char* config_param_name);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  //============================================================
  /*!
   * @if jp
   * @class ConfigurationSetListenerHolder
   * @brief ConfigurationSetListener ۥ饹
   *
   * ʣ ConfigurationSetListener ݻ륯饹
   *
   * @else
   * @class ConfigurationSetListenerHolder
   * @brief ConfigurationSetListener holder class
   *
   * This class manages one ore more instances of
   * ConfigurationSetListener class.
   *
   * @endif
   */
  class ConfigurationSetListenerHolder
  {
    typedef std::pair<ConfigurationSetListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    ConfigurationSetListenerHolder();
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationSetListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(ConfigurationSetListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(ConfigurationSetListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(const coil::Properties& config_set);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  //============================================================
  /*!
   * @if jp
   * @class ConfigurationSetNameListenerHolder 
   * @brief ConfigurationSetNameListener ۥ饹
   *
   * ʣ ConfigurationSetNameListener ݻ륯饹
   *
   * @else
   * @class ConfigurationSetNameListenerHolder
   * @brief ConfigurationSetNameListener holder class
   *
   * This class manages one ore more instances of
   * ConfigurationSetNameListener class.
   *
   * @endif
   */
  class ConfigurationSetNameListenerHolder
  {
    typedef std::pair<ConfigurationSetNameListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    ConfigurationSetNameListenerHolder();
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ConfigurationSetNameListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(ConfigurationSetNameListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(ConfigurationSetNameListener* listener);

    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @endif
     */
    void notify(const char* config_set_name);
      
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };

  //------------------------------------------------------------
  /*!
   * @if jp
   * @class ConfigurationActionListeners
   * @brief ConfigurationActionListeners 饹
   *
   *
   * @else
   * @class ConfigurationActionListeners
   * @brief ConfigurationActionListeners class
   *
   *
   * @endif
   */
  class ConfigurationListeners
  {
  public:
    /*!
     * @if jp
     * @brief ConfigurationParamTypeꥹ
     * ConfigurationParamTypeꥹʤǼ
     * @else
     * @brief ConfigurationParamType listener array
     * The ConfigurationParamType listener is stored.
     * @endif
     */
    ConfigurationParamListenerHolder 
    configparam_[CONFIG_PARAM_LISTENER_NUM];
    /*!
     * @if jp
     * @brief ConfigurationSetTypeꥹ
     * ConfigurationSetTypeꥹʤǼ
     * @else
     * @brief ConfigurationSetType listener array
     * The ConfigurationSetType listener is stored.
     * @endif
     */
    ConfigurationSetListenerHolder
    configset_[CONFIG_SET_LISTENER_NUM];
    /*!
     * @if jp
     * @brief ConfigurationSetNameListenerTypeꥹ
     * ConfigurationSetNameListenerTypeꥹʤǼ
     * @else
     * @brief ConfigurationSetNameListenerType listener array
     * The ConfigurationSetNameListenerType listener is stored. 
     * @endif
     */
    ConfigurationSetNameListenerHolder 
    configsetname_[CONFIG_SET_NAME_LISTENER_NUM];
  };


}; // namespace RTC

#endif // RTC_CONFIGURATIONLISTENER_H
